/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An issue related to an add-on.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AddonIssue implements SdkPojo, Serializable, ToCopyableBuilder<AddonIssue.Builder, AddonIssue> {
    private static final SdkField<String> CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("code")
            .getter(getter(AddonIssue::codeAsString)).setter(setter(Builder::code))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("code").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("message")
            .getter(getter(AddonIssue::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("message").build()).build();

    private static final SdkField<List<String>> RESOURCE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("resourceIds")
            .getter(getter(AddonIssue::resourceIds))
            .setter(setter(Builder::resourceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CODE_FIELD, MESSAGE_FIELD,
            RESOURCE_IDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String code;

    private final String message;

    private final List<String> resourceIds;

    private AddonIssue(BuilderImpl builder) {
        this.code = builder.code;
        this.message = builder.message;
        this.resourceIds = builder.resourceIds;
    }

    /**
     * <p>
     * A code that describes the type of issue.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #code} will return
     * {@link AddonIssueCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #codeAsString}.
     * </p>
     * 
     * @return A code that describes the type of issue.
     * @see AddonIssueCode
     */
    public AddonIssueCode code() {
        return AddonIssueCode.fromValue(code);
    }

    /**
     * <p>
     * A code that describes the type of issue.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #code} will return
     * {@link AddonIssueCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #codeAsString}.
     * </p>
     * 
     * @return A code that describes the type of issue.
     * @see AddonIssueCode
     */
    public String codeAsString() {
        return code;
    }

    /**
     * <p>
     * A message that provides details about the issue and what might cause it.
     * </p>
     * 
     * @return A message that provides details about the issue and what might cause it.
     */
    public String message() {
        return message;
    }

    /**
     * Returns true if the ResourceIds property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasResourceIds() {
        return resourceIds != null && !(resourceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The resource IDs of the issue.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResourceIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The resource IDs of the issue.
     */
    public List<String> resourceIds() {
        return resourceIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(codeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceIds() ? resourceIds() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AddonIssue)) {
            return false;
        }
        AddonIssue other = (AddonIssue) obj;
        return Objects.equals(codeAsString(), other.codeAsString()) && Objects.equals(message(), other.message())
                && hasResourceIds() == other.hasResourceIds() && Objects.equals(resourceIds(), other.resourceIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AddonIssue").add("Code", codeAsString()).add("Message", message())
                .add("ResourceIds", hasResourceIds() ? resourceIds() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "code":
            return Optional.ofNullable(clazz.cast(codeAsString()));
        case "message":
            return Optional.ofNullable(clazz.cast(message()));
        case "resourceIds":
            return Optional.ofNullable(clazz.cast(resourceIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AddonIssue, T> g) {
        return obj -> g.apply((AddonIssue) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AddonIssue> {
        /**
         * <p>
         * A code that describes the type of issue.
         * </p>
         * 
         * @param code
         *        A code that describes the type of issue.
         * @see AddonIssueCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AddonIssueCode
         */
        Builder code(String code);

        /**
         * <p>
         * A code that describes the type of issue.
         * </p>
         * 
         * @param code
         *        A code that describes the type of issue.
         * @see AddonIssueCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AddonIssueCode
         */
        Builder code(AddonIssueCode code);

        /**
         * <p>
         * A message that provides details about the issue and what might cause it.
         * </p>
         * 
         * @param message
         *        A message that provides details about the issue and what might cause it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * The resource IDs of the issue.
         * </p>
         * 
         * @param resourceIds
         *        The resource IDs of the issue.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIds(Collection<String> resourceIds);

        /**
         * <p>
         * The resource IDs of the issue.
         * </p>
         * 
         * @param resourceIds
         *        The resource IDs of the issue.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIds(String... resourceIds);
    }

    static final class BuilderImpl implements Builder {
        private String code;

        private String message;

        private List<String> resourceIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AddonIssue model) {
            code(model.code);
            message(model.message);
            resourceIds(model.resourceIds);
        }

        public final String getCode() {
            return code;
        }

        @Override
        public final Builder code(String code) {
            this.code = code;
            return this;
        }

        @Override
        public final Builder code(AddonIssueCode code) {
            this.code(code == null ? null : code.toString());
            return this;
        }

        public final void setCode(String code) {
            this.code = code;
        }

        public final String getMessage() {
            return message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        public final Collection<String> getResourceIds() {
            if (resourceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceIds;
        }

        @Override
        public final Builder resourceIds(Collection<String> resourceIds) {
            this.resourceIds = StringListCopier.copy(resourceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceIds(String... resourceIds) {
            resourceIds(Arrays.asList(resourceIds));
            return this;
        }

        public final void setResourceIds(Collection<String> resourceIds) {
            this.resourceIds = StringListCopier.copy(resourceIds);
        }

        @Override
        public AddonIssue build() {
            return new AddonIssue(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
