/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing an issue with an Amazon EKS resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Issue implements SdkPojo, Serializable, ToCopyableBuilder<Issue.Builder, Issue> {
    private static final SdkField<String> CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("code")
            .getter(getter(Issue::codeAsString)).setter(setter(Builder::code))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("code").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("message")
            .getter(getter(Issue::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("message").build()).build();

    private static final SdkField<List<String>> RESOURCE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("resourceIds")
            .getter(getter(Issue::resourceIds))
            .setter(setter(Builder::resourceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CODE_FIELD, MESSAGE_FIELD,
            RESOURCE_IDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String code;

    private final String message;

    private final List<String> resourceIds;

    private Issue(BuilderImpl builder) {
        this.code = builder.code;
        this.message = builder.message;
        this.resourceIds = builder.resourceIds;
    }

    /**
     * <p>
     * A brief description of the error.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>AccessDenied</b>: Amazon EKS or one or more of your managed nodes is failing to authenticate or authorize with
     * your Kubernetes cluster API server.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>AsgInstanceLaunchFailures</b>: Your Auto Scaling group is experiencing failures while attempting to launch
     * instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>AutoScalingGroupNotFound</b>: We couldn't find the Auto Scaling group associated with the managed node group.
     * You may be able to recreate an Auto Scaling group with the same settings to recover.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ClusterUnreachable</b>: Amazon EKS or one or more of your managed nodes is unable to to communicate with your
     * Kubernetes cluster API server. This can happen if there are network disruptions or if API servers are timing out
     * processing requests.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Ec2LaunchTemplateNotFound</b>: We couldn't find the Amazon EC2 launch template for your managed node group.
     * You may be able to recreate a launch template with the same settings to recover.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Ec2LaunchTemplateVersionMismatch</b>: The Amazon EC2 launch template version for your managed node group does
     * not match the version that Amazon EKS created. You may be able to revert to the version that Amazon EKS created
     * to recover.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Ec2SecurityGroupDeletionFailure</b>: We could not delete the remote access security group for your managed
     * node group. Remove any dependencies from the security group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Ec2SecurityGroupNotFound</b>: We couldn't find the cluster security group for the cluster. You must recreate
     * your cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Ec2SubnetInvalidConfiguration</b>: One or more Amazon EC2 subnets specified for a node group do not
     * automatically assign public IP addresses to instances launched into it. If you want your instances to be assigned
     * a public IP address, then you need to enable the <code>auto-assign public IP address</code> setting for the
     * subnet. See <a
     * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-ip-addressing.html#subnet-public-ip">Modifying the
     * public IPv4 addressing attribute for your subnet</a> in the Amazon VPC User Guide.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>IamInstanceProfileNotFound</b>: We couldn't find the IAM instance profile for your managed node group. You may
     * be able to recreate an instance profile with the same settings to recover.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>IamNodeRoleNotFound</b>: We couldn't find the IAM role for your managed node group. You may be able to
     * recreate an IAM role with the same settings to recover.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>InstanceLimitExceeded</b>: Your Amazon Web Services account is unable to launch any more instances of the
     * specified instance type. You may be able to request an Amazon EC2 instance limit increase to recover.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>InsufficientFreeAddresses</b>: One or more of the subnets associated with your managed node group does not
     * have enough available IP addresses for new nodes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>InternalFailure</b>: These errors are usually caused by an Amazon EKS server-side issue.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>NodeCreationFailure</b>: Your launched instances are unable to register with your Amazon EKS cluster. Common
     * causes of this failure are insufficient <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/worker_node_IAM_role.html">node IAM role</a> permissions
     * or lack of outbound internet access for the nodes.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #code} will return
     * {@link NodegroupIssueCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #codeAsString}.
     * </p>
     * 
     * @return A brief description of the error.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>AccessDenied</b>: Amazon EKS or one or more of your managed nodes is failing to authenticate or
     *         authorize with your Kubernetes cluster API server.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>AsgInstanceLaunchFailures</b>: Your Auto Scaling group is experiencing failures while attempting to
     *         launch instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>AutoScalingGroupNotFound</b>: We couldn't find the Auto Scaling group associated with the managed node
     *         group. You may be able to recreate an Auto Scaling group with the same settings to recover.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ClusterUnreachable</b>: Amazon EKS or one or more of your managed nodes is unable to to communicate
     *         with your Kubernetes cluster API server. This can happen if there are network disruptions or if API
     *         servers are timing out processing requests.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Ec2LaunchTemplateNotFound</b>: We couldn't find the Amazon EC2 launch template for your managed node
     *         group. You may be able to recreate a launch template with the same settings to recover.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Ec2LaunchTemplateVersionMismatch</b>: The Amazon EC2 launch template version for your managed node
     *         group does not match the version that Amazon EKS created. You may be able to revert to the version that
     *         Amazon EKS created to recover.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Ec2SecurityGroupDeletionFailure</b>: We could not delete the remote access security group for your
     *         managed node group. Remove any dependencies from the security group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Ec2SecurityGroupNotFound</b>: We couldn't find the cluster security group for the cluster. You must
     *         recreate your cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Ec2SubnetInvalidConfiguration</b>: One or more Amazon EC2 subnets specified for a node group do not
     *         automatically assign public IP addresses to instances launched into it. If you want your instances to be
     *         assigned a public IP address, then you need to enable the <code>auto-assign public IP address</code>
     *         setting for the subnet. See <a
     *         href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-ip-addressing.html#subnet-public-ip">Modifying
     *         the public IPv4 addressing attribute for your subnet</a> in the Amazon VPC User Guide.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IamInstanceProfileNotFound</b>: We couldn't find the IAM instance profile for your managed node group.
     *         You may be able to recreate an instance profile with the same settings to recover.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IamNodeRoleNotFound</b>: We couldn't find the IAM role for your managed node group. You may be able to
     *         recreate an IAM role with the same settings to recover.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InstanceLimitExceeded</b>: Your Amazon Web Services account is unable to launch any more instances of
     *         the specified instance type. You may be able to request an Amazon EC2 instance limit increase to recover.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InsufficientFreeAddresses</b>: One or more of the subnets associated with your managed node group does
     *         not have enough available IP addresses for new nodes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InternalFailure</b>: These errors are usually caused by an Amazon EKS server-side issue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>NodeCreationFailure</b>: Your launched instances are unable to register with your Amazon EKS cluster.
     *         Common causes of this failure are insufficient <a
     *         href="https://docs.aws.amazon.com/eks/latest/userguide/worker_node_IAM_role.html">node IAM role</a>
     *         permissions or lack of outbound internet access for the nodes.
     *         </p>
     *         </li>
     * @see NodegroupIssueCode
     */
    public final NodegroupIssueCode code() {
        return NodegroupIssueCode.fromValue(code);
    }

    /**
     * <p>
     * A brief description of the error.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>AccessDenied</b>: Amazon EKS or one or more of your managed nodes is failing to authenticate or authorize with
     * your Kubernetes cluster API server.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>AsgInstanceLaunchFailures</b>: Your Auto Scaling group is experiencing failures while attempting to launch
     * instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>AutoScalingGroupNotFound</b>: We couldn't find the Auto Scaling group associated with the managed node group.
     * You may be able to recreate an Auto Scaling group with the same settings to recover.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ClusterUnreachable</b>: Amazon EKS or one or more of your managed nodes is unable to to communicate with your
     * Kubernetes cluster API server. This can happen if there are network disruptions or if API servers are timing out
     * processing requests.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Ec2LaunchTemplateNotFound</b>: We couldn't find the Amazon EC2 launch template for your managed node group.
     * You may be able to recreate a launch template with the same settings to recover.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Ec2LaunchTemplateVersionMismatch</b>: The Amazon EC2 launch template version for your managed node group does
     * not match the version that Amazon EKS created. You may be able to revert to the version that Amazon EKS created
     * to recover.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Ec2SecurityGroupDeletionFailure</b>: We could not delete the remote access security group for your managed
     * node group. Remove any dependencies from the security group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Ec2SecurityGroupNotFound</b>: We couldn't find the cluster security group for the cluster. You must recreate
     * your cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Ec2SubnetInvalidConfiguration</b>: One or more Amazon EC2 subnets specified for a node group do not
     * automatically assign public IP addresses to instances launched into it. If you want your instances to be assigned
     * a public IP address, then you need to enable the <code>auto-assign public IP address</code> setting for the
     * subnet. See <a
     * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-ip-addressing.html#subnet-public-ip">Modifying the
     * public IPv4 addressing attribute for your subnet</a> in the Amazon VPC User Guide.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>IamInstanceProfileNotFound</b>: We couldn't find the IAM instance profile for your managed node group. You may
     * be able to recreate an instance profile with the same settings to recover.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>IamNodeRoleNotFound</b>: We couldn't find the IAM role for your managed node group. You may be able to
     * recreate an IAM role with the same settings to recover.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>InstanceLimitExceeded</b>: Your Amazon Web Services account is unable to launch any more instances of the
     * specified instance type. You may be able to request an Amazon EC2 instance limit increase to recover.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>InsufficientFreeAddresses</b>: One or more of the subnets associated with your managed node group does not
     * have enough available IP addresses for new nodes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>InternalFailure</b>: These errors are usually caused by an Amazon EKS server-side issue.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>NodeCreationFailure</b>: Your launched instances are unable to register with your Amazon EKS cluster. Common
     * causes of this failure are insufficient <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/worker_node_IAM_role.html">node IAM role</a> permissions
     * or lack of outbound internet access for the nodes.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #code} will return
     * {@link NodegroupIssueCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #codeAsString}.
     * </p>
     * 
     * @return A brief description of the error.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>AccessDenied</b>: Amazon EKS or one or more of your managed nodes is failing to authenticate or
     *         authorize with your Kubernetes cluster API server.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>AsgInstanceLaunchFailures</b>: Your Auto Scaling group is experiencing failures while attempting to
     *         launch instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>AutoScalingGroupNotFound</b>: We couldn't find the Auto Scaling group associated with the managed node
     *         group. You may be able to recreate an Auto Scaling group with the same settings to recover.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ClusterUnreachable</b>: Amazon EKS or one or more of your managed nodes is unable to to communicate
     *         with your Kubernetes cluster API server. This can happen if there are network disruptions or if API
     *         servers are timing out processing requests.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Ec2LaunchTemplateNotFound</b>: We couldn't find the Amazon EC2 launch template for your managed node
     *         group. You may be able to recreate a launch template with the same settings to recover.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Ec2LaunchTemplateVersionMismatch</b>: The Amazon EC2 launch template version for your managed node
     *         group does not match the version that Amazon EKS created. You may be able to revert to the version that
     *         Amazon EKS created to recover.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Ec2SecurityGroupDeletionFailure</b>: We could not delete the remote access security group for your
     *         managed node group. Remove any dependencies from the security group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Ec2SecurityGroupNotFound</b>: We couldn't find the cluster security group for the cluster. You must
     *         recreate your cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Ec2SubnetInvalidConfiguration</b>: One or more Amazon EC2 subnets specified for a node group do not
     *         automatically assign public IP addresses to instances launched into it. If you want your instances to be
     *         assigned a public IP address, then you need to enable the <code>auto-assign public IP address</code>
     *         setting for the subnet. See <a
     *         href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-ip-addressing.html#subnet-public-ip">Modifying
     *         the public IPv4 addressing attribute for your subnet</a> in the Amazon VPC User Guide.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IamInstanceProfileNotFound</b>: We couldn't find the IAM instance profile for your managed node group.
     *         You may be able to recreate an instance profile with the same settings to recover.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IamNodeRoleNotFound</b>: We couldn't find the IAM role for your managed node group. You may be able to
     *         recreate an IAM role with the same settings to recover.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InstanceLimitExceeded</b>: Your Amazon Web Services account is unable to launch any more instances of
     *         the specified instance type. You may be able to request an Amazon EC2 instance limit increase to recover.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InsufficientFreeAddresses</b>: One or more of the subnets associated with your managed node group does
     *         not have enough available IP addresses for new nodes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>InternalFailure</b>: These errors are usually caused by an Amazon EKS server-side issue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>NodeCreationFailure</b>: Your launched instances are unable to register with your Amazon EKS cluster.
     *         Common causes of this failure are insufficient <a
     *         href="https://docs.aws.amazon.com/eks/latest/userguide/worker_node_IAM_role.html">node IAM role</a>
     *         permissions or lack of outbound internet access for the nodes.
     *         </p>
     *         </li>
     * @see NodegroupIssueCode
     */
    public final String codeAsString() {
        return code;
    }

    /**
     * <p>
     * The error message associated with the issue.
     * </p>
     * 
     * @return The error message associated with the issue.
     */
    public final String message() {
        return message;
    }

    /**
     * Returns true if the ResourceIds property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasResourceIds() {
        return resourceIds != null && !(resourceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Web Services resources that are afflicted by this issue.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResourceIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Amazon Web Services resources that are afflicted by this issue.
     */
    public final List<String> resourceIds() {
        return resourceIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(codeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceIds() ? resourceIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Issue)) {
            return false;
        }
        Issue other = (Issue) obj;
        return Objects.equals(codeAsString(), other.codeAsString()) && Objects.equals(message(), other.message())
                && hasResourceIds() == other.hasResourceIds() && Objects.equals(resourceIds(), other.resourceIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Issue").add("Code", codeAsString()).add("Message", message())
                .add("ResourceIds", hasResourceIds() ? resourceIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "code":
            return Optional.ofNullable(clazz.cast(codeAsString()));
        case "message":
            return Optional.ofNullable(clazz.cast(message()));
        case "resourceIds":
            return Optional.ofNullable(clazz.cast(resourceIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Issue, T> g) {
        return obj -> g.apply((Issue) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Issue> {
        /**
         * <p>
         * A brief description of the error.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>AccessDenied</b>: Amazon EKS or one or more of your managed nodes is failing to authenticate or authorize
         * with your Kubernetes cluster API server.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>AsgInstanceLaunchFailures</b>: Your Auto Scaling group is experiencing failures while attempting to launch
         * instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>AutoScalingGroupNotFound</b>: We couldn't find the Auto Scaling group associated with the managed node
         * group. You may be able to recreate an Auto Scaling group with the same settings to recover.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ClusterUnreachable</b>: Amazon EKS or one or more of your managed nodes is unable to to communicate with
         * your Kubernetes cluster API server. This can happen if there are network disruptions or if API servers are
         * timing out processing requests.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Ec2LaunchTemplateNotFound</b>: We couldn't find the Amazon EC2 launch template for your managed node
         * group. You may be able to recreate a launch template with the same settings to recover.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Ec2LaunchTemplateVersionMismatch</b>: The Amazon EC2 launch template version for your managed node group
         * does not match the version that Amazon EKS created. You may be able to revert to the version that Amazon EKS
         * created to recover.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Ec2SecurityGroupDeletionFailure</b>: We could not delete the remote access security group for your managed
         * node group. Remove any dependencies from the security group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Ec2SecurityGroupNotFound</b>: We couldn't find the cluster security group for the cluster. You must
         * recreate your cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Ec2SubnetInvalidConfiguration</b>: One or more Amazon EC2 subnets specified for a node group do not
         * automatically assign public IP addresses to instances launched into it. If you want your instances to be
         * assigned a public IP address, then you need to enable the <code>auto-assign public IP address</code> setting
         * for the subnet. See <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-ip-addressing.html#subnet-public-ip">Modifying the
         * public IPv4 addressing attribute for your subnet</a> in the Amazon VPC User Guide.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>IamInstanceProfileNotFound</b>: We couldn't find the IAM instance profile for your managed node group. You
         * may be able to recreate an instance profile with the same settings to recover.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>IamNodeRoleNotFound</b>: We couldn't find the IAM role for your managed node group. You may be able to
         * recreate an IAM role with the same settings to recover.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>InstanceLimitExceeded</b>: Your Amazon Web Services account is unable to launch any more instances of the
         * specified instance type. You may be able to request an Amazon EC2 instance limit increase to recover.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>InsufficientFreeAddresses</b>: One or more of the subnets associated with your managed node group does not
         * have enough available IP addresses for new nodes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>InternalFailure</b>: These errors are usually caused by an Amazon EKS server-side issue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>NodeCreationFailure</b>: Your launched instances are unable to register with your Amazon EKS cluster.
         * Common causes of this failure are insufficient <a
         * href="https://docs.aws.amazon.com/eks/latest/userguide/worker_node_IAM_role.html">node IAM role</a>
         * permissions or lack of outbound internet access for the nodes.
         * </p>
         * </li>
         * </ul>
         * 
         * @param code
         *        A brief description of the error.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>AccessDenied</b>: Amazon EKS or one or more of your managed nodes is failing to authenticate or
         *        authorize with your Kubernetes cluster API server.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>AsgInstanceLaunchFailures</b>: Your Auto Scaling group is experiencing failures while attempting to
         *        launch instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>AutoScalingGroupNotFound</b>: We couldn't find the Auto Scaling group associated with the managed
         *        node group. You may be able to recreate an Auto Scaling group with the same settings to recover.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ClusterUnreachable</b>: Amazon EKS or one or more of your managed nodes is unable to to communicate
         *        with your Kubernetes cluster API server. This can happen if there are network disruptions or if API
         *        servers are timing out processing requests.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Ec2LaunchTemplateNotFound</b>: We couldn't find the Amazon EC2 launch template for your managed
         *        node group. You may be able to recreate a launch template with the same settings to recover.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Ec2LaunchTemplateVersionMismatch</b>: The Amazon EC2 launch template version for your managed node
         *        group does not match the version that Amazon EKS created. You may be able to revert to the version
         *        that Amazon EKS created to recover.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Ec2SecurityGroupDeletionFailure</b>: We could not delete the remote access security group for your
         *        managed node group. Remove any dependencies from the security group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Ec2SecurityGroupNotFound</b>: We couldn't find the cluster security group for the cluster. You must
         *        recreate your cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Ec2SubnetInvalidConfiguration</b>: One or more Amazon EC2 subnets specified for a node group do not
         *        automatically assign public IP addresses to instances launched into it. If you want your instances to
         *        be assigned a public IP address, then you need to enable the
         *        <code>auto-assign public IP address</code> setting for the subnet. See <a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-ip-addressing.html#subnet-public-ip"
         *        >Modifying the public IPv4 addressing attribute for your subnet</a> in the Amazon VPC User Guide.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>IamInstanceProfileNotFound</b>: We couldn't find the IAM instance profile for your managed node
         *        group. You may be able to recreate an instance profile with the same settings to recover.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>IamNodeRoleNotFound</b>: We couldn't find the IAM role for your managed node group. You may be able
         *        to recreate an IAM role with the same settings to recover.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>InstanceLimitExceeded</b>: Your Amazon Web Services account is unable to launch any more instances
         *        of the specified instance type. You may be able to request an Amazon EC2 instance limit increase to
         *        recover.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>InsufficientFreeAddresses</b>: One or more of the subnets associated with your managed node group
         *        does not have enough available IP addresses for new nodes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>InternalFailure</b>: These errors are usually caused by an Amazon EKS server-side issue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>NodeCreationFailure</b>: Your launched instances are unable to register with your Amazon EKS
         *        cluster. Common causes of this failure are insufficient <a
         *        href="https://docs.aws.amazon.com/eks/latest/userguide/worker_node_IAM_role.html">node IAM role</a>
         *        permissions or lack of outbound internet access for the nodes.
         *        </p>
         *        </li>
         * @see NodegroupIssueCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NodegroupIssueCode
         */
        Builder code(String code);

        /**
         * <p>
         * A brief description of the error.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>AccessDenied</b>: Amazon EKS or one or more of your managed nodes is failing to authenticate or authorize
         * with your Kubernetes cluster API server.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>AsgInstanceLaunchFailures</b>: Your Auto Scaling group is experiencing failures while attempting to launch
         * instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>AutoScalingGroupNotFound</b>: We couldn't find the Auto Scaling group associated with the managed node
         * group. You may be able to recreate an Auto Scaling group with the same settings to recover.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ClusterUnreachable</b>: Amazon EKS or one or more of your managed nodes is unable to to communicate with
         * your Kubernetes cluster API server. This can happen if there are network disruptions or if API servers are
         * timing out processing requests.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Ec2LaunchTemplateNotFound</b>: We couldn't find the Amazon EC2 launch template for your managed node
         * group. You may be able to recreate a launch template with the same settings to recover.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Ec2LaunchTemplateVersionMismatch</b>: The Amazon EC2 launch template version for your managed node group
         * does not match the version that Amazon EKS created. You may be able to revert to the version that Amazon EKS
         * created to recover.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Ec2SecurityGroupDeletionFailure</b>: We could not delete the remote access security group for your managed
         * node group. Remove any dependencies from the security group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Ec2SecurityGroupNotFound</b>: We couldn't find the cluster security group for the cluster. You must
         * recreate your cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Ec2SubnetInvalidConfiguration</b>: One or more Amazon EC2 subnets specified for a node group do not
         * automatically assign public IP addresses to instances launched into it. If you want your instances to be
         * assigned a public IP address, then you need to enable the <code>auto-assign public IP address</code> setting
         * for the subnet. See <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-ip-addressing.html#subnet-public-ip">Modifying the
         * public IPv4 addressing attribute for your subnet</a> in the Amazon VPC User Guide.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>IamInstanceProfileNotFound</b>: We couldn't find the IAM instance profile for your managed node group. You
         * may be able to recreate an instance profile with the same settings to recover.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>IamNodeRoleNotFound</b>: We couldn't find the IAM role for your managed node group. You may be able to
         * recreate an IAM role with the same settings to recover.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>InstanceLimitExceeded</b>: Your Amazon Web Services account is unable to launch any more instances of the
         * specified instance type. You may be able to request an Amazon EC2 instance limit increase to recover.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>InsufficientFreeAddresses</b>: One or more of the subnets associated with your managed node group does not
         * have enough available IP addresses for new nodes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>InternalFailure</b>: These errors are usually caused by an Amazon EKS server-side issue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>NodeCreationFailure</b>: Your launched instances are unable to register with your Amazon EKS cluster.
         * Common causes of this failure are insufficient <a
         * href="https://docs.aws.amazon.com/eks/latest/userguide/worker_node_IAM_role.html">node IAM role</a>
         * permissions or lack of outbound internet access for the nodes.
         * </p>
         * </li>
         * </ul>
         * 
         * @param code
         *        A brief description of the error.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>AccessDenied</b>: Amazon EKS or one or more of your managed nodes is failing to authenticate or
         *        authorize with your Kubernetes cluster API server.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>AsgInstanceLaunchFailures</b>: Your Auto Scaling group is experiencing failures while attempting to
         *        launch instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>AutoScalingGroupNotFound</b>: We couldn't find the Auto Scaling group associated with the managed
         *        node group. You may be able to recreate an Auto Scaling group with the same settings to recover.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ClusterUnreachable</b>: Amazon EKS or one or more of your managed nodes is unable to to communicate
         *        with your Kubernetes cluster API server. This can happen if there are network disruptions or if API
         *        servers are timing out processing requests.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Ec2LaunchTemplateNotFound</b>: We couldn't find the Amazon EC2 launch template for your managed
         *        node group. You may be able to recreate a launch template with the same settings to recover.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Ec2LaunchTemplateVersionMismatch</b>: The Amazon EC2 launch template version for your managed node
         *        group does not match the version that Amazon EKS created. You may be able to revert to the version
         *        that Amazon EKS created to recover.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Ec2SecurityGroupDeletionFailure</b>: We could not delete the remote access security group for your
         *        managed node group. Remove any dependencies from the security group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Ec2SecurityGroupNotFound</b>: We couldn't find the cluster security group for the cluster. You must
         *        recreate your cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Ec2SubnetInvalidConfiguration</b>: One or more Amazon EC2 subnets specified for a node group do not
         *        automatically assign public IP addresses to instances launched into it. If you want your instances to
         *        be assigned a public IP address, then you need to enable the
         *        <code>auto-assign public IP address</code> setting for the subnet. See <a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-ip-addressing.html#subnet-public-ip"
         *        >Modifying the public IPv4 addressing attribute for your subnet</a> in the Amazon VPC User Guide.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>IamInstanceProfileNotFound</b>: We couldn't find the IAM instance profile for your managed node
         *        group. You may be able to recreate an instance profile with the same settings to recover.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>IamNodeRoleNotFound</b>: We couldn't find the IAM role for your managed node group. You may be able
         *        to recreate an IAM role with the same settings to recover.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>InstanceLimitExceeded</b>: Your Amazon Web Services account is unable to launch any more instances
         *        of the specified instance type. You may be able to request an Amazon EC2 instance limit increase to
         *        recover.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>InsufficientFreeAddresses</b>: One or more of the subnets associated with your managed node group
         *        does not have enough available IP addresses for new nodes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>InternalFailure</b>: These errors are usually caused by an Amazon EKS server-side issue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>NodeCreationFailure</b>: Your launched instances are unable to register with your Amazon EKS
         *        cluster. Common causes of this failure are insufficient <a
         *        href="https://docs.aws.amazon.com/eks/latest/userguide/worker_node_IAM_role.html">node IAM role</a>
         *        permissions or lack of outbound internet access for the nodes.
         *        </p>
         *        </li>
         * @see NodegroupIssueCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NodegroupIssueCode
         */
        Builder code(NodegroupIssueCode code);

        /**
         * <p>
         * The error message associated with the issue.
         * </p>
         * 
         * @param message
         *        The error message associated with the issue.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * The Amazon Web Services resources that are afflicted by this issue.
         * </p>
         * 
         * @param resourceIds
         *        The Amazon Web Services resources that are afflicted by this issue.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIds(Collection<String> resourceIds);

        /**
         * <p>
         * The Amazon Web Services resources that are afflicted by this issue.
         * </p>
         * 
         * @param resourceIds
         *        The Amazon Web Services resources that are afflicted by this issue.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIds(String... resourceIds);
    }

    static final class BuilderImpl implements Builder {
        private String code;

        private String message;

        private List<String> resourceIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Issue model) {
            code(model.code);
            message(model.message);
            resourceIds(model.resourceIds);
        }

        public final String getCode() {
            return code;
        }

        @Override
        public final Builder code(String code) {
            this.code = code;
            return this;
        }

        @Override
        public final Builder code(NodegroupIssueCode code) {
            this.code(code == null ? null : code.toString());
            return this;
        }

        public final void setCode(String code) {
            this.code = code;
        }

        public final String getMessage() {
            return message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        public final Collection<String> getResourceIds() {
            if (resourceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceIds;
        }

        @Override
        public final Builder resourceIds(Collection<String> resourceIds) {
            this.resourceIds = StringListCopier.copy(resourceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceIds(String... resourceIds) {
            resourceIds(Arrays.asList(resourceIds));
            return this;
        }

        public final void setResourceIds(Collection<String> resourceIds) {
            this.resourceIds = StringListCopier.copy(resourceIds);
        }

        @Override
        public Issue build() {
            return new Issue(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
