/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing an Amazon EKS cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Cluster implements SdkPojo, Serializable, ToCopyableBuilder<Cluster.Builder, Cluster> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Cluster::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Cluster::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(Cluster::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(Cluster::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<String> ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("endpoint").getter(getter(Cluster::endpoint)).setter(setter(Builder::endpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endpoint").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(Cluster::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<VpcConfigResponse> RESOURCES_VPC_CONFIG_FIELD = SdkField
            .<VpcConfigResponse> builder(MarshallingType.SDK_POJO).memberName("resourcesVpcConfig")
            .getter(getter(Cluster::resourcesVpcConfig)).setter(setter(Builder::resourcesVpcConfig))
            .constructor(VpcConfigResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourcesVpcConfig").build())
            .build();

    private static final SdkField<KubernetesNetworkConfigResponse> KUBERNETES_NETWORK_CONFIG_FIELD = SdkField
            .<KubernetesNetworkConfigResponse> builder(MarshallingType.SDK_POJO).memberName("kubernetesNetworkConfig")
            .getter(getter(Cluster::kubernetesNetworkConfig)).setter(setter(Builder::kubernetesNetworkConfig))
            .constructor(KubernetesNetworkConfigResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kubernetesNetworkConfig").build())
            .build();

    private static final SdkField<Logging> LOGGING_FIELD = SdkField.<Logging> builder(MarshallingType.SDK_POJO)
            .memberName("logging").getter(getter(Cluster::logging)).setter(setter(Builder::logging))
            .constructor(Logging::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logging").build()).build();

    private static final SdkField<Identity> IDENTITY_FIELD = SdkField.<Identity> builder(MarshallingType.SDK_POJO)
            .memberName("identity").getter(getter(Cluster::identity)).setter(setter(Builder::identity))
            .constructor(Identity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("identity").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(Cluster::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Certificate> CERTIFICATE_AUTHORITY_FIELD = SdkField
            .<Certificate> builder(MarshallingType.SDK_POJO).memberName("certificateAuthority")
            .getter(getter(Cluster::certificateAuthority)).setter(setter(Builder::certificateAuthority))
            .constructor(Certificate::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateAuthority").build())
            .build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clientRequestToken").getter(getter(Cluster::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientRequestToken").build())
            .build();

    private static final SdkField<String> PLATFORM_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("platformVersion").getter(getter(Cluster::platformVersion)).setter(setter(Builder::platformVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platformVersion").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Cluster::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<EncryptionConfig>> ENCRYPTION_CONFIG_FIELD = SdkField
            .<List<EncryptionConfig>> builder(MarshallingType.LIST)
            .memberName("encryptionConfig")
            .getter(getter(Cluster::encryptionConfig))
            .setter(setter(Builder::encryptionConfig))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionConfig").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EncryptionConfig> builder(MarshallingType.SDK_POJO)
                                            .constructor(EncryptionConfig::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ConnectorConfigResponse> CONNECTOR_CONFIG_FIELD = SdkField
            .<ConnectorConfigResponse> builder(MarshallingType.SDK_POJO).memberName("connectorConfig")
            .getter(getter(Cluster::connectorConfig)).setter(setter(Builder::connectorConfig))
            .constructor(ConnectorConfigResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("connectorConfig").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(Cluster::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<ClusterHealth> HEALTH_FIELD = SdkField.<ClusterHealth> builder(MarshallingType.SDK_POJO)
            .memberName("health").getter(getter(Cluster::health)).setter(setter(Builder::health))
            .constructor(ClusterHealth::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("health").build()).build();

    private static final SdkField<OutpostConfigResponse> OUTPOST_CONFIG_FIELD = SdkField
            .<OutpostConfigResponse> builder(MarshallingType.SDK_POJO).memberName("outpostConfig")
            .getter(getter(Cluster::outpostConfig)).setter(setter(Builder::outpostConfig))
            .constructor(OutpostConfigResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outpostConfig").build()).build();

    private static final SdkField<AccessConfigResponse> ACCESS_CONFIG_FIELD = SdkField
            .<AccessConfigResponse> builder(MarshallingType.SDK_POJO).memberName("accessConfig")
            .getter(getter(Cluster::accessConfig)).setter(setter(Builder::accessConfig))
            .constructor(AccessConfigResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accessConfig").build()).build();

    private static final SdkField<UpgradePolicyResponse> UPGRADE_POLICY_FIELD = SdkField
            .<UpgradePolicyResponse> builder(MarshallingType.SDK_POJO).memberName("upgradePolicy")
            .getter(getter(Cluster::upgradePolicy)).setter(setter(Builder::upgradePolicy))
            .constructor(UpgradePolicyResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("upgradePolicy").build()).build();

    private static final SdkField<ZonalShiftConfigResponse> ZONAL_SHIFT_CONFIG_FIELD = SdkField
            .<ZonalShiftConfigResponse> builder(MarshallingType.SDK_POJO).memberName("zonalShiftConfig")
            .getter(getter(Cluster::zonalShiftConfig)).setter(setter(Builder::zonalShiftConfig))
            .constructor(ZonalShiftConfigResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("zonalShiftConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            CREATED_AT_FIELD, VERSION_FIELD, ENDPOINT_FIELD, ROLE_ARN_FIELD, RESOURCES_VPC_CONFIG_FIELD,
            KUBERNETES_NETWORK_CONFIG_FIELD, LOGGING_FIELD, IDENTITY_FIELD, STATUS_FIELD, CERTIFICATE_AUTHORITY_FIELD,
            CLIENT_REQUEST_TOKEN_FIELD, PLATFORM_VERSION_FIELD, TAGS_FIELD, ENCRYPTION_CONFIG_FIELD, CONNECTOR_CONFIG_FIELD,
            ID_FIELD, HEALTH_FIELD, OUTPOST_CONFIG_FIELD, ACCESS_CONFIG_FIELD, UPGRADE_POLICY_FIELD, ZONAL_SHIFT_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final Instant createdAt;

    private final String version;

    private final String endpoint;

    private final String roleArn;

    private final VpcConfigResponse resourcesVpcConfig;

    private final KubernetesNetworkConfigResponse kubernetesNetworkConfig;

    private final Logging logging;

    private final Identity identity;

    private final String status;

    private final Certificate certificateAuthority;

    private final String clientRequestToken;

    private final String platformVersion;

    private final Map<String, String> tags;

    private final List<EncryptionConfig> encryptionConfig;

    private final ConnectorConfigResponse connectorConfig;

    private final String id;

    private final ClusterHealth health;

    private final OutpostConfigResponse outpostConfig;

    private final AccessConfigResponse accessConfig;

    private final UpgradePolicyResponse upgradePolicy;

    private final ZonalShiftConfigResponse zonalShiftConfig;

    private Cluster(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.createdAt = builder.createdAt;
        this.version = builder.version;
        this.endpoint = builder.endpoint;
        this.roleArn = builder.roleArn;
        this.resourcesVpcConfig = builder.resourcesVpcConfig;
        this.kubernetesNetworkConfig = builder.kubernetesNetworkConfig;
        this.logging = builder.logging;
        this.identity = builder.identity;
        this.status = builder.status;
        this.certificateAuthority = builder.certificateAuthority;
        this.clientRequestToken = builder.clientRequestToken;
        this.platformVersion = builder.platformVersion;
        this.tags = builder.tags;
        this.encryptionConfig = builder.encryptionConfig;
        this.connectorConfig = builder.connectorConfig;
        this.id = builder.id;
        this.health = builder.health;
        this.outpostConfig = builder.outpostConfig;
        this.accessConfig = builder.accessConfig;
        this.upgradePolicy = builder.upgradePolicy;
        this.zonalShiftConfig = builder.zonalShiftConfig;
    }

    /**
     * <p>
     * The name of your cluster.
     * </p>
     * 
     * @return The name of your cluster.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the cluster.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the cluster.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The Unix epoch timestamp at object creation.
     * </p>
     * 
     * @return The Unix epoch timestamp at object creation.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The Kubernetes server version for the cluster.
     * </p>
     * 
     * @return The Kubernetes server version for the cluster.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The endpoint for your Kubernetes API server.
     * </p>
     * 
     * @return The endpoint for your Kubernetes API server.
     */
    public final String endpoint() {
        return endpoint;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role that provides permissions for the Kubernetes control plane to make
     * calls to Amazon Web Services API operations on your behalf.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role that provides permissions for the Kubernetes control plane
     *         to make calls to Amazon Web Services API operations on your behalf.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The VPC configuration used by the cluster control plane. Amazon EKS VPC resources have specific requirements to
     * work properly with Kubernetes. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html">Cluster VPC considerations</a> and <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html">Cluster security group
     * considerations</a> in the <i>Amazon EKS User Guide</i>.
     * </p>
     * 
     * @return The VPC configuration used by the cluster control plane. Amazon EKS VPC resources have specific
     *         requirements to work properly with Kubernetes. For more information, see <a
     *         href="https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html">Cluster VPC considerations</a>
     *         and <a href="https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html">Cluster security group
     *         considerations</a> in the <i>Amazon EKS User Guide</i>.
     */
    public final VpcConfigResponse resourcesVpcConfig() {
        return resourcesVpcConfig;
    }

    /**
     * <p>
     * The Kubernetes network configuration for the cluster.
     * </p>
     * 
     * @return The Kubernetes network configuration for the cluster.
     */
    public final KubernetesNetworkConfigResponse kubernetesNetworkConfig() {
        return kubernetesNetworkConfig;
    }

    /**
     * <p>
     * The logging configuration for your cluster.
     * </p>
     * 
     * @return The logging configuration for your cluster.
     */
    public final Logging logging() {
        return logging;
    }

    /**
     * <p>
     * The identity provider information for the cluster.
     * </p>
     * 
     * @return The identity provider information for the cluster.
     */
    public final Identity identity() {
        return identity;
    }

    /**
     * <p>
     * The current status of the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ClusterStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the cluster.
     * @see ClusterStatus
     */
    public final ClusterStatus status() {
        return ClusterStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ClusterStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the cluster.
     * @see ClusterStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The <code>certificate-authority-data</code> for your cluster.
     * </p>
     * 
     * @return The <code>certificate-authority-data</code> for your cluster.
     */
    public final Certificate certificateAuthority() {
        return certificateAuthority;
    }

    /**
     * <p>
     * A unique, case-sensitive identifier that you provide to ensure the idempotency of the request.
     * </p>
     * 
     * @return A unique, case-sensitive identifier that you provide to ensure the idempotency of the request.
     */
    public final String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * The platform version of your Amazon EKS cluster. For more information about clusters deployed on the Amazon Web
     * Services Cloud, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/platform-versions.html">Platform
     * versions</a> in the <i> <i>Amazon EKS User Guide</i> </i>. For more information about local clusters deployed on
     * an Outpost, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/eks-outposts-platform-versions.html">Amazon EKS local
     * cluster platform versions</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * 
     * @return The platform version of your Amazon EKS cluster. For more information about clusters deployed on the
     *         Amazon Web Services Cloud, see <a
     *         href="https://docs.aws.amazon.com/eks/latest/userguide/platform-versions.html">Platform versions</a> in
     *         the <i> <i>Amazon EKS User Guide</i> </i>. For more information about local clusters deployed on an
     *         Outpost, see <a
     *         href="https://docs.aws.amazon.com/eks/latest/userguide/eks-outposts-platform-versions.html">Amazon EKS
     *         local cluster platform versions</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     */
    public final String platformVersion() {
        return platformVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Metadata that assists with categorization and organization. Each tag consists of a key and an optional value. You
     * define both. Tags don't propagate to any other cluster or Amazon Web Services resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Metadata that assists with categorization and organization. Each tag consists of a key and an optional
     *         value. You define both. Tags don't propagate to any other cluster or Amazon Web Services resources.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the EncryptionConfig property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEncryptionConfig() {
        return encryptionConfig != null && !(encryptionConfig instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The encryption configuration for the cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEncryptionConfig} method.
     * </p>
     * 
     * @return The encryption configuration for the cluster.
     */
    public final List<EncryptionConfig> encryptionConfig() {
        return encryptionConfig;
    }

    /**
     * <p>
     * The configuration used to connect to a cluster for registration.
     * </p>
     * 
     * @return The configuration used to connect to a cluster for registration.
     */
    public final ConnectorConfigResponse connectorConfig() {
        return connectorConfig;
    }

    /**
     * <p>
     * The ID of your local Amazon EKS cluster on an Amazon Web Services Outpost. This property isn't available for an
     * Amazon EKS cluster on the Amazon Web Services cloud.
     * </p>
     * 
     * @return The ID of your local Amazon EKS cluster on an Amazon Web Services Outpost. This property isn't available
     *         for an Amazon EKS cluster on the Amazon Web Services cloud.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * An object representing the health of your Amazon EKS cluster.
     * </p>
     * 
     * @return An object representing the health of your Amazon EKS cluster.
     */
    public final ClusterHealth health() {
        return health;
    }

    /**
     * <p>
     * An object representing the configuration of your local Amazon EKS cluster on an Amazon Web Services Outpost. This
     * object isn't available for clusters on the Amazon Web Services cloud.
     * </p>
     * 
     * @return An object representing the configuration of your local Amazon EKS cluster on an Amazon Web Services
     *         Outpost. This object isn't available for clusters on the Amazon Web Services cloud.
     */
    public final OutpostConfigResponse outpostConfig() {
        return outpostConfig;
    }

    /**
     * <p>
     * The access configuration for the cluster.
     * </p>
     * 
     * @return The access configuration for the cluster.
     */
    public final AccessConfigResponse accessConfig() {
        return accessConfig;
    }

    /**
     * <p>
     * This value indicates if extended support is enabled or disabled for the cluster.
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/eks/latest/userguide/extended-support-control.html">Learn more about EKS
     * Extended Support in the EKS User Guide.</a>
     * </p>
     * 
     * @return This value indicates if extended support is enabled or disabled for the cluster.</p>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/eks/latest/userguide/extended-support-control.html">Learn more about
     *         EKS Extended Support in the EKS User Guide.</a>
     */
    public final UpgradePolicyResponse upgradePolicy() {
        return upgradePolicy;
    }

    /**
     * <p>
     * The configuration for zonal shift for the cluster.
     * </p>
     * 
     * @return The configuration for zonal shift for the cluster.
     */
    public final ZonalShiftConfigResponse zonalShiftConfig() {
        return zonalShiftConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(endpoint());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(resourcesVpcConfig());
        hashCode = 31 * hashCode + Objects.hashCode(kubernetesNetworkConfig());
        hashCode = 31 * hashCode + Objects.hashCode(logging());
        hashCode = 31 * hashCode + Objects.hashCode(identity());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(certificateAuthority());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(platformVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEncryptionConfig() ? encryptionConfig() : null);
        hashCode = 31 * hashCode + Objects.hashCode(connectorConfig());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(health());
        hashCode = 31 * hashCode + Objects.hashCode(outpostConfig());
        hashCode = 31 * hashCode + Objects.hashCode(accessConfig());
        hashCode = 31 * hashCode + Objects.hashCode(upgradePolicy());
        hashCode = 31 * hashCode + Objects.hashCode(zonalShiftConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Cluster)) {
            return false;
        }
        Cluster other = (Cluster) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(version(), other.version())
                && Objects.equals(endpoint(), other.endpoint()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(resourcesVpcConfig(), other.resourcesVpcConfig())
                && Objects.equals(kubernetesNetworkConfig(), other.kubernetesNetworkConfig())
                && Objects.equals(logging(), other.logging()) && Objects.equals(identity(), other.identity())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(certificateAuthority(), other.certificateAuthority())
                && Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(platformVersion(), other.platformVersion()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && hasEncryptionConfig() == other.hasEncryptionConfig()
                && Objects.equals(encryptionConfig(), other.encryptionConfig())
                && Objects.equals(connectorConfig(), other.connectorConfig()) && Objects.equals(id(), other.id())
                && Objects.equals(health(), other.health()) && Objects.equals(outpostConfig(), other.outpostConfig())
                && Objects.equals(accessConfig(), other.accessConfig()) && Objects.equals(upgradePolicy(), other.upgradePolicy())
                && Objects.equals(zonalShiftConfig(), other.zonalShiftConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Cluster").add("Name", name()).add("Arn", arn()).add("CreatedAt", createdAt())
                .add("Version", version()).add("Endpoint", endpoint()).add("RoleArn", roleArn())
                .add("ResourcesVpcConfig", resourcesVpcConfig()).add("KubernetesNetworkConfig", kubernetesNetworkConfig())
                .add("Logging", logging()).add("Identity", identity()).add("Status", statusAsString())
                .add("CertificateAuthority", certificateAuthority()).add("ClientRequestToken", clientRequestToken())
                .add("PlatformVersion", platformVersion()).add("Tags", hasTags() ? tags() : null)
                .add("EncryptionConfig", hasEncryptionConfig() ? encryptionConfig() : null)
                .add("ConnectorConfig", connectorConfig()).add("Id", id()).add("Health", health())
                .add("OutpostConfig", outpostConfig()).add("AccessConfig", accessConfig()).add("UpgradePolicy", upgradePolicy())
                .add("ZonalShiftConfig", zonalShiftConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "endpoint":
            return Optional.ofNullable(clazz.cast(endpoint()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "resourcesVpcConfig":
            return Optional.ofNullable(clazz.cast(resourcesVpcConfig()));
        case "kubernetesNetworkConfig":
            return Optional.ofNullable(clazz.cast(kubernetesNetworkConfig()));
        case "logging":
            return Optional.ofNullable(clazz.cast(logging()));
        case "identity":
            return Optional.ofNullable(clazz.cast(identity()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "certificateAuthority":
            return Optional.ofNullable(clazz.cast(certificateAuthority()));
        case "clientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "platformVersion":
            return Optional.ofNullable(clazz.cast(platformVersion()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "encryptionConfig":
            return Optional.ofNullable(clazz.cast(encryptionConfig()));
        case "connectorConfig":
            return Optional.ofNullable(clazz.cast(connectorConfig()));
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "health":
            return Optional.ofNullable(clazz.cast(health()));
        case "outpostConfig":
            return Optional.ofNullable(clazz.cast(outpostConfig()));
        case "accessConfig":
            return Optional.ofNullable(clazz.cast(accessConfig()));
        case "upgradePolicy":
            return Optional.ofNullable(clazz.cast(upgradePolicy()));
        case "zonalShiftConfig":
            return Optional.ofNullable(clazz.cast(zonalShiftConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Cluster, T> g) {
        return obj -> g.apply((Cluster) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Cluster> {
        /**
         * <p>
         * The name of your cluster.
         * </p>
         * 
         * @param name
         *        The name of your cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the cluster.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The Unix epoch timestamp at object creation.
         * </p>
         * 
         * @param createdAt
         *        The Unix epoch timestamp at object creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The Kubernetes server version for the cluster.
         * </p>
         * 
         * @param version
         *        The Kubernetes server version for the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The endpoint for your Kubernetes API server.
         * </p>
         * 
         * @param endpoint
         *        The endpoint for your Kubernetes API server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoint(String endpoint);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role that provides permissions for the Kubernetes control plane to
         * make calls to Amazon Web Services API operations on your behalf.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of the IAM role that provides permissions for the Kubernetes control
         *        plane to make calls to Amazon Web Services API operations on your behalf.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The VPC configuration used by the cluster control plane. Amazon EKS VPC resources have specific requirements
         * to work properly with Kubernetes. For more information, see <a
         * href="https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html">Cluster VPC considerations</a> and
         * <a href="https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html">Cluster security group
         * considerations</a> in the <i>Amazon EKS User Guide</i>.
         * </p>
         * 
         * @param resourcesVpcConfig
         *        The VPC configuration used by the cluster control plane. Amazon EKS VPC resources have specific
         *        requirements to work properly with Kubernetes. For more information, see <a
         *        href="https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html">Cluster VPC
         *        considerations</a> and <a
         *        href="https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html">Cluster security group
         *        considerations</a> in the <i>Amazon EKS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourcesVpcConfig(VpcConfigResponse resourcesVpcConfig);

        /**
         * <p>
         * The VPC configuration used by the cluster control plane. Amazon EKS VPC resources have specific requirements
         * to work properly with Kubernetes. For more information, see <a
         * href="https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html">Cluster VPC considerations</a> and
         * <a href="https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html">Cluster security group
         * considerations</a> in the <i>Amazon EKS User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link VpcConfigResponse.Builder} avoiding the
         * need to create one manually via {@link VpcConfigResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VpcConfigResponse.Builder#build()} is called immediately and its
         * result is passed to {@link #resourcesVpcConfig(VpcConfigResponse)}.
         * 
         * @param resourcesVpcConfig
         *        a consumer that will call methods on {@link VpcConfigResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourcesVpcConfig(VpcConfigResponse)
         */
        default Builder resourcesVpcConfig(Consumer<VpcConfigResponse.Builder> resourcesVpcConfig) {
            return resourcesVpcConfig(VpcConfigResponse.builder().applyMutation(resourcesVpcConfig).build());
        }

        /**
         * <p>
         * The Kubernetes network configuration for the cluster.
         * </p>
         * 
         * @param kubernetesNetworkConfig
         *        The Kubernetes network configuration for the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kubernetesNetworkConfig(KubernetesNetworkConfigResponse kubernetesNetworkConfig);

        /**
         * <p>
         * The Kubernetes network configuration for the cluster.
         * </p>
         * This is a convenience method that creates an instance of the {@link KubernetesNetworkConfigResponse.Builder}
         * avoiding the need to create one manually via {@link KubernetesNetworkConfigResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KubernetesNetworkConfigResponse.Builder#build()} is called
         * immediately and its result is passed to {@link #kubernetesNetworkConfig(KubernetesNetworkConfigResponse)}.
         * 
         * @param kubernetesNetworkConfig
         *        a consumer that will call methods on {@link KubernetesNetworkConfigResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kubernetesNetworkConfig(KubernetesNetworkConfigResponse)
         */
        default Builder kubernetesNetworkConfig(Consumer<KubernetesNetworkConfigResponse.Builder> kubernetesNetworkConfig) {
            return kubernetesNetworkConfig(KubernetesNetworkConfigResponse.builder().applyMutation(kubernetesNetworkConfig)
                    .build());
        }

        /**
         * <p>
         * The logging configuration for your cluster.
         * </p>
         * 
         * @param logging
         *        The logging configuration for your cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logging(Logging logging);

        /**
         * <p>
         * The logging configuration for your cluster.
         * </p>
         * This is a convenience method that creates an instance of the {@link Logging.Builder} avoiding the need to
         * create one manually via {@link Logging#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Logging.Builder#build()} is called immediately and its result is
         * passed to {@link #logging(Logging)}.
         * 
         * @param logging
         *        a consumer that will call methods on {@link Logging.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logging(Logging)
         */
        default Builder logging(Consumer<Logging.Builder> logging) {
            return logging(Logging.builder().applyMutation(logging).build());
        }

        /**
         * <p>
         * The identity provider information for the cluster.
         * </p>
         * 
         * @param identity
         *        The identity provider information for the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identity(Identity identity);

        /**
         * <p>
         * The identity provider information for the cluster.
         * </p>
         * This is a convenience method that creates an instance of the {@link Identity.Builder} avoiding the need to
         * create one manually via {@link Identity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Identity.Builder#build()} is called immediately and its result is
         * passed to {@link #identity(Identity)}.
         * 
         * @param identity
         *        a consumer that will call methods on {@link Identity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #identity(Identity)
         */
        default Builder identity(Consumer<Identity.Builder> identity) {
            return identity(Identity.builder().applyMutation(identity).build());
        }

        /**
         * <p>
         * The current status of the cluster.
         * </p>
         * 
         * @param status
         *        The current status of the cluster.
         * @see ClusterStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the cluster.
         * </p>
         * 
         * @param status
         *        The current status of the cluster.
         * @see ClusterStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterStatus
         */
        Builder status(ClusterStatus status);

        /**
         * <p>
         * The <code>certificate-authority-data</code> for your cluster.
         * </p>
         * 
         * @param certificateAuthority
         *        The <code>certificate-authority-data</code> for your cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateAuthority(Certificate certificateAuthority);

        /**
         * <p>
         * The <code>certificate-authority-data</code> for your cluster.
         * </p>
         * This is a convenience method that creates an instance of the {@link Certificate.Builder} avoiding the need to
         * create one manually via {@link Certificate#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Certificate.Builder#build()} is called immediately and its result
         * is passed to {@link #certificateAuthority(Certificate)}.
         * 
         * @param certificateAuthority
         *        a consumer that will call methods on {@link Certificate.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #certificateAuthority(Certificate)
         */
        default Builder certificateAuthority(Consumer<Certificate.Builder> certificateAuthority) {
            return certificateAuthority(Certificate.builder().applyMutation(certificateAuthority).build());
        }

        /**
         * <p>
         * A unique, case-sensitive identifier that you provide to ensure the idempotency of the request.
         * </p>
         * 
         * @param clientRequestToken
         *        A unique, case-sensitive identifier that you provide to ensure the idempotency of the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * The platform version of your Amazon EKS cluster. For more information about clusters deployed on the Amazon
         * Web Services Cloud, see <a
         * href="https://docs.aws.amazon.com/eks/latest/userguide/platform-versions.html">Platform versions</a> in the
         * <i> <i>Amazon EKS User Guide</i> </i>. For more information about local clusters deployed on an Outpost, see
         * <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-outposts-platform-versions.html">Amazon EKS
         * local cluster platform versions</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
         * </p>
         * 
         * @param platformVersion
         *        The platform version of your Amazon EKS cluster. For more information about clusters deployed on the
         *        Amazon Web Services Cloud, see <a
         *        href="https://docs.aws.amazon.com/eks/latest/userguide/platform-versions.html">Platform versions</a>
         *        in the <i> <i>Amazon EKS User Guide</i> </i>. For more information about local clusters deployed on an
         *        Outpost, see <a
         *        href="https://docs.aws.amazon.com/eks/latest/userguide/eks-outposts-platform-versions.html">Amazon EKS
         *        local cluster platform versions</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformVersion(String platformVersion);

        /**
         * <p>
         * Metadata that assists with categorization and organization. Each tag consists of a key and an optional value.
         * You define both. Tags don't propagate to any other cluster or Amazon Web Services resources.
         * </p>
         * 
         * @param tags
         *        Metadata that assists with categorization and organization. Each tag consists of a key and an optional
         *        value. You define both. Tags don't propagate to any other cluster or Amazon Web Services resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The encryption configuration for the cluster.
         * </p>
         * 
         * @param encryptionConfig
         *        The encryption configuration for the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfig(Collection<EncryptionConfig> encryptionConfig);

        /**
         * <p>
         * The encryption configuration for the cluster.
         * </p>
         * 
         * @param encryptionConfig
         *        The encryption configuration for the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfig(EncryptionConfig... encryptionConfig);

        /**
         * <p>
         * The encryption configuration for the cluster.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.eks.model.EncryptionConfig.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.eks.model.EncryptionConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.eks.model.EncryptionConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #encryptionConfig(List<EncryptionConfig>)}.
         * 
         * @param encryptionConfig
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.eks.model.EncryptionConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfig(java.util.Collection<EncryptionConfig>)
         */
        Builder encryptionConfig(Consumer<EncryptionConfig.Builder>... encryptionConfig);

        /**
         * <p>
         * The configuration used to connect to a cluster for registration.
         * </p>
         * 
         * @param connectorConfig
         *        The configuration used to connect to a cluster for registration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectorConfig(ConnectorConfigResponse connectorConfig);

        /**
         * <p>
         * The configuration used to connect to a cluster for registration.
         * </p>
         * This is a convenience method that creates an instance of the {@link ConnectorConfigResponse.Builder} avoiding
         * the need to create one manually via {@link ConnectorConfigResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ConnectorConfigResponse.Builder#build()} is called immediately
         * and its result is passed to {@link #connectorConfig(ConnectorConfigResponse)}.
         * 
         * @param connectorConfig
         *        a consumer that will call methods on {@link ConnectorConfigResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #connectorConfig(ConnectorConfigResponse)
         */
        default Builder connectorConfig(Consumer<ConnectorConfigResponse.Builder> connectorConfig) {
            return connectorConfig(ConnectorConfigResponse.builder().applyMutation(connectorConfig).build());
        }

        /**
         * <p>
         * The ID of your local Amazon EKS cluster on an Amazon Web Services Outpost. This property isn't available for
         * an Amazon EKS cluster on the Amazon Web Services cloud.
         * </p>
         * 
         * @param id
         *        The ID of your local Amazon EKS cluster on an Amazon Web Services Outpost. This property isn't
         *        available for an Amazon EKS cluster on the Amazon Web Services cloud.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * An object representing the health of your Amazon EKS cluster.
         * </p>
         * 
         * @param health
         *        An object representing the health of your Amazon EKS cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder health(ClusterHealth health);

        /**
         * <p>
         * An object representing the health of your Amazon EKS cluster.
         * </p>
         * This is a convenience method that creates an instance of the {@link ClusterHealth.Builder} avoiding the need
         * to create one manually via {@link ClusterHealth#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ClusterHealth.Builder#build()} is called immediately and its
         * result is passed to {@link #health(ClusterHealth)}.
         * 
         * @param health
         *        a consumer that will call methods on {@link ClusterHealth.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #health(ClusterHealth)
         */
        default Builder health(Consumer<ClusterHealth.Builder> health) {
            return health(ClusterHealth.builder().applyMutation(health).build());
        }

        /**
         * <p>
         * An object representing the configuration of your local Amazon EKS cluster on an Amazon Web Services Outpost.
         * This object isn't available for clusters on the Amazon Web Services cloud.
         * </p>
         * 
         * @param outpostConfig
         *        An object representing the configuration of your local Amazon EKS cluster on an Amazon Web Services
         *        Outpost. This object isn't available for clusters on the Amazon Web Services cloud.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outpostConfig(OutpostConfigResponse outpostConfig);

        /**
         * <p>
         * An object representing the configuration of your local Amazon EKS cluster on an Amazon Web Services Outpost.
         * This object isn't available for clusters on the Amazon Web Services cloud.
         * </p>
         * This is a convenience method that creates an instance of the {@link OutpostConfigResponse.Builder} avoiding
         * the need to create one manually via {@link OutpostConfigResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OutpostConfigResponse.Builder#build()} is called immediately and
         * its result is passed to {@link #outpostConfig(OutpostConfigResponse)}.
         * 
         * @param outpostConfig
         *        a consumer that will call methods on {@link OutpostConfigResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outpostConfig(OutpostConfigResponse)
         */
        default Builder outpostConfig(Consumer<OutpostConfigResponse.Builder> outpostConfig) {
            return outpostConfig(OutpostConfigResponse.builder().applyMutation(outpostConfig).build());
        }

        /**
         * <p>
         * The access configuration for the cluster.
         * </p>
         * 
         * @param accessConfig
         *        The access configuration for the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessConfig(AccessConfigResponse accessConfig);

        /**
         * <p>
         * The access configuration for the cluster.
         * </p>
         * This is a convenience method that creates an instance of the {@link AccessConfigResponse.Builder} avoiding
         * the need to create one manually via {@link AccessConfigResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AccessConfigResponse.Builder#build()} is called immediately and
         * its result is passed to {@link #accessConfig(AccessConfigResponse)}.
         * 
         * @param accessConfig
         *        a consumer that will call methods on {@link AccessConfigResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessConfig(AccessConfigResponse)
         */
        default Builder accessConfig(Consumer<AccessConfigResponse.Builder> accessConfig) {
            return accessConfig(AccessConfigResponse.builder().applyMutation(accessConfig).build());
        }

        /**
         * <p>
         * This value indicates if extended support is enabled or disabled for the cluster.
         * </p>
         * <p>
         * <a href="https://docs.aws.amazon.com/eks/latest/userguide/extended-support-control.html">Learn more about EKS
         * Extended Support in the EKS User Guide.</a>
         * </p>
         * 
         * @param upgradePolicy
         *        This value indicates if extended support is enabled or disabled for the cluster.</p>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/eks/latest/userguide/extended-support-control.html">Learn more
         *        about EKS Extended Support in the EKS User Guide.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder upgradePolicy(UpgradePolicyResponse upgradePolicy);

        /**
         * <p>
         * This value indicates if extended support is enabled or disabled for the cluster.
         * </p>
         * <p>
         * <a href="https://docs.aws.amazon.com/eks/latest/userguide/extended-support-control.html">Learn more about EKS
         * Extended Support in the EKS User Guide.</a>
         * </p>
         * This is a convenience method that creates an instance of the {@link UpgradePolicyResponse.Builder} avoiding
         * the need to create one manually via {@link UpgradePolicyResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UpgradePolicyResponse.Builder#build()} is called immediately and
         * its result is passed to {@link #upgradePolicy(UpgradePolicyResponse)}.
         * 
         * @param upgradePolicy
         *        a consumer that will call methods on {@link UpgradePolicyResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #upgradePolicy(UpgradePolicyResponse)
         */
        default Builder upgradePolicy(Consumer<UpgradePolicyResponse.Builder> upgradePolicy) {
            return upgradePolicy(UpgradePolicyResponse.builder().applyMutation(upgradePolicy).build());
        }

        /**
         * <p>
         * The configuration for zonal shift for the cluster.
         * </p>
         * 
         * @param zonalShiftConfig
         *        The configuration for zonal shift for the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder zonalShiftConfig(ZonalShiftConfigResponse zonalShiftConfig);

        /**
         * <p>
         * The configuration for zonal shift for the cluster.
         * </p>
         * This is a convenience method that creates an instance of the {@link ZonalShiftConfigResponse.Builder}
         * avoiding the need to create one manually via {@link ZonalShiftConfigResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ZonalShiftConfigResponse.Builder#build()} is called immediately
         * and its result is passed to {@link #zonalShiftConfig(ZonalShiftConfigResponse)}.
         * 
         * @param zonalShiftConfig
         *        a consumer that will call methods on {@link ZonalShiftConfigResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #zonalShiftConfig(ZonalShiftConfigResponse)
         */
        default Builder zonalShiftConfig(Consumer<ZonalShiftConfigResponse.Builder> zonalShiftConfig) {
            return zonalShiftConfig(ZonalShiftConfigResponse.builder().applyMutation(zonalShiftConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private Instant createdAt;

        private String version;

        private String endpoint;

        private String roleArn;

        private VpcConfigResponse resourcesVpcConfig;

        private KubernetesNetworkConfigResponse kubernetesNetworkConfig;

        private Logging logging;

        private Identity identity;

        private String status;

        private Certificate certificateAuthority;

        private String clientRequestToken;

        private String platformVersion;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private List<EncryptionConfig> encryptionConfig = DefaultSdkAutoConstructList.getInstance();

        private ConnectorConfigResponse connectorConfig;

        private String id;

        private ClusterHealth health;

        private OutpostConfigResponse outpostConfig;

        private AccessConfigResponse accessConfig;

        private UpgradePolicyResponse upgradePolicy;

        private ZonalShiftConfigResponse zonalShiftConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(Cluster model) {
            name(model.name);
            arn(model.arn);
            createdAt(model.createdAt);
            version(model.version);
            endpoint(model.endpoint);
            roleArn(model.roleArn);
            resourcesVpcConfig(model.resourcesVpcConfig);
            kubernetesNetworkConfig(model.kubernetesNetworkConfig);
            logging(model.logging);
            identity(model.identity);
            status(model.status);
            certificateAuthority(model.certificateAuthority);
            clientRequestToken(model.clientRequestToken);
            platformVersion(model.platformVersion);
            tags(model.tags);
            encryptionConfig(model.encryptionConfig);
            connectorConfig(model.connectorConfig);
            id(model.id);
            health(model.health);
            outpostConfig(model.outpostConfig);
            accessConfig(model.accessConfig);
            upgradePolicy(model.upgradePolicy);
            zonalShiftConfig(model.zonalShiftConfig);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getEndpoint() {
            return endpoint;
        }

        public final void setEndpoint(String endpoint) {
            this.endpoint = endpoint;
        }

        @Override
        public final Builder endpoint(String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final VpcConfigResponse.Builder getResourcesVpcConfig() {
            return resourcesVpcConfig != null ? resourcesVpcConfig.toBuilder() : null;
        }

        public final void setResourcesVpcConfig(VpcConfigResponse.BuilderImpl resourcesVpcConfig) {
            this.resourcesVpcConfig = resourcesVpcConfig != null ? resourcesVpcConfig.build() : null;
        }

        @Override
        public final Builder resourcesVpcConfig(VpcConfigResponse resourcesVpcConfig) {
            this.resourcesVpcConfig = resourcesVpcConfig;
            return this;
        }

        public final KubernetesNetworkConfigResponse.Builder getKubernetesNetworkConfig() {
            return kubernetesNetworkConfig != null ? kubernetesNetworkConfig.toBuilder() : null;
        }

        public final void setKubernetesNetworkConfig(KubernetesNetworkConfigResponse.BuilderImpl kubernetesNetworkConfig) {
            this.kubernetesNetworkConfig = kubernetesNetworkConfig != null ? kubernetesNetworkConfig.build() : null;
        }

        @Override
        public final Builder kubernetesNetworkConfig(KubernetesNetworkConfigResponse kubernetesNetworkConfig) {
            this.kubernetesNetworkConfig = kubernetesNetworkConfig;
            return this;
        }

        public final Logging.Builder getLogging() {
            return logging != null ? logging.toBuilder() : null;
        }

        public final void setLogging(Logging.BuilderImpl logging) {
            this.logging = logging != null ? logging.build() : null;
        }

        @Override
        public final Builder logging(Logging logging) {
            this.logging = logging;
            return this;
        }

        public final Identity.Builder getIdentity() {
            return identity != null ? identity.toBuilder() : null;
        }

        public final void setIdentity(Identity.BuilderImpl identity) {
            this.identity = identity != null ? identity.build() : null;
        }

        @Override
        public final Builder identity(Identity identity) {
            this.identity = identity;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ClusterStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Certificate.Builder getCertificateAuthority() {
            return certificateAuthority != null ? certificateAuthority.toBuilder() : null;
        }

        public final void setCertificateAuthority(Certificate.BuilderImpl certificateAuthority) {
            this.certificateAuthority = certificateAuthority != null ? certificateAuthority.build() : null;
        }

        @Override
        public final Builder certificateAuthority(Certificate certificateAuthority) {
            this.certificateAuthority = certificateAuthority;
            return this;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final String getPlatformVersion() {
            return platformVersion;
        }

        public final void setPlatformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
        }

        @Override
        public final Builder platformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final List<EncryptionConfig.Builder> getEncryptionConfig() {
            List<EncryptionConfig.Builder> result = EncryptionConfigListCopier.copyToBuilder(this.encryptionConfig);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEncryptionConfig(Collection<EncryptionConfig.BuilderImpl> encryptionConfig) {
            this.encryptionConfig = EncryptionConfigListCopier.copyFromBuilder(encryptionConfig);
        }

        @Override
        public final Builder encryptionConfig(Collection<EncryptionConfig> encryptionConfig) {
            this.encryptionConfig = EncryptionConfigListCopier.copy(encryptionConfig);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder encryptionConfig(EncryptionConfig... encryptionConfig) {
            encryptionConfig(Arrays.asList(encryptionConfig));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder encryptionConfig(Consumer<EncryptionConfig.Builder>... encryptionConfig) {
            encryptionConfig(Stream.of(encryptionConfig).map(c -> EncryptionConfig.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final ConnectorConfigResponse.Builder getConnectorConfig() {
            return connectorConfig != null ? connectorConfig.toBuilder() : null;
        }

        public final void setConnectorConfig(ConnectorConfigResponse.BuilderImpl connectorConfig) {
            this.connectorConfig = connectorConfig != null ? connectorConfig.build() : null;
        }

        @Override
        public final Builder connectorConfig(ConnectorConfigResponse connectorConfig) {
            this.connectorConfig = connectorConfig;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final ClusterHealth.Builder getHealth() {
            return health != null ? health.toBuilder() : null;
        }

        public final void setHealth(ClusterHealth.BuilderImpl health) {
            this.health = health != null ? health.build() : null;
        }

        @Override
        public final Builder health(ClusterHealth health) {
            this.health = health;
            return this;
        }

        public final OutpostConfigResponse.Builder getOutpostConfig() {
            return outpostConfig != null ? outpostConfig.toBuilder() : null;
        }

        public final void setOutpostConfig(OutpostConfigResponse.BuilderImpl outpostConfig) {
            this.outpostConfig = outpostConfig != null ? outpostConfig.build() : null;
        }

        @Override
        public final Builder outpostConfig(OutpostConfigResponse outpostConfig) {
            this.outpostConfig = outpostConfig;
            return this;
        }

        public final AccessConfigResponse.Builder getAccessConfig() {
            return accessConfig != null ? accessConfig.toBuilder() : null;
        }

        public final void setAccessConfig(AccessConfigResponse.BuilderImpl accessConfig) {
            this.accessConfig = accessConfig != null ? accessConfig.build() : null;
        }

        @Override
        public final Builder accessConfig(AccessConfigResponse accessConfig) {
            this.accessConfig = accessConfig;
            return this;
        }

        public final UpgradePolicyResponse.Builder getUpgradePolicy() {
            return upgradePolicy != null ? upgradePolicy.toBuilder() : null;
        }

        public final void setUpgradePolicy(UpgradePolicyResponse.BuilderImpl upgradePolicy) {
            this.upgradePolicy = upgradePolicy != null ? upgradePolicy.build() : null;
        }

        @Override
        public final Builder upgradePolicy(UpgradePolicyResponse upgradePolicy) {
            this.upgradePolicy = upgradePolicy;
            return this;
        }

        public final ZonalShiftConfigResponse.Builder getZonalShiftConfig() {
            return zonalShiftConfig != null ? zonalShiftConfig.toBuilder() : null;
        }

        public final void setZonalShiftConfig(ZonalShiftConfigResponse.BuilderImpl zonalShiftConfig) {
            this.zonalShiftConfig = zonalShiftConfig != null ? zonalShiftConfig.build() : null;
        }

        @Override
        public final Builder zonalShiftConfig(ZonalShiftConfigResponse zonalShiftConfig) {
            this.zonalShiftConfig = zonalShiftConfig;
            return this;
        }

        @Override
        public Cluster build() {
            return new Cluster(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
