/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Compatibility information.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Compatibility implements SdkPojo, Serializable, ToCopyableBuilder<Compatibility.Builder, Compatibility> {
    private static final SdkField<String> CLUSTER_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clusterVersion").getter(getter(Compatibility::clusterVersion)).setter(setter(Builder::clusterVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clusterVersion").build()).build();

    private static final SdkField<List<String>> PLATFORM_VERSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("platformVersions")
            .getter(getter(Compatibility::platformVersions))
            .setter(setter(Builder::platformVersions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platformVersions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> DEFAULT_VERSION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("defaultVersion").getter(getter(Compatibility::defaultVersion)).setter(setter(Builder::defaultVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("defaultVersion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLUSTER_VERSION_FIELD,
            PLATFORM_VERSIONS_FIELD, DEFAULT_VERSION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String clusterVersion;

    private final List<String> platformVersions;

    private final Boolean defaultVersion;

    private Compatibility(BuilderImpl builder) {
        this.clusterVersion = builder.clusterVersion;
        this.platformVersions = builder.platformVersions;
        this.defaultVersion = builder.defaultVersion;
    }

    /**
     * <p>
     * The supported Kubernetes version of the cluster.
     * </p>
     * 
     * @return The supported Kubernetes version of the cluster.
     */
    public final String clusterVersion() {
        return clusterVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the PlatformVersions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPlatformVersions() {
        return platformVersions != null && !(platformVersions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The supported compute platform.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlatformVersions} method.
     * </p>
     * 
     * @return The supported compute platform.
     */
    public final List<String> platformVersions() {
        return platformVersions;
    }

    /**
     * <p>
     * The supported default version.
     * </p>
     * 
     * @return The supported default version.
     */
    public final Boolean defaultVersion() {
        return defaultVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clusterVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasPlatformVersions() ? platformVersions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(defaultVersion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Compatibility)) {
            return false;
        }
        Compatibility other = (Compatibility) obj;
        return Objects.equals(clusterVersion(), other.clusterVersion()) && hasPlatformVersions() == other.hasPlatformVersions()
                && Objects.equals(platformVersions(), other.platformVersions())
                && Objects.equals(defaultVersion(), other.defaultVersion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Compatibility").add("ClusterVersion", clusterVersion())
                .add("PlatformVersions", hasPlatformVersions() ? platformVersions() : null)
                .add("DefaultVersion", defaultVersion()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "clusterVersion":
            return Optional.ofNullable(clazz.cast(clusterVersion()));
        case "platformVersions":
            return Optional.ofNullable(clazz.cast(platformVersions()));
        case "defaultVersion":
            return Optional.ofNullable(clazz.cast(defaultVersion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("clusterVersion", CLUSTER_VERSION_FIELD);
        map.put("platformVersions", PLATFORM_VERSIONS_FIELD);
        map.put("defaultVersion", DEFAULT_VERSION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Compatibility, T> g) {
        return obj -> g.apply((Compatibility) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Compatibility> {
        /**
         * <p>
         * The supported Kubernetes version of the cluster.
         * </p>
         * 
         * @param clusterVersion
         *        The supported Kubernetes version of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterVersion(String clusterVersion);

        /**
         * <p>
         * The supported compute platform.
         * </p>
         * 
         * @param platformVersions
         *        The supported compute platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformVersions(Collection<String> platformVersions);

        /**
         * <p>
         * The supported compute platform.
         * </p>
         * 
         * @param platformVersions
         *        The supported compute platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformVersions(String... platformVersions);

        /**
         * <p>
         * The supported default version.
         * </p>
         * 
         * @param defaultVersion
         *        The supported default version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultVersion(Boolean defaultVersion);
    }

    static final class BuilderImpl implements Builder {
        private String clusterVersion;

        private List<String> platformVersions = DefaultSdkAutoConstructList.getInstance();

        private Boolean defaultVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(Compatibility model) {
            clusterVersion(model.clusterVersion);
            platformVersions(model.platformVersions);
            defaultVersion(model.defaultVersion);
        }

        public final String getClusterVersion() {
            return clusterVersion;
        }

        public final void setClusterVersion(String clusterVersion) {
            this.clusterVersion = clusterVersion;
        }

        @Override
        public final Builder clusterVersion(String clusterVersion) {
            this.clusterVersion = clusterVersion;
            return this;
        }

        public final Collection<String> getPlatformVersions() {
            if (platformVersions instanceof SdkAutoConstructList) {
                return null;
            }
            return platformVersions;
        }

        public final void setPlatformVersions(Collection<String> platformVersions) {
            this.platformVersions = StringListCopier.copy(platformVersions);
        }

        @Override
        public final Builder platformVersions(Collection<String> platformVersions) {
            this.platformVersions = StringListCopier.copy(platformVersions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformVersions(String... platformVersions) {
            platformVersions(Arrays.asList(platformVersions));
            return this;
        }

        public final Boolean getDefaultVersion() {
            return defaultVersion;
        }

        public final void setDefaultVersion(Boolean defaultVersion) {
            this.defaultVersion = defaultVersion;
        }

        @Override
        public final Builder defaultVersion(Boolean defaultVersion) {
            this.defaultVersion = defaultVersion;
            return this;
        }

        @Override
        public Compatibility build() {
            return new Compatibility(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
