/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A check that provides recommendations to remedy potential upgrade-impacting issues.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Insight implements SdkPojo, Serializable, ToCopyableBuilder<Insight.Builder, Insight> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(Insight::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Insight::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> CATEGORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("category").getter(getter(Insight::categoryAsString)).setter(setter(Builder::category))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("category").build()).build();

    private static final SdkField<String> KUBERNETES_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("kubernetesVersion").getter(getter(Insight::kubernetesVersion))
            .setter(setter(Builder::kubernetesVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kubernetesVersion").build()).build();

    private static final SdkField<Instant> LAST_REFRESH_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastRefreshTime").getter(getter(Insight::lastRefreshTime)).setter(setter(Builder::lastRefreshTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRefreshTime").build()).build();

    private static final SdkField<Instant> LAST_TRANSITION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastTransitionTime").getter(getter(Insight::lastTransitionTime))
            .setter(setter(Builder::lastTransitionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastTransitionTime").build())
            .build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(Insight::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<InsightStatus> INSIGHT_STATUS_FIELD = SdkField
            .<InsightStatus> builder(MarshallingType.SDK_POJO).memberName("insightStatus").getter(getter(Insight::insightStatus))
            .setter(setter(Builder::insightStatus)).constructor(InsightStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("insightStatus").build()).build();

    private static final SdkField<String> RECOMMENDATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("recommendation").getter(getter(Insight::recommendation)).setter(setter(Builder::recommendation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendation").build()).build();

    private static final SdkField<Map<String, String>> ADDITIONAL_INFO_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("additionalInfo")
            .getter(getter(Insight::additionalInfo))
            .setter(setter(Builder::additionalInfo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("additionalInfo").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<InsightResourceDetail>> RESOURCES_FIELD = SdkField
            .<List<InsightResourceDetail>> builder(MarshallingType.LIST)
            .memberName("resources")
            .getter(getter(Insight::resources))
            .setter(setter(Builder::resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InsightResourceDetail> builder(MarshallingType.SDK_POJO)
                                            .constructor(InsightResourceDetail::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<InsightCategorySpecificSummary> CATEGORY_SPECIFIC_SUMMARY_FIELD = SdkField
            .<InsightCategorySpecificSummary> builder(MarshallingType.SDK_POJO).memberName("categorySpecificSummary")
            .getter(getter(Insight::categorySpecificSummary)).setter(setter(Builder::categorySpecificSummary))
            .constructor(InsightCategorySpecificSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("categorySpecificSummary").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, NAME_FIELD,
            CATEGORY_FIELD, KUBERNETES_VERSION_FIELD, LAST_REFRESH_TIME_FIELD, LAST_TRANSITION_TIME_FIELD, DESCRIPTION_FIELD,
            INSIGHT_STATUS_FIELD, RECOMMENDATION_FIELD, ADDITIONAL_INFO_FIELD, RESOURCES_FIELD, CATEGORY_SPECIFIC_SUMMARY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String name;

    private final String category;

    private final String kubernetesVersion;

    private final Instant lastRefreshTime;

    private final Instant lastTransitionTime;

    private final String description;

    private final InsightStatus insightStatus;

    private final String recommendation;

    private final Map<String, String> additionalInfo;

    private final List<InsightResourceDetail> resources;

    private final InsightCategorySpecificSummary categorySpecificSummary;

    private Insight(BuilderImpl builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.category = builder.category;
        this.kubernetesVersion = builder.kubernetesVersion;
        this.lastRefreshTime = builder.lastRefreshTime;
        this.lastTransitionTime = builder.lastTransitionTime;
        this.description = builder.description;
        this.insightStatus = builder.insightStatus;
        this.recommendation = builder.recommendation;
        this.additionalInfo = builder.additionalInfo;
        this.resources = builder.resources;
        this.categorySpecificSummary = builder.categorySpecificSummary;
    }

    /**
     * <p>
     * The ID of the insight.
     * </p>
     * 
     * @return The ID of the insight.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The name of the insight.
     * </p>
     * 
     * @return The name of the insight.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The category of the insight.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #category} will
     * return {@link Category#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #categoryAsString}.
     * </p>
     * 
     * @return The category of the insight.
     * @see Category
     */
    public final Category category() {
        return Category.fromValue(category);
    }

    /**
     * <p>
     * The category of the insight.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #category} will
     * return {@link Category#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #categoryAsString}.
     * </p>
     * 
     * @return The category of the insight.
     * @see Category
     */
    public final String categoryAsString() {
        return category;
    }

    /**
     * <p>
     * The Kubernetes minor version associated with an insight if applicable.
     * </p>
     * 
     * @return The Kubernetes minor version associated with an insight if applicable.
     */
    public final String kubernetesVersion() {
        return kubernetesVersion;
    }

    /**
     * <p>
     * The time Amazon EKS last successfully completed a refresh of this insight check on the cluster.
     * </p>
     * 
     * @return The time Amazon EKS last successfully completed a refresh of this insight check on the cluster.
     */
    public final Instant lastRefreshTime() {
        return lastRefreshTime;
    }

    /**
     * <p>
     * The time the status of the insight last changed.
     * </p>
     * 
     * @return The time the status of the insight last changed.
     */
    public final Instant lastTransitionTime() {
        return lastTransitionTime;
    }

    /**
     * <p>
     * The description of the insight which includes alert criteria, remediation recommendation, and additional
     * resources (contains Markdown).
     * </p>
     * 
     * @return The description of the insight which includes alert criteria, remediation recommendation, and additional
     *         resources (contains Markdown).
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * An object containing more detail on the status of the insight resource.
     * </p>
     * 
     * @return An object containing more detail on the status of the insight resource.
     */
    public final InsightStatus insightStatus() {
        return insightStatus;
    }

    /**
     * <p>
     * A summary of how to remediate the finding of this insight if applicable.
     * </p>
     * 
     * @return A summary of how to remediate the finding of this insight if applicable.
     */
    public final String recommendation() {
        return recommendation;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdditionalInfo property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAdditionalInfo() {
        return additionalInfo != null && !(additionalInfo instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Links to sources that provide additional context on the insight.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdditionalInfo} method.
     * </p>
     * 
     * @return Links to sources that provide additional context on the insight.
     */
    public final Map<String, String> additionalInfo() {
        return additionalInfo;
    }

    /**
     * For responses, this returns true if the service returned a value for the Resources property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasResources() {
        return resources != null && !(resources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The details about each resource listed in the insight check result.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResources} method.
     * </p>
     * 
     * @return The details about each resource listed in the insight check result.
     */
    public final List<InsightResourceDetail> resources() {
        return resources;
    }

    /**
     * <p>
     * Summary information that relates to the category of the insight. Currently only returned with certain insights
     * having category <code>UPGRADE_READINESS</code>.
     * </p>
     * 
     * @return Summary information that relates to the category of the insight. Currently only returned with certain
     *         insights having category <code>UPGRADE_READINESS</code>.
     */
    public final InsightCategorySpecificSummary categorySpecificSummary() {
        return categorySpecificSummary;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(categoryAsString());
        hashCode = 31 * hashCode + Objects.hashCode(kubernetesVersion());
        hashCode = 31 * hashCode + Objects.hashCode(lastRefreshTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastTransitionTime());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(insightStatus());
        hashCode = 31 * hashCode + Objects.hashCode(recommendation());
        hashCode = 31 * hashCode + Objects.hashCode(hasAdditionalInfo() ? additionalInfo() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResources() ? resources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(categorySpecificSummary());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Insight)) {
            return false;
        }
        Insight other = (Insight) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(name(), other.name())
                && Objects.equals(categoryAsString(), other.categoryAsString())
                && Objects.equals(kubernetesVersion(), other.kubernetesVersion())
                && Objects.equals(lastRefreshTime(), other.lastRefreshTime())
                && Objects.equals(lastTransitionTime(), other.lastTransitionTime())
                && Objects.equals(description(), other.description()) && Objects.equals(insightStatus(), other.insightStatus())
                && Objects.equals(recommendation(), other.recommendation()) && hasAdditionalInfo() == other.hasAdditionalInfo()
                && Objects.equals(additionalInfo(), other.additionalInfo()) && hasResources() == other.hasResources()
                && Objects.equals(resources(), other.resources())
                && Objects.equals(categorySpecificSummary(), other.categorySpecificSummary());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Insight").add("Id", id()).add("Name", name()).add("Category", categoryAsString())
                .add("KubernetesVersion", kubernetesVersion()).add("LastRefreshTime", lastRefreshTime())
                .add("LastTransitionTime", lastTransitionTime()).add("Description", description())
                .add("InsightStatus", insightStatus()).add("Recommendation", recommendation())
                .add("AdditionalInfo", hasAdditionalInfo() ? additionalInfo() : null)
                .add("Resources", hasResources() ? resources() : null).add("CategorySpecificSummary", categorySpecificSummary())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "category":
            return Optional.ofNullable(clazz.cast(categoryAsString()));
        case "kubernetesVersion":
            return Optional.ofNullable(clazz.cast(kubernetesVersion()));
        case "lastRefreshTime":
            return Optional.ofNullable(clazz.cast(lastRefreshTime()));
        case "lastTransitionTime":
            return Optional.ofNullable(clazz.cast(lastTransitionTime()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "insightStatus":
            return Optional.ofNullable(clazz.cast(insightStatus()));
        case "recommendation":
            return Optional.ofNullable(clazz.cast(recommendation()));
        case "additionalInfo":
            return Optional.ofNullable(clazz.cast(additionalInfo()));
        case "resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "categorySpecificSummary":
            return Optional.ofNullable(clazz.cast(categorySpecificSummary()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("id", ID_FIELD);
        map.put("name", NAME_FIELD);
        map.put("category", CATEGORY_FIELD);
        map.put("kubernetesVersion", KUBERNETES_VERSION_FIELD);
        map.put("lastRefreshTime", LAST_REFRESH_TIME_FIELD);
        map.put("lastTransitionTime", LAST_TRANSITION_TIME_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("insightStatus", INSIGHT_STATUS_FIELD);
        map.put("recommendation", RECOMMENDATION_FIELD);
        map.put("additionalInfo", ADDITIONAL_INFO_FIELD);
        map.put("resources", RESOURCES_FIELD);
        map.put("categorySpecificSummary", CATEGORY_SPECIFIC_SUMMARY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Insight, T> g) {
        return obj -> g.apply((Insight) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Insight> {
        /**
         * <p>
         * The ID of the insight.
         * </p>
         * 
         * @param id
         *        The ID of the insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the insight.
         * </p>
         * 
         * @param name
         *        The name of the insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The category of the insight.
         * </p>
         * 
         * @param category
         *        The category of the insight.
         * @see Category
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Category
         */
        Builder category(String category);

        /**
         * <p>
         * The category of the insight.
         * </p>
         * 
         * @param category
         *        The category of the insight.
         * @see Category
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Category
         */
        Builder category(Category category);

        /**
         * <p>
         * The Kubernetes minor version associated with an insight if applicable.
         * </p>
         * 
         * @param kubernetesVersion
         *        The Kubernetes minor version associated with an insight if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kubernetesVersion(String kubernetesVersion);

        /**
         * <p>
         * The time Amazon EKS last successfully completed a refresh of this insight check on the cluster.
         * </p>
         * 
         * @param lastRefreshTime
         *        The time Amazon EKS last successfully completed a refresh of this insight check on the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRefreshTime(Instant lastRefreshTime);

        /**
         * <p>
         * The time the status of the insight last changed.
         * </p>
         * 
         * @param lastTransitionTime
         *        The time the status of the insight last changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastTransitionTime(Instant lastTransitionTime);

        /**
         * <p>
         * The description of the insight which includes alert criteria, remediation recommendation, and additional
         * resources (contains Markdown).
         * </p>
         * 
         * @param description
         *        The description of the insight which includes alert criteria, remediation recommendation, and
         *        additional resources (contains Markdown).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * An object containing more detail on the status of the insight resource.
         * </p>
         * 
         * @param insightStatus
         *        An object containing more detail on the status of the insight resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder insightStatus(InsightStatus insightStatus);

        /**
         * <p>
         * An object containing more detail on the status of the insight resource.
         * </p>
         * This is a convenience method that creates an instance of the {@link InsightStatus.Builder} avoiding the need
         * to create one manually via {@link InsightStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InsightStatus.Builder#build()} is called immediately and its
         * result is passed to {@link #insightStatus(InsightStatus)}.
         * 
         * @param insightStatus
         *        a consumer that will call methods on {@link InsightStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #insightStatus(InsightStatus)
         */
        default Builder insightStatus(Consumer<InsightStatus.Builder> insightStatus) {
            return insightStatus(InsightStatus.builder().applyMutation(insightStatus).build());
        }

        /**
         * <p>
         * A summary of how to remediate the finding of this insight if applicable.
         * </p>
         * 
         * @param recommendation
         *        A summary of how to remediate the finding of this insight if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendation(String recommendation);

        /**
         * <p>
         * Links to sources that provide additional context on the insight.
         * </p>
         * 
         * @param additionalInfo
         *        Links to sources that provide additional context on the insight.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalInfo(Map<String, String> additionalInfo);

        /**
         * <p>
         * The details about each resource listed in the insight check result.
         * </p>
         * 
         * @param resources
         *        The details about each resource listed in the insight check result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<InsightResourceDetail> resources);

        /**
         * <p>
         * The details about each resource listed in the insight check result.
         * </p>
         * 
         * @param resources
         *        The details about each resource listed in the insight check result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(InsightResourceDetail... resources);

        /**
         * <p>
         * The details about each resource listed in the insight check result.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.eks.model.InsightResourceDetail.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.eks.model.InsightResourceDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.eks.model.InsightResourceDetail.Builder#build()} is called immediately
         * and its result is passed to {@link #resources(List<InsightResourceDetail>)}.
         * 
         * @param resources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.eks.model.InsightResourceDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resources(java.util.Collection<InsightResourceDetail>)
         */
        Builder resources(Consumer<InsightResourceDetail.Builder>... resources);

        /**
         * <p>
         * Summary information that relates to the category of the insight. Currently only returned with certain
         * insights having category <code>UPGRADE_READINESS</code>.
         * </p>
         * 
         * @param categorySpecificSummary
         *        Summary information that relates to the category of the insight. Currently only returned with certain
         *        insights having category <code>UPGRADE_READINESS</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder categorySpecificSummary(InsightCategorySpecificSummary categorySpecificSummary);

        /**
         * <p>
         * Summary information that relates to the category of the insight. Currently only returned with certain
         * insights having category <code>UPGRADE_READINESS</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link InsightCategorySpecificSummary.Builder}
         * avoiding the need to create one manually via {@link InsightCategorySpecificSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InsightCategorySpecificSummary.Builder#build()} is called
         * immediately and its result is passed to {@link #categorySpecificSummary(InsightCategorySpecificSummary)}.
         * 
         * @param categorySpecificSummary
         *        a consumer that will call methods on {@link InsightCategorySpecificSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #categorySpecificSummary(InsightCategorySpecificSummary)
         */
        default Builder categorySpecificSummary(Consumer<InsightCategorySpecificSummary.Builder> categorySpecificSummary) {
            return categorySpecificSummary(InsightCategorySpecificSummary.builder().applyMutation(categorySpecificSummary)
                    .build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String name;

        private String category;

        private String kubernetesVersion;

        private Instant lastRefreshTime;

        private Instant lastTransitionTime;

        private String description;

        private InsightStatus insightStatus;

        private String recommendation;

        private Map<String, String> additionalInfo = DefaultSdkAutoConstructMap.getInstance();

        private List<InsightResourceDetail> resources = DefaultSdkAutoConstructList.getInstance();

        private InsightCategorySpecificSummary categorySpecificSummary;

        private BuilderImpl() {
        }

        private BuilderImpl(Insight model) {
            id(model.id);
            name(model.name);
            category(model.category);
            kubernetesVersion(model.kubernetesVersion);
            lastRefreshTime(model.lastRefreshTime);
            lastTransitionTime(model.lastTransitionTime);
            description(model.description);
            insightStatus(model.insightStatus);
            recommendation(model.recommendation);
            additionalInfo(model.additionalInfo);
            resources(model.resources);
            categorySpecificSummary(model.categorySpecificSummary);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getCategory() {
            return category;
        }

        public final void setCategory(String category) {
            this.category = category;
        }

        @Override
        public final Builder category(String category) {
            this.category = category;
            return this;
        }

        @Override
        public final Builder category(Category category) {
            this.category(category == null ? null : category.toString());
            return this;
        }

        public final String getKubernetesVersion() {
            return kubernetesVersion;
        }

        public final void setKubernetesVersion(String kubernetesVersion) {
            this.kubernetesVersion = kubernetesVersion;
        }

        @Override
        public final Builder kubernetesVersion(String kubernetesVersion) {
            this.kubernetesVersion = kubernetesVersion;
            return this;
        }

        public final Instant getLastRefreshTime() {
            return lastRefreshTime;
        }

        public final void setLastRefreshTime(Instant lastRefreshTime) {
            this.lastRefreshTime = lastRefreshTime;
        }

        @Override
        public final Builder lastRefreshTime(Instant lastRefreshTime) {
            this.lastRefreshTime = lastRefreshTime;
            return this;
        }

        public final Instant getLastTransitionTime() {
            return lastTransitionTime;
        }

        public final void setLastTransitionTime(Instant lastTransitionTime) {
            this.lastTransitionTime = lastTransitionTime;
        }

        @Override
        public final Builder lastTransitionTime(Instant lastTransitionTime) {
            this.lastTransitionTime = lastTransitionTime;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final InsightStatus.Builder getInsightStatus() {
            return insightStatus != null ? insightStatus.toBuilder() : null;
        }

        public final void setInsightStatus(InsightStatus.BuilderImpl insightStatus) {
            this.insightStatus = insightStatus != null ? insightStatus.build() : null;
        }

        @Override
        public final Builder insightStatus(InsightStatus insightStatus) {
            this.insightStatus = insightStatus;
            return this;
        }

        public final String getRecommendation() {
            return recommendation;
        }

        public final void setRecommendation(String recommendation) {
            this.recommendation = recommendation;
        }

        @Override
        public final Builder recommendation(String recommendation) {
            this.recommendation = recommendation;
            return this;
        }

        public final Map<String, String> getAdditionalInfo() {
            if (additionalInfo instanceof SdkAutoConstructMap) {
                return null;
            }
            return additionalInfo;
        }

        public final void setAdditionalInfo(Map<String, String> additionalInfo) {
            this.additionalInfo = AdditionalInfoMapCopier.copy(additionalInfo);
        }

        @Override
        public final Builder additionalInfo(Map<String, String> additionalInfo) {
            this.additionalInfo = AdditionalInfoMapCopier.copy(additionalInfo);
            return this;
        }

        public final List<InsightResourceDetail.Builder> getResources() {
            List<InsightResourceDetail.Builder> result = InsightResourceDetailsCopier.copyToBuilder(this.resources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResources(Collection<InsightResourceDetail.BuilderImpl> resources) {
            this.resources = InsightResourceDetailsCopier.copyFromBuilder(resources);
        }

        @Override
        public final Builder resources(Collection<InsightResourceDetail> resources) {
            this.resources = InsightResourceDetailsCopier.copy(resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(InsightResourceDetail... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(Consumer<InsightResourceDetail.Builder>... resources) {
            resources(Stream.of(resources).map(c -> InsightResourceDetail.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final InsightCategorySpecificSummary.Builder getCategorySpecificSummary() {
            return categorySpecificSummary != null ? categorySpecificSummary.toBuilder() : null;
        }

        public final void setCategorySpecificSummary(InsightCategorySpecificSummary.BuilderImpl categorySpecificSummary) {
            this.categorySpecificSummary = categorySpecificSummary != null ? categorySpecificSummary.build() : null;
        }

        @Override
        public final Builder categorySpecificSummary(InsightCategorySpecificSummary categorySpecificSummary) {
            this.categorySpecificSummary = categorySpecificSummary;
            return this;
        }

        @Override
        public Insight build() {
            return new Insight(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
