/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A mapping between an Argo CD role and IAM Identity CenterIAM; Identity Center identities. This defines which users or
 * groups have specific permissions in Argo CD.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ArgoCdRoleMapping implements SdkPojo, Serializable,
        ToCopyableBuilder<ArgoCdRoleMapping.Builder, ArgoCdRoleMapping> {
    private static final SdkField<String> ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("role")
            .getter(getter(ArgoCdRoleMapping::roleAsString)).setter(setter(Builder::role))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("role").build()).build();

    private static final SdkField<List<SsoIdentity>> IDENTITIES_FIELD = SdkField
            .<List<SsoIdentity>> builder(MarshallingType.LIST)
            .memberName("identities")
            .getter(getter(ArgoCdRoleMapping::identities))
            .setter(setter(Builder::identities))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("identities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SsoIdentity> builder(MarshallingType.SDK_POJO)
                                            .constructor(SsoIdentity::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ROLE_FIELD, IDENTITIES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String role;

    private final List<SsoIdentity> identities;

    private ArgoCdRoleMapping(BuilderImpl builder) {
        this.role = builder.role;
        this.identities = builder.identities;
    }

    /**
     * <p>
     * The Argo CD role to assign. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ADMIN</code> – Full administrative access to Argo CD.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EDITOR</code> – Edit access to Argo CD resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VIEWER</code> – Read-only access to Argo CD resources.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #role} will return
     * {@link ArgoCdRole#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #roleAsString}.
     * </p>
     * 
     * @return The Argo CD role to assign. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ADMIN</code> – Full administrative access to Argo CD.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EDITOR</code> – Edit access to Argo CD resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VIEWER</code> – Read-only access to Argo CD resources.
     *         </p>
     *         </li>
     * @see ArgoCdRole
     */
    public final ArgoCdRole role() {
        return ArgoCdRole.fromValue(role);
    }

    /**
     * <p>
     * The Argo CD role to assign. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ADMIN</code> – Full administrative access to Argo CD.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EDITOR</code> – Edit access to Argo CD resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VIEWER</code> – Read-only access to Argo CD resources.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #role} will return
     * {@link ArgoCdRole#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #roleAsString}.
     * </p>
     * 
     * @return The Argo CD role to assign. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ADMIN</code> – Full administrative access to Argo CD.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EDITOR</code> – Edit access to Argo CD resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VIEWER</code> – Read-only access to Argo CD resources.
     *         </p>
     *         </li>
     * @see ArgoCdRole
     */
    public final String roleAsString() {
        return role;
    }

    /**
     * For responses, this returns true if the service returned a value for the Identities property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasIdentities() {
        return identities != null && !(identities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of IAM Identity CenterIAM; Identity Center identities (users or groups) that should be assigned this Argo
     * CD role.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIdentities} method.
     * </p>
     * 
     * @return A list of IAM Identity CenterIAM; Identity Center identities (users or groups) that should be assigned
     *         this Argo CD role.
     */
    public final List<SsoIdentity> identities() {
        return identities;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(roleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasIdentities() ? identities() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ArgoCdRoleMapping)) {
            return false;
        }
        ArgoCdRoleMapping other = (ArgoCdRoleMapping) obj;
        return Objects.equals(roleAsString(), other.roleAsString()) && hasIdentities() == other.hasIdentities()
                && Objects.equals(identities(), other.identities());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ArgoCdRoleMapping").add("Role", roleAsString())
                .add("Identities", hasIdentities() ? identities() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "role":
            return Optional.ofNullable(clazz.cast(roleAsString()));
        case "identities":
            return Optional.ofNullable(clazz.cast(identities()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("role", ROLE_FIELD);
        map.put("identities", IDENTITIES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ArgoCdRoleMapping, T> g) {
        return obj -> g.apply((ArgoCdRoleMapping) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ArgoCdRoleMapping> {
        /**
         * <p>
         * The Argo CD role to assign. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ADMIN</code> – Full administrative access to Argo CD.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EDITOR</code> – Edit access to Argo CD resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VIEWER</code> – Read-only access to Argo CD resources.
         * </p>
         * </li>
         * </ul>
         * 
         * @param role
         *        The Argo CD role to assign. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ADMIN</code> – Full administrative access to Argo CD.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EDITOR</code> – Edit access to Argo CD resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VIEWER</code> – Read-only access to Argo CD resources.
         *        </p>
         *        </li>
         * @see ArgoCdRole
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ArgoCdRole
         */
        Builder role(String role);

        /**
         * <p>
         * The Argo CD role to assign. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ADMIN</code> – Full administrative access to Argo CD.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EDITOR</code> – Edit access to Argo CD resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VIEWER</code> – Read-only access to Argo CD resources.
         * </p>
         * </li>
         * </ul>
         * 
         * @param role
         *        The Argo CD role to assign. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ADMIN</code> – Full administrative access to Argo CD.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EDITOR</code> – Edit access to Argo CD resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VIEWER</code> – Read-only access to Argo CD resources.
         *        </p>
         *        </li>
         * @see ArgoCdRole
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ArgoCdRole
         */
        Builder role(ArgoCdRole role);

        /**
         * <p>
         * A list of IAM Identity CenterIAM; Identity Center identities (users or groups) that should be assigned this
         * Argo CD role.
         * </p>
         * 
         * @param identities
         *        A list of IAM Identity CenterIAM; Identity Center identities (users or groups) that should be assigned
         *        this Argo CD role.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identities(Collection<SsoIdentity> identities);

        /**
         * <p>
         * A list of IAM Identity CenterIAM; Identity Center identities (users or groups) that should be assigned this
         * Argo CD role.
         * </p>
         * 
         * @param identities
         *        A list of IAM Identity CenterIAM; Identity Center identities (users or groups) that should be assigned
         *        this Argo CD role.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identities(SsoIdentity... identities);

        /**
         * <p>
         * A list of IAM Identity CenterIAM; Identity Center identities (users or groups) that should be assigned this
         * Argo CD role.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.eks.model.SsoIdentity.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.eks.model.SsoIdentity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.eks.model.SsoIdentity.Builder#build()} is called immediately and its
         * result is passed to {@link #identities(List<SsoIdentity>)}.
         * 
         * @param identities
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.eks.model.SsoIdentity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #identities(java.util.Collection<SsoIdentity>)
         */
        Builder identities(Consumer<SsoIdentity.Builder>... identities);
    }

    static final class BuilderImpl implements Builder {
        private String role;

        private List<SsoIdentity> identities = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ArgoCdRoleMapping model) {
            role(model.role);
            identities(model.identities);
        }

        public final String getRole() {
            return role;
        }

        public final void setRole(String role) {
            this.role = role;
        }

        @Override
        public final Builder role(String role) {
            this.role = role;
            return this;
        }

        @Override
        public final Builder role(ArgoCdRole role) {
            this.role(role == null ? null : role.toString());
            return this;
        }

        public final List<SsoIdentity.Builder> getIdentities() {
            List<SsoIdentity.Builder> result = SsoIdentityListCopier.copyToBuilder(this.identities);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setIdentities(Collection<SsoIdentity.BuilderImpl> identities) {
            this.identities = SsoIdentityListCopier.copyFromBuilder(identities);
        }

        @Override
        public final Builder identities(Collection<SsoIdentity> identities) {
            this.identities = SsoIdentityListCopier.copy(identities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder identities(SsoIdentity... identities) {
            identities(Arrays.asList(identities));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder identities(Consumer<SsoIdentity.Builder>... identities) {
            identities(Stream.of(identities).map(c -> SsoIdentity.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ArgoCdRoleMapping build() {
            return new ArgoCdRoleMapping(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
