/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.services.eks.model.CreateClusterRequest;
import software.amazon.awssdk.services.eks.model.CreateClusterResponse;
import software.amazon.awssdk.services.eks.model.DeleteClusterRequest;
import software.amazon.awssdk.services.eks.model.DeleteClusterResponse;
import software.amazon.awssdk.services.eks.model.DescribeClusterRequest;
import software.amazon.awssdk.services.eks.model.DescribeClusterResponse;
import software.amazon.awssdk.services.eks.model.DescribeUpdateRequest;
import software.amazon.awssdk.services.eks.model.DescribeUpdateResponse;
import software.amazon.awssdk.services.eks.model.ListClustersRequest;
import software.amazon.awssdk.services.eks.model.ListClustersResponse;
import software.amazon.awssdk.services.eks.model.ListUpdatesRequest;
import software.amazon.awssdk.services.eks.model.ListUpdatesResponse;
import software.amazon.awssdk.services.eks.model.UpdateClusterConfigRequest;
import software.amazon.awssdk.services.eks.model.UpdateClusterConfigResponse;
import software.amazon.awssdk.services.eks.model.UpdateClusterVersionRequest;
import software.amazon.awssdk.services.eks.model.UpdateClusterVersionResponse;

/**
 * Service client for accessing Amazon EKS asynchronously. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * Amazon Elastic Kubernetes Service (Amazon EKS) is a managed service that makes it easy for you to run Kubernetes on
 * AWS without needing to stand up or maintain your own Kubernetes control plane. Kubernetes is an open-source system
 * for automating the deployment, scaling, and management of containerized applications.
 * </p>
 * <p>
 * Amazon EKS runs up-to-date versions of the open-source Kubernetes software, so you can use all the existing plugins
 * and tooling from the Kubernetes community. Applications running on Amazon EKS are fully compatible with applications
 * running on any standard Kubernetes environment, whether running in on-premises data centers or public clouds. This
 * means that you can easily migrate any standard Kubernetes application to Amazon EKS without any code modification
 * required.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface EksAsyncClient extends SdkClient {
    String SERVICE_NAME = "eks";

    /**
     * Create a {@link EksAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static EksAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link EksAsyncClient}.
     */
    static EksAsyncClientBuilder builder() {
        return new DefaultEksAsyncClientBuilder();
    }

    /**
     * <p>
     * Creates an Amazon EKS control plane.
     * </p>
     * <p>
     * The Amazon EKS control plane consists of control plane instances that run the Kubernetes software, such as
     * <code>etcd</code> and the API server. The control plane runs in an account managed by AWS, and the Kubernetes API
     * is exposed via the Amazon EKS API server endpoint. Each Amazon EKS cluster control plane is single-tenant and
     * unique and runs on its own set of Amazon EC2 instances.
     * </p>
     * <p>
     * The cluster control plane is provisioned across multiple Availability Zones and fronted by an Elastic Load
     * Balancing Network Load Balancer. Amazon EKS also provisions elastic network interfaces in your VPC subnets to
     * provide connectivity from the control plane instances to the worker nodes (for example, to support
     * <code>kubectl exec</code>, <code>logs</code>, and <code>proxy</code> data flows).
     * </p>
     * <p>
     * Amazon EKS worker nodes run in your AWS account and connect to your cluster's control plane via the Kubernetes
     * API server endpoint and a certificate file that is created for your cluster.
     * </p>
     * <p>
     * You can use the <code>endpointPublicAccess</code> and <code>endpointPrivateAccess</code> parameters to enable or
     * disable public and private access to your cluster's Kubernetes API server endpoint. By default, public access is
     * enabled, and private access is disabled. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html">Amazon EKS Cluster Endpoint Access
     * Control</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <p>
     * You can use the <code>logging</code> parameter to enable or disable exporting the Kubernetes control plane logs
     * for your cluster to CloudWatch Logs. By default, cluster control plane logs aren't exported to CloudWatch Logs.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html">Amazon EKS Cluster Control Plane
     * Logs</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <note>
     * <p>
     * CloudWatch Logs ingestion, archive storage, and data scanning rates apply to exported control plane logs. For
     * more information, see <a href="http://aws.amazon.com/cloudwatch/pricing/">Amazon CloudWatch Pricing</a>.
     * </p>
     * </note>
     * <p>
     * Cluster creation typically takes between 10 and 15 minutes. After you create an Amazon EKS cluster, you must
     * configure your Kubernetes tooling to communicate with the API server and launch worker nodes into your cluster.
     * For more information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/managing-auth.html">Managing
     * Cluster Authentication</a> and <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html">Launching Amazon EKS Worker Nodes</a>
     * in the <i>Amazon EKS User Guide</i>.
     * </p>
     *
     * @param createClusterRequest
     * @return A Java Future containing the result of the CreateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceLimitExceededException You have encountered a service limit on the specified resource.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>UnsupportedAvailabilityZoneException At least one of your specified cluster subnets is in an
     *         Availability Zone that does not support Amazon EKS. The exception output specifies the supported
     *         Availability Zones for your account, from which you can choose subnets for your cluster.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.CreateCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateClusterResponse> createCluster(CreateClusterRequest createClusterRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an Amazon EKS control plane.
     * </p>
     * <p>
     * The Amazon EKS control plane consists of control plane instances that run the Kubernetes software, such as
     * <code>etcd</code> and the API server. The control plane runs in an account managed by AWS, and the Kubernetes API
     * is exposed via the Amazon EKS API server endpoint. Each Amazon EKS cluster control plane is single-tenant and
     * unique and runs on its own set of Amazon EC2 instances.
     * </p>
     * <p>
     * The cluster control plane is provisioned across multiple Availability Zones and fronted by an Elastic Load
     * Balancing Network Load Balancer. Amazon EKS also provisions elastic network interfaces in your VPC subnets to
     * provide connectivity from the control plane instances to the worker nodes (for example, to support
     * <code>kubectl exec</code>, <code>logs</code>, and <code>proxy</code> data flows).
     * </p>
     * <p>
     * Amazon EKS worker nodes run in your AWS account and connect to your cluster's control plane via the Kubernetes
     * API server endpoint and a certificate file that is created for your cluster.
     * </p>
     * <p>
     * You can use the <code>endpointPublicAccess</code> and <code>endpointPrivateAccess</code> parameters to enable or
     * disable public and private access to your cluster's Kubernetes API server endpoint. By default, public access is
     * enabled, and private access is disabled. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html">Amazon EKS Cluster Endpoint Access
     * Control</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <p>
     * You can use the <code>logging</code> parameter to enable or disable exporting the Kubernetes control plane logs
     * for your cluster to CloudWatch Logs. By default, cluster control plane logs aren't exported to CloudWatch Logs.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html">Amazon EKS Cluster Control Plane
     * Logs</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <note>
     * <p>
     * CloudWatch Logs ingestion, archive storage, and data scanning rates apply to exported control plane logs. For
     * more information, see <a href="http://aws.amazon.com/cloudwatch/pricing/">Amazon CloudWatch Pricing</a>.
     * </p>
     * </note>
     * <p>
     * Cluster creation typically takes between 10 and 15 minutes. After you create an Amazon EKS cluster, you must
     * configure your Kubernetes tooling to communicate with the API server and launch worker nodes into your cluster.
     * For more information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/managing-auth.html">Managing
     * Cluster Authentication</a> and <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html">Launching Amazon EKS Worker Nodes</a>
     * in the <i>Amazon EKS User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateClusterRequest.Builder} avoiding the need to
     * create one manually via {@link CreateClusterRequest#builder()}
     * </p>
     *
     * @param createClusterRequest
     *        A {@link Consumer} that will call methods on {@link CreateClusterRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceLimitExceededException You have encountered a service limit on the specified resource.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>UnsupportedAvailabilityZoneException At least one of your specified cluster subnets is in an
     *         Availability Zone that does not support Amazon EKS. The exception output specifies the supported
     *         Availability Zones for your account, from which you can choose subnets for your cluster.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.CreateCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateClusterResponse> createCluster(Consumer<CreateClusterRequest.Builder> createClusterRequest) {
        return createCluster(CreateClusterRequest.builder().applyMutation(createClusterRequest).build());
    }

    /**
     * <p>
     * Deletes the Amazon EKS cluster control plane.
     * </p>
     * <note>
     * <p>
     * If you have active services in your cluster that are associated with a load balancer, you must delete those
     * services before deleting the cluster so that the load balancers are deleted properly. Otherwise, you can have
     * orphaned resources in your VPC that prevent you from being able to delete the VPC. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/delete-cluster.html">Deleting a Cluster</a> in the
     * <i>Amazon EKS User Guide</i>.
     * </p>
     * </note>
     *
     * @param deleteClusterRequest
     * @return A Java Future containing the result of the DeleteCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <a>ListClusters</a>. Amazon EKS clusters are Region-specific.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DeleteCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteClusterResponse> deleteCluster(DeleteClusterRequest deleteClusterRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the Amazon EKS cluster control plane.
     * </p>
     * <note>
     * <p>
     * If you have active services in your cluster that are associated with a load balancer, you must delete those
     * services before deleting the cluster so that the load balancers are deleted properly. Otherwise, you can have
     * orphaned resources in your VPC that prevent you from being able to delete the VPC. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/delete-cluster.html">Deleting a Cluster</a> in the
     * <i>Amazon EKS User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteClusterRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteClusterRequest#builder()}
     * </p>
     *
     * @param deleteClusterRequest
     *        A {@link Consumer} that will call methods on {@link DeleteClusterRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <a>ListClusters</a>. Amazon EKS clusters are Region-specific.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DeleteCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteClusterResponse> deleteCluster(Consumer<DeleteClusterRequest.Builder> deleteClusterRequest) {
        return deleteCluster(DeleteClusterRequest.builder().applyMutation(deleteClusterRequest).build());
    }

    /**
     * <p>
     * Returns descriptive information about an Amazon EKS cluster.
     * </p>
     * <p>
     * The API server endpoint and certificate authority data returned by this operation are required for
     * <code>kubelet</code> and <code>kubectl</code> to communicate with your Kubernetes API server. For more
     * information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/create-kubeconfig.html">Create a
     * kubeconfig for Amazon EKS</a>.
     * </p>
     * <note>
     * <p>
     * The API server endpoint and certificate authority data aren't available until the cluster reaches the
     * <code>ACTIVE</code> state.
     * </p>
     * </note>
     *
     * @param describeClusterRequest
     * @return A Java Future containing the result of the DescribeCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <a>ListClusters</a>. Amazon EKS clusters are Region-specific.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeClusterResponse> describeCluster(DescribeClusterRequest describeClusterRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns descriptive information about an Amazon EKS cluster.
     * </p>
     * <p>
     * The API server endpoint and certificate authority data returned by this operation are required for
     * <code>kubelet</code> and <code>kubectl</code> to communicate with your Kubernetes API server. For more
     * information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/create-kubeconfig.html">Create a
     * kubeconfig for Amazon EKS</a>.
     * </p>
     * <note>
     * <p>
     * The API server endpoint and certificate authority data aren't available until the cluster reaches the
     * <code>ACTIVE</code> state.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeClusterRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeClusterRequest#builder()}
     * </p>
     *
     * @param describeClusterRequest
     *        A {@link Consumer} that will call methods on {@link DescribeClusterRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DescribeCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <a>ListClusters</a>. Amazon EKS clusters are Region-specific.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeClusterResponse> describeCluster(
            Consumer<DescribeClusterRequest.Builder> describeClusterRequest) {
        return describeCluster(DescribeClusterRequest.builder().applyMutation(describeClusterRequest).build());
    }

    /**
     * <p>
     * Returns descriptive information about an update against your Amazon EKS cluster.
     * </p>
     * <p>
     * When the status of the update is <code>Succeeded</code>, the update is complete. If an update fails, the status
     * is <code>Failed</code>, and an error detail explains the reason for the failure.
     * </p>
     *
     * @param describeUpdateRequest
     * @return A Java Future containing the result of the DescribeUpdate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <a>ListClusters</a>. Amazon EKS clusters are Region-specific.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeUpdate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeUpdate" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeUpdateResponse> describeUpdate(DescribeUpdateRequest describeUpdateRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns descriptive information about an update against your Amazon EKS cluster.
     * </p>
     * <p>
     * When the status of the update is <code>Succeeded</code>, the update is complete. If an update fails, the status
     * is <code>Failed</code>, and an error detail explains the reason for the failure.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeUpdateRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeUpdateRequest#builder()}
     * </p>
     *
     * @param describeUpdateRequest
     *        A {@link Consumer} that will call methods on {@link DescribeUpdateRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DescribeUpdate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <a>ListClusters</a>. Amazon EKS clusters are Region-specific.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.DescribeUpdate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/DescribeUpdate" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeUpdateResponse> describeUpdate(Consumer<DescribeUpdateRequest.Builder> describeUpdateRequest) {
        return describeUpdate(DescribeUpdateRequest.builder().applyMutation(describeUpdateRequest).build());
    }

    /**
     * <p>
     * Lists the Amazon EKS clusters in your AWS account in the specified Region.
     * </p>
     *
     * @param listClustersRequest
     * @return A Java Future containing the result of the ListClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListClusters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListClustersResponse> listClusters(ListClustersRequest listClustersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the Amazon EKS clusters in your AWS account in the specified Region.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListClustersRequest.Builder} avoiding the need to
     * create one manually via {@link ListClustersRequest#builder()}
     * </p>
     *
     * @param listClustersRequest
     *        A {@link Consumer} that will call methods on {@link ListClustersRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListClusters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListClustersResponse> listClusters(Consumer<ListClustersRequest.Builder> listClustersRequest) {
        return listClusters(ListClustersRequest.builder().applyMutation(listClustersRequest).build());
    }

    /**
     * <p>
     * Lists the Amazon EKS clusters in your AWS account in the specified Region.
     * </p>
     *
     * @return A Java Future containing the result of the ListClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Back off and retry the operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListClusters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListClustersResponse> listClusters() {
        return listClusters(ListClustersRequest.builder().build());
    }

    /**
     * <p>
     * Lists the updates associated with an Amazon EKS cluster in your AWS account, in the specified Region.
     * </p>
     *
     * @param listUpdatesRequest
     * @return A Java Future containing the result of the ListUpdates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <a>ListClusters</a>. Amazon EKS clusters are Region-specific.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListUpdates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListUpdates" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListUpdatesResponse> listUpdates(ListUpdatesRequest listUpdatesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the updates associated with an Amazon EKS cluster in your AWS account, in the specified Region.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListUpdatesRequest.Builder} avoiding the need to
     * create one manually via {@link ListUpdatesRequest#builder()}
     * </p>
     *
     * @param listUpdatesRequest
     *        A {@link Consumer} that will call methods on {@link ListUpdatesRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListUpdates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <a>ListClusters</a>. Amazon EKS clusters are Region-specific.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.ListUpdates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/ListUpdates" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListUpdatesResponse> listUpdates(Consumer<ListUpdatesRequest.Builder> listUpdatesRequest) {
        return listUpdates(ListUpdatesRequest.builder().applyMutation(listUpdatesRequest).build());
    }

    /**
     * <p>
     * Updates an Amazon EKS cluster configuration. Your cluster continues to function during the update. The response
     * output includes an update ID that you can use to track the status of your cluster update with the
     * <a>DescribeUpdate</a> API operation.
     * </p>
     * <p>
     * You can use this API operation to enable or disable exporting the Kubernetes control plane logs for your cluster
     * to CloudWatch Logs. By default, cluster control plane logs aren't exported to CloudWatch Logs. For more
     * information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html">Amazon EKS
     * Cluster Control Plane Logs</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <note>
     * <p>
     * CloudWatch Logs ingestion, archive storage, and data scanning rates apply to exported control plane logs. For
     * more information, see <a href="http://aws.amazon.com/cloudwatch/pricing/">Amazon CloudWatch Pricing</a>.
     * </p>
     * </note>
     * <p>
     * You can also use this API operation to enable or disable public and private access to your cluster's Kubernetes
     * API server endpoint. By default, public access is enabled, and private access is disabled. For more information,
     * see <a href="https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html">Amazon EKS Cluster Endpoint
     * Access Control</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <important>
     * <p>
     * At this time, you can not update the subnets or security group IDs for an existing cluster.
     * </p>
     * </important>
     * <p>
     * Cluster updates are asynchronous, and they should finish within a few minutes. During an update, the cluster
     * status moves to <code>UPDATING</code> (this status transition is eventually consistent). When the update is
     * complete (either <code>Failed</code> or <code>Successful</code>), the cluster status moves to <code>Active</code>
     * .
     * </p>
     *
     * @param updateClusterConfigRequest
     * @return A Java Future containing the result of the UpdateClusterConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <a>ListClusters</a>. Amazon EKS clusters are Region-specific.</li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.UpdateClusterConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateClusterConfig" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateClusterConfigResponse> updateClusterConfig(
            UpdateClusterConfigRequest updateClusterConfigRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates an Amazon EKS cluster configuration. Your cluster continues to function during the update. The response
     * output includes an update ID that you can use to track the status of your cluster update with the
     * <a>DescribeUpdate</a> API operation.
     * </p>
     * <p>
     * You can use this API operation to enable or disable exporting the Kubernetes control plane logs for your cluster
     * to CloudWatch Logs. By default, cluster control plane logs aren't exported to CloudWatch Logs. For more
     * information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/control-plane-logs.html">Amazon EKS
     * Cluster Control Plane Logs</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <note>
     * <p>
     * CloudWatch Logs ingestion, archive storage, and data scanning rates apply to exported control plane logs. For
     * more information, see <a href="http://aws.amazon.com/cloudwatch/pricing/">Amazon CloudWatch Pricing</a>.
     * </p>
     * </note>
     * <p>
     * You can also use this API operation to enable or disable public and private access to your cluster's Kubernetes
     * API server endpoint. By default, public access is enabled, and private access is disabled. For more information,
     * see <a href="https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html">Amazon EKS Cluster Endpoint
     * Access Control</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * <important>
     * <p>
     * At this time, you can not update the subnets or security group IDs for an existing cluster.
     * </p>
     * </important>
     * <p>
     * Cluster updates are asynchronous, and they should finish within a few minutes. During an update, the cluster
     * status moves to <code>UPDATING</code> (this status transition is eventually consistent). When the update is
     * complete (either <code>Failed</code> or <code>Successful</code>), the cluster status moves to <code>Active</code>
     * .
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateClusterConfigRequest.Builder} avoiding the
     * need to create one manually via {@link UpdateClusterConfigRequest#builder()}
     * </p>
     *
     * @param updateClusterConfigRequest
     *        A {@link Consumer} that will call methods on {@link UpdateClusterConfigRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the UpdateClusterConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <a>ListClusters</a>. Amazon EKS clusters are Region-specific.</li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.UpdateClusterConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateClusterConfig" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateClusterConfigResponse> updateClusterConfig(
            Consumer<UpdateClusterConfigRequest.Builder> updateClusterConfigRequest) {
        return updateClusterConfig(UpdateClusterConfigRequest.builder().applyMutation(updateClusterConfigRequest).build());
    }

    /**
     * <p>
     * Updates an Amazon EKS cluster to the specified Kubernetes version. Your cluster continues to function during the
     * update. The response output includes an update ID that you can use to track the status of your cluster update
     * with the <a>DescribeUpdate</a> API operation.
     * </p>
     * <p>
     * Cluster updates are asynchronous, and they should finish within a few minutes. During an update, the cluster
     * status moves to <code>UPDATING</code> (this status transition is eventually consistent). When the update is
     * complete (either <code>Failed</code> or <code>Successful</code>), the cluster status moves to <code>Active</code>
     * .
     * </p>
     *
     * @param updateClusterVersionRequest
     * @return A Java Future containing the result of the UpdateClusterVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <a>ListClusters</a>. Amazon EKS clusters are Region-specific.</li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.UpdateClusterVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateClusterVersion" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateClusterVersionResponse> updateClusterVersion(
            UpdateClusterVersionRequest updateClusterVersionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates an Amazon EKS cluster to the specified Kubernetes version. Your cluster continues to function during the
     * update. The response output includes an update ID that you can use to track the status of your cluster update
     * with the <a>DescribeUpdate</a> API operation.
     * </p>
     * <p>
     * Cluster updates are asynchronous, and they should finish within a few minutes. During an update, the cluster
     * status moves to <code>UPDATING</code> (this status transition is eventually consistent). When the update is
     * complete (either <code>Failed</code> or <code>Successful</code>), the cluster status moves to <code>Active</code>
     * .
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateClusterVersionRequest.Builder} avoiding the
     * need to create one manually via {@link UpdateClusterVersionRequest#builder()}
     * </p>
     *
     * @param updateClusterVersionRequest
     *        A {@link Consumer} that will call methods on {@link UpdateClusterVersionRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the UpdateClusterVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ClientException These errors are usually caused by a client action. Actions can include using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource or
     *         specifying an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ResourceInUseException The specified resource is in use.</li>
     *         <li>ResourceNotFoundException The specified resource could not be found. You can view your available
     *         clusters with <a>ListClusters</a>. Amazon EKS clusters are Region-specific.</li>
     *         <li>InvalidRequestException The request is invalid given the state of the cluster. Check the state of the
     *         cluster and the associated operations.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EksException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EksAsyncClient.UpdateClusterVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/eks-2017-11-01/UpdateClusterVersion" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateClusterVersionResponse> updateClusterVersion(
            Consumer<UpdateClusterVersionRequest.Builder> updateClusterVersionRequest) {
        return updateClusterVersion(UpdateClusterVersionRequest.builder().applyMutation(updateClusterVersionRequest).build());
    }
}
