/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a collection of cache nodes in a replication group. One node in the node group is the read/write primary
 * node. All the other nodes are read-only Replica nodes.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NodeGroup implements SdkPojo, Serializable, ToCopyableBuilder<NodeGroup.Builder, NodeGroup> {
    private static final SdkField<String> NODE_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NodeGroupId").getter(getter(NodeGroup::nodeGroupId)).setter(setter(Builder::nodeGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeGroupId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(NodeGroup::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Endpoint> PRIMARY_ENDPOINT_FIELD = SdkField.<Endpoint> builder(MarshallingType.SDK_POJO)
            .memberName("PrimaryEndpoint").getter(getter(NodeGroup::primaryEndpoint)).setter(setter(Builder::primaryEndpoint))
            .constructor(Endpoint::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrimaryEndpoint").build()).build();

    private static final SdkField<Endpoint> READER_ENDPOINT_FIELD = SdkField.<Endpoint> builder(MarshallingType.SDK_POJO)
            .memberName("ReaderEndpoint").getter(getter(NodeGroup::readerEndpoint)).setter(setter(Builder::readerEndpoint))
            .constructor(Endpoint::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReaderEndpoint").build()).build();

    private static final SdkField<String> SLOTS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Slots")
            .getter(getter(NodeGroup::slots)).setter(setter(Builder::slots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Slots").build()).build();

    private static final SdkField<List<NodeGroupMember>> NODE_GROUP_MEMBERS_FIELD = SdkField
            .<List<NodeGroupMember>> builder(MarshallingType.LIST)
            .memberName("NodeGroupMembers")
            .getter(getter(NodeGroup::nodeGroupMembers))
            .setter(setter(Builder::nodeGroupMembers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeGroupMembers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("NodeGroupMember")
                            .memberFieldInfo(
                                    SdkField.<NodeGroupMember> builder(MarshallingType.SDK_POJO)
                                            .constructor(NodeGroupMember::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("NodeGroupMember").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NODE_GROUP_ID_FIELD,
            STATUS_FIELD, PRIMARY_ENDPOINT_FIELD, READER_ENDPOINT_FIELD, SLOTS_FIELD, NODE_GROUP_MEMBERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String nodeGroupId;

    private final String status;

    private final Endpoint primaryEndpoint;

    private final Endpoint readerEndpoint;

    private final String slots;

    private final List<NodeGroupMember> nodeGroupMembers;

    private NodeGroup(BuilderImpl builder) {
        this.nodeGroupId = builder.nodeGroupId;
        this.status = builder.status;
        this.primaryEndpoint = builder.primaryEndpoint;
        this.readerEndpoint = builder.readerEndpoint;
        this.slots = builder.slots;
        this.nodeGroupMembers = builder.nodeGroupMembers;
    }

    /**
     * <p>
     * The identifier for the node group (shard). A Valkey or Redis OSS (cluster mode disabled) replication group
     * contains only 1 node group; therefore, the node group ID is 0001. A Valkey or Redis OSS (cluster mode enabled)
     * replication group contains 1 to 90 node groups numbered 0001 to 0090. Optionally, the user can provide the id for
     * a node group.
     * </p>
     * 
     * @return The identifier for the node group (shard). A Valkey or Redis OSS (cluster mode disabled) replication
     *         group contains only 1 node group; therefore, the node group ID is 0001. A Valkey or Redis OSS (cluster
     *         mode enabled) replication group contains 1 to 90 node groups numbered 0001 to 0090. Optionally, the user
     *         can provide the id for a node group.
     */
    public final String nodeGroupId() {
        return nodeGroupId;
    }

    /**
     * <p>
     * The current state of this replication group - <code>creating</code>, <code>available</code>,
     * <code>modifying</code>, <code>deleting</code>.
     * </p>
     * 
     * @return The current state of this replication group - <code>creating</code>, <code>available</code>,
     *         <code>modifying</code>, <code>deleting</code>.
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * The endpoint of the primary node in this node group (shard).
     * </p>
     * 
     * @return The endpoint of the primary node in this node group (shard).
     */
    public final Endpoint primaryEndpoint() {
        return primaryEndpoint;
    }

    /**
     * <p>
     * The endpoint of the replica nodes in this node group (shard). This value is read-only.
     * </p>
     * 
     * @return The endpoint of the replica nodes in this node group (shard). This value is read-only.
     */
    public final Endpoint readerEndpoint() {
        return readerEndpoint;
    }

    /**
     * <p>
     * The keyspace for this node group (shard).
     * </p>
     * 
     * @return The keyspace for this node group (shard).
     */
    public final String slots() {
        return slots;
    }

    /**
     * For responses, this returns true if the service returned a value for the NodeGroupMembers property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNodeGroupMembers() {
        return nodeGroupMembers != null && !(nodeGroupMembers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list containing information about individual nodes within the node group (shard).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNodeGroupMembers} method.
     * </p>
     * 
     * @return A list containing information about individual nodes within the node group (shard).
     */
    public final List<NodeGroupMember> nodeGroupMembers() {
        return nodeGroupMembers;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nodeGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(primaryEndpoint());
        hashCode = 31 * hashCode + Objects.hashCode(readerEndpoint());
        hashCode = 31 * hashCode + Objects.hashCode(slots());
        hashCode = 31 * hashCode + Objects.hashCode(hasNodeGroupMembers() ? nodeGroupMembers() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NodeGroup)) {
            return false;
        }
        NodeGroup other = (NodeGroup) obj;
        return Objects.equals(nodeGroupId(), other.nodeGroupId()) && Objects.equals(status(), other.status())
                && Objects.equals(primaryEndpoint(), other.primaryEndpoint())
                && Objects.equals(readerEndpoint(), other.readerEndpoint()) && Objects.equals(slots(), other.slots())
                && hasNodeGroupMembers() == other.hasNodeGroupMembers()
                && Objects.equals(nodeGroupMembers(), other.nodeGroupMembers());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NodeGroup").add("NodeGroupId", nodeGroupId()).add("Status", status())
                .add("PrimaryEndpoint", primaryEndpoint()).add("ReaderEndpoint", readerEndpoint()).add("Slots", slots())
                .add("NodeGroupMembers", hasNodeGroupMembers() ? nodeGroupMembers() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NodeGroupId":
            return Optional.ofNullable(clazz.cast(nodeGroupId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "PrimaryEndpoint":
            return Optional.ofNullable(clazz.cast(primaryEndpoint()));
        case "ReaderEndpoint":
            return Optional.ofNullable(clazz.cast(readerEndpoint()));
        case "Slots":
            return Optional.ofNullable(clazz.cast(slots()));
        case "NodeGroupMembers":
            return Optional.ofNullable(clazz.cast(nodeGroupMembers()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NodeGroup, T> g) {
        return obj -> g.apply((NodeGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NodeGroup> {
        /**
         * <p>
         * The identifier for the node group (shard). A Valkey or Redis OSS (cluster mode disabled) replication group
         * contains only 1 node group; therefore, the node group ID is 0001. A Valkey or Redis OSS (cluster mode
         * enabled) replication group contains 1 to 90 node groups numbered 0001 to 0090. Optionally, the user can
         * provide the id for a node group.
         * </p>
         * 
         * @param nodeGroupId
         *        The identifier for the node group (shard). A Valkey or Redis OSS (cluster mode disabled) replication
         *        group contains only 1 node group; therefore, the node group ID is 0001. A Valkey or Redis OSS (cluster
         *        mode enabled) replication group contains 1 to 90 node groups numbered 0001 to 0090. Optionally, the
         *        user can provide the id for a node group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeGroupId(String nodeGroupId);

        /**
         * <p>
         * The current state of this replication group - <code>creating</code>, <code>available</code>,
         * <code>modifying</code>, <code>deleting</code>.
         * </p>
         * 
         * @param status
         *        The current state of this replication group - <code>creating</code>, <code>available</code>,
         *        <code>modifying</code>, <code>deleting</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The endpoint of the primary node in this node group (shard).
         * </p>
         * 
         * @param primaryEndpoint
         *        The endpoint of the primary node in this node group (shard).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder primaryEndpoint(Endpoint primaryEndpoint);

        /**
         * <p>
         * The endpoint of the primary node in this node group (shard).
         * </p>
         * This is a convenience method that creates an instance of the {@link Endpoint.Builder} avoiding the need to
         * create one manually via {@link Endpoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Endpoint.Builder#build()} is called immediately and its result is
         * passed to {@link #primaryEndpoint(Endpoint)}.
         * 
         * @param primaryEndpoint
         *        a consumer that will call methods on {@link Endpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #primaryEndpoint(Endpoint)
         */
        default Builder primaryEndpoint(Consumer<Endpoint.Builder> primaryEndpoint) {
            return primaryEndpoint(Endpoint.builder().applyMutation(primaryEndpoint).build());
        }

        /**
         * <p>
         * The endpoint of the replica nodes in this node group (shard). This value is read-only.
         * </p>
         * 
         * @param readerEndpoint
         *        The endpoint of the replica nodes in this node group (shard). This value is read-only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readerEndpoint(Endpoint readerEndpoint);

        /**
         * <p>
         * The endpoint of the replica nodes in this node group (shard). This value is read-only.
         * </p>
         * This is a convenience method that creates an instance of the {@link Endpoint.Builder} avoiding the need to
         * create one manually via {@link Endpoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Endpoint.Builder#build()} is called immediately and its result is
         * passed to {@link #readerEndpoint(Endpoint)}.
         * 
         * @param readerEndpoint
         *        a consumer that will call methods on {@link Endpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #readerEndpoint(Endpoint)
         */
        default Builder readerEndpoint(Consumer<Endpoint.Builder> readerEndpoint) {
            return readerEndpoint(Endpoint.builder().applyMutation(readerEndpoint).build());
        }

        /**
         * <p>
         * The keyspace for this node group (shard).
         * </p>
         * 
         * @param slots
         *        The keyspace for this node group (shard).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slots(String slots);

        /**
         * <p>
         * A list containing information about individual nodes within the node group (shard).
         * </p>
         * 
         * @param nodeGroupMembers
         *        A list containing information about individual nodes within the node group (shard).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeGroupMembers(Collection<NodeGroupMember> nodeGroupMembers);

        /**
         * <p>
         * A list containing information about individual nodes within the node group (shard).
         * </p>
         * 
         * @param nodeGroupMembers
         *        A list containing information about individual nodes within the node group (shard).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeGroupMembers(NodeGroupMember... nodeGroupMembers);

        /**
         * <p>
         * A list containing information about individual nodes within the node group (shard).
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.elasticache.model.NodeGroupMember.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.elasticache.model.NodeGroupMember#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.elasticache.model.NodeGroupMember.Builder#build()} is called
         * immediately and its result is passed to {@link #nodeGroupMembers(List<NodeGroupMember>)}.
         * 
         * @param nodeGroupMembers
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.elasticache.model.NodeGroupMember.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodeGroupMembers(java.util.Collection<NodeGroupMember>)
         */
        Builder nodeGroupMembers(Consumer<NodeGroupMember.Builder>... nodeGroupMembers);
    }

    static final class BuilderImpl implements Builder {
        private String nodeGroupId;

        private String status;

        private Endpoint primaryEndpoint;

        private Endpoint readerEndpoint;

        private String slots;

        private List<NodeGroupMember> nodeGroupMembers = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NodeGroup model) {
            nodeGroupId(model.nodeGroupId);
            status(model.status);
            primaryEndpoint(model.primaryEndpoint);
            readerEndpoint(model.readerEndpoint);
            slots(model.slots);
            nodeGroupMembers(model.nodeGroupMembers);
        }

        public final String getNodeGroupId() {
            return nodeGroupId;
        }

        public final void setNodeGroupId(String nodeGroupId) {
            this.nodeGroupId = nodeGroupId;
        }

        @Override
        public final Builder nodeGroupId(String nodeGroupId) {
            this.nodeGroupId = nodeGroupId;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final Endpoint.Builder getPrimaryEndpoint() {
            return primaryEndpoint != null ? primaryEndpoint.toBuilder() : null;
        }

        public final void setPrimaryEndpoint(Endpoint.BuilderImpl primaryEndpoint) {
            this.primaryEndpoint = primaryEndpoint != null ? primaryEndpoint.build() : null;
        }

        @Override
        public final Builder primaryEndpoint(Endpoint primaryEndpoint) {
            this.primaryEndpoint = primaryEndpoint;
            return this;
        }

        public final Endpoint.Builder getReaderEndpoint() {
            return readerEndpoint != null ? readerEndpoint.toBuilder() : null;
        }

        public final void setReaderEndpoint(Endpoint.BuilderImpl readerEndpoint) {
            this.readerEndpoint = readerEndpoint != null ? readerEndpoint.build() : null;
        }

        @Override
        public final Builder readerEndpoint(Endpoint readerEndpoint) {
            this.readerEndpoint = readerEndpoint;
            return this;
        }

        public final String getSlots() {
            return slots;
        }

        public final void setSlots(String slots) {
            this.slots = slots;
        }

        @Override
        public final Builder slots(String slots) {
            this.slots = slots;
            return this;
        }

        public final List<NodeGroupMember.Builder> getNodeGroupMembers() {
            List<NodeGroupMember.Builder> result = NodeGroupMemberListCopier.copyToBuilder(this.nodeGroupMembers);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNodeGroupMembers(Collection<NodeGroupMember.BuilderImpl> nodeGroupMembers) {
            this.nodeGroupMembers = NodeGroupMemberListCopier.copyFromBuilder(nodeGroupMembers);
        }

        @Override
        public final Builder nodeGroupMembers(Collection<NodeGroupMember> nodeGroupMembers) {
            this.nodeGroupMembers = NodeGroupMemberListCopier.copy(nodeGroupMembers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodeGroupMembers(NodeGroupMember... nodeGroupMembers) {
            nodeGroupMembers(Arrays.asList(nodeGroupMembers));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodeGroupMembers(Consumer<NodeGroupMember.Builder>... nodeGroupMembers) {
            nodeGroupMembers(Stream.of(nodeGroupMembers).map(c -> NodeGroupMember.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public NodeGroup build() {
            return new NodeGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
