/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The status of the service update for a specific replication group
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateAction implements SdkPojo, Serializable, ToCopyableBuilder<UpdateAction.Builder, UpdateAction> {
    private static final SdkField<String> REPLICATION_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReplicationGroupId").getter(getter(UpdateAction::replicationGroupId))
            .setter(setter(Builder::replicationGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationGroupId").build())
            .build();

    private static final SdkField<String> CACHE_CLUSTER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CacheClusterId").getter(getter(UpdateAction::cacheClusterId)).setter(setter(Builder::cacheClusterId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheClusterId").build()).build();

    private static final SdkField<String> SERVICE_UPDATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceUpdateName").getter(getter(UpdateAction::serviceUpdateName))
            .setter(setter(Builder::serviceUpdateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceUpdateName").build()).build();

    private static final SdkField<Instant> SERVICE_UPDATE_RELEASE_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT).memberName("ServiceUpdateReleaseDate")
            .getter(getter(UpdateAction::serviceUpdateReleaseDate)).setter(setter(Builder::serviceUpdateReleaseDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceUpdateReleaseDate").build())
            .build();

    private static final SdkField<String> SERVICE_UPDATE_SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceUpdateSeverity").getter(getter(UpdateAction::serviceUpdateSeverityAsString))
            .setter(setter(Builder::serviceUpdateSeverity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceUpdateSeverity").build())
            .build();

    private static final SdkField<String> SERVICE_UPDATE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceUpdateStatus").getter(getter(UpdateAction::serviceUpdateStatusAsString))
            .setter(setter(Builder::serviceUpdateStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceUpdateStatus").build())
            .build();

    private static final SdkField<Instant> SERVICE_UPDATE_RECOMMENDED_APPLY_BY_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ServiceUpdateRecommendedApplyByDate")
            .getter(getter(UpdateAction::serviceUpdateRecommendedApplyByDate))
            .setter(setter(Builder::serviceUpdateRecommendedApplyByDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("ServiceUpdateRecommendedApplyByDate").build()).build();

    private static final SdkField<String> SERVICE_UPDATE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceUpdateType").getter(getter(UpdateAction::serviceUpdateTypeAsString))
            .setter(setter(Builder::serviceUpdateType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceUpdateType").build()).build();

    private static final SdkField<Instant> UPDATE_ACTION_AVAILABLE_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT).memberName("UpdateActionAvailableDate")
            .getter(getter(UpdateAction::updateActionAvailableDate)).setter(setter(Builder::updateActionAvailableDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdateActionAvailableDate").build())
            .build();

    private static final SdkField<String> UPDATE_ACTION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UpdateActionStatus").getter(getter(UpdateAction::updateActionStatusAsString))
            .setter(setter(Builder::updateActionStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdateActionStatus").build())
            .build();

    private static final SdkField<String> NODES_UPDATED_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NodesUpdated").getter(getter(UpdateAction::nodesUpdated)).setter(setter(Builder::nodesUpdated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodesUpdated").build()).build();

    private static final SdkField<Instant> UPDATE_ACTION_STATUS_MODIFIED_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdateActionStatusModifiedDate")
            .getter(getter(UpdateAction::updateActionStatusModifiedDate))
            .setter(setter(Builder::updateActionStatusModifiedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdateActionStatusModifiedDate")
                    .build()).build();

    private static final SdkField<String> SLA_MET_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("SlaMet")
            .getter(getter(UpdateAction::slaMetAsString)).setter(setter(Builder::slaMet))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SlaMet").build()).build();

    private static final SdkField<List<NodeGroupUpdateStatus>> NODE_GROUP_UPDATE_STATUS_FIELD = SdkField
            .<List<NodeGroupUpdateStatus>> builder(MarshallingType.LIST)
            .memberName("NodeGroupUpdateStatus")
            .getter(getter(UpdateAction::nodeGroupUpdateStatus))
            .setter(setter(Builder::nodeGroupUpdateStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeGroupUpdateStatus").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("NodeGroupUpdateStatus")
                            .memberFieldInfo(
                                    SdkField.<NodeGroupUpdateStatus> builder(MarshallingType.SDK_POJO)
                                            .constructor(NodeGroupUpdateStatus::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("NodeGroupUpdateStatus").build()).build()).build()).build();

    private static final SdkField<List<CacheNodeUpdateStatus>> CACHE_NODE_UPDATE_STATUS_FIELD = SdkField
            .<List<CacheNodeUpdateStatus>> builder(MarshallingType.LIST)
            .memberName("CacheNodeUpdateStatus")
            .getter(getter(UpdateAction::cacheNodeUpdateStatus))
            .setter(setter(Builder::cacheNodeUpdateStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheNodeUpdateStatus").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("CacheNodeUpdateStatus")
                            .memberFieldInfo(
                                    SdkField.<CacheNodeUpdateStatus> builder(MarshallingType.SDK_POJO)
                                            .constructor(CacheNodeUpdateStatus::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("CacheNodeUpdateStatus").build()).build()).build()).build();

    private static final SdkField<String> ESTIMATED_UPDATE_TIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EstimatedUpdateTime").getter(getter(UpdateAction::estimatedUpdateTime))
            .setter(setter(Builder::estimatedUpdateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EstimatedUpdateTime").build())
            .build();

    private static final SdkField<String> ENGINE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Engine")
            .getter(getter(UpdateAction::engine)).setter(setter(Builder::engine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Engine").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REPLICATION_GROUP_ID_FIELD,
            CACHE_CLUSTER_ID_FIELD, SERVICE_UPDATE_NAME_FIELD, SERVICE_UPDATE_RELEASE_DATE_FIELD, SERVICE_UPDATE_SEVERITY_FIELD,
            SERVICE_UPDATE_STATUS_FIELD, SERVICE_UPDATE_RECOMMENDED_APPLY_BY_DATE_FIELD, SERVICE_UPDATE_TYPE_FIELD,
            UPDATE_ACTION_AVAILABLE_DATE_FIELD, UPDATE_ACTION_STATUS_FIELD, NODES_UPDATED_FIELD,
            UPDATE_ACTION_STATUS_MODIFIED_DATE_FIELD, SLA_MET_FIELD, NODE_GROUP_UPDATE_STATUS_FIELD,
            CACHE_NODE_UPDATE_STATUS_FIELD, ESTIMATED_UPDATE_TIME_FIELD, ENGINE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String replicationGroupId;

    private final String cacheClusterId;

    private final String serviceUpdateName;

    private final Instant serviceUpdateReleaseDate;

    private final String serviceUpdateSeverity;

    private final String serviceUpdateStatus;

    private final Instant serviceUpdateRecommendedApplyByDate;

    private final String serviceUpdateType;

    private final Instant updateActionAvailableDate;

    private final String updateActionStatus;

    private final String nodesUpdated;

    private final Instant updateActionStatusModifiedDate;

    private final String slaMet;

    private final List<NodeGroupUpdateStatus> nodeGroupUpdateStatus;

    private final List<CacheNodeUpdateStatus> cacheNodeUpdateStatus;

    private final String estimatedUpdateTime;

    private final String engine;

    private UpdateAction(BuilderImpl builder) {
        this.replicationGroupId = builder.replicationGroupId;
        this.cacheClusterId = builder.cacheClusterId;
        this.serviceUpdateName = builder.serviceUpdateName;
        this.serviceUpdateReleaseDate = builder.serviceUpdateReleaseDate;
        this.serviceUpdateSeverity = builder.serviceUpdateSeverity;
        this.serviceUpdateStatus = builder.serviceUpdateStatus;
        this.serviceUpdateRecommendedApplyByDate = builder.serviceUpdateRecommendedApplyByDate;
        this.serviceUpdateType = builder.serviceUpdateType;
        this.updateActionAvailableDate = builder.updateActionAvailableDate;
        this.updateActionStatus = builder.updateActionStatus;
        this.nodesUpdated = builder.nodesUpdated;
        this.updateActionStatusModifiedDate = builder.updateActionStatusModifiedDate;
        this.slaMet = builder.slaMet;
        this.nodeGroupUpdateStatus = builder.nodeGroupUpdateStatus;
        this.cacheNodeUpdateStatus = builder.cacheNodeUpdateStatus;
        this.estimatedUpdateTime = builder.estimatedUpdateTime;
        this.engine = builder.engine;
    }

    /**
     * <p>
     * The ID of the replication group
     * </p>
     * 
     * @return The ID of the replication group
     */
    public final String replicationGroupId() {
        return replicationGroupId;
    }

    /**
     * <p>
     * The ID of the cache cluster
     * </p>
     * 
     * @return The ID of the cache cluster
     */
    public final String cacheClusterId() {
        return cacheClusterId;
    }

    /**
     * <p>
     * The unique ID of the service update
     * </p>
     * 
     * @return The unique ID of the service update
     */
    public final String serviceUpdateName() {
        return serviceUpdateName;
    }

    /**
     * <p>
     * The date the update is first available
     * </p>
     * 
     * @return The date the update is first available
     */
    public final Instant serviceUpdateReleaseDate() {
        return serviceUpdateReleaseDate;
    }

    /**
     * <p>
     * The severity of the service update
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #serviceUpdateSeverity} will return {@link ServiceUpdateSeverity#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #serviceUpdateSeverityAsString}.
     * </p>
     * 
     * @return The severity of the service update
     * @see ServiceUpdateSeverity
     */
    public final ServiceUpdateSeverity serviceUpdateSeverity() {
        return ServiceUpdateSeverity.fromValue(serviceUpdateSeverity);
    }

    /**
     * <p>
     * The severity of the service update
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #serviceUpdateSeverity} will return {@link ServiceUpdateSeverity#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #serviceUpdateSeverityAsString}.
     * </p>
     * 
     * @return The severity of the service update
     * @see ServiceUpdateSeverity
     */
    public final String serviceUpdateSeverityAsString() {
        return serviceUpdateSeverity;
    }

    /**
     * <p>
     * The status of the service update
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #serviceUpdateStatus} will return {@link ServiceUpdateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #serviceUpdateStatusAsString}.
     * </p>
     * 
     * @return The status of the service update
     * @see ServiceUpdateStatus
     */
    public final ServiceUpdateStatus serviceUpdateStatus() {
        return ServiceUpdateStatus.fromValue(serviceUpdateStatus);
    }

    /**
     * <p>
     * The status of the service update
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #serviceUpdateStatus} will return {@link ServiceUpdateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #serviceUpdateStatusAsString}.
     * </p>
     * 
     * @return The status of the service update
     * @see ServiceUpdateStatus
     */
    public final String serviceUpdateStatusAsString() {
        return serviceUpdateStatus;
    }

    /**
     * <p>
     * The recommended date to apply the service update to ensure compliance. For information on compliance, see <a
     * href=
     * "https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/elasticache-compliance.html#elasticache-compliance-self-service"
     * >Self-Service Security Updates for Compliance</a>.
     * </p>
     * 
     * @return The recommended date to apply the service update to ensure compliance. For information on compliance, see
     *         <a href=
     *         "https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/elasticache-compliance.html#elasticache-compliance-self-service"
     *         >Self-Service Security Updates for Compliance</a>.
     */
    public final Instant serviceUpdateRecommendedApplyByDate() {
        return serviceUpdateRecommendedApplyByDate;
    }

    /**
     * <p>
     * Reflects the nature of the service update
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serviceUpdateType}
     * will return {@link ServiceUpdateType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #serviceUpdateTypeAsString}.
     * </p>
     * 
     * @return Reflects the nature of the service update
     * @see ServiceUpdateType
     */
    public final ServiceUpdateType serviceUpdateType() {
        return ServiceUpdateType.fromValue(serviceUpdateType);
    }

    /**
     * <p>
     * Reflects the nature of the service update
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serviceUpdateType}
     * will return {@link ServiceUpdateType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #serviceUpdateTypeAsString}.
     * </p>
     * 
     * @return Reflects the nature of the service update
     * @see ServiceUpdateType
     */
    public final String serviceUpdateTypeAsString() {
        return serviceUpdateType;
    }

    /**
     * <p>
     * The date that the service update is available to a replication group
     * </p>
     * 
     * @return The date that the service update is available to a replication group
     */
    public final Instant updateActionAvailableDate() {
        return updateActionAvailableDate;
    }

    /**
     * <p>
     * The status of the update action
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #updateActionStatus} will return {@link UpdateActionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #updateActionStatusAsString}.
     * </p>
     * 
     * @return The status of the update action
     * @see UpdateActionStatus
     */
    public final UpdateActionStatus updateActionStatus() {
        return UpdateActionStatus.fromValue(updateActionStatus);
    }

    /**
     * <p>
     * The status of the update action
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #updateActionStatus} will return {@link UpdateActionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #updateActionStatusAsString}.
     * </p>
     * 
     * @return The status of the update action
     * @see UpdateActionStatus
     */
    public final String updateActionStatusAsString() {
        return updateActionStatus;
    }

    /**
     * <p>
     * The progress of the service update on the replication group
     * </p>
     * 
     * @return The progress of the service update on the replication group
     */
    public final String nodesUpdated() {
        return nodesUpdated;
    }

    /**
     * <p>
     * The date when the UpdateActionStatus was last modified
     * </p>
     * 
     * @return The date when the UpdateActionStatus was last modified
     */
    public final Instant updateActionStatusModifiedDate() {
        return updateActionStatusModifiedDate;
    }

    /**
     * <p>
     * If yes, all nodes in the replication group have been updated by the recommended apply-by date. If no, at least
     * one node in the replication group have not been updated by the recommended apply-by date. If N/A, the replication
     * group was created after the recommended apply-by date.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #slaMet} will
     * return {@link SlaMet#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #slaMetAsString}.
     * </p>
     * 
     * @return If yes, all nodes in the replication group have been updated by the recommended apply-by date. If no, at
     *         least one node in the replication group have not been updated by the recommended apply-by date. If N/A,
     *         the replication group was created after the recommended apply-by date.
     * @see SlaMet
     */
    public final SlaMet slaMet() {
        return SlaMet.fromValue(slaMet);
    }

    /**
     * <p>
     * If yes, all nodes in the replication group have been updated by the recommended apply-by date. If no, at least
     * one node in the replication group have not been updated by the recommended apply-by date. If N/A, the replication
     * group was created after the recommended apply-by date.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #slaMet} will
     * return {@link SlaMet#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #slaMetAsString}.
     * </p>
     * 
     * @return If yes, all nodes in the replication group have been updated by the recommended apply-by date. If no, at
     *         least one node in the replication group have not been updated by the recommended apply-by date. If N/A,
     *         the replication group was created after the recommended apply-by date.
     * @see SlaMet
     */
    public final String slaMetAsString() {
        return slaMet;
    }

    /**
     * For responses, this returns true if the service returned a value for the NodeGroupUpdateStatus property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasNodeGroupUpdateStatus() {
        return nodeGroupUpdateStatus != null && !(nodeGroupUpdateStatus instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The status of the service update on the node group
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNodeGroupUpdateStatus} method.
     * </p>
     * 
     * @return The status of the service update on the node group
     */
    public final List<NodeGroupUpdateStatus> nodeGroupUpdateStatus() {
        return nodeGroupUpdateStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the CacheNodeUpdateStatus property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCacheNodeUpdateStatus() {
        return cacheNodeUpdateStatus != null && !(cacheNodeUpdateStatus instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The status of the service update on the cache node
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCacheNodeUpdateStatus} method.
     * </p>
     * 
     * @return The status of the service update on the cache node
     */
    public final List<CacheNodeUpdateStatus> cacheNodeUpdateStatus() {
        return cacheNodeUpdateStatus;
    }

    /**
     * <p>
     * The estimated length of time for the update to complete
     * </p>
     * 
     * @return The estimated length of time for the update to complete
     */
    public final String estimatedUpdateTime() {
        return estimatedUpdateTime;
    }

    /**
     * <p>
     * The Elasticache engine to which the update applies. Either Valkey, Redis OSS or Memcached.
     * </p>
     * 
     * @return The Elasticache engine to which the update applies. Either Valkey, Redis OSS or Memcached.
     */
    public final String engine() {
        return engine;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(replicationGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(cacheClusterId());
        hashCode = 31 * hashCode + Objects.hashCode(serviceUpdateName());
        hashCode = 31 * hashCode + Objects.hashCode(serviceUpdateReleaseDate());
        hashCode = 31 * hashCode + Objects.hashCode(serviceUpdateSeverityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(serviceUpdateStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(serviceUpdateRecommendedApplyByDate());
        hashCode = 31 * hashCode + Objects.hashCode(serviceUpdateTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(updateActionAvailableDate());
        hashCode = 31 * hashCode + Objects.hashCode(updateActionStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(nodesUpdated());
        hashCode = 31 * hashCode + Objects.hashCode(updateActionStatusModifiedDate());
        hashCode = 31 * hashCode + Objects.hashCode(slaMetAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasNodeGroupUpdateStatus() ? nodeGroupUpdateStatus() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCacheNodeUpdateStatus() ? cacheNodeUpdateStatus() : null);
        hashCode = 31 * hashCode + Objects.hashCode(estimatedUpdateTime());
        hashCode = 31 * hashCode + Objects.hashCode(engine());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateAction)) {
            return false;
        }
        UpdateAction other = (UpdateAction) obj;
        return Objects.equals(replicationGroupId(), other.replicationGroupId())
                && Objects.equals(cacheClusterId(), other.cacheClusterId())
                && Objects.equals(serviceUpdateName(), other.serviceUpdateName())
                && Objects.equals(serviceUpdateReleaseDate(), other.serviceUpdateReleaseDate())
                && Objects.equals(serviceUpdateSeverityAsString(), other.serviceUpdateSeverityAsString())
                && Objects.equals(serviceUpdateStatusAsString(), other.serviceUpdateStatusAsString())
                && Objects.equals(serviceUpdateRecommendedApplyByDate(), other.serviceUpdateRecommendedApplyByDate())
                && Objects.equals(serviceUpdateTypeAsString(), other.serviceUpdateTypeAsString())
                && Objects.equals(updateActionAvailableDate(), other.updateActionAvailableDate())
                && Objects.equals(updateActionStatusAsString(), other.updateActionStatusAsString())
                && Objects.equals(nodesUpdated(), other.nodesUpdated())
                && Objects.equals(updateActionStatusModifiedDate(), other.updateActionStatusModifiedDate())
                && Objects.equals(slaMetAsString(), other.slaMetAsString())
                && hasNodeGroupUpdateStatus() == other.hasNodeGroupUpdateStatus()
                && Objects.equals(nodeGroupUpdateStatus(), other.nodeGroupUpdateStatus())
                && hasCacheNodeUpdateStatus() == other.hasCacheNodeUpdateStatus()
                && Objects.equals(cacheNodeUpdateStatus(), other.cacheNodeUpdateStatus())
                && Objects.equals(estimatedUpdateTime(), other.estimatedUpdateTime()) && Objects.equals(engine(), other.engine());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateAction").add("ReplicationGroupId", replicationGroupId())
                .add("CacheClusterId", cacheClusterId()).add("ServiceUpdateName", serviceUpdateName())
                .add("ServiceUpdateReleaseDate", serviceUpdateReleaseDate())
                .add("ServiceUpdateSeverity", serviceUpdateSeverityAsString())
                .add("ServiceUpdateStatus", serviceUpdateStatusAsString())
                .add("ServiceUpdateRecommendedApplyByDate", serviceUpdateRecommendedApplyByDate())
                .add("ServiceUpdateType", serviceUpdateTypeAsString())
                .add("UpdateActionAvailableDate", updateActionAvailableDate())
                .add("UpdateActionStatus", updateActionStatusAsString()).add("NodesUpdated", nodesUpdated())
                .add("UpdateActionStatusModifiedDate", updateActionStatusModifiedDate()).add("SlaMet", slaMetAsString())
                .add("NodeGroupUpdateStatus", hasNodeGroupUpdateStatus() ? nodeGroupUpdateStatus() : null)
                .add("CacheNodeUpdateStatus", hasCacheNodeUpdateStatus() ? cacheNodeUpdateStatus() : null)
                .add("EstimatedUpdateTime", estimatedUpdateTime()).add("Engine", engine()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ReplicationGroupId":
            return Optional.ofNullable(clazz.cast(replicationGroupId()));
        case "CacheClusterId":
            return Optional.ofNullable(clazz.cast(cacheClusterId()));
        case "ServiceUpdateName":
            return Optional.ofNullable(clazz.cast(serviceUpdateName()));
        case "ServiceUpdateReleaseDate":
            return Optional.ofNullable(clazz.cast(serviceUpdateReleaseDate()));
        case "ServiceUpdateSeverity":
            return Optional.ofNullable(clazz.cast(serviceUpdateSeverityAsString()));
        case "ServiceUpdateStatus":
            return Optional.ofNullable(clazz.cast(serviceUpdateStatusAsString()));
        case "ServiceUpdateRecommendedApplyByDate":
            return Optional.ofNullable(clazz.cast(serviceUpdateRecommendedApplyByDate()));
        case "ServiceUpdateType":
            return Optional.ofNullable(clazz.cast(serviceUpdateTypeAsString()));
        case "UpdateActionAvailableDate":
            return Optional.ofNullable(clazz.cast(updateActionAvailableDate()));
        case "UpdateActionStatus":
            return Optional.ofNullable(clazz.cast(updateActionStatusAsString()));
        case "NodesUpdated":
            return Optional.ofNullable(clazz.cast(nodesUpdated()));
        case "UpdateActionStatusModifiedDate":
            return Optional.ofNullable(clazz.cast(updateActionStatusModifiedDate()));
        case "SlaMet":
            return Optional.ofNullable(clazz.cast(slaMetAsString()));
        case "NodeGroupUpdateStatus":
            return Optional.ofNullable(clazz.cast(nodeGroupUpdateStatus()));
        case "CacheNodeUpdateStatus":
            return Optional.ofNullable(clazz.cast(cacheNodeUpdateStatus()));
        case "EstimatedUpdateTime":
            return Optional.ofNullable(clazz.cast(estimatedUpdateTime()));
        case "Engine":
            return Optional.ofNullable(clazz.cast(engine()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpdateAction, T> g) {
        return obj -> g.apply((UpdateAction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UpdateAction> {
        /**
         * <p>
         * The ID of the replication group
         * </p>
         * 
         * @param replicationGroupId
         *        The ID of the replication group
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationGroupId(String replicationGroupId);

        /**
         * <p>
         * The ID of the cache cluster
         * </p>
         * 
         * @param cacheClusterId
         *        The ID of the cache cluster
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheClusterId(String cacheClusterId);

        /**
         * <p>
         * The unique ID of the service update
         * </p>
         * 
         * @param serviceUpdateName
         *        The unique ID of the service update
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceUpdateName(String serviceUpdateName);

        /**
         * <p>
         * The date the update is first available
         * </p>
         * 
         * @param serviceUpdateReleaseDate
         *        The date the update is first available
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceUpdateReleaseDate(Instant serviceUpdateReleaseDate);

        /**
         * <p>
         * The severity of the service update
         * </p>
         * 
         * @param serviceUpdateSeverity
         *        The severity of the service update
         * @see ServiceUpdateSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceUpdateSeverity
         */
        Builder serviceUpdateSeverity(String serviceUpdateSeverity);

        /**
         * <p>
         * The severity of the service update
         * </p>
         * 
         * @param serviceUpdateSeverity
         *        The severity of the service update
         * @see ServiceUpdateSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceUpdateSeverity
         */
        Builder serviceUpdateSeverity(ServiceUpdateSeverity serviceUpdateSeverity);

        /**
         * <p>
         * The status of the service update
         * </p>
         * 
         * @param serviceUpdateStatus
         *        The status of the service update
         * @see ServiceUpdateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceUpdateStatus
         */
        Builder serviceUpdateStatus(String serviceUpdateStatus);

        /**
         * <p>
         * The status of the service update
         * </p>
         * 
         * @param serviceUpdateStatus
         *        The status of the service update
         * @see ServiceUpdateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceUpdateStatus
         */
        Builder serviceUpdateStatus(ServiceUpdateStatus serviceUpdateStatus);

        /**
         * <p>
         * The recommended date to apply the service update to ensure compliance. For information on compliance, see <a
         * href=
         * "https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/elasticache-compliance.html#elasticache-compliance-self-service"
         * >Self-Service Security Updates for Compliance</a>.
         * </p>
         * 
         * @param serviceUpdateRecommendedApplyByDate
         *        The recommended date to apply the service update to ensure compliance. For information on compliance,
         *        see <a href=
         *        "https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/elasticache-compliance.html#elasticache-compliance-self-service"
         *        >Self-Service Security Updates for Compliance</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceUpdateRecommendedApplyByDate(Instant serviceUpdateRecommendedApplyByDate);

        /**
         * <p>
         * Reflects the nature of the service update
         * </p>
         * 
         * @param serviceUpdateType
         *        Reflects the nature of the service update
         * @see ServiceUpdateType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceUpdateType
         */
        Builder serviceUpdateType(String serviceUpdateType);

        /**
         * <p>
         * Reflects the nature of the service update
         * </p>
         * 
         * @param serviceUpdateType
         *        Reflects the nature of the service update
         * @see ServiceUpdateType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceUpdateType
         */
        Builder serviceUpdateType(ServiceUpdateType serviceUpdateType);

        /**
         * <p>
         * The date that the service update is available to a replication group
         * </p>
         * 
         * @param updateActionAvailableDate
         *        The date that the service update is available to a replication group
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updateActionAvailableDate(Instant updateActionAvailableDate);

        /**
         * <p>
         * The status of the update action
         * </p>
         * 
         * @param updateActionStatus
         *        The status of the update action
         * @see UpdateActionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UpdateActionStatus
         */
        Builder updateActionStatus(String updateActionStatus);

        /**
         * <p>
         * The status of the update action
         * </p>
         * 
         * @param updateActionStatus
         *        The status of the update action
         * @see UpdateActionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UpdateActionStatus
         */
        Builder updateActionStatus(UpdateActionStatus updateActionStatus);

        /**
         * <p>
         * The progress of the service update on the replication group
         * </p>
         * 
         * @param nodesUpdated
         *        The progress of the service update on the replication group
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodesUpdated(String nodesUpdated);

        /**
         * <p>
         * The date when the UpdateActionStatus was last modified
         * </p>
         * 
         * @param updateActionStatusModifiedDate
         *        The date when the UpdateActionStatus was last modified
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updateActionStatusModifiedDate(Instant updateActionStatusModifiedDate);

        /**
         * <p>
         * If yes, all nodes in the replication group have been updated by the recommended apply-by date. If no, at
         * least one node in the replication group have not been updated by the recommended apply-by date. If N/A, the
         * replication group was created after the recommended apply-by date.
         * </p>
         * 
         * @param slaMet
         *        If yes, all nodes in the replication group have been updated by the recommended apply-by date. If no,
         *        at least one node in the replication group have not been updated by the recommended apply-by date. If
         *        N/A, the replication group was created after the recommended apply-by date.
         * @see SlaMet
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SlaMet
         */
        Builder slaMet(String slaMet);

        /**
         * <p>
         * If yes, all nodes in the replication group have been updated by the recommended apply-by date. If no, at
         * least one node in the replication group have not been updated by the recommended apply-by date. If N/A, the
         * replication group was created after the recommended apply-by date.
         * </p>
         * 
         * @param slaMet
         *        If yes, all nodes in the replication group have been updated by the recommended apply-by date. If no,
         *        at least one node in the replication group have not been updated by the recommended apply-by date. If
         *        N/A, the replication group was created after the recommended apply-by date.
         * @see SlaMet
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SlaMet
         */
        Builder slaMet(SlaMet slaMet);

        /**
         * <p>
         * The status of the service update on the node group
         * </p>
         * 
         * @param nodeGroupUpdateStatus
         *        The status of the service update on the node group
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeGroupUpdateStatus(Collection<NodeGroupUpdateStatus> nodeGroupUpdateStatus);

        /**
         * <p>
         * The status of the service update on the node group
         * </p>
         * 
         * @param nodeGroupUpdateStatus
         *        The status of the service update on the node group
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeGroupUpdateStatus(NodeGroupUpdateStatus... nodeGroupUpdateStatus);

        /**
         * <p>
         * The status of the service update on the node group
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.elasticache.model.NodeGroupUpdateStatus.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.elasticache.model.NodeGroupUpdateStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.elasticache.model.NodeGroupUpdateStatus.Builder#build()} is called
         * immediately and its result is passed to {@link #nodeGroupUpdateStatus(List<NodeGroupUpdateStatus>)}.
         * 
         * @param nodeGroupUpdateStatus
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.elasticache.model.NodeGroupUpdateStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodeGroupUpdateStatus(java.util.Collection<NodeGroupUpdateStatus>)
         */
        Builder nodeGroupUpdateStatus(Consumer<NodeGroupUpdateStatus.Builder>... nodeGroupUpdateStatus);

        /**
         * <p>
         * The status of the service update on the cache node
         * </p>
         * 
         * @param cacheNodeUpdateStatus
         *        The status of the service update on the cache node
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheNodeUpdateStatus(Collection<CacheNodeUpdateStatus> cacheNodeUpdateStatus);

        /**
         * <p>
         * The status of the service update on the cache node
         * </p>
         * 
         * @param cacheNodeUpdateStatus
         *        The status of the service update on the cache node
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheNodeUpdateStatus(CacheNodeUpdateStatus... cacheNodeUpdateStatus);

        /**
         * <p>
         * The status of the service update on the cache node
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.elasticache.model.CacheNodeUpdateStatus.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.elasticache.model.CacheNodeUpdateStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.elasticache.model.CacheNodeUpdateStatus.Builder#build()} is called
         * immediately and its result is passed to {@link #cacheNodeUpdateStatus(List<CacheNodeUpdateStatus>)}.
         * 
         * @param cacheNodeUpdateStatus
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.elasticache.model.CacheNodeUpdateStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cacheNodeUpdateStatus(java.util.Collection<CacheNodeUpdateStatus>)
         */
        Builder cacheNodeUpdateStatus(Consumer<CacheNodeUpdateStatus.Builder>... cacheNodeUpdateStatus);

        /**
         * <p>
         * The estimated length of time for the update to complete
         * </p>
         * 
         * @param estimatedUpdateTime
         *        The estimated length of time for the update to complete
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder estimatedUpdateTime(String estimatedUpdateTime);

        /**
         * <p>
         * The Elasticache engine to which the update applies. Either Valkey, Redis OSS or Memcached.
         * </p>
         * 
         * @param engine
         *        The Elasticache engine to which the update applies. Either Valkey, Redis OSS or Memcached.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engine(String engine);
    }

    static final class BuilderImpl implements Builder {
        private String replicationGroupId;

        private String cacheClusterId;

        private String serviceUpdateName;

        private Instant serviceUpdateReleaseDate;

        private String serviceUpdateSeverity;

        private String serviceUpdateStatus;

        private Instant serviceUpdateRecommendedApplyByDate;

        private String serviceUpdateType;

        private Instant updateActionAvailableDate;

        private String updateActionStatus;

        private String nodesUpdated;

        private Instant updateActionStatusModifiedDate;

        private String slaMet;

        private List<NodeGroupUpdateStatus> nodeGroupUpdateStatus = DefaultSdkAutoConstructList.getInstance();

        private List<CacheNodeUpdateStatus> cacheNodeUpdateStatus = DefaultSdkAutoConstructList.getInstance();

        private String estimatedUpdateTime;

        private String engine;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateAction model) {
            replicationGroupId(model.replicationGroupId);
            cacheClusterId(model.cacheClusterId);
            serviceUpdateName(model.serviceUpdateName);
            serviceUpdateReleaseDate(model.serviceUpdateReleaseDate);
            serviceUpdateSeverity(model.serviceUpdateSeverity);
            serviceUpdateStatus(model.serviceUpdateStatus);
            serviceUpdateRecommendedApplyByDate(model.serviceUpdateRecommendedApplyByDate);
            serviceUpdateType(model.serviceUpdateType);
            updateActionAvailableDate(model.updateActionAvailableDate);
            updateActionStatus(model.updateActionStatus);
            nodesUpdated(model.nodesUpdated);
            updateActionStatusModifiedDate(model.updateActionStatusModifiedDate);
            slaMet(model.slaMet);
            nodeGroupUpdateStatus(model.nodeGroupUpdateStatus);
            cacheNodeUpdateStatus(model.cacheNodeUpdateStatus);
            estimatedUpdateTime(model.estimatedUpdateTime);
            engine(model.engine);
        }

        public final String getReplicationGroupId() {
            return replicationGroupId;
        }

        public final void setReplicationGroupId(String replicationGroupId) {
            this.replicationGroupId = replicationGroupId;
        }

        @Override
        public final Builder replicationGroupId(String replicationGroupId) {
            this.replicationGroupId = replicationGroupId;
            return this;
        }

        public final String getCacheClusterId() {
            return cacheClusterId;
        }

        public final void setCacheClusterId(String cacheClusterId) {
            this.cacheClusterId = cacheClusterId;
        }

        @Override
        public final Builder cacheClusterId(String cacheClusterId) {
            this.cacheClusterId = cacheClusterId;
            return this;
        }

        public final String getServiceUpdateName() {
            return serviceUpdateName;
        }

        public final void setServiceUpdateName(String serviceUpdateName) {
            this.serviceUpdateName = serviceUpdateName;
        }

        @Override
        public final Builder serviceUpdateName(String serviceUpdateName) {
            this.serviceUpdateName = serviceUpdateName;
            return this;
        }

        public final Instant getServiceUpdateReleaseDate() {
            return serviceUpdateReleaseDate;
        }

        public final void setServiceUpdateReleaseDate(Instant serviceUpdateReleaseDate) {
            this.serviceUpdateReleaseDate = serviceUpdateReleaseDate;
        }

        @Override
        public final Builder serviceUpdateReleaseDate(Instant serviceUpdateReleaseDate) {
            this.serviceUpdateReleaseDate = serviceUpdateReleaseDate;
            return this;
        }

        public final String getServiceUpdateSeverity() {
            return serviceUpdateSeverity;
        }

        public final void setServiceUpdateSeverity(String serviceUpdateSeverity) {
            this.serviceUpdateSeverity = serviceUpdateSeverity;
        }

        @Override
        public final Builder serviceUpdateSeverity(String serviceUpdateSeverity) {
            this.serviceUpdateSeverity = serviceUpdateSeverity;
            return this;
        }

        @Override
        public final Builder serviceUpdateSeverity(ServiceUpdateSeverity serviceUpdateSeverity) {
            this.serviceUpdateSeverity(serviceUpdateSeverity == null ? null : serviceUpdateSeverity.toString());
            return this;
        }

        public final String getServiceUpdateStatus() {
            return serviceUpdateStatus;
        }

        public final void setServiceUpdateStatus(String serviceUpdateStatus) {
            this.serviceUpdateStatus = serviceUpdateStatus;
        }

        @Override
        public final Builder serviceUpdateStatus(String serviceUpdateStatus) {
            this.serviceUpdateStatus = serviceUpdateStatus;
            return this;
        }

        @Override
        public final Builder serviceUpdateStatus(ServiceUpdateStatus serviceUpdateStatus) {
            this.serviceUpdateStatus(serviceUpdateStatus == null ? null : serviceUpdateStatus.toString());
            return this;
        }

        public final Instant getServiceUpdateRecommendedApplyByDate() {
            return serviceUpdateRecommendedApplyByDate;
        }

        public final void setServiceUpdateRecommendedApplyByDate(Instant serviceUpdateRecommendedApplyByDate) {
            this.serviceUpdateRecommendedApplyByDate = serviceUpdateRecommendedApplyByDate;
        }

        @Override
        public final Builder serviceUpdateRecommendedApplyByDate(Instant serviceUpdateRecommendedApplyByDate) {
            this.serviceUpdateRecommendedApplyByDate = serviceUpdateRecommendedApplyByDate;
            return this;
        }

        public final String getServiceUpdateType() {
            return serviceUpdateType;
        }

        public final void setServiceUpdateType(String serviceUpdateType) {
            this.serviceUpdateType = serviceUpdateType;
        }

        @Override
        public final Builder serviceUpdateType(String serviceUpdateType) {
            this.serviceUpdateType = serviceUpdateType;
            return this;
        }

        @Override
        public final Builder serviceUpdateType(ServiceUpdateType serviceUpdateType) {
            this.serviceUpdateType(serviceUpdateType == null ? null : serviceUpdateType.toString());
            return this;
        }

        public final Instant getUpdateActionAvailableDate() {
            return updateActionAvailableDate;
        }

        public final void setUpdateActionAvailableDate(Instant updateActionAvailableDate) {
            this.updateActionAvailableDate = updateActionAvailableDate;
        }

        @Override
        public final Builder updateActionAvailableDate(Instant updateActionAvailableDate) {
            this.updateActionAvailableDate = updateActionAvailableDate;
            return this;
        }

        public final String getUpdateActionStatus() {
            return updateActionStatus;
        }

        public final void setUpdateActionStatus(String updateActionStatus) {
            this.updateActionStatus = updateActionStatus;
        }

        @Override
        public final Builder updateActionStatus(String updateActionStatus) {
            this.updateActionStatus = updateActionStatus;
            return this;
        }

        @Override
        public final Builder updateActionStatus(UpdateActionStatus updateActionStatus) {
            this.updateActionStatus(updateActionStatus == null ? null : updateActionStatus.toString());
            return this;
        }

        public final String getNodesUpdated() {
            return nodesUpdated;
        }

        public final void setNodesUpdated(String nodesUpdated) {
            this.nodesUpdated = nodesUpdated;
        }

        @Override
        public final Builder nodesUpdated(String nodesUpdated) {
            this.nodesUpdated = nodesUpdated;
            return this;
        }

        public final Instant getUpdateActionStatusModifiedDate() {
            return updateActionStatusModifiedDate;
        }

        public final void setUpdateActionStatusModifiedDate(Instant updateActionStatusModifiedDate) {
            this.updateActionStatusModifiedDate = updateActionStatusModifiedDate;
        }

        @Override
        public final Builder updateActionStatusModifiedDate(Instant updateActionStatusModifiedDate) {
            this.updateActionStatusModifiedDate = updateActionStatusModifiedDate;
            return this;
        }

        public final String getSlaMet() {
            return slaMet;
        }

        public final void setSlaMet(String slaMet) {
            this.slaMet = slaMet;
        }

        @Override
        public final Builder slaMet(String slaMet) {
            this.slaMet = slaMet;
            return this;
        }

        @Override
        public final Builder slaMet(SlaMet slaMet) {
            this.slaMet(slaMet == null ? null : slaMet.toString());
            return this;
        }

        public final List<NodeGroupUpdateStatus.Builder> getNodeGroupUpdateStatus() {
            List<NodeGroupUpdateStatus.Builder> result = NodeGroupUpdateStatusListCopier
                    .copyToBuilder(this.nodeGroupUpdateStatus);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNodeGroupUpdateStatus(Collection<NodeGroupUpdateStatus.BuilderImpl> nodeGroupUpdateStatus) {
            this.nodeGroupUpdateStatus = NodeGroupUpdateStatusListCopier.copyFromBuilder(nodeGroupUpdateStatus);
        }

        @Override
        public final Builder nodeGroupUpdateStatus(Collection<NodeGroupUpdateStatus> nodeGroupUpdateStatus) {
            this.nodeGroupUpdateStatus = NodeGroupUpdateStatusListCopier.copy(nodeGroupUpdateStatus);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodeGroupUpdateStatus(NodeGroupUpdateStatus... nodeGroupUpdateStatus) {
            nodeGroupUpdateStatus(Arrays.asList(nodeGroupUpdateStatus));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodeGroupUpdateStatus(Consumer<NodeGroupUpdateStatus.Builder>... nodeGroupUpdateStatus) {
            nodeGroupUpdateStatus(Stream.of(nodeGroupUpdateStatus)
                    .map(c -> NodeGroupUpdateStatus.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<CacheNodeUpdateStatus.Builder> getCacheNodeUpdateStatus() {
            List<CacheNodeUpdateStatus.Builder> result = CacheNodeUpdateStatusListCopier
                    .copyToBuilder(this.cacheNodeUpdateStatus);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCacheNodeUpdateStatus(Collection<CacheNodeUpdateStatus.BuilderImpl> cacheNodeUpdateStatus) {
            this.cacheNodeUpdateStatus = CacheNodeUpdateStatusListCopier.copyFromBuilder(cacheNodeUpdateStatus);
        }

        @Override
        public final Builder cacheNodeUpdateStatus(Collection<CacheNodeUpdateStatus> cacheNodeUpdateStatus) {
            this.cacheNodeUpdateStatus = CacheNodeUpdateStatusListCopier.copy(cacheNodeUpdateStatus);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cacheNodeUpdateStatus(CacheNodeUpdateStatus... cacheNodeUpdateStatus) {
            cacheNodeUpdateStatus(Arrays.asList(cacheNodeUpdateStatus));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cacheNodeUpdateStatus(Consumer<CacheNodeUpdateStatus.Builder>... cacheNodeUpdateStatus) {
            cacheNodeUpdateStatus(Stream.of(cacheNodeUpdateStatus)
                    .map(c -> CacheNodeUpdateStatus.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getEstimatedUpdateTime() {
            return estimatedUpdateTime;
        }

        public final void setEstimatedUpdateTime(String estimatedUpdateTime) {
            this.estimatedUpdateTime = estimatedUpdateTime;
        }

        @Override
        public final Builder estimatedUpdateTime(String estimatedUpdateTime) {
            this.estimatedUpdateTime = estimatedUpdateTime;
            return this;
        }

        public final String getEngine() {
            return engine;
        }

        public final void setEngine(String engine) {
            this.engine = engine;
        }

        @Override
        public final Builder engine(String engine) {
            this.engine = engine;
            return this;
        }

        @Override
        public UpdateAction build() {
            return new UpdateAction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
