/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a copy of an entire Valkey or Redis OSS cluster as of the time when the snapshot was taken.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Snapshot implements SdkPojo, Serializable, ToCopyableBuilder<Snapshot.Builder, Snapshot> {
    private static final SdkField<String> SNAPSHOT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SnapshotName").getter(getter(Snapshot::snapshotName)).setter(setter(Builder::snapshotName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotName").build()).build();

    private static final SdkField<String> REPLICATION_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReplicationGroupId").getter(getter(Snapshot::replicationGroupId))
            .setter(setter(Builder::replicationGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationGroupId").build())
            .build();

    private static final SdkField<String> REPLICATION_GROUP_DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ReplicationGroupDescription")
            .getter(getter(Snapshot::replicationGroupDescription))
            .setter(setter(Builder::replicationGroupDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationGroupDescription")
                    .build()).build();

    private static final SdkField<String> CACHE_CLUSTER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CacheClusterId").getter(getter(Snapshot::cacheClusterId)).setter(setter(Builder::cacheClusterId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheClusterId").build()).build();

    private static final SdkField<String> SNAPSHOT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SnapshotStatus").getter(getter(Snapshot::snapshotStatus)).setter(setter(Builder::snapshotStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotStatus").build()).build();

    private static final SdkField<String> SNAPSHOT_SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SnapshotSource").getter(getter(Snapshot::snapshotSource)).setter(setter(Builder::snapshotSource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotSource").build()).build();

    private static final SdkField<String> CACHE_NODE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CacheNodeType").getter(getter(Snapshot::cacheNodeType)).setter(setter(Builder::cacheNodeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheNodeType").build()).build();

    private static final SdkField<String> ENGINE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Engine")
            .getter(getter(Snapshot::engine)).setter(setter(Builder::engine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Engine").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineVersion").getter(getter(Snapshot::engineVersion)).setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<Integer> NUM_CACHE_NODES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumCacheNodes").getter(getter(Snapshot::numCacheNodes)).setter(setter(Builder::numCacheNodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumCacheNodes").build()).build();

    private static final SdkField<String> PREFERRED_AVAILABILITY_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreferredAvailabilityZone").getter(getter(Snapshot::preferredAvailabilityZone))
            .setter(setter(Builder::preferredAvailabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredAvailabilityZone").build())
            .build();

    private static final SdkField<String> PREFERRED_OUTPOST_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreferredOutpostArn").getter(getter(Snapshot::preferredOutpostArn))
            .setter(setter(Builder::preferredOutpostArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredOutpostArn").build())
            .build();

    private static final SdkField<Instant> CACHE_CLUSTER_CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CacheClusterCreateTime").getter(getter(Snapshot::cacheClusterCreateTime))
            .setter(setter(Builder::cacheClusterCreateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheClusterCreateTime").build())
            .build();

    private static final SdkField<String> PREFERRED_MAINTENANCE_WINDOW_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PreferredMaintenanceWindow")
            .getter(getter(Snapshot::preferredMaintenanceWindow))
            .setter(setter(Builder::preferredMaintenanceWindow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredMaintenanceWindow").build())
            .build();

    private static final SdkField<String> TOPIC_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TopicArn").getter(getter(Snapshot::topicArn)).setter(setter(Builder::topicArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TopicArn").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(Snapshot::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> CACHE_PARAMETER_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CacheParameterGroupName").getter(getter(Snapshot::cacheParameterGroupName))
            .setter(setter(Builder::cacheParameterGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheParameterGroupName").build())
            .build();

    private static final SdkField<String> CACHE_SUBNET_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CacheSubnetGroupName").getter(getter(Snapshot::cacheSubnetGroupName))
            .setter(setter(Builder::cacheSubnetGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheSubnetGroupName").build())
            .build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(Snapshot::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<Boolean> AUTO_MINOR_VERSION_UPGRADE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AutoMinorVersionUpgrade").getter(getter(Snapshot::autoMinorVersionUpgrade))
            .setter(setter(Builder::autoMinorVersionUpgrade))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoMinorVersionUpgrade").build())
            .build();

    private static final SdkField<Integer> SNAPSHOT_RETENTION_LIMIT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("SnapshotRetentionLimit").getter(getter(Snapshot::snapshotRetentionLimit))
            .setter(setter(Builder::snapshotRetentionLimit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotRetentionLimit").build())
            .build();

    private static final SdkField<String> SNAPSHOT_WINDOW_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SnapshotWindow").getter(getter(Snapshot::snapshotWindow)).setter(setter(Builder::snapshotWindow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotWindow").build()).build();

    private static final SdkField<Integer> NUM_NODE_GROUPS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumNodeGroups").getter(getter(Snapshot::numNodeGroups)).setter(setter(Builder::numNodeGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumNodeGroups").build()).build();

    private static final SdkField<String> AUTOMATIC_FAILOVER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AutomaticFailover").getter(getter(Snapshot::automaticFailoverAsString))
            .setter(setter(Builder::automaticFailover))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutomaticFailover").build()).build();

    private static final SdkField<List<NodeSnapshot>> NODE_SNAPSHOTS_FIELD = SdkField
            .<List<NodeSnapshot>> builder(MarshallingType.LIST)
            .memberName("NodeSnapshots")
            .getter(getter(Snapshot::nodeSnapshots))
            .setter(setter(Builder::nodeSnapshots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeSnapshots").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("NodeSnapshot")
                            .memberFieldInfo(
                                    SdkField.<NodeSnapshot> builder(MarshallingType.SDK_POJO)
                                            .constructor(NodeSnapshot::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("NodeSnapshot").build()).build()).build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(Snapshot::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ARN")
            .getter(getter(Snapshot::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").build()).build();

    private static final SdkField<String> DATA_TIERING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataTiering").getter(getter(Snapshot::dataTieringAsString)).setter(setter(Builder::dataTiering))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataTiering").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SNAPSHOT_NAME_FIELD,
            REPLICATION_GROUP_ID_FIELD, REPLICATION_GROUP_DESCRIPTION_FIELD, CACHE_CLUSTER_ID_FIELD, SNAPSHOT_STATUS_FIELD,
            SNAPSHOT_SOURCE_FIELD, CACHE_NODE_TYPE_FIELD, ENGINE_FIELD, ENGINE_VERSION_FIELD, NUM_CACHE_NODES_FIELD,
            PREFERRED_AVAILABILITY_ZONE_FIELD, PREFERRED_OUTPOST_ARN_FIELD, CACHE_CLUSTER_CREATE_TIME_FIELD,
            PREFERRED_MAINTENANCE_WINDOW_FIELD, TOPIC_ARN_FIELD, PORT_FIELD, CACHE_PARAMETER_GROUP_NAME_FIELD,
            CACHE_SUBNET_GROUP_NAME_FIELD, VPC_ID_FIELD, AUTO_MINOR_VERSION_UPGRADE_FIELD, SNAPSHOT_RETENTION_LIMIT_FIELD,
            SNAPSHOT_WINDOW_FIELD, NUM_NODE_GROUPS_FIELD, AUTOMATIC_FAILOVER_FIELD, NODE_SNAPSHOTS_FIELD, KMS_KEY_ID_FIELD,
            ARN_FIELD, DATA_TIERING_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String snapshotName;

    private final String replicationGroupId;

    private final String replicationGroupDescription;

    private final String cacheClusterId;

    private final String snapshotStatus;

    private final String snapshotSource;

    private final String cacheNodeType;

    private final String engine;

    private final String engineVersion;

    private final Integer numCacheNodes;

    private final String preferredAvailabilityZone;

    private final String preferredOutpostArn;

    private final Instant cacheClusterCreateTime;

    private final String preferredMaintenanceWindow;

    private final String topicArn;

    private final Integer port;

    private final String cacheParameterGroupName;

    private final String cacheSubnetGroupName;

    private final String vpcId;

    private final Boolean autoMinorVersionUpgrade;

    private final Integer snapshotRetentionLimit;

    private final String snapshotWindow;

    private final Integer numNodeGroups;

    private final String automaticFailover;

    private final List<NodeSnapshot> nodeSnapshots;

    private final String kmsKeyId;

    private final String arn;

    private final String dataTiering;

    private Snapshot(BuilderImpl builder) {
        this.snapshotName = builder.snapshotName;
        this.replicationGroupId = builder.replicationGroupId;
        this.replicationGroupDescription = builder.replicationGroupDescription;
        this.cacheClusterId = builder.cacheClusterId;
        this.snapshotStatus = builder.snapshotStatus;
        this.snapshotSource = builder.snapshotSource;
        this.cacheNodeType = builder.cacheNodeType;
        this.engine = builder.engine;
        this.engineVersion = builder.engineVersion;
        this.numCacheNodes = builder.numCacheNodes;
        this.preferredAvailabilityZone = builder.preferredAvailabilityZone;
        this.preferredOutpostArn = builder.preferredOutpostArn;
        this.cacheClusterCreateTime = builder.cacheClusterCreateTime;
        this.preferredMaintenanceWindow = builder.preferredMaintenanceWindow;
        this.topicArn = builder.topicArn;
        this.port = builder.port;
        this.cacheParameterGroupName = builder.cacheParameterGroupName;
        this.cacheSubnetGroupName = builder.cacheSubnetGroupName;
        this.vpcId = builder.vpcId;
        this.autoMinorVersionUpgrade = builder.autoMinorVersionUpgrade;
        this.snapshotRetentionLimit = builder.snapshotRetentionLimit;
        this.snapshotWindow = builder.snapshotWindow;
        this.numNodeGroups = builder.numNodeGroups;
        this.automaticFailover = builder.automaticFailover;
        this.nodeSnapshots = builder.nodeSnapshots;
        this.kmsKeyId = builder.kmsKeyId;
        this.arn = builder.arn;
        this.dataTiering = builder.dataTiering;
    }

    /**
     * <p>
     * The name of a snapshot. For an automatic snapshot, the name is system-generated. For a manual snapshot, this is
     * the user-provided name.
     * </p>
     * 
     * @return The name of a snapshot. For an automatic snapshot, the name is system-generated. For a manual snapshot,
     *         this is the user-provided name.
     */
    public final String snapshotName() {
        return snapshotName;
    }

    /**
     * <p>
     * The unique identifier of the source replication group.
     * </p>
     * 
     * @return The unique identifier of the source replication group.
     */
    public final String replicationGroupId() {
        return replicationGroupId;
    }

    /**
     * <p>
     * A description of the source replication group.
     * </p>
     * 
     * @return A description of the source replication group.
     */
    public final String replicationGroupDescription() {
        return replicationGroupDescription;
    }

    /**
     * <p>
     * The user-supplied identifier of the source cluster.
     * </p>
     * 
     * @return The user-supplied identifier of the source cluster.
     */
    public final String cacheClusterId() {
        return cacheClusterId;
    }

    /**
     * <p>
     * The status of the snapshot. Valid values: <code>creating</code> | <code>available</code> | <code>restoring</code>
     * | <code>copying</code> | <code>deleting</code>.
     * </p>
     * 
     * @return The status of the snapshot. Valid values: <code>creating</code> | <code>available</code> |
     *         <code>restoring</code> | <code>copying</code> | <code>deleting</code>.
     */
    public final String snapshotStatus() {
        return snapshotStatus;
    }

    /**
     * <p>
     * Indicates whether the snapshot is from an automatic backup (<code>automated</code>) or was created manually (
     * <code>manual</code>).
     * </p>
     * 
     * @return Indicates whether the snapshot is from an automatic backup (<code>automated</code>) or was created
     *         manually (<code>manual</code>).
     */
    public final String snapshotSource() {
        return snapshotSource;
    }

    /**
     * <p>
     * The name of the compute and memory capacity node type for the source cluster.
     * </p>
     * <p>
     * The following node types are supported by ElastiCache. Generally speaking, the current generation types provide
     * more memory and computational power at lower cost when compared to their equivalent previous generation
     * counterparts.
     * </p>
     * <ul>
     * <li>
     * <p>
     * General purpose:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Current generation:
     * </p>
     * <p>
     * <b>M7g node types</b>: <code>cache.m7g.large</code>, <code>cache.m7g.xlarge</code>,
     * <code>cache.m7g.2xlarge</code>, <code>cache.m7g.4xlarge</code>, <code>cache.m7g.8xlarge</code>,
     * <code>cache.m7g.12xlarge</code>, <code>cache.m7g.16xlarge</code>
     * </p>
     * <note>
     * <p>
     * For region availability, see <a href=
     * "https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/CacheNodes.SupportedTypes.html#CacheNodes.SupportedTypesByRegion"
     * >Supported Node Types</a>
     * </p>
     * </note>
     * <p>
     * <b>M6g node types</b> (available only for Redis OSS engine version 5.0.6 onward and for Memcached engine version
     * 1.5.16 onward): <code>cache.m6g.large</code>, <code>cache.m6g.xlarge</code>, <code>cache.m6g.2xlarge</code>,
     * <code>cache.m6g.4xlarge</code>, <code>cache.m6g.8xlarge</code>, <code>cache.m6g.12xlarge</code>,
     * <code>cache.m6g.16xlarge</code>
     * </p>
     * <p>
     * <b>M5 node types:</b> <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>, <code>cache.m5.2xlarge</code>,
     * <code>cache.m5.4xlarge</code>, <code>cache.m5.12xlarge</code>, <code>cache.m5.24xlarge</code>
     * </p>
     * <p>
     * <b>M4 node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>, <code>cache.m4.2xlarge</code>,
     * <code>cache.m4.4xlarge</code>, <code>cache.m4.10xlarge</code>
     * </p>
     * <p>
     * <b>T4g node types</b> (available only for Redis OSS engine version 5.0.6 onward and Memcached engine version
     * 1.5.16 onward): <code>cache.t4g.micro</code>, <code>cache.t4g.small</code>, <code>cache.t4g.medium</code>
     * </p>
     * <p>
     * <b>T3 node types:</b> <code>cache.t3.micro</code>, <code>cache.t3.small</code>, <code>cache.t3.medium</code>
     * </p>
     * <p>
     * <b>T2 node types:</b> <code>cache.t2.micro</code>, <code>cache.t2.small</code>, <code>cache.t2.medium</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Previous generation: (not recommended. Existing clusters are still supported but creation of new clusters is not
     * supported for these types.)
     * </p>
     * <p>
     * <b>T1 node types:</b> <code>cache.t1.micro</code>
     * </p>
     * <p>
     * <b>M1 node types:</b> <code>cache.m1.small</code>, <code>cache.m1.medium</code>, <code>cache.m1.large</code>,
     * <code>cache.m1.xlarge</code>
     * </p>
     * <p>
     * <b>M3 node types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>, <code>cache.m3.xlarge</code>,
     * <code>cache.m3.2xlarge</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Compute optimized:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Previous generation: (not recommended. Existing clusters are still supported but creation of new clusters is not
     * supported for these types.)
     * </p>
     * <p>
     * <b>C1 node types:</b> <code>cache.c1.xlarge</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Memory optimized:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Current generation:
     * </p>
     * <p>
     * <b>R7g node types</b>: <code>cache.r7g.large</code>, <code>cache.r7g.xlarge</code>,
     * <code>cache.r7g.2xlarge</code>, <code>cache.r7g.4xlarge</code>, <code>cache.r7g.8xlarge</code>,
     * <code>cache.r7g.12xlarge</code>, <code>cache.r7g.16xlarge</code>
     * </p>
     * <note>
     * <p>
     * For region availability, see <a href=
     * "https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/CacheNodes.SupportedTypes.html#CacheNodes.SupportedTypesByRegion"
     * >Supported Node Types</a>
     * </p>
     * </note>
     * <p>
     * <b>R6g node types</b> (available only for Redis OSS engine version 5.0.6 onward and for Memcached engine version
     * 1.5.16 onward): <code>cache.r6g.large</code>, <code>cache.r6g.xlarge</code>, <code>cache.r6g.2xlarge</code>,
     * <code>cache.r6g.4xlarge</code>, <code>cache.r6g.8xlarge</code>, <code>cache.r6g.12xlarge</code>,
     * <code>cache.r6g.16xlarge</code>
     * </p>
     * <p>
     * <b>R5 node types:</b> <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>, <code>cache.r5.2xlarge</code>,
     * <code>cache.r5.4xlarge</code>, <code>cache.r5.12xlarge</code>, <code>cache.r5.24xlarge</code>
     * </p>
     * <p>
     * <b>R4 node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>, <code>cache.r4.2xlarge</code>,
     * <code>cache.r4.4xlarge</code>, <code>cache.r4.8xlarge</code>, <code>cache.r4.16xlarge</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Previous generation: (not recommended. Existing clusters are still supported but creation of new clusters is not
     * supported for these types.)
     * </p>
     * <p>
     * <b>M2 node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>, <code>cache.m2.4xlarge</code>
     * </p>
     * <p>
     * <b>R3 node types:</b> <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>, <code>cache.r3.2xlarge</code>,
     * <code>cache.r3.4xlarge</code>, <code>cache.r3.8xlarge</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * <b>Additional node type info</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * All current generation instance types are created in Amazon VPC by default.
     * </p>
     * </li>
     * <li>
     * <p>
     * Valkey or Redis OSS append-only files (AOF) are not supported for T1 or T2 instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * Valkey or Redis OSS Multi-AZ with automatic failover is not supported on T1 instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * The configuration variables <code>appendonly</code> and <code>appendfsync</code> are not supported on Valkey, or
     * on Redis OSS version 2.8.22 and later.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of the compute and memory capacity node type for the source cluster.</p>
     *         <p>
     *         The following node types are supported by ElastiCache. Generally speaking, the current generation types
     *         provide more memory and computational power at lower cost when compared to their equivalent previous
     *         generation counterparts.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         General purpose:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Current generation:
     *         </p>
     *         <p>
     *         <b>M7g node types</b>: <code>cache.m7g.large</code>, <code>cache.m7g.xlarge</code>,
     *         <code>cache.m7g.2xlarge</code>, <code>cache.m7g.4xlarge</code>, <code>cache.m7g.8xlarge</code>,
     *         <code>cache.m7g.12xlarge</code>, <code>cache.m7g.16xlarge</code>
     *         </p>
     *         <note>
     *         <p>
     *         For region availability, see <a href=
     *         "https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/CacheNodes.SupportedTypes.html#CacheNodes.SupportedTypesByRegion"
     *         >Supported Node Types</a>
     *         </p>
     *         </note>
     *         <p>
     *         <b>M6g node types</b> (available only for Redis OSS engine version 5.0.6 onward and for Memcached engine
     *         version 1.5.16 onward): <code>cache.m6g.large</code>, <code>cache.m6g.xlarge</code>,
     *         <code>cache.m6g.2xlarge</code>, <code>cache.m6g.4xlarge</code>, <code>cache.m6g.8xlarge</code>,
     *         <code>cache.m6g.12xlarge</code>, <code>cache.m6g.16xlarge</code>
     *         </p>
     *         <p>
     *         <b>M5 node types:</b> <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>,
     *         <code>cache.m5.2xlarge</code>, <code>cache.m5.4xlarge</code>, <code>cache.m5.12xlarge</code>,
     *         <code>cache.m5.24xlarge</code>
     *         </p>
     *         <p>
     *         <b>M4 node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>,
     *         <code>cache.m4.2xlarge</code>, <code>cache.m4.4xlarge</code>, <code>cache.m4.10xlarge</code>
     *         </p>
     *         <p>
     *         <b>T4g node types</b> (available only for Redis OSS engine version 5.0.6 onward and Memcached engine
     *         version 1.5.16 onward): <code>cache.t4g.micro</code>, <code>cache.t4g.small</code>,
     *         <code>cache.t4g.medium</code>
     *         </p>
     *         <p>
     *         <b>T3 node types:</b> <code>cache.t3.micro</code>, <code>cache.t3.small</code>,
     *         <code>cache.t3.medium</code>
     *         </p>
     *         <p>
     *         <b>T2 node types:</b> <code>cache.t2.micro</code>, <code>cache.t2.small</code>,
     *         <code>cache.t2.medium</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Previous generation: (not recommended. Existing clusters are still supported but creation of new clusters
     *         is not supported for these types.)
     *         </p>
     *         <p>
     *         <b>T1 node types:</b> <code>cache.t1.micro</code>
     *         </p>
     *         <p>
     *         <b>M1 node types:</b> <code>cache.m1.small</code>, <code>cache.m1.medium</code>,
     *         <code>cache.m1.large</code>, <code>cache.m1.xlarge</code>
     *         </p>
     *         <p>
     *         <b>M3 node types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>,
     *         <code>cache.m3.xlarge</code>, <code>cache.m3.2xlarge</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Compute optimized:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Previous generation: (not recommended. Existing clusters are still supported but creation of new clusters
     *         is not supported for these types.)
     *         </p>
     *         <p>
     *         <b>C1 node types:</b> <code>cache.c1.xlarge</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Memory optimized:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Current generation:
     *         </p>
     *         <p>
     *         <b>R7g node types</b>: <code>cache.r7g.large</code>, <code>cache.r7g.xlarge</code>,
     *         <code>cache.r7g.2xlarge</code>, <code>cache.r7g.4xlarge</code>, <code>cache.r7g.8xlarge</code>,
     *         <code>cache.r7g.12xlarge</code>, <code>cache.r7g.16xlarge</code>
     *         </p>
     *         <note>
     *         <p>
     *         For region availability, see <a href=
     *         "https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/CacheNodes.SupportedTypes.html#CacheNodes.SupportedTypesByRegion"
     *         >Supported Node Types</a>
     *         </p>
     *         </note>
     *         <p>
     *         <b>R6g node types</b> (available only for Redis OSS engine version 5.0.6 onward and for Memcached engine
     *         version 1.5.16 onward): <code>cache.r6g.large</code>, <code>cache.r6g.xlarge</code>,
     *         <code>cache.r6g.2xlarge</code>, <code>cache.r6g.4xlarge</code>, <code>cache.r6g.8xlarge</code>,
     *         <code>cache.r6g.12xlarge</code>, <code>cache.r6g.16xlarge</code>
     *         </p>
     *         <p>
     *         <b>R5 node types:</b> <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>,
     *         <code>cache.r5.2xlarge</code>, <code>cache.r5.4xlarge</code>, <code>cache.r5.12xlarge</code>,
     *         <code>cache.r5.24xlarge</code>
     *         </p>
     *         <p>
     *         <b>R4 node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>,
     *         <code>cache.r4.2xlarge</code>, <code>cache.r4.4xlarge</code>, <code>cache.r4.8xlarge</code>,
     *         <code>cache.r4.16xlarge</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Previous generation: (not recommended. Existing clusters are still supported but creation of new clusters
     *         is not supported for these types.)
     *         </p>
     *         <p>
     *         <b>M2 node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>,
     *         <code>cache.m2.4xlarge</code>
     *         </p>
     *         <p>
     *         <b>R3 node types:</b> <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>,
     *         <code>cache.r3.2xlarge</code>, <code>cache.r3.4xlarge</code>, <code>cache.r3.8xlarge</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Additional node type info</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         All current generation instance types are created in Amazon VPC by default.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Valkey or Redis OSS append-only files (AOF) are not supported for T1 or T2 instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Valkey or Redis OSS Multi-AZ with automatic failover is not supported on T1 instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The configuration variables <code>appendonly</code> and <code>appendfsync</code> are not supported on
     *         Valkey, or on Redis OSS version 2.8.22 and later.
     *         </p>
     *         </li>
     */
    public final String cacheNodeType() {
        return cacheNodeType;
    }

    /**
     * <p>
     * The name of the cache engine (<code>memcached</code> or <code>redis</code>) used by the source cluster.
     * </p>
     * 
     * @return The name of the cache engine (<code>memcached</code> or <code>redis</code>) used by the source cluster.
     */
    public final String engine() {
        return engine;
    }

    /**
     * <p>
     * The version of the cache engine version that is used by the source cluster.
     * </p>
     * 
     * @return The version of the cache engine version that is used by the source cluster.
     */
    public final String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * The number of cache nodes in the source cluster.
     * </p>
     * <p>
     * For clusters running Valkey or Redis OSS, this value must be 1. For clusters running Memcached, this value must
     * be between 1 and 40.
     * </p>
     * 
     * @return The number of cache nodes in the source cluster.</p>
     *         <p>
     *         For clusters running Valkey or Redis OSS, this value must be 1. For clusters running Memcached, this
     *         value must be between 1 and 40.
     */
    public final Integer numCacheNodes() {
        return numCacheNodes;
    }

    /**
     * <p>
     * The name of the Availability Zone in which the source cluster is located.
     * </p>
     * 
     * @return The name of the Availability Zone in which the source cluster is located.
     */
    public final String preferredAvailabilityZone() {
        return preferredAvailabilityZone;
    }

    /**
     * <p>
     * The ARN (Amazon Resource Name) of the preferred outpost.
     * </p>
     * 
     * @return The ARN (Amazon Resource Name) of the preferred outpost.
     */
    public final String preferredOutpostArn() {
        return preferredOutpostArn;
    }

    /**
     * <p>
     * The date and time when the source cluster was created.
     * </p>
     * 
     * @return The date and time when the source cluster was created.
     */
    public final Instant cacheClusterCreateTime() {
        return cacheClusterCreateTime;
    }

    /**
     * <p>
     * Specifies the weekly time range during which maintenance on the cluster is performed. It is specified as a range
     * in the format ddd:hh24:mi-ddd:hh24:mi (24H Clock UTC). The minimum maintenance window is a 60 minute period.
     * </p>
     * <p>
     * Valid values for <code>ddd</code> are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>sun</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>mon</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tue</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>wed</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>thu</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>fri</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sat</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Example: <code>sun:23:00-mon:01:30</code>
     * </p>
     * 
     * @return Specifies the weekly time range during which maintenance on the cluster is performed. It is specified as
     *         a range in the format ddd:hh24:mi-ddd:hh24:mi (24H Clock UTC). The minimum maintenance window is a 60
     *         minute period.</p>
     *         <p>
     *         Valid values for <code>ddd</code> are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>sun</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>mon</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>tue</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>wed</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>thu</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>fri</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sat</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Example: <code>sun:23:00-mon:01:30</code>
     */
    public final String preferredMaintenanceWindow() {
        return preferredMaintenanceWindow;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the topic used by the source cluster for publishing notifications.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the topic used by the source cluster for publishing notifications.
     */
    public final String topicArn() {
        return topicArn;
    }

    /**
     * <p>
     * The port number used by each cache nodes in the source cluster.
     * </p>
     * 
     * @return The port number used by each cache nodes in the source cluster.
     */
    public final Integer port() {
        return port;
    }

    /**
     * <p>
     * The cache parameter group that is associated with the source cluster.
     * </p>
     * 
     * @return The cache parameter group that is associated with the source cluster.
     */
    public final String cacheParameterGroupName() {
        return cacheParameterGroupName;
    }

    /**
     * <p>
     * The name of the cache subnet group associated with the source cluster.
     * </p>
     * 
     * @return The name of the cache subnet group associated with the source cluster.
     */
    public final String cacheSubnetGroupName() {
        return cacheSubnetGroupName;
    }

    /**
     * <p>
     * The Amazon Virtual Private Cloud identifier (VPC ID) of the cache subnet group for the source cluster.
     * </p>
     * 
     * @return The Amazon Virtual Private Cloud identifier (VPC ID) of the cache subnet group for the source cluster.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     *  If you are running Valkey 7.2 and above or Redis OSS engine version 6.0 and above, set this parameter to yes if
     * you want to opt-in to the next auto minor version upgrade campaign. This parameter is disabled for previous
     * versions. 
     * </p>
     * 
     * @return  If you are running Valkey 7.2 and above or Redis OSS engine version 6.0 and above, set this parameter to
     *         yes if you want to opt-in to the next auto minor version upgrade campaign. This parameter is disabled for
     *         previous versions. 
     */
    public final Boolean autoMinorVersionUpgrade() {
        return autoMinorVersionUpgrade;
    }

    /**
     * <p>
     * For an automatic snapshot, the number of days for which ElastiCache retains the snapshot before deleting it.
     * </p>
     * <p>
     * For manual snapshots, this field reflects the <code>SnapshotRetentionLimit</code> for the source cluster when the
     * snapshot was created. This field is otherwise ignored: Manual snapshots do not expire, and can only be deleted
     * using the <code>DeleteSnapshot</code> operation.
     * </p>
     * <p>
     * <b>Important</b> If the value of SnapshotRetentionLimit is set to zero (0), backups are turned off.
     * </p>
     * 
     * @return For an automatic snapshot, the number of days for which ElastiCache retains the snapshot before deleting
     *         it.</p>
     *         <p>
     *         For manual snapshots, this field reflects the <code>SnapshotRetentionLimit</code> for the source cluster
     *         when the snapshot was created. This field is otherwise ignored: Manual snapshots do not expire, and can
     *         only be deleted using the <code>DeleteSnapshot</code> operation.
     *         </p>
     *         <p>
     *         <b>Important</b> If the value of SnapshotRetentionLimit is set to zero (0), backups are turned off.
     */
    public final Integer snapshotRetentionLimit() {
        return snapshotRetentionLimit;
    }

    /**
     * <p>
     * The daily time range during which ElastiCache takes daily snapshots of the source cluster.
     * </p>
     * 
     * @return The daily time range during which ElastiCache takes daily snapshots of the source cluster.
     */
    public final String snapshotWindow() {
        return snapshotWindow;
    }

    /**
     * <p>
     * The number of node groups (shards) in this snapshot. When restoring from a snapshot, the number of node groups
     * (shards) in the snapshot and in the restored replication group must be the same.
     * </p>
     * 
     * @return The number of node groups (shards) in this snapshot. When restoring from a snapshot, the number of node
     *         groups (shards) in the snapshot and in the restored replication group must be the same.
     */
    public final Integer numNodeGroups() {
        return numNodeGroups;
    }

    /**
     * <p>
     * Indicates the status of automatic failover for the source Valkey or Redis OSS replication group.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #automaticFailover}
     * will return {@link AutomaticFailoverStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #automaticFailoverAsString}.
     * </p>
     * 
     * @return Indicates the status of automatic failover for the source Valkey or Redis OSS replication group.
     * @see AutomaticFailoverStatus
     */
    public final AutomaticFailoverStatus automaticFailover() {
        return AutomaticFailoverStatus.fromValue(automaticFailover);
    }

    /**
     * <p>
     * Indicates the status of automatic failover for the source Valkey or Redis OSS replication group.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #automaticFailover}
     * will return {@link AutomaticFailoverStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #automaticFailoverAsString}.
     * </p>
     * 
     * @return Indicates the status of automatic failover for the source Valkey or Redis OSS replication group.
     * @see AutomaticFailoverStatus
     */
    public final String automaticFailoverAsString() {
        return automaticFailover;
    }

    /**
     * For responses, this returns true if the service returned a value for the NodeSnapshots property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNodeSnapshots() {
        return nodeSnapshots != null && !(nodeSnapshots instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the cache nodes in the source cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNodeSnapshots} method.
     * </p>
     * 
     * @return A list of the cache nodes in the source cluster.
     */
    public final List<NodeSnapshot> nodeSnapshots() {
        return nodeSnapshots;
    }

    /**
     * <p>
     * The ID of the KMS key used to encrypt the snapshot.
     * </p>
     * 
     * @return The ID of the KMS key used to encrypt the snapshot.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The ARN (Amazon Resource Name) of the snapshot.
     * </p>
     * 
     * @return The ARN (Amazon Resource Name) of the snapshot.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * Enables data tiering. Data tiering is only supported for replication groups using the r6gd node type. This
     * parameter must be set to true when using r6gd nodes. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/data-tiering.html">Data tiering</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dataTiering} will
     * return {@link DataTieringStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #dataTieringAsString}.
     * </p>
     * 
     * @return Enables data tiering. Data tiering is only supported for replication groups using the r6gd node type.
     *         This parameter must be set to true when using r6gd nodes. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/data-tiering.html">Data tiering</a>.
     * @see DataTieringStatus
     */
    public final DataTieringStatus dataTiering() {
        return DataTieringStatus.fromValue(dataTiering);
    }

    /**
     * <p>
     * Enables data tiering. Data tiering is only supported for replication groups using the r6gd node type. This
     * parameter must be set to true when using r6gd nodes. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/data-tiering.html">Data tiering</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dataTiering} will
     * return {@link DataTieringStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #dataTieringAsString}.
     * </p>
     * 
     * @return Enables data tiering. Data tiering is only supported for replication groups using the r6gd node type.
     *         This parameter must be set to true when using r6gd nodes. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/data-tiering.html">Data tiering</a>.
     * @see DataTieringStatus
     */
    public final String dataTieringAsString() {
        return dataTiering;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(snapshotName());
        hashCode = 31 * hashCode + Objects.hashCode(replicationGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(replicationGroupDescription());
        hashCode = 31 * hashCode + Objects.hashCode(cacheClusterId());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotStatus());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotSource());
        hashCode = 31 * hashCode + Objects.hashCode(cacheNodeType());
        hashCode = 31 * hashCode + Objects.hashCode(engine());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(numCacheNodes());
        hashCode = 31 * hashCode + Objects.hashCode(preferredAvailabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(preferredOutpostArn());
        hashCode = 31 * hashCode + Objects.hashCode(cacheClusterCreateTime());
        hashCode = 31 * hashCode + Objects.hashCode(preferredMaintenanceWindow());
        hashCode = 31 * hashCode + Objects.hashCode(topicArn());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(cacheParameterGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(cacheSubnetGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(autoMinorVersionUpgrade());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotRetentionLimit());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotWindow());
        hashCode = 31 * hashCode + Objects.hashCode(numNodeGroups());
        hashCode = 31 * hashCode + Objects.hashCode(automaticFailoverAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasNodeSnapshots() ? nodeSnapshots() : null);
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(dataTieringAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Snapshot)) {
            return false;
        }
        Snapshot other = (Snapshot) obj;
        return Objects.equals(snapshotName(), other.snapshotName())
                && Objects.equals(replicationGroupId(), other.replicationGroupId())
                && Objects.equals(replicationGroupDescription(), other.replicationGroupDescription())
                && Objects.equals(cacheClusterId(), other.cacheClusterId())
                && Objects.equals(snapshotStatus(), other.snapshotStatus())
                && Objects.equals(snapshotSource(), other.snapshotSource())
                && Objects.equals(cacheNodeType(), other.cacheNodeType()) && Objects.equals(engine(), other.engine())
                && Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(numCacheNodes(), other.numCacheNodes())
                && Objects.equals(preferredAvailabilityZone(), other.preferredAvailabilityZone())
                && Objects.equals(preferredOutpostArn(), other.preferredOutpostArn())
                && Objects.equals(cacheClusterCreateTime(), other.cacheClusterCreateTime())
                && Objects.equals(preferredMaintenanceWindow(), other.preferredMaintenanceWindow())
                && Objects.equals(topicArn(), other.topicArn()) && Objects.equals(port(), other.port())
                && Objects.equals(cacheParameterGroupName(), other.cacheParameterGroupName())
                && Objects.equals(cacheSubnetGroupName(), other.cacheSubnetGroupName()) && Objects.equals(vpcId(), other.vpcId())
                && Objects.equals(autoMinorVersionUpgrade(), other.autoMinorVersionUpgrade())
                && Objects.equals(snapshotRetentionLimit(), other.snapshotRetentionLimit())
                && Objects.equals(snapshotWindow(), other.snapshotWindow())
                && Objects.equals(numNodeGroups(), other.numNodeGroups())
                && Objects.equals(automaticFailoverAsString(), other.automaticFailoverAsString())
                && hasNodeSnapshots() == other.hasNodeSnapshots() && Objects.equals(nodeSnapshots(), other.nodeSnapshots())
                && Objects.equals(kmsKeyId(), other.kmsKeyId()) && Objects.equals(arn(), other.arn())
                && Objects.equals(dataTieringAsString(), other.dataTieringAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Snapshot").add("SnapshotName", snapshotName()).add("ReplicationGroupId", replicationGroupId())
                .add("ReplicationGroupDescription", replicationGroupDescription()).add("CacheClusterId", cacheClusterId())
                .add("SnapshotStatus", snapshotStatus()).add("SnapshotSource", snapshotSource())
                .add("CacheNodeType", cacheNodeType()).add("Engine", engine()).add("EngineVersion", engineVersion())
                .add("NumCacheNodes", numCacheNodes()).add("PreferredAvailabilityZone", preferredAvailabilityZone())
                .add("PreferredOutpostArn", preferredOutpostArn()).add("CacheClusterCreateTime", cacheClusterCreateTime())
                .add("PreferredMaintenanceWindow", preferredMaintenanceWindow()).add("TopicArn", topicArn()).add("Port", port())
                .add("CacheParameterGroupName", cacheParameterGroupName()).add("CacheSubnetGroupName", cacheSubnetGroupName())
                .add("VpcId", vpcId()).add("AutoMinorVersionUpgrade", autoMinorVersionUpgrade())
                .add("SnapshotRetentionLimit", snapshotRetentionLimit()).add("SnapshotWindow", snapshotWindow())
                .add("NumNodeGroups", numNodeGroups()).add("AutomaticFailover", automaticFailoverAsString())
                .add("NodeSnapshots", hasNodeSnapshots() ? nodeSnapshots() : null).add("KmsKeyId", kmsKeyId()).add("ARN", arn())
                .add("DataTiering", dataTieringAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SnapshotName":
            return Optional.ofNullable(clazz.cast(snapshotName()));
        case "ReplicationGroupId":
            return Optional.ofNullable(clazz.cast(replicationGroupId()));
        case "ReplicationGroupDescription":
            return Optional.ofNullable(clazz.cast(replicationGroupDescription()));
        case "CacheClusterId":
            return Optional.ofNullable(clazz.cast(cacheClusterId()));
        case "SnapshotStatus":
            return Optional.ofNullable(clazz.cast(snapshotStatus()));
        case "SnapshotSource":
            return Optional.ofNullable(clazz.cast(snapshotSource()));
        case "CacheNodeType":
            return Optional.ofNullable(clazz.cast(cacheNodeType()));
        case "Engine":
            return Optional.ofNullable(clazz.cast(engine()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "NumCacheNodes":
            return Optional.ofNullable(clazz.cast(numCacheNodes()));
        case "PreferredAvailabilityZone":
            return Optional.ofNullable(clazz.cast(preferredAvailabilityZone()));
        case "PreferredOutpostArn":
            return Optional.ofNullable(clazz.cast(preferredOutpostArn()));
        case "CacheClusterCreateTime":
            return Optional.ofNullable(clazz.cast(cacheClusterCreateTime()));
        case "PreferredMaintenanceWindow":
            return Optional.ofNullable(clazz.cast(preferredMaintenanceWindow()));
        case "TopicArn":
            return Optional.ofNullable(clazz.cast(topicArn()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "CacheParameterGroupName":
            return Optional.ofNullable(clazz.cast(cacheParameterGroupName()));
        case "CacheSubnetGroupName":
            return Optional.ofNullable(clazz.cast(cacheSubnetGroupName()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "AutoMinorVersionUpgrade":
            return Optional.ofNullable(clazz.cast(autoMinorVersionUpgrade()));
        case "SnapshotRetentionLimit":
            return Optional.ofNullable(clazz.cast(snapshotRetentionLimit()));
        case "SnapshotWindow":
            return Optional.ofNullable(clazz.cast(snapshotWindow()));
        case "NumNodeGroups":
            return Optional.ofNullable(clazz.cast(numNodeGroups()));
        case "AutomaticFailover":
            return Optional.ofNullable(clazz.cast(automaticFailoverAsString()));
        case "NodeSnapshots":
            return Optional.ofNullable(clazz.cast(nodeSnapshots()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        case "DataTiering":
            return Optional.ofNullable(clazz.cast(dataTieringAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SnapshotName", SNAPSHOT_NAME_FIELD);
        map.put("ReplicationGroupId", REPLICATION_GROUP_ID_FIELD);
        map.put("ReplicationGroupDescription", REPLICATION_GROUP_DESCRIPTION_FIELD);
        map.put("CacheClusterId", CACHE_CLUSTER_ID_FIELD);
        map.put("SnapshotStatus", SNAPSHOT_STATUS_FIELD);
        map.put("SnapshotSource", SNAPSHOT_SOURCE_FIELD);
        map.put("CacheNodeType", CACHE_NODE_TYPE_FIELD);
        map.put("Engine", ENGINE_FIELD);
        map.put("EngineVersion", ENGINE_VERSION_FIELD);
        map.put("NumCacheNodes", NUM_CACHE_NODES_FIELD);
        map.put("PreferredAvailabilityZone", PREFERRED_AVAILABILITY_ZONE_FIELD);
        map.put("PreferredOutpostArn", PREFERRED_OUTPOST_ARN_FIELD);
        map.put("CacheClusterCreateTime", CACHE_CLUSTER_CREATE_TIME_FIELD);
        map.put("PreferredMaintenanceWindow", PREFERRED_MAINTENANCE_WINDOW_FIELD);
        map.put("TopicArn", TOPIC_ARN_FIELD);
        map.put("Port", PORT_FIELD);
        map.put("CacheParameterGroupName", CACHE_PARAMETER_GROUP_NAME_FIELD);
        map.put("CacheSubnetGroupName", CACHE_SUBNET_GROUP_NAME_FIELD);
        map.put("VpcId", VPC_ID_FIELD);
        map.put("AutoMinorVersionUpgrade", AUTO_MINOR_VERSION_UPGRADE_FIELD);
        map.put("SnapshotRetentionLimit", SNAPSHOT_RETENTION_LIMIT_FIELD);
        map.put("SnapshotWindow", SNAPSHOT_WINDOW_FIELD);
        map.put("NumNodeGroups", NUM_NODE_GROUPS_FIELD);
        map.put("AutomaticFailover", AUTOMATIC_FAILOVER_FIELD);
        map.put("NodeSnapshots", NODE_SNAPSHOTS_FIELD);
        map.put("KmsKeyId", KMS_KEY_ID_FIELD);
        map.put("ARN", ARN_FIELD);
        map.put("DataTiering", DATA_TIERING_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Snapshot, T> g) {
        return obj -> g.apply((Snapshot) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Snapshot> {
        /**
         * <p>
         * The name of a snapshot. For an automatic snapshot, the name is system-generated. For a manual snapshot, this
         * is the user-provided name.
         * </p>
         * 
         * @param snapshotName
         *        The name of a snapshot. For an automatic snapshot, the name is system-generated. For a manual
         *        snapshot, this is the user-provided name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotName(String snapshotName);

        /**
         * <p>
         * The unique identifier of the source replication group.
         * </p>
         * 
         * @param replicationGroupId
         *        The unique identifier of the source replication group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationGroupId(String replicationGroupId);

        /**
         * <p>
         * A description of the source replication group.
         * </p>
         * 
         * @param replicationGroupDescription
         *        A description of the source replication group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationGroupDescription(String replicationGroupDescription);

        /**
         * <p>
         * The user-supplied identifier of the source cluster.
         * </p>
         * 
         * @param cacheClusterId
         *        The user-supplied identifier of the source cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheClusterId(String cacheClusterId);

        /**
         * <p>
         * The status of the snapshot. Valid values: <code>creating</code> | <code>available</code> |
         * <code>restoring</code> | <code>copying</code> | <code>deleting</code>.
         * </p>
         * 
         * @param snapshotStatus
         *        The status of the snapshot. Valid values: <code>creating</code> | <code>available</code> |
         *        <code>restoring</code> | <code>copying</code> | <code>deleting</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotStatus(String snapshotStatus);

        /**
         * <p>
         * Indicates whether the snapshot is from an automatic backup (<code>automated</code>) or was created manually (
         * <code>manual</code>).
         * </p>
         * 
         * @param snapshotSource
         *        Indicates whether the snapshot is from an automatic backup (<code>automated</code>) or was created
         *        manually (<code>manual</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotSource(String snapshotSource);

        /**
         * <p>
         * The name of the compute and memory capacity node type for the source cluster.
         * </p>
         * <p>
         * The following node types are supported by ElastiCache. Generally speaking, the current generation types
         * provide more memory and computational power at lower cost when compared to their equivalent previous
         * generation counterparts.
         * </p>
         * <ul>
         * <li>
         * <p>
         * General purpose:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Current generation:
         * </p>
         * <p>
         * <b>M7g node types</b>: <code>cache.m7g.large</code>, <code>cache.m7g.xlarge</code>,
         * <code>cache.m7g.2xlarge</code>, <code>cache.m7g.4xlarge</code>, <code>cache.m7g.8xlarge</code>,
         * <code>cache.m7g.12xlarge</code>, <code>cache.m7g.16xlarge</code>
         * </p>
         * <note>
         * <p>
         * For region availability, see <a href=
         * "https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/CacheNodes.SupportedTypes.html#CacheNodes.SupportedTypesByRegion"
         * >Supported Node Types</a>
         * </p>
         * </note>
         * <p>
         * <b>M6g node types</b> (available only for Redis OSS engine version 5.0.6 onward and for Memcached engine
         * version 1.5.16 onward): <code>cache.m6g.large</code>, <code>cache.m6g.xlarge</code>,
         * <code>cache.m6g.2xlarge</code>, <code>cache.m6g.4xlarge</code>, <code>cache.m6g.8xlarge</code>,
         * <code>cache.m6g.12xlarge</code>, <code>cache.m6g.16xlarge</code>
         * </p>
         * <p>
         * <b>M5 node types:</b> <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>,
         * <code>cache.m5.2xlarge</code>, <code>cache.m5.4xlarge</code>, <code>cache.m5.12xlarge</code>,
         * <code>cache.m5.24xlarge</code>
         * </p>
         * <p>
         * <b>M4 node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>,
         * <code>cache.m4.2xlarge</code>, <code>cache.m4.4xlarge</code>, <code>cache.m4.10xlarge</code>
         * </p>
         * <p>
         * <b>T4g node types</b> (available only for Redis OSS engine version 5.0.6 onward and Memcached engine version
         * 1.5.16 onward): <code>cache.t4g.micro</code>, <code>cache.t4g.small</code>, <code>cache.t4g.medium</code>
         * </p>
         * <p>
         * <b>T3 node types:</b> <code>cache.t3.micro</code>, <code>cache.t3.small</code>, <code>cache.t3.medium</code>
         * </p>
         * <p>
         * <b>T2 node types:</b> <code>cache.t2.micro</code>, <code>cache.t2.small</code>, <code>cache.t2.medium</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Previous generation: (not recommended. Existing clusters are still supported but creation of new clusters is
         * not supported for these types.)
         * </p>
         * <p>
         * <b>T1 node types:</b> <code>cache.t1.micro</code>
         * </p>
         * <p>
         * <b>M1 node types:</b> <code>cache.m1.small</code>, <code>cache.m1.medium</code>, <code>cache.m1.large</code>,
         * <code>cache.m1.xlarge</code>
         * </p>
         * <p>
         * <b>M3 node types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>, <code>cache.m3.xlarge</code>, <code>cache.m3.2xlarge</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Compute optimized:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Previous generation: (not recommended. Existing clusters are still supported but creation of new clusters is
         * not supported for these types.)
         * </p>
         * <p>
         * <b>C1 node types:</b> <code>cache.c1.xlarge</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Memory optimized:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Current generation:
         * </p>
         * <p>
         * <b>R7g node types</b>: <code>cache.r7g.large</code>, <code>cache.r7g.xlarge</code>,
         * <code>cache.r7g.2xlarge</code>, <code>cache.r7g.4xlarge</code>, <code>cache.r7g.8xlarge</code>,
         * <code>cache.r7g.12xlarge</code>, <code>cache.r7g.16xlarge</code>
         * </p>
         * <note>
         * <p>
         * For region availability, see <a href=
         * "https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/CacheNodes.SupportedTypes.html#CacheNodes.SupportedTypesByRegion"
         * >Supported Node Types</a>
         * </p>
         * </note>
         * <p>
         * <b>R6g node types</b> (available only for Redis OSS engine version 5.0.6 onward and for Memcached engine
         * version 1.5.16 onward): <code>cache.r6g.large</code>, <code>cache.r6g.xlarge</code>,
         * <code>cache.r6g.2xlarge</code>, <code>cache.r6g.4xlarge</code>, <code>cache.r6g.8xlarge</code>,
         * <code>cache.r6g.12xlarge</code>, <code>cache.r6g.16xlarge</code>
         * </p>
         * <p>
         * <b>R5 node types:</b> <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>,
         * <code>cache.r5.2xlarge</code>, <code>cache.r5.4xlarge</code>, <code>cache.r5.12xlarge</code>,
         * <code>cache.r5.24xlarge</code>
         * </p>
         * <p>
         * <b>R4 node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>,
         * <code>cache.r4.2xlarge</code>, <code>cache.r4.4xlarge</code>, <code>cache.r4.8xlarge</code>,
         * <code>cache.r4.16xlarge</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Previous generation: (not recommended. Existing clusters are still supported but creation of new clusters is
         * not supported for these types.)
         * </p>
         * <p>
         * <b>M2 node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>,
         * <code>cache.m2.4xlarge</code>
         * </p>
         * <p>
         * <b>R3 node types:</b> <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>,
         * <code>cache.r3.2xlarge</code>, <code>cache.r3.4xlarge</code>, <code>cache.r3.8xlarge</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <p>
         * <b>Additional node type info</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * All current generation instance types are created in Amazon VPC by default.
         * </p>
         * </li>
         * <li>
         * <p>
         * Valkey or Redis OSS append-only files (AOF) are not supported for T1 or T2 instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * Valkey or Redis OSS Multi-AZ with automatic failover is not supported on T1 instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * The configuration variables <code>appendonly</code> and <code>appendfsync</code> are not supported on Valkey,
         * or on Redis OSS version 2.8.22 and later.
         * </p>
         * </li>
         * </ul>
         * 
         * @param cacheNodeType
         *        The name of the compute and memory capacity node type for the source cluster.</p>
         *        <p>
         *        The following node types are supported by ElastiCache. Generally speaking, the current generation
         *        types provide more memory and computational power at lower cost when compared to their equivalent
         *        previous generation counterparts.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        General purpose:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Current generation:
         *        </p>
         *        <p>
         *        <b>M7g node types</b>: <code>cache.m7g.large</code>, <code>cache.m7g.xlarge</code>,
         *        <code>cache.m7g.2xlarge</code>, <code>cache.m7g.4xlarge</code>, <code>cache.m7g.8xlarge</code>,
         *        <code>cache.m7g.12xlarge</code>, <code>cache.m7g.16xlarge</code>
         *        </p>
         *        <note>
         *        <p>
         *        For region availability, see <a href=
         *        "https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/CacheNodes.SupportedTypes.html#CacheNodes.SupportedTypesByRegion"
         *        >Supported Node Types</a>
         *        </p>
         *        </note>
         *        <p>
         *        <b>M6g node types</b> (available only for Redis OSS engine version 5.0.6 onward and for Memcached
         *        engine version 1.5.16 onward): <code>cache.m6g.large</code>, <code>cache.m6g.xlarge</code>,
         *        <code>cache.m6g.2xlarge</code>, <code>cache.m6g.4xlarge</code>, <code>cache.m6g.8xlarge</code>,
         *        <code>cache.m6g.12xlarge</code>, <code>cache.m6g.16xlarge</code>
         *        </p>
         *        <p>
         *        <b>M5 node types:</b> <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>,
         *        <code>cache.m5.2xlarge</code>, <code>cache.m5.4xlarge</code>, <code>cache.m5.12xlarge</code>,
         *        <code>cache.m5.24xlarge</code>
         *        </p>
         *        <p>
         *        <b>M4 node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>,
         *        <code>cache.m4.2xlarge</code>, <code>cache.m4.4xlarge</code>, <code>cache.m4.10xlarge</code>
         *        </p>
         *        <p>
         *        <b>T4g node types</b> (available only for Redis OSS engine version 5.0.6 onward and Memcached engine
         *        version 1.5.16 onward): <code>cache.t4g.micro</code>, <code>cache.t4g.small</code>,
         *        <code>cache.t4g.medium</code>
         *        </p>
         *        <p>
         *        <b>T3 node types:</b> <code>cache.t3.micro</code>, <code>cache.t3.small</code>,
         *        <code>cache.t3.medium</code>
         *        </p>
         *        <p>
         *        <b>T2 node types:</b> <code>cache.t2.micro</code>, <code>cache.t2.small</code>,
         *        <code>cache.t2.medium</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Previous generation: (not recommended. Existing clusters are still supported but creation of new
         *        clusters is not supported for these types.)
         *        </p>
         *        <p>
         *        <b>T1 node types:</b> <code>cache.t1.micro</code>
         *        </p>
         *        <p>
         *        <b>M1 node types:</b> <code>cache.m1.small</code>, <code>cache.m1.medium</code>,
         *        <code>cache.m1.large</code>, <code>cache.m1.xlarge</code>
         *        </p>
         *        <p>
         *        <b>M3 node types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>,
         *        <code>cache.m3.xlarge</code>, <code>cache.m3.2xlarge</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Compute optimized:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Previous generation: (not recommended. Existing clusters are still supported but creation of new
         *        clusters is not supported for these types.)
         *        </p>
         *        <p>
         *        <b>C1 node types:</b> <code>cache.c1.xlarge</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Memory optimized:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Current generation:
         *        </p>
         *        <p>
         *        <b>R7g node types</b>: <code>cache.r7g.large</code>, <code>cache.r7g.xlarge</code>,
         *        <code>cache.r7g.2xlarge</code>, <code>cache.r7g.4xlarge</code>, <code>cache.r7g.8xlarge</code>,
         *        <code>cache.r7g.12xlarge</code>, <code>cache.r7g.16xlarge</code>
         *        </p>
         *        <note>
         *        <p>
         *        For region availability, see <a href=
         *        "https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/CacheNodes.SupportedTypes.html#CacheNodes.SupportedTypesByRegion"
         *        >Supported Node Types</a>
         *        </p>
         *        </note>
         *        <p>
         *        <b>R6g node types</b> (available only for Redis OSS engine version 5.0.6 onward and for Memcached
         *        engine version 1.5.16 onward): <code>cache.r6g.large</code>, <code>cache.r6g.xlarge</code>,
         *        <code>cache.r6g.2xlarge</code>, <code>cache.r6g.4xlarge</code>, <code>cache.r6g.8xlarge</code>,
         *        <code>cache.r6g.12xlarge</code>, <code>cache.r6g.16xlarge</code>
         *        </p>
         *        <p>
         *        <b>R5 node types:</b> <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>,
         *        <code>cache.r5.2xlarge</code>, <code>cache.r5.4xlarge</code>, <code>cache.r5.12xlarge</code>,
         *        <code>cache.r5.24xlarge</code>
         *        </p>
         *        <p>
         *        <b>R4 node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>,
         *        <code>cache.r4.2xlarge</code>, <code>cache.r4.4xlarge</code>, <code>cache.r4.8xlarge</code>,
         *        <code>cache.r4.16xlarge</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Previous generation: (not recommended. Existing clusters are still supported but creation of new
         *        clusters is not supported for these types.)
         *        </p>
         *        <p>
         *        <b>M2 node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>,
         *        <code>cache.m2.4xlarge</code>
         *        </p>
         *        <p>
         *        <b>R3 node types:</b> <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>,
         *        <code>cache.r3.2xlarge</code>, <code>cache.r3.4xlarge</code>, <code>cache.r3.8xlarge</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Additional node type info</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        All current generation instance types are created in Amazon VPC by default.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Valkey or Redis OSS append-only files (AOF) are not supported for T1 or T2 instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Valkey or Redis OSS Multi-AZ with automatic failover is not supported on T1 instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The configuration variables <code>appendonly</code> and <code>appendfsync</code> are not supported on
         *        Valkey, or on Redis OSS version 2.8.22 and later.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheNodeType(String cacheNodeType);

        /**
         * <p>
         * The name of the cache engine (<code>memcached</code> or <code>redis</code>) used by the source cluster.
         * </p>
         * 
         * @param engine
         *        The name of the cache engine (<code>memcached</code> or <code>redis</code>) used by the source
         *        cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engine(String engine);

        /**
         * <p>
         * The version of the cache engine version that is used by the source cluster.
         * </p>
         * 
         * @param engineVersion
         *        The version of the cache engine version that is used by the source cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * The number of cache nodes in the source cluster.
         * </p>
         * <p>
         * For clusters running Valkey or Redis OSS, this value must be 1. For clusters running Memcached, this value
         * must be between 1 and 40.
         * </p>
         * 
         * @param numCacheNodes
         *        The number of cache nodes in the source cluster.</p>
         *        <p>
         *        For clusters running Valkey or Redis OSS, this value must be 1. For clusters running Memcached, this
         *        value must be between 1 and 40.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numCacheNodes(Integer numCacheNodes);

        /**
         * <p>
         * The name of the Availability Zone in which the source cluster is located.
         * </p>
         * 
         * @param preferredAvailabilityZone
         *        The name of the Availability Zone in which the source cluster is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredAvailabilityZone(String preferredAvailabilityZone);

        /**
         * <p>
         * The ARN (Amazon Resource Name) of the preferred outpost.
         * </p>
         * 
         * @param preferredOutpostArn
         *        The ARN (Amazon Resource Name) of the preferred outpost.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredOutpostArn(String preferredOutpostArn);

        /**
         * <p>
         * The date and time when the source cluster was created.
         * </p>
         * 
         * @param cacheClusterCreateTime
         *        The date and time when the source cluster was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheClusterCreateTime(Instant cacheClusterCreateTime);

        /**
         * <p>
         * Specifies the weekly time range during which maintenance on the cluster is performed. It is specified as a
         * range in the format ddd:hh24:mi-ddd:hh24:mi (24H Clock UTC). The minimum maintenance window is a 60 minute
         * period.
         * </p>
         * <p>
         * Valid values for <code>ddd</code> are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>sun</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>mon</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>tue</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>wed</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>thu</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>fri</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sat</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * Example: <code>sun:23:00-mon:01:30</code>
         * </p>
         * 
         * @param preferredMaintenanceWindow
         *        Specifies the weekly time range during which maintenance on the cluster is performed. It is specified
         *        as a range in the format ddd:hh24:mi-ddd:hh24:mi (24H Clock UTC). The minimum maintenance window is a
         *        60 minute period.</p>
         *        <p>
         *        Valid values for <code>ddd</code> are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>sun</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>mon</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>tue</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>wed</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>thu</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>fri</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sat</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Example: <code>sun:23:00-mon:01:30</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredMaintenanceWindow(String preferredMaintenanceWindow);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the topic used by the source cluster for publishing notifications.
         * </p>
         * 
         * @param topicArn
         *        The Amazon Resource Name (ARN) for the topic used by the source cluster for publishing notifications.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topicArn(String topicArn);

        /**
         * <p>
         * The port number used by each cache nodes in the source cluster.
         * </p>
         * 
         * @param port
         *        The port number used by each cache nodes in the source cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * The cache parameter group that is associated with the source cluster.
         * </p>
         * 
         * @param cacheParameterGroupName
         *        The cache parameter group that is associated with the source cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheParameterGroupName(String cacheParameterGroupName);

        /**
         * <p>
         * The name of the cache subnet group associated with the source cluster.
         * </p>
         * 
         * @param cacheSubnetGroupName
         *        The name of the cache subnet group associated with the source cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheSubnetGroupName(String cacheSubnetGroupName);

        /**
         * <p>
         * The Amazon Virtual Private Cloud identifier (VPC ID) of the cache subnet group for the source cluster.
         * </p>
         * 
         * @param vpcId
         *        The Amazon Virtual Private Cloud identifier (VPC ID) of the cache subnet group for the source cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         *  If you are running Valkey 7.2 and above or Redis OSS engine version 6.0 and above, set this parameter to yes
         * if you want to opt-in to the next auto minor version upgrade campaign. This parameter is disabled for
         * previous versions. 
         * </p>
         * 
         * @param autoMinorVersionUpgrade
         *         If you are running Valkey 7.2 and above or Redis OSS engine version 6.0 and above, set this parameter
         *        to yes if you want to opt-in to the next auto minor version upgrade campaign. This parameter is
         *        disabled for previous versions. 
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade);

        /**
         * <p>
         * For an automatic snapshot, the number of days for which ElastiCache retains the snapshot before deleting it.
         * </p>
         * <p>
         * For manual snapshots, this field reflects the <code>SnapshotRetentionLimit</code> for the source cluster when
         * the snapshot was created. This field is otherwise ignored: Manual snapshots do not expire, and can only be
         * deleted using the <code>DeleteSnapshot</code> operation.
         * </p>
         * <p>
         * <b>Important</b> If the value of SnapshotRetentionLimit is set to zero (0), backups are turned off.
         * </p>
         * 
         * @param snapshotRetentionLimit
         *        For an automatic snapshot, the number of days for which ElastiCache retains the snapshot before
         *        deleting it.</p>
         *        <p>
         *        For manual snapshots, this field reflects the <code>SnapshotRetentionLimit</code> for the source
         *        cluster when the snapshot was created. This field is otherwise ignored: Manual snapshots do not
         *        expire, and can only be deleted using the <code>DeleteSnapshot</code> operation.
         *        </p>
         *        <p>
         *        <b>Important</b> If the value of SnapshotRetentionLimit is set to zero (0), backups are turned off.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotRetentionLimit(Integer snapshotRetentionLimit);

        /**
         * <p>
         * The daily time range during which ElastiCache takes daily snapshots of the source cluster.
         * </p>
         * 
         * @param snapshotWindow
         *        The daily time range during which ElastiCache takes daily snapshots of the source cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotWindow(String snapshotWindow);

        /**
         * <p>
         * The number of node groups (shards) in this snapshot. When restoring from a snapshot, the number of node
         * groups (shards) in the snapshot and in the restored replication group must be the same.
         * </p>
         * 
         * @param numNodeGroups
         *        The number of node groups (shards) in this snapshot. When restoring from a snapshot, the number of
         *        node groups (shards) in the snapshot and in the restored replication group must be the same.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numNodeGroups(Integer numNodeGroups);

        /**
         * <p>
         * Indicates the status of automatic failover for the source Valkey or Redis OSS replication group.
         * </p>
         * 
         * @param automaticFailover
         *        Indicates the status of automatic failover for the source Valkey or Redis OSS replication group.
         * @see AutomaticFailoverStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomaticFailoverStatus
         */
        Builder automaticFailover(String automaticFailover);

        /**
         * <p>
         * Indicates the status of automatic failover for the source Valkey or Redis OSS replication group.
         * </p>
         * 
         * @param automaticFailover
         *        Indicates the status of automatic failover for the source Valkey or Redis OSS replication group.
         * @see AutomaticFailoverStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomaticFailoverStatus
         */
        Builder automaticFailover(AutomaticFailoverStatus automaticFailover);

        /**
         * <p>
         * A list of the cache nodes in the source cluster.
         * </p>
         * 
         * @param nodeSnapshots
         *        A list of the cache nodes in the source cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeSnapshots(Collection<NodeSnapshot> nodeSnapshots);

        /**
         * <p>
         * A list of the cache nodes in the source cluster.
         * </p>
         * 
         * @param nodeSnapshots
         *        A list of the cache nodes in the source cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeSnapshots(NodeSnapshot... nodeSnapshots);

        /**
         * <p>
         * A list of the cache nodes in the source cluster.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.elasticache.model.NodeSnapshot.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.elasticache.model.NodeSnapshot#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.elasticache.model.NodeSnapshot.Builder#build()} is called immediately
         * and its result is passed to {@link #nodeSnapshots(List<NodeSnapshot>)}.
         * 
         * @param nodeSnapshots
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.elasticache.model.NodeSnapshot.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodeSnapshots(java.util.Collection<NodeSnapshot>)
         */
        Builder nodeSnapshots(Consumer<NodeSnapshot.Builder>... nodeSnapshots);

        /**
         * <p>
         * The ID of the KMS key used to encrypt the snapshot.
         * </p>
         * 
         * @param kmsKeyId
         *        The ID of the KMS key used to encrypt the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The ARN (Amazon Resource Name) of the snapshot.
         * </p>
         * 
         * @param arn
         *        The ARN (Amazon Resource Name) of the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * Enables data tiering. Data tiering is only supported for replication groups using the r6gd node type. This
         * parameter must be set to true when using r6gd nodes. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/data-tiering.html">Data tiering</a>.
         * </p>
         * 
         * @param dataTiering
         *        Enables data tiering. Data tiering is only supported for replication groups using the r6gd node type.
         *        This parameter must be set to true when using r6gd nodes. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/data-tiering.html">Data tiering</a>.
         * @see DataTieringStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataTieringStatus
         */
        Builder dataTiering(String dataTiering);

        /**
         * <p>
         * Enables data tiering. Data tiering is only supported for replication groups using the r6gd node type. This
         * parameter must be set to true when using r6gd nodes. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/data-tiering.html">Data tiering</a>.
         * </p>
         * 
         * @param dataTiering
         *        Enables data tiering. Data tiering is only supported for replication groups using the r6gd node type.
         *        This parameter must be set to true when using r6gd nodes. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/data-tiering.html">Data tiering</a>.
         * @see DataTieringStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataTieringStatus
         */
        Builder dataTiering(DataTieringStatus dataTiering);
    }

    static final class BuilderImpl implements Builder {
        private String snapshotName;

        private String replicationGroupId;

        private String replicationGroupDescription;

        private String cacheClusterId;

        private String snapshotStatus;

        private String snapshotSource;

        private String cacheNodeType;

        private String engine;

        private String engineVersion;

        private Integer numCacheNodes;

        private String preferredAvailabilityZone;

        private String preferredOutpostArn;

        private Instant cacheClusterCreateTime;

        private String preferredMaintenanceWindow;

        private String topicArn;

        private Integer port;

        private String cacheParameterGroupName;

        private String cacheSubnetGroupName;

        private String vpcId;

        private Boolean autoMinorVersionUpgrade;

        private Integer snapshotRetentionLimit;

        private String snapshotWindow;

        private Integer numNodeGroups;

        private String automaticFailover;

        private List<NodeSnapshot> nodeSnapshots = DefaultSdkAutoConstructList.getInstance();

        private String kmsKeyId;

        private String arn;

        private String dataTiering;

        private BuilderImpl() {
        }

        private BuilderImpl(Snapshot model) {
            snapshotName(model.snapshotName);
            replicationGroupId(model.replicationGroupId);
            replicationGroupDescription(model.replicationGroupDescription);
            cacheClusterId(model.cacheClusterId);
            snapshotStatus(model.snapshotStatus);
            snapshotSource(model.snapshotSource);
            cacheNodeType(model.cacheNodeType);
            engine(model.engine);
            engineVersion(model.engineVersion);
            numCacheNodes(model.numCacheNodes);
            preferredAvailabilityZone(model.preferredAvailabilityZone);
            preferredOutpostArn(model.preferredOutpostArn);
            cacheClusterCreateTime(model.cacheClusterCreateTime);
            preferredMaintenanceWindow(model.preferredMaintenanceWindow);
            topicArn(model.topicArn);
            port(model.port);
            cacheParameterGroupName(model.cacheParameterGroupName);
            cacheSubnetGroupName(model.cacheSubnetGroupName);
            vpcId(model.vpcId);
            autoMinorVersionUpgrade(model.autoMinorVersionUpgrade);
            snapshotRetentionLimit(model.snapshotRetentionLimit);
            snapshotWindow(model.snapshotWindow);
            numNodeGroups(model.numNodeGroups);
            automaticFailover(model.automaticFailover);
            nodeSnapshots(model.nodeSnapshots);
            kmsKeyId(model.kmsKeyId);
            arn(model.arn);
            dataTiering(model.dataTiering);
        }

        public final String getSnapshotName() {
            return snapshotName;
        }

        public final void setSnapshotName(String snapshotName) {
            this.snapshotName = snapshotName;
        }

        @Override
        public final Builder snapshotName(String snapshotName) {
            this.snapshotName = snapshotName;
            return this;
        }

        public final String getReplicationGroupId() {
            return replicationGroupId;
        }

        public final void setReplicationGroupId(String replicationGroupId) {
            this.replicationGroupId = replicationGroupId;
        }

        @Override
        public final Builder replicationGroupId(String replicationGroupId) {
            this.replicationGroupId = replicationGroupId;
            return this;
        }

        public final String getReplicationGroupDescription() {
            return replicationGroupDescription;
        }

        public final void setReplicationGroupDescription(String replicationGroupDescription) {
            this.replicationGroupDescription = replicationGroupDescription;
        }

        @Override
        public final Builder replicationGroupDescription(String replicationGroupDescription) {
            this.replicationGroupDescription = replicationGroupDescription;
            return this;
        }

        public final String getCacheClusterId() {
            return cacheClusterId;
        }

        public final void setCacheClusterId(String cacheClusterId) {
            this.cacheClusterId = cacheClusterId;
        }

        @Override
        public final Builder cacheClusterId(String cacheClusterId) {
            this.cacheClusterId = cacheClusterId;
            return this;
        }

        public final String getSnapshotStatus() {
            return snapshotStatus;
        }

        public final void setSnapshotStatus(String snapshotStatus) {
            this.snapshotStatus = snapshotStatus;
        }

        @Override
        public final Builder snapshotStatus(String snapshotStatus) {
            this.snapshotStatus = snapshotStatus;
            return this;
        }

        public final String getSnapshotSource() {
            return snapshotSource;
        }

        public final void setSnapshotSource(String snapshotSource) {
            this.snapshotSource = snapshotSource;
        }

        @Override
        public final Builder snapshotSource(String snapshotSource) {
            this.snapshotSource = snapshotSource;
            return this;
        }

        public final String getCacheNodeType() {
            return cacheNodeType;
        }

        public final void setCacheNodeType(String cacheNodeType) {
            this.cacheNodeType = cacheNodeType;
        }

        @Override
        public final Builder cacheNodeType(String cacheNodeType) {
            this.cacheNodeType = cacheNodeType;
            return this;
        }

        public final String getEngine() {
            return engine;
        }

        public final void setEngine(String engine) {
            this.engine = engine;
        }

        @Override
        public final Builder engine(String engine) {
            this.engine = engine;
            return this;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final Integer getNumCacheNodes() {
            return numCacheNodes;
        }

        public final void setNumCacheNodes(Integer numCacheNodes) {
            this.numCacheNodes = numCacheNodes;
        }

        @Override
        public final Builder numCacheNodes(Integer numCacheNodes) {
            this.numCacheNodes = numCacheNodes;
            return this;
        }

        public final String getPreferredAvailabilityZone() {
            return preferredAvailabilityZone;
        }

        public final void setPreferredAvailabilityZone(String preferredAvailabilityZone) {
            this.preferredAvailabilityZone = preferredAvailabilityZone;
        }

        @Override
        public final Builder preferredAvailabilityZone(String preferredAvailabilityZone) {
            this.preferredAvailabilityZone = preferredAvailabilityZone;
            return this;
        }

        public final String getPreferredOutpostArn() {
            return preferredOutpostArn;
        }

        public final void setPreferredOutpostArn(String preferredOutpostArn) {
            this.preferredOutpostArn = preferredOutpostArn;
        }

        @Override
        public final Builder preferredOutpostArn(String preferredOutpostArn) {
            this.preferredOutpostArn = preferredOutpostArn;
            return this;
        }

        public final Instant getCacheClusterCreateTime() {
            return cacheClusterCreateTime;
        }

        public final void setCacheClusterCreateTime(Instant cacheClusterCreateTime) {
            this.cacheClusterCreateTime = cacheClusterCreateTime;
        }

        @Override
        public final Builder cacheClusterCreateTime(Instant cacheClusterCreateTime) {
            this.cacheClusterCreateTime = cacheClusterCreateTime;
            return this;
        }

        public final String getPreferredMaintenanceWindow() {
            return preferredMaintenanceWindow;
        }

        public final void setPreferredMaintenanceWindow(String preferredMaintenanceWindow) {
            this.preferredMaintenanceWindow = preferredMaintenanceWindow;
        }

        @Override
        public final Builder preferredMaintenanceWindow(String preferredMaintenanceWindow) {
            this.preferredMaintenanceWindow = preferredMaintenanceWindow;
            return this;
        }

        public final String getTopicArn() {
            return topicArn;
        }

        public final void setTopicArn(String topicArn) {
            this.topicArn = topicArn;
        }

        @Override
        public final Builder topicArn(String topicArn) {
            this.topicArn = topicArn;
            return this;
        }

        public final Integer getPort() {
            return port;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final String getCacheParameterGroupName() {
            return cacheParameterGroupName;
        }

        public final void setCacheParameterGroupName(String cacheParameterGroupName) {
            this.cacheParameterGroupName = cacheParameterGroupName;
        }

        @Override
        public final Builder cacheParameterGroupName(String cacheParameterGroupName) {
            this.cacheParameterGroupName = cacheParameterGroupName;
            return this;
        }

        public final String getCacheSubnetGroupName() {
            return cacheSubnetGroupName;
        }

        public final void setCacheSubnetGroupName(String cacheSubnetGroupName) {
            this.cacheSubnetGroupName = cacheSubnetGroupName;
        }

        @Override
        public final Builder cacheSubnetGroupName(String cacheSubnetGroupName) {
            this.cacheSubnetGroupName = cacheSubnetGroupName;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final Boolean getAutoMinorVersionUpgrade() {
            return autoMinorVersionUpgrade;
        }

        public final void setAutoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade) {
            this.autoMinorVersionUpgrade = autoMinorVersionUpgrade;
        }

        @Override
        public final Builder autoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade) {
            this.autoMinorVersionUpgrade = autoMinorVersionUpgrade;
            return this;
        }

        public final Integer getSnapshotRetentionLimit() {
            return snapshotRetentionLimit;
        }

        public final void setSnapshotRetentionLimit(Integer snapshotRetentionLimit) {
            this.snapshotRetentionLimit = snapshotRetentionLimit;
        }

        @Override
        public final Builder snapshotRetentionLimit(Integer snapshotRetentionLimit) {
            this.snapshotRetentionLimit = snapshotRetentionLimit;
            return this;
        }

        public final String getSnapshotWindow() {
            return snapshotWindow;
        }

        public final void setSnapshotWindow(String snapshotWindow) {
            this.snapshotWindow = snapshotWindow;
        }

        @Override
        public final Builder snapshotWindow(String snapshotWindow) {
            this.snapshotWindow = snapshotWindow;
            return this;
        }

        public final Integer getNumNodeGroups() {
            return numNodeGroups;
        }

        public final void setNumNodeGroups(Integer numNodeGroups) {
            this.numNodeGroups = numNodeGroups;
        }

        @Override
        public final Builder numNodeGroups(Integer numNodeGroups) {
            this.numNodeGroups = numNodeGroups;
            return this;
        }

        public final String getAutomaticFailover() {
            return automaticFailover;
        }

        public final void setAutomaticFailover(String automaticFailover) {
            this.automaticFailover = automaticFailover;
        }

        @Override
        public final Builder automaticFailover(String automaticFailover) {
            this.automaticFailover = automaticFailover;
            return this;
        }

        @Override
        public final Builder automaticFailover(AutomaticFailoverStatus automaticFailover) {
            this.automaticFailover(automaticFailover == null ? null : automaticFailover.toString());
            return this;
        }

        public final List<NodeSnapshot.Builder> getNodeSnapshots() {
            List<NodeSnapshot.Builder> result = NodeSnapshotListCopier.copyToBuilder(this.nodeSnapshots);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNodeSnapshots(Collection<NodeSnapshot.BuilderImpl> nodeSnapshots) {
            this.nodeSnapshots = NodeSnapshotListCopier.copyFromBuilder(nodeSnapshots);
        }

        @Override
        public final Builder nodeSnapshots(Collection<NodeSnapshot> nodeSnapshots) {
            this.nodeSnapshots = NodeSnapshotListCopier.copy(nodeSnapshots);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodeSnapshots(NodeSnapshot... nodeSnapshots) {
            nodeSnapshots(Arrays.asList(nodeSnapshots));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodeSnapshots(Consumer<NodeSnapshot.Builder>... nodeSnapshots) {
            nodeSnapshots(Stream.of(nodeSnapshots).map(c -> NodeSnapshot.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getDataTiering() {
            return dataTiering;
        }

        public final void setDataTiering(String dataTiering) {
            this.dataTiering = dataTiering;
        }

        @Override
        public final Builder dataTiering(String dataTiering) {
            this.dataTiering = dataTiering;
            return this;
        }

        @Override
        public final Builder dataTiering(DataTieringStatus dataTiering) {
            this.dataTiering(dataTiering == null ? null : dataTiering.toString());
            return this;
        }

        @Override
        public Snapshot build() {
            return new Snapshot(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
