/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticbeanstalk.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Request to create a configuration template.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateConfigurationTemplateRequest extends ElasticBeanstalkRequest implements
        ToCopyableBuilder<CreateConfigurationTemplateRequest.Builder, CreateConfigurationTemplateRequest> {
    private static final SdkField<String> APPLICATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationName").getter(getter(CreateConfigurationTemplateRequest::applicationName))
            .setter(setter(Builder::applicationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationName").build()).build();

    private static final SdkField<String> TEMPLATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TemplateName").getter(getter(CreateConfigurationTemplateRequest::templateName))
            .setter(setter(Builder::templateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateName").build()).build();

    private static final SdkField<String> SOLUTION_STACK_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SolutionStackName").getter(getter(CreateConfigurationTemplateRequest::solutionStackName))
            .setter(setter(Builder::solutionStackName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SolutionStackName").build()).build();

    private static final SdkField<String> PLATFORM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PlatformArn").getter(getter(CreateConfigurationTemplateRequest::platformArn))
            .setter(setter(Builder::platformArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlatformArn").build()).build();

    private static final SdkField<SourceConfiguration> SOURCE_CONFIGURATION_FIELD = SdkField
            .<SourceConfiguration> builder(MarshallingType.SDK_POJO).memberName("SourceConfiguration")
            .getter(getter(CreateConfigurationTemplateRequest::sourceConfiguration)).setter(setter(Builder::sourceConfiguration))
            .constructor(SourceConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceConfiguration").build())
            .build();

    private static final SdkField<String> ENVIRONMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EnvironmentId").getter(getter(CreateConfigurationTemplateRequest::environmentId))
            .setter(setter(Builder::environmentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(CreateConfigurationTemplateRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<ConfigurationOptionSetting>> OPTION_SETTINGS_FIELD = SdkField
            .<List<ConfigurationOptionSetting>> builder(MarshallingType.LIST)
            .memberName("OptionSettings")
            .getter(getter(CreateConfigurationTemplateRequest::optionSettings))
            .setter(setter(Builder::optionSettings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OptionSettings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ConfigurationOptionSetting> builder(MarshallingType.SDK_POJO)
                                            .constructor(ConfigurationOptionSetting::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateConfigurationTemplateRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLICATION_NAME_FIELD,
            TEMPLATE_NAME_FIELD, SOLUTION_STACK_NAME_FIELD, PLATFORM_ARN_FIELD, SOURCE_CONFIGURATION_FIELD, ENVIRONMENT_ID_FIELD,
            DESCRIPTION_FIELD, OPTION_SETTINGS_FIELD, TAGS_FIELD));

    private final String applicationName;

    private final String templateName;

    private final String solutionStackName;

    private final String platformArn;

    private final SourceConfiguration sourceConfiguration;

    private final String environmentId;

    private final String description;

    private final List<ConfigurationOptionSetting> optionSettings;

    private final List<Tag> tags;

    private CreateConfigurationTemplateRequest(BuilderImpl builder) {
        super(builder);
        this.applicationName = builder.applicationName;
        this.templateName = builder.templateName;
        this.solutionStackName = builder.solutionStackName;
        this.platformArn = builder.platformArn;
        this.sourceConfiguration = builder.sourceConfiguration;
        this.environmentId = builder.environmentId;
        this.description = builder.description;
        this.optionSettings = builder.optionSettings;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the Elastic Beanstalk application to associate with this configuration template.
     * </p>
     * 
     * @return The name of the Elastic Beanstalk application to associate with this configuration template.
     */
    public String applicationName() {
        return applicationName;
    }

    /**
     * <p>
     * The name of the configuration template.
     * </p>
     * <p>
     * Constraint: This name must be unique per application.
     * </p>
     * 
     * @return The name of the configuration template.</p>
     *         <p>
     *         Constraint: This name must be unique per application.
     */
    public String templateName() {
        return templateName;
    }

    /**
     * <p>
     * The name of an Elastic Beanstalk solution stack (platform version) that this configuration uses. For example,
     * <code>64bit Amazon Linux 2013.09 running Tomcat 7 Java 7</code>. A solution stack specifies the operating system,
     * runtime, and application server for a configuration template. It also determines the set of configuration options
     * as well as the possible and default values. For more information, see <a
     * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/concepts.platforms.html">Supported Platforms</a> in
     * the <i>AWS Elastic Beanstalk Developer Guide</i>.
     * </p>
     * <p>
     * You must specify <code>SolutionStackName</code> if you don't specify <code>PlatformArn</code>,
     * <code>EnvironmentId</code>, or <code>SourceConfiguration</code>.
     * </p>
     * <p>
     * Use the <a href="https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListAvailableSolutionStacks.html">
     * <code>ListAvailableSolutionStacks</code> </a> API to obtain a list of available solution stacks.
     * </p>
     * 
     * @return The name of an Elastic Beanstalk solution stack (platform version) that this configuration uses. For
     *         example, <code>64bit Amazon Linux 2013.09 running Tomcat 7 Java 7</code>. A solution stack specifies the
     *         operating system, runtime, and application server for a configuration template. It also determines the
     *         set of configuration options as well as the possible and default values. For more information, see <a
     *         href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/concepts.platforms.html">Supported
     *         Platforms</a> in the <i>AWS Elastic Beanstalk Developer Guide</i>.</p>
     *         <p>
     *         You must specify <code>SolutionStackName</code> if you don't specify <code>PlatformArn</code>,
     *         <code>EnvironmentId</code>, or <code>SourceConfiguration</code>.
     *         </p>
     *         <p>
     *         Use the <a
     *         href="https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListAvailableSolutionStacks.html">
     *         <code>ListAvailableSolutionStacks</code> </a> API to obtain a list of available solution stacks.
     */
    public String solutionStackName() {
        return solutionStackName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the custom platform. For more information, see <a
     * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/custom-platforms.html"> Custom Platforms</a> in the
     * <i>AWS Elastic Beanstalk Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * If you specify <code>PlatformArn</code>, then don't specify <code>SolutionStackName</code>.
     * </p>
     * </note>
     * 
     * @return The Amazon Resource Name (ARN) of the custom platform. For more information, see <a
     *         href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/custom-platforms.html"> Custom Platforms</a>
     *         in the <i>AWS Elastic Beanstalk Developer Guide</i>.</p> <note>
     *         <p>
     *         If you specify <code>PlatformArn</code>, then don't specify <code>SolutionStackName</code>.
     *         </p>
     */
    public String platformArn() {
        return platformArn;
    }

    /**
     * <p>
     * An Elastic Beanstalk configuration template to base this one on. If specified, Elastic Beanstalk uses the
     * configuration values from the specified configuration template to create a new configuration.
     * </p>
     * <p>
     * Values specified in <code>OptionSettings</code> override any values obtained from the
     * <code>SourceConfiguration</code>.
     * </p>
     * <p>
     * You must specify <code>SourceConfiguration</code> if you don't specify <code>PlatformArn</code>,
     * <code>EnvironmentId</code>, or <code>SolutionStackName</code>.
     * </p>
     * <p>
     * Constraint: If both solution stack name and source configuration are specified, the solution stack of the source
     * configuration template must match the specified solution stack name.
     * </p>
     * 
     * @return An Elastic Beanstalk configuration template to base this one on. If specified, Elastic Beanstalk uses the
     *         configuration values from the specified configuration template to create a new configuration.</p>
     *         <p>
     *         Values specified in <code>OptionSettings</code> override any values obtained from the
     *         <code>SourceConfiguration</code>.
     *         </p>
     *         <p>
     *         You must specify <code>SourceConfiguration</code> if you don't specify <code>PlatformArn</code>,
     *         <code>EnvironmentId</code>, or <code>SolutionStackName</code>.
     *         </p>
     *         <p>
     *         Constraint: If both solution stack name and source configuration are specified, the solution stack of the
     *         source configuration template must match the specified solution stack name.
     */
    public SourceConfiguration sourceConfiguration() {
        return sourceConfiguration;
    }

    /**
     * <p>
     * The ID of an environment whose settings you want to use to create the configuration template. You must specify
     * <code>EnvironmentId</code> if you don't specify <code>PlatformArn</code>, <code>SolutionStackName</code>, or
     * <code>SourceConfiguration</code>.
     * </p>
     * 
     * @return The ID of an environment whose settings you want to use to create the configuration template. You must
     *         specify <code>EnvironmentId</code> if you don't specify <code>PlatformArn</code>,
     *         <code>SolutionStackName</code>, or <code>SourceConfiguration</code>.
     */
    public String environmentId() {
        return environmentId;
    }

    /**
     * <p>
     * An optional description for this configuration.
     * </p>
     * 
     * @return An optional description for this configuration.
     */
    public String description() {
        return description;
    }

    /**
     * Returns true if the OptionSettings property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasOptionSettings() {
        return optionSettings != null && !(optionSettings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Option values for the Elastic Beanstalk configuration, such as the instance type. If specified, these values
     * override the values obtained from the solution stack or the source configuration template. For a complete list of
     * Elastic Beanstalk configuration options, see <a
     * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options.html">Option Values</a> in the
     * <i>AWS Elastic Beanstalk Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasOptionSettings()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Option values for the Elastic Beanstalk configuration, such as the instance type. If specified, these
     *         values override the values obtained from the solution stack or the source configuration template. For a
     *         complete list of Elastic Beanstalk configuration options, see <a
     *         href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options.html">Option Values</a> in
     *         the <i>AWS Elastic Beanstalk Developer Guide</i>.
     */
    public List<ConfigurationOptionSetting> optionSettings() {
        return optionSettings;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the tags applied to the configuration template.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the tags applied to the configuration template.
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(applicationName());
        hashCode = 31 * hashCode + Objects.hashCode(templateName());
        hashCode = 31 * hashCode + Objects.hashCode(solutionStackName());
        hashCode = 31 * hashCode + Objects.hashCode(platformArn());
        hashCode = 31 * hashCode + Objects.hashCode(sourceConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(environmentId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasOptionSettings() ? optionSettings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateConfigurationTemplateRequest)) {
            return false;
        }
        CreateConfigurationTemplateRequest other = (CreateConfigurationTemplateRequest) obj;
        return Objects.equals(applicationName(), other.applicationName()) && Objects.equals(templateName(), other.templateName())
                && Objects.equals(solutionStackName(), other.solutionStackName())
                && Objects.equals(platformArn(), other.platformArn())
                && Objects.equals(sourceConfiguration(), other.sourceConfiguration())
                && Objects.equals(environmentId(), other.environmentId()) && Objects.equals(description(), other.description())
                && hasOptionSettings() == other.hasOptionSettings() && Objects.equals(optionSettings(), other.optionSettings())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateConfigurationTemplateRequest").add("ApplicationName", applicationName())
                .add("TemplateName", templateName()).add("SolutionStackName", solutionStackName())
                .add("PlatformArn", platformArn()).add("SourceConfiguration", sourceConfiguration())
                .add("EnvironmentId", environmentId()).add("Description", description())
                .add("OptionSettings", hasOptionSettings() ? optionSettings() : null).add("Tags", hasTags() ? tags() : null)
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApplicationName":
            return Optional.ofNullable(clazz.cast(applicationName()));
        case "TemplateName":
            return Optional.ofNullable(clazz.cast(templateName()));
        case "SolutionStackName":
            return Optional.ofNullable(clazz.cast(solutionStackName()));
        case "PlatformArn":
            return Optional.ofNullable(clazz.cast(platformArn()));
        case "SourceConfiguration":
            return Optional.ofNullable(clazz.cast(sourceConfiguration()));
        case "EnvironmentId":
            return Optional.ofNullable(clazz.cast(environmentId()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "OptionSettings":
            return Optional.ofNullable(clazz.cast(optionSettings()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateConfigurationTemplateRequest, T> g) {
        return obj -> g.apply((CreateConfigurationTemplateRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ElasticBeanstalkRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, CreateConfigurationTemplateRequest> {
        /**
         * <p>
         * The name of the Elastic Beanstalk application to associate with this configuration template.
         * </p>
         * 
         * @param applicationName
         *        The name of the Elastic Beanstalk application to associate with this configuration template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationName(String applicationName);

        /**
         * <p>
         * The name of the configuration template.
         * </p>
         * <p>
         * Constraint: This name must be unique per application.
         * </p>
         * 
         * @param templateName
         *        The name of the configuration template.</p>
         *        <p>
         *        Constraint: This name must be unique per application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateName(String templateName);

        /**
         * <p>
         * The name of an Elastic Beanstalk solution stack (platform version) that this configuration uses. For example,
         * <code>64bit Amazon Linux 2013.09 running Tomcat 7 Java 7</code>. A solution stack specifies the operating
         * system, runtime, and application server for a configuration template. It also determines the set of
         * configuration options as well as the possible and default values. For more information, see <a
         * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/concepts.platforms.html">Supported Platforms</a>
         * in the <i>AWS Elastic Beanstalk Developer Guide</i>.
         * </p>
         * <p>
         * You must specify <code>SolutionStackName</code> if you don't specify <code>PlatformArn</code>,
         * <code>EnvironmentId</code>, or <code>SourceConfiguration</code>.
         * </p>
         * <p>
         * Use the <a
         * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListAvailableSolutionStacks.html">
         * <code>ListAvailableSolutionStacks</code> </a> API to obtain a list of available solution stacks.
         * </p>
         * 
         * @param solutionStackName
         *        The name of an Elastic Beanstalk solution stack (platform version) that this configuration uses. For
         *        example, <code>64bit Amazon Linux 2013.09 running Tomcat 7 Java 7</code>. A solution stack specifies
         *        the operating system, runtime, and application server for a configuration template. It also determines
         *        the set of configuration options as well as the possible and default values. For more information, see
         *        <a href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/concepts.platforms.html">Supported
         *        Platforms</a> in the <i>AWS Elastic Beanstalk Developer Guide</i>.</p>
         *        <p>
         *        You must specify <code>SolutionStackName</code> if you don't specify <code>PlatformArn</code>,
         *        <code>EnvironmentId</code>, or <code>SourceConfiguration</code>.
         *        </p>
         *        <p>
         *        Use the <a
         *        href="https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListAvailableSolutionStacks.html">
         *        <code>ListAvailableSolutionStacks</code> </a> API to obtain a list of available solution stacks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder solutionStackName(String solutionStackName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the custom platform. For more information, see <a
         * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/custom-platforms.html"> Custom Platforms</a> in
         * the <i>AWS Elastic Beanstalk Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * If you specify <code>PlatformArn</code>, then don't specify <code>SolutionStackName</code>.
         * </p>
         * </note>
         * 
         * @param platformArn
         *        The Amazon Resource Name (ARN) of the custom platform. For more information, see <a
         *        href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/custom-platforms.html"> Custom
         *        Platforms</a> in the <i>AWS Elastic Beanstalk Developer Guide</i>.</p> <note>
         *        <p>
         *        If you specify <code>PlatformArn</code>, then don't specify <code>SolutionStackName</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformArn(String platformArn);

        /**
         * <p>
         * An Elastic Beanstalk configuration template to base this one on. If specified, Elastic Beanstalk uses the
         * configuration values from the specified configuration template to create a new configuration.
         * </p>
         * <p>
         * Values specified in <code>OptionSettings</code> override any values obtained from the
         * <code>SourceConfiguration</code>.
         * </p>
         * <p>
         * You must specify <code>SourceConfiguration</code> if you don't specify <code>PlatformArn</code>,
         * <code>EnvironmentId</code>, or <code>SolutionStackName</code>.
         * </p>
         * <p>
         * Constraint: If both solution stack name and source configuration are specified, the solution stack of the
         * source configuration template must match the specified solution stack name.
         * </p>
         * 
         * @param sourceConfiguration
         *        An Elastic Beanstalk configuration template to base this one on. If specified, Elastic Beanstalk uses
         *        the configuration values from the specified configuration template to create a new configuration.</p>
         *        <p>
         *        Values specified in <code>OptionSettings</code> override any values obtained from the
         *        <code>SourceConfiguration</code>.
         *        </p>
         *        <p>
         *        You must specify <code>SourceConfiguration</code> if you don't specify <code>PlatformArn</code>,
         *        <code>EnvironmentId</code>, or <code>SolutionStackName</code>.
         *        </p>
         *        <p>
         *        Constraint: If both solution stack name and source configuration are specified, the solution stack of
         *        the source configuration template must match the specified solution stack name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceConfiguration(SourceConfiguration sourceConfiguration);

        /**
         * <p>
         * An Elastic Beanstalk configuration template to base this one on. If specified, Elastic Beanstalk uses the
         * configuration values from the specified configuration template to create a new configuration.
         * </p>
         * <p>
         * Values specified in <code>OptionSettings</code> override any values obtained from the
         * <code>SourceConfiguration</code>.
         * </p>
         * <p>
         * You must specify <code>SourceConfiguration</code> if you don't specify <code>PlatformArn</code>,
         * <code>EnvironmentId</code>, or <code>SolutionStackName</code>.
         * </p>
         * <p>
         * Constraint: If both solution stack name and source configuration are specified, the solution stack of the
         * source configuration template must match the specified solution stack name.
         * </p>
         * This is a convenience that creates an instance of the {@link SourceConfiguration.Builder} avoiding the need
         * to create one manually via {@link SourceConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link SourceConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #sourceConfiguration(SourceConfiguration)}.
         * 
         * @param sourceConfiguration
         *        a consumer that will call methods on {@link SourceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceConfiguration(SourceConfiguration)
         */
        default Builder sourceConfiguration(Consumer<SourceConfiguration.Builder> sourceConfiguration) {
            return sourceConfiguration(SourceConfiguration.builder().applyMutation(sourceConfiguration).build());
        }

        /**
         * <p>
         * The ID of an environment whose settings you want to use to create the configuration template. You must
         * specify <code>EnvironmentId</code> if you don't specify <code>PlatformArn</code>,
         * <code>SolutionStackName</code>, or <code>SourceConfiguration</code>.
         * </p>
         * 
         * @param environmentId
         *        The ID of an environment whose settings you want to use to create the configuration template. You must
         *        specify <code>EnvironmentId</code> if you don't specify <code>PlatformArn</code>,
         *        <code>SolutionStackName</code>, or <code>SourceConfiguration</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentId(String environmentId);

        /**
         * <p>
         * An optional description for this configuration.
         * </p>
         * 
         * @param description
         *        An optional description for this configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Option values for the Elastic Beanstalk configuration, such as the instance type. If specified, these values
         * override the values obtained from the solution stack or the source configuration template. For a complete
         * list of Elastic Beanstalk configuration options, see <a
         * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options.html">Option Values</a> in the
         * <i>AWS Elastic Beanstalk Developer Guide</i>.
         * </p>
         * 
         * @param optionSettings
         *        Option values for the Elastic Beanstalk configuration, such as the instance type. If specified, these
         *        values override the values obtained from the solution stack or the source configuration template. For
         *        a complete list of Elastic Beanstalk configuration options, see <a
         *        href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options.html">Option Values</a>
         *        in the <i>AWS Elastic Beanstalk Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optionSettings(Collection<ConfigurationOptionSetting> optionSettings);

        /**
         * <p>
         * Option values for the Elastic Beanstalk configuration, such as the instance type. If specified, these values
         * override the values obtained from the solution stack or the source configuration template. For a complete
         * list of Elastic Beanstalk configuration options, see <a
         * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options.html">Option Values</a> in the
         * <i>AWS Elastic Beanstalk Developer Guide</i>.
         * </p>
         * 
         * @param optionSettings
         *        Option values for the Elastic Beanstalk configuration, such as the instance type. If specified, these
         *        values override the values obtained from the solution stack or the source configuration template. For
         *        a complete list of Elastic Beanstalk configuration options, see <a
         *        href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options.html">Option Values</a>
         *        in the <i>AWS Elastic Beanstalk Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optionSettings(ConfigurationOptionSetting... optionSettings);

        /**
         * <p>
         * Option values for the Elastic Beanstalk configuration, such as the instance type. If specified, these values
         * override the values obtained from the solution stack or the source configuration template. For a complete
         * list of Elastic Beanstalk configuration options, see <a
         * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options.html">Option Values</a> in the
         * <i>AWS Elastic Beanstalk Developer Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ConfigurationOptionSetting>.Builder}
         * avoiding the need to create one manually via {@link List<ConfigurationOptionSetting>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ConfigurationOptionSetting>.Builder#build()} is called
         * immediately and its result is passed to {@link #optionSettings(List<ConfigurationOptionSetting>)}.
         * 
         * @param optionSettings
         *        a consumer that will call methods on {@link List<ConfigurationOptionSetting>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #optionSettings(List<ConfigurationOptionSetting>)
         */
        Builder optionSettings(Consumer<ConfigurationOptionSetting.Builder>... optionSettings);

        /**
         * <p>
         * Specifies the tags applied to the configuration template.
         * </p>
         * 
         * @param tags
         *        Specifies the tags applied to the configuration template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Specifies the tags applied to the configuration template.
         * </p>
         * 
         * @param tags
         *        Specifies the tags applied to the configuration template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Specifies the tags applied to the configuration template.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ElasticBeanstalkRequest.BuilderImpl implements Builder {
        private String applicationName;

        private String templateName;

        private String solutionStackName;

        private String platformArn;

        private SourceConfiguration sourceConfiguration;

        private String environmentId;

        private String description;

        private List<ConfigurationOptionSetting> optionSettings = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateConfigurationTemplateRequest model) {
            super(model);
            applicationName(model.applicationName);
            templateName(model.templateName);
            solutionStackName(model.solutionStackName);
            platformArn(model.platformArn);
            sourceConfiguration(model.sourceConfiguration);
            environmentId(model.environmentId);
            description(model.description);
            optionSettings(model.optionSettings);
            tags(model.tags);
        }

        public final String getApplicationName() {
            return applicationName;
        }

        @Override
        public final Builder applicationName(String applicationName) {
            this.applicationName = applicationName;
            return this;
        }

        public final void setApplicationName(String applicationName) {
            this.applicationName = applicationName;
        }

        public final String getTemplateName() {
            return templateName;
        }

        @Override
        public final Builder templateName(String templateName) {
            this.templateName = templateName;
            return this;
        }

        public final void setTemplateName(String templateName) {
            this.templateName = templateName;
        }

        public final String getSolutionStackName() {
            return solutionStackName;
        }

        @Override
        public final Builder solutionStackName(String solutionStackName) {
            this.solutionStackName = solutionStackName;
            return this;
        }

        public final void setSolutionStackName(String solutionStackName) {
            this.solutionStackName = solutionStackName;
        }

        public final String getPlatformArn() {
            return platformArn;
        }

        @Override
        public final Builder platformArn(String platformArn) {
            this.platformArn = platformArn;
            return this;
        }

        public final void setPlatformArn(String platformArn) {
            this.platformArn = platformArn;
        }

        public final SourceConfiguration.Builder getSourceConfiguration() {
            return sourceConfiguration != null ? sourceConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder sourceConfiguration(SourceConfiguration sourceConfiguration) {
            this.sourceConfiguration = sourceConfiguration;
            return this;
        }

        public final void setSourceConfiguration(SourceConfiguration.BuilderImpl sourceConfiguration) {
            this.sourceConfiguration = sourceConfiguration != null ? sourceConfiguration.build() : null;
        }

        public final String getEnvironmentId() {
            return environmentId;
        }

        @Override
        public final Builder environmentId(String environmentId) {
            this.environmentId = environmentId;
            return this;
        }

        public final void setEnvironmentId(String environmentId) {
            this.environmentId = environmentId;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<ConfigurationOptionSetting.Builder> getOptionSettings() {
            if (optionSettings instanceof SdkAutoConstructList) {
                return null;
            }
            return optionSettings != null ? optionSettings.stream().map(ConfigurationOptionSetting::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder optionSettings(Collection<ConfigurationOptionSetting> optionSettings) {
            this.optionSettings = ConfigurationOptionSettingsListCopier.copy(optionSettings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder optionSettings(ConfigurationOptionSetting... optionSettings) {
            optionSettings(Arrays.asList(optionSettings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder optionSettings(Consumer<ConfigurationOptionSetting.Builder>... optionSettings) {
            optionSettings(Stream.of(optionSettings).map(c -> ConfigurationOptionSetting.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setOptionSettings(Collection<ConfigurationOptionSetting.BuilderImpl> optionSettings) {
            this.optionSettings = ConfigurationOptionSettingsListCopier.copyFromBuilder(optionSettings);
        }

        public final Collection<Tag.Builder> getTags() {
            if (tags instanceof SdkAutoConstructList) {
                return null;
            }
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateConfigurationTemplateRequest build() {
            return new CreateConfigurationTemplateRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
