/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticbeanstalk.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the properties of an environment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EnvironmentDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<EnvironmentDescription.Builder, EnvironmentDescription> {
    private static final SdkField<String> ENVIRONMENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EnvironmentName").getter(getter(EnvironmentDescription::environmentName))
            .setter(setter(Builder::environmentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentName").build()).build();

    private static final SdkField<String> ENVIRONMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EnvironmentId").getter(getter(EnvironmentDescription::environmentId))
            .setter(setter(Builder::environmentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentId").build()).build();

    private static final SdkField<String> APPLICATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationName").getter(getter(EnvironmentDescription::applicationName))
            .setter(setter(Builder::applicationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationName").build()).build();

    private static final SdkField<String> VERSION_LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VersionLabel").getter(getter(EnvironmentDescription::versionLabel))
            .setter(setter(Builder::versionLabel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionLabel").build()).build();

    private static final SdkField<String> SOLUTION_STACK_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SolutionStackName").getter(getter(EnvironmentDescription::solutionStackName))
            .setter(setter(Builder::solutionStackName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SolutionStackName").build()).build();

    private static final SdkField<String> PLATFORM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PlatformArn").getter(getter(EnvironmentDescription::platformArn)).setter(setter(Builder::platformArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlatformArn").build()).build();

    private static final SdkField<String> TEMPLATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TemplateName").getter(getter(EnvironmentDescription::templateName))
            .setter(setter(Builder::templateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateName").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(EnvironmentDescription::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> ENDPOINT_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointURL").getter(getter(EnvironmentDescription::endpointURL)).setter(setter(Builder::endpointURL))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointURL").build()).build();

    private static final SdkField<String> CNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("CNAME")
            .getter(getter(EnvironmentDescription::cname)).setter(setter(Builder::cname))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CNAME").build()).build();

    private static final SdkField<Instant> DATE_CREATED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("DateCreated").getter(getter(EnvironmentDescription::dateCreated)).setter(setter(Builder::dateCreated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DateCreated").build()).build();

    private static final SdkField<Instant> DATE_UPDATED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("DateUpdated").getter(getter(EnvironmentDescription::dateUpdated)).setter(setter(Builder::dateUpdated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DateUpdated").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(EnvironmentDescription::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Boolean> ABORTABLE_OPERATION_IN_PROGRESS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AbortableOperationInProgress")
            .getter(getter(EnvironmentDescription::abortableOperationInProgress))
            .setter(setter(Builder::abortableOperationInProgress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AbortableOperationInProgress")
                    .build()).build();

    private static final SdkField<String> HEALTH_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Health")
            .getter(getter(EnvironmentDescription::healthAsString)).setter(setter(Builder::health))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Health").build()).build();

    private static final SdkField<String> HEALTH_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HealthStatus").getter(getter(EnvironmentDescription::healthStatusAsString))
            .setter(setter(Builder::healthStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthStatus").build()).build();

    private static final SdkField<EnvironmentResourcesDescription> RESOURCES_FIELD = SdkField
            .<EnvironmentResourcesDescription> builder(MarshallingType.SDK_POJO).memberName("Resources")
            .getter(getter(EnvironmentDescription::resources)).setter(setter(Builder::resources))
            .constructor(EnvironmentResourcesDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resources").build()).build();

    private static final SdkField<EnvironmentTier> TIER_FIELD = SdkField.<EnvironmentTier> builder(MarshallingType.SDK_POJO)
            .memberName("Tier").getter(getter(EnvironmentDescription::tier)).setter(setter(Builder::tier))
            .constructor(EnvironmentTier::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tier").build()).build();

    private static final SdkField<List<EnvironmentLink>> ENVIRONMENT_LINKS_FIELD = SdkField
            .<List<EnvironmentLink>> builder(MarshallingType.LIST)
            .memberName("EnvironmentLinks")
            .getter(getter(EnvironmentDescription::environmentLinks))
            .setter(setter(Builder::environmentLinks))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentLinks").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EnvironmentLink> builder(MarshallingType.SDK_POJO)
                                            .constructor(EnvironmentLink::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ENVIRONMENT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EnvironmentArn").getter(getter(EnvironmentDescription::environmentArn))
            .setter(setter(Builder::environmentArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentArn").build()).build();

    private static final SdkField<String> OPERATIONS_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OperationsRole").getter(getter(EnvironmentDescription::operationsRole))
            .setter(setter(Builder::operationsRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationsRole").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENVIRONMENT_NAME_FIELD,
            ENVIRONMENT_ID_FIELD, APPLICATION_NAME_FIELD, VERSION_LABEL_FIELD, SOLUTION_STACK_NAME_FIELD, PLATFORM_ARN_FIELD,
            TEMPLATE_NAME_FIELD, DESCRIPTION_FIELD, ENDPOINT_URL_FIELD, CNAME_FIELD, DATE_CREATED_FIELD, DATE_UPDATED_FIELD,
            STATUS_FIELD, ABORTABLE_OPERATION_IN_PROGRESS_FIELD, HEALTH_FIELD, HEALTH_STATUS_FIELD, RESOURCES_FIELD, TIER_FIELD,
            ENVIRONMENT_LINKS_FIELD, ENVIRONMENT_ARN_FIELD, OPERATIONS_ROLE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String environmentName;

    private final String environmentId;

    private final String applicationName;

    private final String versionLabel;

    private final String solutionStackName;

    private final String platformArn;

    private final String templateName;

    private final String description;

    private final String endpointURL;

    private final String cname;

    private final Instant dateCreated;

    private final Instant dateUpdated;

    private final String status;

    private final Boolean abortableOperationInProgress;

    private final String health;

    private final String healthStatus;

    private final EnvironmentResourcesDescription resources;

    private final EnvironmentTier tier;

    private final List<EnvironmentLink> environmentLinks;

    private final String environmentArn;

    private final String operationsRole;

    private EnvironmentDescription(BuilderImpl builder) {
        this.environmentName = builder.environmentName;
        this.environmentId = builder.environmentId;
        this.applicationName = builder.applicationName;
        this.versionLabel = builder.versionLabel;
        this.solutionStackName = builder.solutionStackName;
        this.platformArn = builder.platformArn;
        this.templateName = builder.templateName;
        this.description = builder.description;
        this.endpointURL = builder.endpointURL;
        this.cname = builder.cname;
        this.dateCreated = builder.dateCreated;
        this.dateUpdated = builder.dateUpdated;
        this.status = builder.status;
        this.abortableOperationInProgress = builder.abortableOperationInProgress;
        this.health = builder.health;
        this.healthStatus = builder.healthStatus;
        this.resources = builder.resources;
        this.tier = builder.tier;
        this.environmentLinks = builder.environmentLinks;
        this.environmentArn = builder.environmentArn;
        this.operationsRole = builder.operationsRole;
    }

    /**
     * <p>
     * The name of this environment.
     * </p>
     * 
     * @return The name of this environment.
     */
    public final String environmentName() {
        return environmentName;
    }

    /**
     * <p>
     * The ID of this environment.
     * </p>
     * 
     * @return The ID of this environment.
     */
    public final String environmentId() {
        return environmentId;
    }

    /**
     * <p>
     * The name of the application associated with this environment.
     * </p>
     * 
     * @return The name of the application associated with this environment.
     */
    public final String applicationName() {
        return applicationName;
    }

    /**
     * <p>
     * The application version deployed in this environment.
     * </p>
     * 
     * @return The application version deployed in this environment.
     */
    public final String versionLabel() {
        return versionLabel;
    }

    /**
     * <p>
     * The name of the <code>SolutionStack</code> deployed with this environment.
     * </p>
     * 
     * @return The name of the <code>SolutionStack</code> deployed with this environment.
     */
    public final String solutionStackName() {
        return solutionStackName;
    }

    /**
     * <p>
     * The ARN of the platform version.
     * </p>
     * 
     * @return The ARN of the platform version.
     */
    public final String platformArn() {
        return platformArn;
    }

    /**
     * <p>
     * The name of the configuration template used to originally launch this environment.
     * </p>
     * 
     * @return The name of the configuration template used to originally launch this environment.
     */
    public final String templateName() {
        return templateName;
    }

    /**
     * <p>
     * Describes this environment.
     * </p>
     * 
     * @return Describes this environment.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * For load-balanced, autoscaling environments, the URL to the LoadBalancer. For single-instance environments, the
     * IP address of the instance.
     * </p>
     * 
     * @return For load-balanced, autoscaling environments, the URL to the LoadBalancer. For single-instance
     *         environments, the IP address of the instance.
     */
    public final String endpointURL() {
        return endpointURL;
    }

    /**
     * <p>
     * The URL to the CNAME for this environment.
     * </p>
     * 
     * @return The URL to the CNAME for this environment.
     */
    public final String cname() {
        return cname;
    }

    /**
     * <p>
     * The creation date for this environment.
     * </p>
     * 
     * @return The creation date for this environment.
     */
    public final Instant dateCreated() {
        return dateCreated;
    }

    /**
     * <p>
     * The last modified date for this environment.
     * </p>
     * 
     * @return The last modified date for this environment.
     */
    public final Instant dateUpdated() {
        return dateUpdated;
    }

    /**
     * <p>
     * The current operational status of the environment:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Launching</code>: Environment is in the process of initial deployment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Updating</code>: Environment is in the process of updating its configuration settings or application
     * version.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Ready</code>: Environment is available to have an action performed on it, such as update or terminate.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Terminating</code>: Environment is in the shut-down process.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Terminated</code>: Environment is not running.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link EnvironmentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current operational status of the environment:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Launching</code>: Environment is in the process of initial deployment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Updating</code>: Environment is in the process of updating its configuration settings or
     *         application version.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Ready</code>: Environment is available to have an action performed on it, such as update or
     *         terminate.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Terminating</code>: Environment is in the shut-down process.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Terminated</code>: Environment is not running.
     *         </p>
     *         </li>
     * @see EnvironmentStatus
     */
    public final EnvironmentStatus status() {
        return EnvironmentStatus.fromValue(status);
    }

    /**
     * <p>
     * The current operational status of the environment:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Launching</code>: Environment is in the process of initial deployment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Updating</code>: Environment is in the process of updating its configuration settings or application
     * version.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Ready</code>: Environment is available to have an action performed on it, such as update or terminate.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Terminating</code>: Environment is in the shut-down process.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Terminated</code>: Environment is not running.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link EnvironmentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current operational status of the environment:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Launching</code>: Environment is in the process of initial deployment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Updating</code>: Environment is in the process of updating its configuration settings or
     *         application version.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Ready</code>: Environment is available to have an action performed on it, such as update or
     *         terminate.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Terminating</code>: Environment is in the shut-down process.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Terminated</code>: Environment is not running.
     *         </p>
     *         </li>
     * @see EnvironmentStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Indicates if there is an in-progress environment configuration update or application version deployment that you
     * can cancel.
     * </p>
     * <p>
     * <code>true:</code> There is an update in progress.
     * </p>
     * <p>
     * <code>false:</code> There are no updates currently in progress.
     * </p>
     * 
     * @return Indicates if there is an in-progress environment configuration update or application version deployment
     *         that you can cancel.</p>
     *         <p>
     *         <code>true:</code> There is an update in progress.
     *         </p>
     *         <p>
     *         <code>false:</code> There are no updates currently in progress.
     */
    public final Boolean abortableOperationInProgress() {
        return abortableOperationInProgress;
    }

    /**
     * <p>
     * Describes the health status of the environment. AWS Elastic Beanstalk indicates the failure levels for a running
     * environment:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Red</code>: Indicates the environment is not responsive. Occurs when three or more consecutive failures
     * occur for an environment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Yellow</code>: Indicates that something is wrong. Occurs when two consecutive failures occur for an
     * environment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Green</code>: Indicates the environment is healthy and fully functional.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Grey</code>: Default health for a new environment. The environment is not fully launched and health checks
     * have not started or health checks are suspended during an <code>UpdateEnvironment</code> or
     * <code>RestartEnvironment</code> request.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>Grey</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #health} will
     * return {@link EnvironmentHealth#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #healthAsString}.
     * </p>
     * 
     * @return Describes the health status of the environment. AWS Elastic Beanstalk indicates the failure levels for a
     *         running environment:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Red</code>: Indicates the environment is not responsive. Occurs when three or more consecutive
     *         failures occur for an environment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Yellow</code>: Indicates that something is wrong. Occurs when two consecutive failures occur for an
     *         environment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Green</code>: Indicates the environment is healthy and fully functional.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Grey</code>: Default health for a new environment. The environment is not fully launched and health
     *         checks have not started or health checks are suspended during an <code>UpdateEnvironment</code> or
     *         <code>RestartEnvironment</code> request.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>Grey</code>
     * @see EnvironmentHealth
     */
    public final EnvironmentHealth health() {
        return EnvironmentHealth.fromValue(health);
    }

    /**
     * <p>
     * Describes the health status of the environment. AWS Elastic Beanstalk indicates the failure levels for a running
     * environment:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Red</code>: Indicates the environment is not responsive. Occurs when three or more consecutive failures
     * occur for an environment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Yellow</code>: Indicates that something is wrong. Occurs when two consecutive failures occur for an
     * environment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Green</code>: Indicates the environment is healthy and fully functional.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Grey</code>: Default health for a new environment. The environment is not fully launched and health checks
     * have not started or health checks are suspended during an <code>UpdateEnvironment</code> or
     * <code>RestartEnvironment</code> request.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>Grey</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #health} will
     * return {@link EnvironmentHealth#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #healthAsString}.
     * </p>
     * 
     * @return Describes the health status of the environment. AWS Elastic Beanstalk indicates the failure levels for a
     *         running environment:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Red</code>: Indicates the environment is not responsive. Occurs when three or more consecutive
     *         failures occur for an environment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Yellow</code>: Indicates that something is wrong. Occurs when two consecutive failures occur for an
     *         environment.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Green</code>: Indicates the environment is healthy and fully functional.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Grey</code>: Default health for a new environment. The environment is not fully launched and health
     *         checks have not started or health checks are suspended during an <code>UpdateEnvironment</code> or
     *         <code>RestartEnvironment</code> request.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>Grey</code>
     * @see EnvironmentHealth
     */
    public final String healthAsString() {
        return health;
    }

    /**
     * <p>
     * Returns the health status of the application running in your environment. For more information, see <a
     * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-status.html">Health Colors and
     * Statuses</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #healthStatus} will
     * return {@link EnvironmentHealthStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #healthStatusAsString}.
     * </p>
     * 
     * @return Returns the health status of the application running in your environment. For more information, see <a
     *         href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-status.html">Health Colors
     *         and Statuses</a>.
     * @see EnvironmentHealthStatus
     */
    public final EnvironmentHealthStatus healthStatus() {
        return EnvironmentHealthStatus.fromValue(healthStatus);
    }

    /**
     * <p>
     * Returns the health status of the application running in your environment. For more information, see <a
     * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-status.html">Health Colors and
     * Statuses</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #healthStatus} will
     * return {@link EnvironmentHealthStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #healthStatusAsString}.
     * </p>
     * 
     * @return Returns the health status of the application running in your environment. For more information, see <a
     *         href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-status.html">Health Colors
     *         and Statuses</a>.
     * @see EnvironmentHealthStatus
     */
    public final String healthStatusAsString() {
        return healthStatus;
    }

    /**
     * <p>
     * The description of the AWS resources used by this environment.
     * </p>
     * 
     * @return The description of the AWS resources used by this environment.
     */
    public final EnvironmentResourcesDescription resources() {
        return resources;
    }

    /**
     * <p>
     * Describes the current tier of this environment.
     * </p>
     * 
     * @return Describes the current tier of this environment.
     */
    public final EnvironmentTier tier() {
        return tier;
    }

    /**
     * Returns true if the EnvironmentLinks property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasEnvironmentLinks() {
        return environmentLinks != null && !(environmentLinks instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of links to other environments in the same group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEnvironmentLinks()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of links to other environments in the same group.
     */
    public final List<EnvironmentLink> environmentLinks() {
        return environmentLinks;
    }

    /**
     * <p>
     * The environment's Amazon Resource Name (ARN), which can be used in other API requests that require an ARN.
     * </p>
     * 
     * @return The environment's Amazon Resource Name (ARN), which can be used in other API requests that require an
     *         ARN.
     */
    public final String environmentArn() {
        return environmentArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the environment's operations role. For more information, see <a
     * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/iam-operationsrole.html">Operations roles</a> in the
     * <i>AWS Elastic Beanstalk Developer Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the environment's operations role. For more information, see <a
     *         href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/iam-operationsrole.html">Operations
     *         roles</a> in the <i>AWS Elastic Beanstalk Developer Guide</i>.
     */
    public final String operationsRole() {
        return operationsRole;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(environmentName());
        hashCode = 31 * hashCode + Objects.hashCode(environmentId());
        hashCode = 31 * hashCode + Objects.hashCode(applicationName());
        hashCode = 31 * hashCode + Objects.hashCode(versionLabel());
        hashCode = 31 * hashCode + Objects.hashCode(solutionStackName());
        hashCode = 31 * hashCode + Objects.hashCode(platformArn());
        hashCode = 31 * hashCode + Objects.hashCode(templateName());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(endpointURL());
        hashCode = 31 * hashCode + Objects.hashCode(cname());
        hashCode = 31 * hashCode + Objects.hashCode(dateCreated());
        hashCode = 31 * hashCode + Objects.hashCode(dateUpdated());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(abortableOperationInProgress());
        hashCode = 31 * hashCode + Objects.hashCode(healthAsString());
        hashCode = 31 * hashCode + Objects.hashCode(healthStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resources());
        hashCode = 31 * hashCode + Objects.hashCode(tier());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironmentLinks() ? environmentLinks() : null);
        hashCode = 31 * hashCode + Objects.hashCode(environmentArn());
        hashCode = 31 * hashCode + Objects.hashCode(operationsRole());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EnvironmentDescription)) {
            return false;
        }
        EnvironmentDescription other = (EnvironmentDescription) obj;
        return Objects.equals(environmentName(), other.environmentName())
                && Objects.equals(environmentId(), other.environmentId())
                && Objects.equals(applicationName(), other.applicationName())
                && Objects.equals(versionLabel(), other.versionLabel())
                && Objects.equals(solutionStackName(), other.solutionStackName())
                && Objects.equals(platformArn(), other.platformArn()) && Objects.equals(templateName(), other.templateName())
                && Objects.equals(description(), other.description()) && Objects.equals(endpointURL(), other.endpointURL())
                && Objects.equals(cname(), other.cname()) && Objects.equals(dateCreated(), other.dateCreated())
                && Objects.equals(dateUpdated(), other.dateUpdated()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(abortableOperationInProgress(), other.abortableOperationInProgress())
                && Objects.equals(healthAsString(), other.healthAsString())
                && Objects.equals(healthStatusAsString(), other.healthStatusAsString())
                && Objects.equals(resources(), other.resources()) && Objects.equals(tier(), other.tier())
                && hasEnvironmentLinks() == other.hasEnvironmentLinks()
                && Objects.equals(environmentLinks(), other.environmentLinks())
                && Objects.equals(environmentArn(), other.environmentArn())
                && Objects.equals(operationsRole(), other.operationsRole());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EnvironmentDescription").add("EnvironmentName", environmentName())
                .add("EnvironmentId", environmentId()).add("ApplicationName", applicationName())
                .add("VersionLabel", versionLabel()).add("SolutionStackName", solutionStackName())
                .add("PlatformArn", platformArn()).add("TemplateName", templateName()).add("Description", description())
                .add("EndpointURL", endpointURL()).add("CNAME", cname()).add("DateCreated", dateCreated())
                .add("DateUpdated", dateUpdated()).add("Status", statusAsString())
                .add("AbortableOperationInProgress", abortableOperationInProgress()).add("Health", healthAsString())
                .add("HealthStatus", healthStatusAsString()).add("Resources", resources()).add("Tier", tier())
                .add("EnvironmentLinks", hasEnvironmentLinks() ? environmentLinks() : null)
                .add("EnvironmentArn", environmentArn()).add("OperationsRole", operationsRole()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EnvironmentName":
            return Optional.ofNullable(clazz.cast(environmentName()));
        case "EnvironmentId":
            return Optional.ofNullable(clazz.cast(environmentId()));
        case "ApplicationName":
            return Optional.ofNullable(clazz.cast(applicationName()));
        case "VersionLabel":
            return Optional.ofNullable(clazz.cast(versionLabel()));
        case "SolutionStackName":
            return Optional.ofNullable(clazz.cast(solutionStackName()));
        case "PlatformArn":
            return Optional.ofNullable(clazz.cast(platformArn()));
        case "TemplateName":
            return Optional.ofNullable(clazz.cast(templateName()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "EndpointURL":
            return Optional.ofNullable(clazz.cast(endpointURL()));
        case "CNAME":
            return Optional.ofNullable(clazz.cast(cname()));
        case "DateCreated":
            return Optional.ofNullable(clazz.cast(dateCreated()));
        case "DateUpdated":
            return Optional.ofNullable(clazz.cast(dateUpdated()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "AbortableOperationInProgress":
            return Optional.ofNullable(clazz.cast(abortableOperationInProgress()));
        case "Health":
            return Optional.ofNullable(clazz.cast(healthAsString()));
        case "HealthStatus":
            return Optional.ofNullable(clazz.cast(healthStatusAsString()));
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "Tier":
            return Optional.ofNullable(clazz.cast(tier()));
        case "EnvironmentLinks":
            return Optional.ofNullable(clazz.cast(environmentLinks()));
        case "EnvironmentArn":
            return Optional.ofNullable(clazz.cast(environmentArn()));
        case "OperationsRole":
            return Optional.ofNullable(clazz.cast(operationsRole()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EnvironmentDescription, T> g) {
        return obj -> g.apply((EnvironmentDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EnvironmentDescription> {
        /**
         * <p>
         * The name of this environment.
         * </p>
         * 
         * @param environmentName
         *        The name of this environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentName(String environmentName);

        /**
         * <p>
         * The ID of this environment.
         * </p>
         * 
         * @param environmentId
         *        The ID of this environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentId(String environmentId);

        /**
         * <p>
         * The name of the application associated with this environment.
         * </p>
         * 
         * @param applicationName
         *        The name of the application associated with this environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationName(String applicationName);

        /**
         * <p>
         * The application version deployed in this environment.
         * </p>
         * 
         * @param versionLabel
         *        The application version deployed in this environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionLabel(String versionLabel);

        /**
         * <p>
         * The name of the <code>SolutionStack</code> deployed with this environment.
         * </p>
         * 
         * @param solutionStackName
         *        The name of the <code>SolutionStack</code> deployed with this environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder solutionStackName(String solutionStackName);

        /**
         * <p>
         * The ARN of the platform version.
         * </p>
         * 
         * @param platformArn
         *        The ARN of the platform version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformArn(String platformArn);

        /**
         * <p>
         * The name of the configuration template used to originally launch this environment.
         * </p>
         * 
         * @param templateName
         *        The name of the configuration template used to originally launch this environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateName(String templateName);

        /**
         * <p>
         * Describes this environment.
         * </p>
         * 
         * @param description
         *        Describes this environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * For load-balanced, autoscaling environments, the URL to the LoadBalancer. For single-instance environments,
         * the IP address of the instance.
         * </p>
         * 
         * @param endpointURL
         *        For load-balanced, autoscaling environments, the URL to the LoadBalancer. For single-instance
         *        environments, the IP address of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointURL(String endpointURL);

        /**
         * <p>
         * The URL to the CNAME for this environment.
         * </p>
         * 
         * @param cname
         *        The URL to the CNAME for this environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cname(String cname);

        /**
         * <p>
         * The creation date for this environment.
         * </p>
         * 
         * @param dateCreated
         *        The creation date for this environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dateCreated(Instant dateCreated);

        /**
         * <p>
         * The last modified date for this environment.
         * </p>
         * 
         * @param dateUpdated
         *        The last modified date for this environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dateUpdated(Instant dateUpdated);

        /**
         * <p>
         * The current operational status of the environment:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Launching</code>: Environment is in the process of initial deployment.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Updating</code>: Environment is in the process of updating its configuration settings or application
         * version.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Ready</code>: Environment is available to have an action performed on it, such as update or terminate.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Terminating</code>: Environment is in the shut-down process.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Terminated</code>: Environment is not running.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current operational status of the environment:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Launching</code>: Environment is in the process of initial deployment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Updating</code>: Environment is in the process of updating its configuration settings or
         *        application version.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Ready</code>: Environment is available to have an action performed on it, such as update or
         *        terminate.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Terminating</code>: Environment is in the shut-down process.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Terminated</code>: Environment is not running.
         *        </p>
         *        </li>
         * @see EnvironmentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnvironmentStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current operational status of the environment:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Launching</code>: Environment is in the process of initial deployment.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Updating</code>: Environment is in the process of updating its configuration settings or application
         * version.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Ready</code>: Environment is available to have an action performed on it, such as update or terminate.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Terminating</code>: Environment is in the shut-down process.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Terminated</code>: Environment is not running.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current operational status of the environment:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Launching</code>: Environment is in the process of initial deployment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Updating</code>: Environment is in the process of updating its configuration settings or
         *        application version.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Ready</code>: Environment is available to have an action performed on it, such as update or
         *        terminate.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Terminating</code>: Environment is in the shut-down process.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Terminated</code>: Environment is not running.
         *        </p>
         *        </li>
         * @see EnvironmentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnvironmentStatus
         */
        Builder status(EnvironmentStatus status);

        /**
         * <p>
         * Indicates if there is an in-progress environment configuration update or application version deployment that
         * you can cancel.
         * </p>
         * <p>
         * <code>true:</code> There is an update in progress.
         * </p>
         * <p>
         * <code>false:</code> There are no updates currently in progress.
         * </p>
         * 
         * @param abortableOperationInProgress
         *        Indicates if there is an in-progress environment configuration update or application version
         *        deployment that you can cancel.</p>
         *        <p>
         *        <code>true:</code> There is an update in progress.
         *        </p>
         *        <p>
         *        <code>false:</code> There are no updates currently in progress.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder abortableOperationInProgress(Boolean abortableOperationInProgress);

        /**
         * <p>
         * Describes the health status of the environment. AWS Elastic Beanstalk indicates the failure levels for a
         * running environment:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Red</code>: Indicates the environment is not responsive. Occurs when three or more consecutive failures
         * occur for an environment.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Yellow</code>: Indicates that something is wrong. Occurs when two consecutive failures occur for an
         * environment.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Green</code>: Indicates the environment is healthy and fully functional.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Grey</code>: Default health for a new environment. The environment is not fully launched and health
         * checks have not started or health checks are suspended during an <code>UpdateEnvironment</code> or
         * <code>RestartEnvironment</code> request.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>Grey</code>
         * </p>
         * 
         * @param health
         *        Describes the health status of the environment. AWS Elastic Beanstalk indicates the failure levels for
         *        a running environment:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Red</code>: Indicates the environment is not responsive. Occurs when three or more consecutive
         *        failures occur for an environment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Yellow</code>: Indicates that something is wrong. Occurs when two consecutive failures occur for
         *        an environment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Green</code>: Indicates the environment is healthy and fully functional.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Grey</code>: Default health for a new environment. The environment is not fully launched and
         *        health checks have not started or health checks are suspended during an <code>UpdateEnvironment</code>
         *        or <code>RestartEnvironment</code> request.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>Grey</code>
         * @see EnvironmentHealth
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnvironmentHealth
         */
        Builder health(String health);

        /**
         * <p>
         * Describes the health status of the environment. AWS Elastic Beanstalk indicates the failure levels for a
         * running environment:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Red</code>: Indicates the environment is not responsive. Occurs when three or more consecutive failures
         * occur for an environment.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Yellow</code>: Indicates that something is wrong. Occurs when two consecutive failures occur for an
         * environment.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Green</code>: Indicates the environment is healthy and fully functional.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Grey</code>: Default health for a new environment. The environment is not fully launched and health
         * checks have not started or health checks are suspended during an <code>UpdateEnvironment</code> or
         * <code>RestartEnvironment</code> request.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>Grey</code>
         * </p>
         * 
         * @param health
         *        Describes the health status of the environment. AWS Elastic Beanstalk indicates the failure levels for
         *        a running environment:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Red</code>: Indicates the environment is not responsive. Occurs when three or more consecutive
         *        failures occur for an environment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Yellow</code>: Indicates that something is wrong. Occurs when two consecutive failures occur for
         *        an environment.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Green</code>: Indicates the environment is healthy and fully functional.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Grey</code>: Default health for a new environment. The environment is not fully launched and
         *        health checks have not started or health checks are suspended during an <code>UpdateEnvironment</code>
         *        or <code>RestartEnvironment</code> request.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>Grey</code>
         * @see EnvironmentHealth
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnvironmentHealth
         */
        Builder health(EnvironmentHealth health);

        /**
         * <p>
         * Returns the health status of the application running in your environment. For more information, see <a
         * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-status.html">Health Colors and
         * Statuses</a>.
         * </p>
         * 
         * @param healthStatus
         *        Returns the health status of the application running in your environment. For more information, see <a
         *        href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-status.html">Health
         *        Colors and Statuses</a>.
         * @see EnvironmentHealthStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnvironmentHealthStatus
         */
        Builder healthStatus(String healthStatus);

        /**
         * <p>
         * Returns the health status of the application running in your environment. For more information, see <a
         * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-status.html">Health Colors and
         * Statuses</a>.
         * </p>
         * 
         * @param healthStatus
         *        Returns the health status of the application running in your environment. For more information, see <a
         *        href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-status.html">Health
         *        Colors and Statuses</a>.
         * @see EnvironmentHealthStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnvironmentHealthStatus
         */
        Builder healthStatus(EnvironmentHealthStatus healthStatus);

        /**
         * <p>
         * The description of the AWS resources used by this environment.
         * </p>
         * 
         * @param resources
         *        The description of the AWS resources used by this environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(EnvironmentResourcesDescription resources);

        /**
         * <p>
         * The description of the AWS resources used by this environment.
         * </p>
         * This is a convenience that creates an instance of the {@link EnvironmentResourcesDescription.Builder}
         * avoiding the need to create one manually via {@link EnvironmentResourcesDescription#builder()}.
         *
         * When the {@link Consumer} completes, {@link EnvironmentResourcesDescription.Builder#build()} is called
         * immediately and its result is passed to {@link #resources(EnvironmentResourcesDescription)}.
         * 
         * @param resources
         *        a consumer that will call methods on {@link EnvironmentResourcesDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resources(EnvironmentResourcesDescription)
         */
        default Builder resources(Consumer<EnvironmentResourcesDescription.Builder> resources) {
            return resources(EnvironmentResourcesDescription.builder().applyMutation(resources).build());
        }

        /**
         * <p>
         * Describes the current tier of this environment.
         * </p>
         * 
         * @param tier
         *        Describes the current tier of this environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tier(EnvironmentTier tier);

        /**
         * <p>
         * Describes the current tier of this environment.
         * </p>
         * This is a convenience that creates an instance of the {@link EnvironmentTier.Builder} avoiding the need to
         * create one manually via {@link EnvironmentTier#builder()}.
         *
         * When the {@link Consumer} completes, {@link EnvironmentTier.Builder#build()} is called immediately and its
         * result is passed to {@link #tier(EnvironmentTier)}.
         * 
         * @param tier
         *        a consumer that will call methods on {@link EnvironmentTier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tier(EnvironmentTier)
         */
        default Builder tier(Consumer<EnvironmentTier.Builder> tier) {
            return tier(EnvironmentTier.builder().applyMutation(tier).build());
        }

        /**
         * <p>
         * A list of links to other environments in the same group.
         * </p>
         * 
         * @param environmentLinks
         *        A list of links to other environments in the same group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentLinks(Collection<EnvironmentLink> environmentLinks);

        /**
         * <p>
         * A list of links to other environments in the same group.
         * </p>
         * 
         * @param environmentLinks
         *        A list of links to other environments in the same group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentLinks(EnvironmentLink... environmentLinks);

        /**
         * <p>
         * A list of links to other environments in the same group.
         * </p>
         * This is a convenience that creates an instance of the {@link List<EnvironmentLink>.Builder} avoiding the need
         * to create one manually via {@link List<EnvironmentLink>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<EnvironmentLink>.Builder#build()} is called immediately and
         * its result is passed to {@link #environmentLinks(List<EnvironmentLink>)}.
         * 
         * @param environmentLinks
         *        a consumer that will call methods on {@link List<EnvironmentLink>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environmentLinks(List<EnvironmentLink>)
         */
        Builder environmentLinks(Consumer<EnvironmentLink.Builder>... environmentLinks);

        /**
         * <p>
         * The environment's Amazon Resource Name (ARN), which can be used in other API requests that require an ARN.
         * </p>
         * 
         * @param environmentArn
         *        The environment's Amazon Resource Name (ARN), which can be used in other API requests that require an
         *        ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentArn(String environmentArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the environment's operations role. For more information, see <a
         * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/iam-operationsrole.html">Operations roles</a> in
         * the <i>AWS Elastic Beanstalk Developer Guide</i>.
         * </p>
         * 
         * @param operationsRole
         *        The Amazon Resource Name (ARN) of the environment's operations role. For more information, see <a
         *        href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/iam-operationsrole.html">Operations
         *        roles</a> in the <i>AWS Elastic Beanstalk Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationsRole(String operationsRole);
    }

    static final class BuilderImpl implements Builder {
        private String environmentName;

        private String environmentId;

        private String applicationName;

        private String versionLabel;

        private String solutionStackName;

        private String platformArn;

        private String templateName;

        private String description;

        private String endpointURL;

        private String cname;

        private Instant dateCreated;

        private Instant dateUpdated;

        private String status;

        private Boolean abortableOperationInProgress;

        private String health;

        private String healthStatus;

        private EnvironmentResourcesDescription resources;

        private EnvironmentTier tier;

        private List<EnvironmentLink> environmentLinks = DefaultSdkAutoConstructList.getInstance();

        private String environmentArn;

        private String operationsRole;

        private BuilderImpl() {
        }

        private BuilderImpl(EnvironmentDescription model) {
            environmentName(model.environmentName);
            environmentId(model.environmentId);
            applicationName(model.applicationName);
            versionLabel(model.versionLabel);
            solutionStackName(model.solutionStackName);
            platformArn(model.platformArn);
            templateName(model.templateName);
            description(model.description);
            endpointURL(model.endpointURL);
            cname(model.cname);
            dateCreated(model.dateCreated);
            dateUpdated(model.dateUpdated);
            status(model.status);
            abortableOperationInProgress(model.abortableOperationInProgress);
            health(model.health);
            healthStatus(model.healthStatus);
            resources(model.resources);
            tier(model.tier);
            environmentLinks(model.environmentLinks);
            environmentArn(model.environmentArn);
            operationsRole(model.operationsRole);
        }

        public final String getEnvironmentName() {
            return environmentName;
        }

        @Override
        public final Builder environmentName(String environmentName) {
            this.environmentName = environmentName;
            return this;
        }

        public final void setEnvironmentName(String environmentName) {
            this.environmentName = environmentName;
        }

        public final String getEnvironmentId() {
            return environmentId;
        }

        @Override
        public final Builder environmentId(String environmentId) {
            this.environmentId = environmentId;
            return this;
        }

        public final void setEnvironmentId(String environmentId) {
            this.environmentId = environmentId;
        }

        public final String getApplicationName() {
            return applicationName;
        }

        @Override
        public final Builder applicationName(String applicationName) {
            this.applicationName = applicationName;
            return this;
        }

        public final void setApplicationName(String applicationName) {
            this.applicationName = applicationName;
        }

        public final String getVersionLabel() {
            return versionLabel;
        }

        @Override
        public final Builder versionLabel(String versionLabel) {
            this.versionLabel = versionLabel;
            return this;
        }

        public final void setVersionLabel(String versionLabel) {
            this.versionLabel = versionLabel;
        }

        public final String getSolutionStackName() {
            return solutionStackName;
        }

        @Override
        public final Builder solutionStackName(String solutionStackName) {
            this.solutionStackName = solutionStackName;
            return this;
        }

        public final void setSolutionStackName(String solutionStackName) {
            this.solutionStackName = solutionStackName;
        }

        public final String getPlatformArn() {
            return platformArn;
        }

        @Override
        public final Builder platformArn(String platformArn) {
            this.platformArn = platformArn;
            return this;
        }

        public final void setPlatformArn(String platformArn) {
            this.platformArn = platformArn;
        }

        public final String getTemplateName() {
            return templateName;
        }

        @Override
        public final Builder templateName(String templateName) {
            this.templateName = templateName;
            return this;
        }

        public final void setTemplateName(String templateName) {
            this.templateName = templateName;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getEndpointURL() {
            return endpointURL;
        }

        @Override
        public final Builder endpointURL(String endpointURL) {
            this.endpointURL = endpointURL;
            return this;
        }

        public final void setEndpointURL(String endpointURL) {
            this.endpointURL = endpointURL;
        }

        public final String getCname() {
            return cname;
        }

        @Override
        public final Builder cname(String cname) {
            this.cname = cname;
            return this;
        }

        public final void setCname(String cname) {
            this.cname = cname;
        }

        public final Instant getDateCreated() {
            return dateCreated;
        }

        @Override
        public final Builder dateCreated(Instant dateCreated) {
            this.dateCreated = dateCreated;
            return this;
        }

        public final void setDateCreated(Instant dateCreated) {
            this.dateCreated = dateCreated;
        }

        public final Instant getDateUpdated() {
            return dateUpdated;
        }

        @Override
        public final Builder dateUpdated(Instant dateUpdated) {
            this.dateUpdated = dateUpdated;
            return this;
        }

        public final void setDateUpdated(Instant dateUpdated) {
            this.dateUpdated = dateUpdated;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(EnvironmentStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Boolean getAbortableOperationInProgress() {
            return abortableOperationInProgress;
        }

        @Override
        public final Builder abortableOperationInProgress(Boolean abortableOperationInProgress) {
            this.abortableOperationInProgress = abortableOperationInProgress;
            return this;
        }

        public final void setAbortableOperationInProgress(Boolean abortableOperationInProgress) {
            this.abortableOperationInProgress = abortableOperationInProgress;
        }

        public final String getHealth() {
            return health;
        }

        @Override
        public final Builder health(String health) {
            this.health = health;
            return this;
        }

        @Override
        public final Builder health(EnvironmentHealth health) {
            this.health(health == null ? null : health.toString());
            return this;
        }

        public final void setHealth(String health) {
            this.health = health;
        }

        public final String getHealthStatus() {
            return healthStatus;
        }

        @Override
        public final Builder healthStatus(String healthStatus) {
            this.healthStatus = healthStatus;
            return this;
        }

        @Override
        public final Builder healthStatus(EnvironmentHealthStatus healthStatus) {
            this.healthStatus(healthStatus == null ? null : healthStatus.toString());
            return this;
        }

        public final void setHealthStatus(String healthStatus) {
            this.healthStatus = healthStatus;
        }

        public final EnvironmentResourcesDescription.Builder getResources() {
            return resources != null ? resources.toBuilder() : null;
        }

        @Override
        public final Builder resources(EnvironmentResourcesDescription resources) {
            this.resources = resources;
            return this;
        }

        public final void setResources(EnvironmentResourcesDescription.BuilderImpl resources) {
            this.resources = resources != null ? resources.build() : null;
        }

        public final EnvironmentTier.Builder getTier() {
            return tier != null ? tier.toBuilder() : null;
        }

        @Override
        public final Builder tier(EnvironmentTier tier) {
            this.tier = tier;
            return this;
        }

        public final void setTier(EnvironmentTier.BuilderImpl tier) {
            this.tier = tier != null ? tier.build() : null;
        }

        public final List<EnvironmentLink.Builder> getEnvironmentLinks() {
            List<EnvironmentLink.Builder> result = EnvironmentLinksCopier.copyToBuilder(this.environmentLinks);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder environmentLinks(Collection<EnvironmentLink> environmentLinks) {
            this.environmentLinks = EnvironmentLinksCopier.copy(environmentLinks);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environmentLinks(EnvironmentLink... environmentLinks) {
            environmentLinks(Arrays.asList(environmentLinks));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environmentLinks(Consumer<EnvironmentLink.Builder>... environmentLinks) {
            environmentLinks(Stream.of(environmentLinks).map(c -> EnvironmentLink.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEnvironmentLinks(Collection<EnvironmentLink.BuilderImpl> environmentLinks) {
            this.environmentLinks = EnvironmentLinksCopier.copyFromBuilder(environmentLinks);
        }

        public final String getEnvironmentArn() {
            return environmentArn;
        }

        @Override
        public final Builder environmentArn(String environmentArn) {
            this.environmentArn = environmentArn;
            return this;
        }

        public final void setEnvironmentArn(String environmentArn) {
            this.environmentArn = environmentArn;
        }

        public final String getOperationsRole() {
            return operationsRole;
        }

        @Override
        public final Builder operationsRole(String operationsRole) {
            this.operationsRole = operationsRole;
            return this;
        }

        public final void setOperationsRole(String operationsRole) {
            this.operationsRole = operationsRole;
        }

        @Override
        public EnvironmentDescription build() {
            return new EnvironmentDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
