/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticbeanstalk.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the possible values for a configuration option.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConfigurationOptionDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<ConfigurationOptionDescription.Builder, ConfigurationOptionDescription> {
    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Namespace").getter(getter(ConfigurationOptionDescription::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Namespace").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ConfigurationOptionDescription::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DEFAULT_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DefaultValue").getter(getter(ConfigurationOptionDescription::defaultValue))
            .setter(setter(Builder::defaultValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultValue").build()).build();

    private static final SdkField<String> CHANGE_SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ChangeSeverity").getter(getter(ConfigurationOptionDescription::changeSeverity))
            .setter(setter(Builder::changeSeverity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChangeSeverity").build()).build();

    private static final SdkField<Boolean> USER_DEFINED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("UserDefined").getter(getter(ConfigurationOptionDescription::userDefined))
            .setter(setter(Builder::userDefined))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserDefined").build()).build();

    private static final SdkField<String> VALUE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ValueType").getter(getter(ConfigurationOptionDescription::valueTypeAsString))
            .setter(setter(Builder::valueType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValueType").build()).build();

    private static final SdkField<List<String>> VALUE_OPTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ValueOptions")
            .getter(getter(ConfigurationOptionDescription::valueOptions))
            .setter(setter(Builder::valueOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValueOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MIN_VALUE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinValue").getter(getter(ConfigurationOptionDescription::minValue)).setter(setter(Builder::minValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinValue").build()).build();

    private static final SdkField<Integer> MAX_VALUE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxValue").getter(getter(ConfigurationOptionDescription::maxValue)).setter(setter(Builder::maxValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxValue").build()).build();

    private static final SdkField<Integer> MAX_LENGTH_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxLength").getter(getter(ConfigurationOptionDescription::maxLength)).setter(setter(Builder::maxLength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxLength").build()).build();

    private static final SdkField<OptionRestrictionRegex> REGEX_FIELD = SdkField
            .<OptionRestrictionRegex> builder(MarshallingType.SDK_POJO).memberName("Regex")
            .getter(getter(ConfigurationOptionDescription::regex)).setter(setter(Builder::regex))
            .constructor(OptionRestrictionRegex::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Regex").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAMESPACE_FIELD, NAME_FIELD,
            DEFAULT_VALUE_FIELD, CHANGE_SEVERITY_FIELD, USER_DEFINED_FIELD, VALUE_TYPE_FIELD, VALUE_OPTIONS_FIELD,
            MIN_VALUE_FIELD, MAX_VALUE_FIELD, MAX_LENGTH_FIELD, REGEX_FIELD));

    private static final long serialVersionUID = 1L;

    private final String namespace;

    private final String name;

    private final String defaultValue;

    private final String changeSeverity;

    private final Boolean userDefined;

    private final String valueType;

    private final List<String> valueOptions;

    private final Integer minValue;

    private final Integer maxValue;

    private final Integer maxLength;

    private final OptionRestrictionRegex regex;

    private ConfigurationOptionDescription(BuilderImpl builder) {
        this.namespace = builder.namespace;
        this.name = builder.name;
        this.defaultValue = builder.defaultValue;
        this.changeSeverity = builder.changeSeverity;
        this.userDefined = builder.userDefined;
        this.valueType = builder.valueType;
        this.valueOptions = builder.valueOptions;
        this.minValue = builder.minValue;
        this.maxValue = builder.maxValue;
        this.maxLength = builder.maxLength;
        this.regex = builder.regex;
    }

    /**
     * <p>
     * A unique namespace identifying the option's associated AWS resource.
     * </p>
     * 
     * @return A unique namespace identifying the option's associated AWS resource.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * <p>
     * The name of the configuration option.
     * </p>
     * 
     * @return The name of the configuration option.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The default value for this configuration option.
     * </p>
     * 
     * @return The default value for this configuration option.
     */
    public final String defaultValue() {
        return defaultValue;
    }

    /**
     * <p>
     * An indication of which action is required if the value for this configuration option changes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NoInterruption</code> : There is no interruption to the environment or application availability.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RestartEnvironment</code> : The environment is entirely restarted, all AWS resources are deleted and
     * recreated, and the environment is unavailable during the process.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RestartApplicationServer</code> : The environment is available the entire time. However, a short
     * application outage occurs when the application servers on the running Amazon EC2 instances are restarted.
     * </p>
     * </li>
     * </ul>
     * 
     * @return An indication of which action is required if the value for this configuration option changes:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NoInterruption</code> : There is no interruption to the environment or application availability.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RestartEnvironment</code> : The environment is entirely restarted, all AWS resources are deleted
     *         and recreated, and the environment is unavailable during the process.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RestartApplicationServer</code> : The environment is available the entire time. However, a short
     *         application outage occurs when the application servers on the running Amazon EC2 instances are restarted.
     *         </p>
     *         </li>
     */
    public final String changeSeverity() {
        return changeSeverity;
    }

    /**
     * <p>
     * An indication of whether the user defined this configuration option:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>true</code> : This configuration option was defined by the user. It is a valid choice for specifying if
     * this as an <code>Option to Remove</code> when updating configuration settings.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>false</code> : This configuration was not defined by the user.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Constraint: You can remove only <code>UserDefined</code> options from a configuration.
     * </p>
     * <p>
     * Valid Values: <code>true</code> | <code>false</code>
     * </p>
     * 
     * @return An indication of whether the user defined this configuration option:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>true</code> : This configuration option was defined by the user. It is a valid choice for
     *         specifying if this as an <code>Option to Remove</code> when updating configuration settings.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>false</code> : This configuration was not defined by the user.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Constraint: You can remove only <code>UserDefined</code> options from a configuration.
     *         </p>
     *         <p>
     *         Valid Values: <code>true</code> | <code>false</code>
     */
    public final Boolean userDefined() {
        return userDefined;
    }

    /**
     * <p>
     * An indication of which type of values this option has and whether it is allowable to select one or more than one
     * of the possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Scalar</code> : Values for this option are a single selection from the possible values, or an unformatted
     * string, or numeric value governed by the <code>MIN/MAX/Regex</code> constraints.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>List</code> : Values for this option are multiple selections from the possible values.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Boolean</code> : Values for this option are either <code>true</code> or <code>false</code> .
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Json</code> : Values for this option are a JSON representation of a <code>ConfigDocument</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #valueType} will
     * return {@link ConfigurationOptionValueType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #valueTypeAsString}.
     * </p>
     * 
     * @return An indication of which type of values this option has and whether it is allowable to select one or more
     *         than one of the possible values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Scalar</code> : Values for this option are a single selection from the possible values, or an
     *         unformatted string, or numeric value governed by the <code>MIN/MAX/Regex</code> constraints.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>List</code> : Values for this option are multiple selections from the possible values.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Boolean</code> : Values for this option are either <code>true</code> or <code>false</code> .
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Json</code> : Values for this option are a JSON representation of a <code>ConfigDocument</code>.
     *         </p>
     *         </li>
     * @see ConfigurationOptionValueType
     */
    public final ConfigurationOptionValueType valueType() {
        return ConfigurationOptionValueType.fromValue(valueType);
    }

    /**
     * <p>
     * An indication of which type of values this option has and whether it is allowable to select one or more than one
     * of the possible values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Scalar</code> : Values for this option are a single selection from the possible values, or an unformatted
     * string, or numeric value governed by the <code>MIN/MAX/Regex</code> constraints.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>List</code> : Values for this option are multiple selections from the possible values.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Boolean</code> : Values for this option are either <code>true</code> or <code>false</code> .
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Json</code> : Values for this option are a JSON representation of a <code>ConfigDocument</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #valueType} will
     * return {@link ConfigurationOptionValueType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #valueTypeAsString}.
     * </p>
     * 
     * @return An indication of which type of values this option has and whether it is allowable to select one or more
     *         than one of the possible values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Scalar</code> : Values for this option are a single selection from the possible values, or an
     *         unformatted string, or numeric value governed by the <code>MIN/MAX/Regex</code> constraints.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>List</code> : Values for this option are multiple selections from the possible values.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Boolean</code> : Values for this option are either <code>true</code> or <code>false</code> .
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Json</code> : Values for this option are a JSON representation of a <code>ConfigDocument</code>.
     *         </p>
     *         </li>
     * @see ConfigurationOptionValueType
     */
    public final String valueTypeAsString() {
        return valueType;
    }

    /**
     * For responses, this returns true if the service returned a value for the ValueOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasValueOptions() {
        return valueOptions != null && !(valueOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If specified, values for the configuration option are selected from this list.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValueOptions} method.
     * </p>
     * 
     * @return If specified, values for the configuration option are selected from this list.
     */
    public final List<String> valueOptions() {
        return valueOptions;
    }

    /**
     * <p>
     * If specified, the configuration option must be a numeric value greater than this value.
     * </p>
     * 
     * @return If specified, the configuration option must be a numeric value greater than this value.
     */
    public final Integer minValue() {
        return minValue;
    }

    /**
     * <p>
     * If specified, the configuration option must be a numeric value less than this value.
     * </p>
     * 
     * @return If specified, the configuration option must be a numeric value less than this value.
     */
    public final Integer maxValue() {
        return maxValue;
    }

    /**
     * <p>
     * If specified, the configuration option must be a string value no longer than this value.
     * </p>
     * 
     * @return If specified, the configuration option must be a string value no longer than this value.
     */
    public final Integer maxLength() {
        return maxLength;
    }

    /**
     * <p>
     * If specified, the configuration option must be a string value that satisfies this regular expression.
     * </p>
     * 
     * @return If specified, the configuration option must be a string value that satisfies this regular expression.
     */
    public final OptionRestrictionRegex regex() {
        return regex;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(defaultValue());
        hashCode = 31 * hashCode + Objects.hashCode(changeSeverity());
        hashCode = 31 * hashCode + Objects.hashCode(userDefined());
        hashCode = 31 * hashCode + Objects.hashCode(valueTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasValueOptions() ? valueOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(minValue());
        hashCode = 31 * hashCode + Objects.hashCode(maxValue());
        hashCode = 31 * hashCode + Objects.hashCode(maxLength());
        hashCode = 31 * hashCode + Objects.hashCode(regex());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConfigurationOptionDescription)) {
            return false;
        }
        ConfigurationOptionDescription other = (ConfigurationOptionDescription) obj;
        return Objects.equals(namespace(), other.namespace()) && Objects.equals(name(), other.name())
                && Objects.equals(defaultValue(), other.defaultValue())
                && Objects.equals(changeSeverity(), other.changeSeverity()) && Objects.equals(userDefined(), other.userDefined())
                && Objects.equals(valueTypeAsString(), other.valueTypeAsString()) && hasValueOptions() == other.hasValueOptions()
                && Objects.equals(valueOptions(), other.valueOptions()) && Objects.equals(minValue(), other.minValue())
                && Objects.equals(maxValue(), other.maxValue()) && Objects.equals(maxLength(), other.maxLength())
                && Objects.equals(regex(), other.regex());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ConfigurationOptionDescription").add("Namespace", namespace()).add("Name", name())
                .add("DefaultValue", defaultValue()).add("ChangeSeverity", changeSeverity()).add("UserDefined", userDefined())
                .add("ValueType", valueTypeAsString()).add("ValueOptions", hasValueOptions() ? valueOptions() : null)
                .add("MinValue", minValue()).add("MaxValue", maxValue()).add("MaxLength", maxLength()).add("Regex", regex())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "DefaultValue":
            return Optional.ofNullable(clazz.cast(defaultValue()));
        case "ChangeSeverity":
            return Optional.ofNullable(clazz.cast(changeSeverity()));
        case "UserDefined":
            return Optional.ofNullable(clazz.cast(userDefined()));
        case "ValueType":
            return Optional.ofNullable(clazz.cast(valueTypeAsString()));
        case "ValueOptions":
            return Optional.ofNullable(clazz.cast(valueOptions()));
        case "MinValue":
            return Optional.ofNullable(clazz.cast(minValue()));
        case "MaxValue":
            return Optional.ofNullable(clazz.cast(maxValue()));
        case "MaxLength":
            return Optional.ofNullable(clazz.cast(maxLength()));
        case "Regex":
            return Optional.ofNullable(clazz.cast(regex()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ConfigurationOptionDescription, T> g) {
        return obj -> g.apply((ConfigurationOptionDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConfigurationOptionDescription> {
        /**
         * <p>
         * A unique namespace identifying the option's associated AWS resource.
         * </p>
         * 
         * @param namespace
         *        A unique namespace identifying the option's associated AWS resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The name of the configuration option.
         * </p>
         * 
         * @param name
         *        The name of the configuration option.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The default value for this configuration option.
         * </p>
         * 
         * @param defaultValue
         *        The default value for this configuration option.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultValue(String defaultValue);

        /**
         * <p>
         * An indication of which action is required if the value for this configuration option changes:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NoInterruption</code> : There is no interruption to the environment or application availability.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RestartEnvironment</code> : The environment is entirely restarted, all AWS resources are deleted and
         * recreated, and the environment is unavailable during the process.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RestartApplicationServer</code> : The environment is available the entire time. However, a short
         * application outage occurs when the application servers on the running Amazon EC2 instances are restarted.
         * </p>
         * </li>
         * </ul>
         * 
         * @param changeSeverity
         *        An indication of which action is required if the value for this configuration option changes:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NoInterruption</code> : There is no interruption to the environment or application availability.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RestartEnvironment</code> : The environment is entirely restarted, all AWS resources are deleted
         *        and recreated, and the environment is unavailable during the process.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RestartApplicationServer</code> : The environment is available the entire time. However, a short
         *        application outage occurs when the application servers on the running Amazon EC2 instances are
         *        restarted.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder changeSeverity(String changeSeverity);

        /**
         * <p>
         * An indication of whether the user defined this configuration option:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>true</code> : This configuration option was defined by the user. It is a valid choice for specifying if
         * this as an <code>Option to Remove</code> when updating configuration settings.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>false</code> : This configuration was not defined by the user.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Constraint: You can remove only <code>UserDefined</code> options from a configuration.
         * </p>
         * <p>
         * Valid Values: <code>true</code> | <code>false</code>
         * </p>
         * 
         * @param userDefined
         *        An indication of whether the user defined this configuration option:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>true</code> : This configuration option was defined by the user. It is a valid choice for
         *        specifying if this as an <code>Option to Remove</code> when updating configuration settings.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>false</code> : This configuration was not defined by the user.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Constraint: You can remove only <code>UserDefined</code> options from a configuration.
         *        </p>
         *        <p>
         *        Valid Values: <code>true</code> | <code>false</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userDefined(Boolean userDefined);

        /**
         * <p>
         * An indication of which type of values this option has and whether it is allowable to select one or more than
         * one of the possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Scalar</code> : Values for this option are a single selection from the possible values, or an
         * unformatted string, or numeric value governed by the <code>MIN/MAX/Regex</code> constraints.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>List</code> : Values for this option are multiple selections from the possible values.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Boolean</code> : Values for this option are either <code>true</code> or <code>false</code> .
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Json</code> : Values for this option are a JSON representation of a <code>ConfigDocument</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param valueType
         *        An indication of which type of values this option has and whether it is allowable to select one or
         *        more than one of the possible values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Scalar</code> : Values for this option are a single selection from the possible values, or an
         *        unformatted string, or numeric value governed by the <code>MIN/MAX/Regex</code> constraints.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>List</code> : Values for this option are multiple selections from the possible values.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Boolean</code> : Values for this option are either <code>true</code> or <code>false</code> .
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Json</code> : Values for this option are a JSON representation of a <code>ConfigDocument</code>.
         *        </p>
         *        </li>
         * @see ConfigurationOptionValueType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigurationOptionValueType
         */
        Builder valueType(String valueType);

        /**
         * <p>
         * An indication of which type of values this option has and whether it is allowable to select one or more than
         * one of the possible values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Scalar</code> : Values for this option are a single selection from the possible values, or an
         * unformatted string, or numeric value governed by the <code>MIN/MAX/Regex</code> constraints.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>List</code> : Values for this option are multiple selections from the possible values.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Boolean</code> : Values for this option are either <code>true</code> or <code>false</code> .
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Json</code> : Values for this option are a JSON representation of a <code>ConfigDocument</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param valueType
         *        An indication of which type of values this option has and whether it is allowable to select one or
         *        more than one of the possible values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Scalar</code> : Values for this option are a single selection from the possible values, or an
         *        unformatted string, or numeric value governed by the <code>MIN/MAX/Regex</code> constraints.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>List</code> : Values for this option are multiple selections from the possible values.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Boolean</code> : Values for this option are either <code>true</code> or <code>false</code> .
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Json</code> : Values for this option are a JSON representation of a <code>ConfigDocument</code>.
         *        </p>
         *        </li>
         * @see ConfigurationOptionValueType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigurationOptionValueType
         */
        Builder valueType(ConfigurationOptionValueType valueType);

        /**
         * <p>
         * If specified, values for the configuration option are selected from this list.
         * </p>
         * 
         * @param valueOptions
         *        If specified, values for the configuration option are selected from this list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder valueOptions(Collection<String> valueOptions);

        /**
         * <p>
         * If specified, values for the configuration option are selected from this list.
         * </p>
         * 
         * @param valueOptions
         *        If specified, values for the configuration option are selected from this list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder valueOptions(String... valueOptions);

        /**
         * <p>
         * If specified, the configuration option must be a numeric value greater than this value.
         * </p>
         * 
         * @param minValue
         *        If specified, the configuration option must be a numeric value greater than this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minValue(Integer minValue);

        /**
         * <p>
         * If specified, the configuration option must be a numeric value less than this value.
         * </p>
         * 
         * @param maxValue
         *        If specified, the configuration option must be a numeric value less than this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxValue(Integer maxValue);

        /**
         * <p>
         * If specified, the configuration option must be a string value no longer than this value.
         * </p>
         * 
         * @param maxLength
         *        If specified, the configuration option must be a string value no longer than this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxLength(Integer maxLength);

        /**
         * <p>
         * If specified, the configuration option must be a string value that satisfies this regular expression.
         * </p>
         * 
         * @param regex
         *        If specified, the configuration option must be a string value that satisfies this regular expression.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regex(OptionRestrictionRegex regex);

        /**
         * <p>
         * If specified, the configuration option must be a string value that satisfies this regular expression.
         * </p>
         * This is a convenience that creates an instance of the {@link OptionRestrictionRegex.Builder} avoiding the
         * need to create one manually via {@link OptionRestrictionRegex#builder()}.
         *
         * When the {@link Consumer} completes, {@link OptionRestrictionRegex.Builder#build()} is called immediately and
         * its result is passed to {@link #regex(OptionRestrictionRegex)}.
         * 
         * @param regex
         *        a consumer that will call methods on {@link OptionRestrictionRegex.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #regex(OptionRestrictionRegex)
         */
        default Builder regex(Consumer<OptionRestrictionRegex.Builder> regex) {
            return regex(OptionRestrictionRegex.builder().applyMutation(regex).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String namespace;

        private String name;

        private String defaultValue;

        private String changeSeverity;

        private Boolean userDefined;

        private String valueType;

        private List<String> valueOptions = DefaultSdkAutoConstructList.getInstance();

        private Integer minValue;

        private Integer maxValue;

        private Integer maxLength;

        private OptionRestrictionRegex regex;

        private BuilderImpl() {
        }

        private BuilderImpl(ConfigurationOptionDescription model) {
            namespace(model.namespace);
            name(model.name);
            defaultValue(model.defaultValue);
            changeSeverity(model.changeSeverity);
            userDefined(model.userDefined);
            valueType(model.valueType);
            valueOptions(model.valueOptions);
            minValue(model.minValue);
            maxValue(model.maxValue);
            maxLength(model.maxLength);
            regex(model.regex);
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        @Transient
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDefaultValue() {
            return defaultValue;
        }

        public final void setDefaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        @Override
        @Transient
        public final Builder defaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
            return this;
        }

        public final String getChangeSeverity() {
            return changeSeverity;
        }

        public final void setChangeSeverity(String changeSeverity) {
            this.changeSeverity = changeSeverity;
        }

        @Override
        @Transient
        public final Builder changeSeverity(String changeSeverity) {
            this.changeSeverity = changeSeverity;
            return this;
        }

        public final Boolean getUserDefined() {
            return userDefined;
        }

        public final void setUserDefined(Boolean userDefined) {
            this.userDefined = userDefined;
        }

        @Override
        @Transient
        public final Builder userDefined(Boolean userDefined) {
            this.userDefined = userDefined;
            return this;
        }

        public final String getValueType() {
            return valueType;
        }

        public final void setValueType(String valueType) {
            this.valueType = valueType;
        }

        @Override
        @Transient
        public final Builder valueType(String valueType) {
            this.valueType = valueType;
            return this;
        }

        @Override
        @Transient
        public final Builder valueType(ConfigurationOptionValueType valueType) {
            this.valueType(valueType == null ? null : valueType.toString());
            return this;
        }

        public final Collection<String> getValueOptions() {
            if (valueOptions instanceof SdkAutoConstructList) {
                return null;
            }
            return valueOptions;
        }

        public final void setValueOptions(Collection<String> valueOptions) {
            this.valueOptions = ConfigurationOptionPossibleValuesCopier.copy(valueOptions);
        }

        @Override
        @Transient
        public final Builder valueOptions(Collection<String> valueOptions) {
            this.valueOptions = ConfigurationOptionPossibleValuesCopier.copy(valueOptions);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder valueOptions(String... valueOptions) {
            valueOptions(Arrays.asList(valueOptions));
            return this;
        }

        public final Integer getMinValue() {
            return minValue;
        }

        public final void setMinValue(Integer minValue) {
            this.minValue = minValue;
        }

        @Override
        @Transient
        public final Builder minValue(Integer minValue) {
            this.minValue = minValue;
            return this;
        }

        public final Integer getMaxValue() {
            return maxValue;
        }

        public final void setMaxValue(Integer maxValue) {
            this.maxValue = maxValue;
        }

        @Override
        @Transient
        public final Builder maxValue(Integer maxValue) {
            this.maxValue = maxValue;
            return this;
        }

        public final Integer getMaxLength() {
            return maxLength;
        }

        public final void setMaxLength(Integer maxLength) {
            this.maxLength = maxLength;
        }

        @Override
        @Transient
        public final Builder maxLength(Integer maxLength) {
            this.maxLength = maxLength;
            return this;
        }

        public final OptionRestrictionRegex.Builder getRegex() {
            return regex != null ? regex.toBuilder() : null;
        }

        public final void setRegex(OptionRestrictionRegex.BuilderImpl regex) {
            this.regex = regex != null ? regex.build() : null;
        }

        @Override
        @Transient
        public final Builder regex(OptionRestrictionRegex regex) {
            this.regex = regex;
            return this;
        }

        @Override
        public ConfigurationOptionDescription build() {
            return new ConfigurationOptionDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
