/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticbeanstalk.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.elasticbeanstalk.ElasticBeanstalkClient;
import software.amazon.awssdk.services.elasticbeanstalk.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.elasticbeanstalk.model.DescribeEnvironmentsRequest;
import software.amazon.awssdk.services.elasticbeanstalk.model.DescribeEnvironmentsResponse;
import software.amazon.awssdk.services.elasticbeanstalk.model.ElasticBeanstalkRequest;
import software.amazon.awssdk.services.elasticbeanstalk.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultElasticBeanstalkWaiter implements ElasticBeanstalkWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final ElasticBeanstalkClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeEnvironmentsResponse> environmentExistsWaiter;

    private final Waiter<DescribeEnvironmentsResponse> environmentUpdatedWaiter;

    private final Waiter<DescribeEnvironmentsResponse> environmentTerminatedWaiter;

    private DefaultElasticBeanstalkWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = ElasticBeanstalkClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.environmentExistsWaiter = Waiter.builder(DescribeEnvironmentsResponse.class)
                .acceptors(environmentExistsWaiterAcceptors())
                .overrideConfiguration(environmentExistsWaiterConfig(builder.overrideConfiguration)).build();
        this.environmentUpdatedWaiter = Waiter.builder(DescribeEnvironmentsResponse.class)
                .acceptors(environmentUpdatedWaiterAcceptors())
                .overrideConfiguration(environmentUpdatedWaiterConfig(builder.overrideConfiguration)).build();
        this.environmentTerminatedWaiter = Waiter.builder(DescribeEnvironmentsResponse.class)
                .acceptors(environmentTerminatedWaiterAcceptors())
                .overrideConfiguration(environmentTerminatedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeEnvironmentsResponse> waitUntilEnvironmentExists(
            DescribeEnvironmentsRequest describeEnvironmentsRequest) {
        return environmentExistsWaiter.run(() -> client.describeEnvironments(applyWaitersUserAgent(describeEnvironmentsRequest)));
    }

    @Override
    public WaiterResponse<DescribeEnvironmentsResponse> waitUntilEnvironmentExists(
            DescribeEnvironmentsRequest describeEnvironmentsRequest, WaiterOverrideConfiguration overrideConfig) {
        return environmentExistsWaiter.run(() -> client.describeEnvironments(applyWaitersUserAgent(describeEnvironmentsRequest)),
                environmentExistsWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeEnvironmentsResponse> waitUntilEnvironmentTerminated(
            DescribeEnvironmentsRequest describeEnvironmentsRequest) {
        return environmentTerminatedWaiter.run(() -> client
                .describeEnvironments(applyWaitersUserAgent(describeEnvironmentsRequest)));
    }

    @Override
    public WaiterResponse<DescribeEnvironmentsResponse> waitUntilEnvironmentTerminated(
            DescribeEnvironmentsRequest describeEnvironmentsRequest, WaiterOverrideConfiguration overrideConfig) {
        return environmentTerminatedWaiter.run(
                () -> client.describeEnvironments(applyWaitersUserAgent(describeEnvironmentsRequest)),
                environmentTerminatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeEnvironmentsResponse> waitUntilEnvironmentUpdated(
            DescribeEnvironmentsRequest describeEnvironmentsRequest) {
        return environmentUpdatedWaiter
                .run(() -> client.describeEnvironments(applyWaitersUserAgent(describeEnvironmentsRequest)));
    }

    @Override
    public WaiterResponse<DescribeEnvironmentsResponse> waitUntilEnvironmentUpdated(
            DescribeEnvironmentsRequest describeEnvironmentsRequest, WaiterOverrideConfiguration overrideConfig) {
        return environmentUpdatedWaiter.run(
                () -> client.describeEnvironments(applyWaitersUserAgent(describeEnvironmentsRequest)),
                environmentUpdatedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeEnvironmentsResponse>> environmentExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeEnvironmentsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Environments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "Ready"));
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Environments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "Launching"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeEnvironmentsResponse>> environmentUpdatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeEnvironmentsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Environments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "Ready"));
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Environments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "Updating"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeEnvironmentsResponse>> environmentTerminatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeEnvironmentsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Environments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "Terminated"));
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Environments").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "Terminating"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration environmentExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration environmentUpdatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration environmentTerminatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static ElasticBeanstalkWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends ElasticBeanstalkRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements ElasticBeanstalkWaiter.Builder {
        private ElasticBeanstalkClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public ElasticBeanstalkWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public ElasticBeanstalkWaiter.Builder client(ElasticBeanstalkClient client) {
            this.client = client;
            return this;
        }

        public ElasticBeanstalkWaiter build() {
            return new DefaultElasticBeanstalkWaiter(this);
        }
    }
}
