/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticbeanstalk.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Request to create a configuration template.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateConfigurationTemplateRequest extends ElasticBeanstalkRequest implements
        ToCopyableBuilder<CreateConfigurationTemplateRequest.Builder, CreateConfigurationTemplateRequest> {
    private static final SdkField<String> APPLICATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateConfigurationTemplateRequest::applicationName)).setter(setter(Builder::applicationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationName").build()).build();

    private static final SdkField<String> TEMPLATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateConfigurationTemplateRequest::templateName)).setter(setter(Builder::templateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateName").build()).build();

    private static final SdkField<String> SOLUTION_STACK_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateConfigurationTemplateRequest::solutionStackName)).setter(setter(Builder::solutionStackName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SolutionStackName").build()).build();

    private static final SdkField<String> PLATFORM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateConfigurationTemplateRequest::platformArn)).setter(setter(Builder::platformArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlatformArn").build()).build();

    private static final SdkField<SourceConfiguration> SOURCE_CONFIGURATION_FIELD = SdkField
            .<SourceConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(CreateConfigurationTemplateRequest::sourceConfiguration)).setter(setter(Builder::sourceConfiguration))
            .constructor(SourceConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceConfiguration").build())
            .build();

    private static final SdkField<String> ENVIRONMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateConfigurationTemplateRequest::environmentId)).setter(setter(Builder::environmentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateConfigurationTemplateRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<ConfigurationOptionSetting>> OPTION_SETTINGS_FIELD = SdkField
            .<List<ConfigurationOptionSetting>> builder(MarshallingType.LIST)
            .getter(getter(CreateConfigurationTemplateRequest::optionSettings))
            .setter(setter(Builder::optionSettings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OptionSettings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ConfigurationOptionSetting> builder(MarshallingType.SDK_POJO)
                                            .constructor(ConfigurationOptionSetting::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLICATION_NAME_FIELD,
            TEMPLATE_NAME_FIELD, SOLUTION_STACK_NAME_FIELD, PLATFORM_ARN_FIELD, SOURCE_CONFIGURATION_FIELD, ENVIRONMENT_ID_FIELD,
            DESCRIPTION_FIELD, OPTION_SETTINGS_FIELD));

    private final String applicationName;

    private final String templateName;

    private final String solutionStackName;

    private final String platformArn;

    private final SourceConfiguration sourceConfiguration;

    private final String environmentId;

    private final String description;

    private final List<ConfigurationOptionSetting> optionSettings;

    private CreateConfigurationTemplateRequest(BuilderImpl builder) {
        super(builder);
        this.applicationName = builder.applicationName;
        this.templateName = builder.templateName;
        this.solutionStackName = builder.solutionStackName;
        this.platformArn = builder.platformArn;
        this.sourceConfiguration = builder.sourceConfiguration;
        this.environmentId = builder.environmentId;
        this.description = builder.description;
        this.optionSettings = builder.optionSettings;
    }

    /**
     * <p>
     * The name of the application to associate with this configuration template. If no application is found with this
     * name, AWS Elastic Beanstalk returns an <code>InvalidParameterValue</code> error.
     * </p>
     * 
     * @return The name of the application to associate with this configuration template. If no application is found
     *         with this name, AWS Elastic Beanstalk returns an <code>InvalidParameterValue</code> error.
     */
    public String applicationName() {
        return applicationName;
    }

    /**
     * <p>
     * The name of the configuration template.
     * </p>
     * <p>
     * Constraint: This name must be unique per application.
     * </p>
     * <p>
     * Default: If a configuration template already exists with this name, AWS Elastic Beanstalk returns an
     * <code>InvalidParameterValue</code> error.
     * </p>
     * 
     * @return The name of the configuration template.</p>
     *         <p>
     *         Constraint: This name must be unique per application.
     *         </p>
     *         <p>
     *         Default: If a configuration template already exists with this name, AWS Elastic Beanstalk returns an
     *         <code>InvalidParameterValue</code> error.
     */
    public String templateName() {
        return templateName;
    }

    /**
     * <p>
     * The name of the solution stack used by this configuration. The solution stack specifies the operating system,
     * architecture, and application server for a configuration template. It determines the set of configuration options
     * as well as the possible and default values.
     * </p>
     * <p>
     * Use <a>ListAvailableSolutionStacks</a> to obtain a list of available solution stacks.
     * </p>
     * <p>
     * A solution stack name or a source configuration parameter must be specified, otherwise AWS Elastic Beanstalk
     * returns an <code>InvalidParameterValue</code> error.
     * </p>
     * <p>
     * If a solution stack name is not specified and the source configuration parameter is specified, AWS Elastic
     * Beanstalk uses the same solution stack as the source configuration template.
     * </p>
     * 
     * @return The name of the solution stack used by this configuration. The solution stack specifies the operating
     *         system, architecture, and application server for a configuration template. It determines the set of
     *         configuration options as well as the possible and default values.</p>
     *         <p>
     *         Use <a>ListAvailableSolutionStacks</a> to obtain a list of available solution stacks.
     *         </p>
     *         <p>
     *         A solution stack name or a source configuration parameter must be specified, otherwise AWS Elastic
     *         Beanstalk returns an <code>InvalidParameterValue</code> error.
     *         </p>
     *         <p>
     *         If a solution stack name is not specified and the source configuration parameter is specified, AWS
     *         Elastic Beanstalk uses the same solution stack as the source configuration template.
     */
    public String solutionStackName() {
        return solutionStackName;
    }

    /**
     * <p>
     * The ARN of the custom platform.
     * </p>
     * 
     * @return The ARN of the custom platform.
     */
    public String platformArn() {
        return platformArn;
    }

    /**
     * <p>
     * If specified, AWS Elastic Beanstalk uses the configuration values from the specified configuration template to
     * create a new configuration.
     * </p>
     * <p>
     * Values specified in the <code>OptionSettings</code> parameter of this call overrides any values obtained from the
     * <code>SourceConfiguration</code>.
     * </p>
     * <p>
     * If no configuration template is found, returns an <code>InvalidParameterValue</code> error.
     * </p>
     * <p>
     * Constraint: If both the solution stack name parameter and the source configuration parameters are specified, the
     * solution stack of the source configuration template must match the specified solution stack name or else AWS
     * Elastic Beanstalk returns an <code>InvalidParameterCombination</code> error.
     * </p>
     * 
     * @return If specified, AWS Elastic Beanstalk uses the configuration values from the specified configuration
     *         template to create a new configuration.</p>
     *         <p>
     *         Values specified in the <code>OptionSettings</code> parameter of this call overrides any values obtained
     *         from the <code>SourceConfiguration</code>.
     *         </p>
     *         <p>
     *         If no configuration template is found, returns an <code>InvalidParameterValue</code> error.
     *         </p>
     *         <p>
     *         Constraint: If both the solution stack name parameter and the source configuration parameters are
     *         specified, the solution stack of the source configuration template must match the specified solution
     *         stack name or else AWS Elastic Beanstalk returns an <code>InvalidParameterCombination</code> error.
     */
    public SourceConfiguration sourceConfiguration() {
        return sourceConfiguration;
    }

    /**
     * <p>
     * The ID of the environment used with this configuration template.
     * </p>
     * 
     * @return The ID of the environment used with this configuration template.
     */
    public String environmentId() {
        return environmentId;
    }

    /**
     * <p>
     * Describes this configuration.
     * </p>
     * 
     * @return Describes this configuration.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * If specified, AWS Elastic Beanstalk sets the specified configuration option to the requested value. The new value
     * overrides the value obtained from the solution stack or the source configuration template.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return If specified, AWS Elastic Beanstalk sets the specified configuration option to the requested value. The
     *         new value overrides the value obtained from the solution stack or the source configuration template.
     */
    public List<ConfigurationOptionSetting> optionSettings() {
        return optionSettings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(applicationName());
        hashCode = 31 * hashCode + Objects.hashCode(templateName());
        hashCode = 31 * hashCode + Objects.hashCode(solutionStackName());
        hashCode = 31 * hashCode + Objects.hashCode(platformArn());
        hashCode = 31 * hashCode + Objects.hashCode(sourceConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(environmentId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(optionSettings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateConfigurationTemplateRequest)) {
            return false;
        }
        CreateConfigurationTemplateRequest other = (CreateConfigurationTemplateRequest) obj;
        return Objects.equals(applicationName(), other.applicationName()) && Objects.equals(templateName(), other.templateName())
                && Objects.equals(solutionStackName(), other.solutionStackName())
                && Objects.equals(platformArn(), other.platformArn())
                && Objects.equals(sourceConfiguration(), other.sourceConfiguration())
                && Objects.equals(environmentId(), other.environmentId()) && Objects.equals(description(), other.description())
                && Objects.equals(optionSettings(), other.optionSettings());
    }

    @Override
    public String toString() {
        return ToString.builder("CreateConfigurationTemplateRequest").add("ApplicationName", applicationName())
                .add("TemplateName", templateName()).add("SolutionStackName", solutionStackName())
                .add("PlatformArn", platformArn()).add("SourceConfiguration", sourceConfiguration())
                .add("EnvironmentId", environmentId()).add("Description", description()).add("OptionSettings", optionSettings())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApplicationName":
            return Optional.ofNullable(clazz.cast(applicationName()));
        case "TemplateName":
            return Optional.ofNullable(clazz.cast(templateName()));
        case "SolutionStackName":
            return Optional.ofNullable(clazz.cast(solutionStackName()));
        case "PlatformArn":
            return Optional.ofNullable(clazz.cast(platformArn()));
        case "SourceConfiguration":
            return Optional.ofNullable(clazz.cast(sourceConfiguration()));
        case "EnvironmentId":
            return Optional.ofNullable(clazz.cast(environmentId()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "OptionSettings":
            return Optional.ofNullable(clazz.cast(optionSettings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateConfigurationTemplateRequest, T> g) {
        return obj -> g.apply((CreateConfigurationTemplateRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ElasticBeanstalkRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, CreateConfigurationTemplateRequest> {
        /**
         * <p>
         * The name of the application to associate with this configuration template. If no application is found with
         * this name, AWS Elastic Beanstalk returns an <code>InvalidParameterValue</code> error.
         * </p>
         * 
         * @param applicationName
         *        The name of the application to associate with this configuration template. If no application is found
         *        with this name, AWS Elastic Beanstalk returns an <code>InvalidParameterValue</code> error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationName(String applicationName);

        /**
         * <p>
         * The name of the configuration template.
         * </p>
         * <p>
         * Constraint: This name must be unique per application.
         * </p>
         * <p>
         * Default: If a configuration template already exists with this name, AWS Elastic Beanstalk returns an
         * <code>InvalidParameterValue</code> error.
         * </p>
         * 
         * @param templateName
         *        The name of the configuration template.</p>
         *        <p>
         *        Constraint: This name must be unique per application.
         *        </p>
         *        <p>
         *        Default: If a configuration template already exists with this name, AWS Elastic Beanstalk returns an
         *        <code>InvalidParameterValue</code> error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateName(String templateName);

        /**
         * <p>
         * The name of the solution stack used by this configuration. The solution stack specifies the operating system,
         * architecture, and application server for a configuration template. It determines the set of configuration
         * options as well as the possible and default values.
         * </p>
         * <p>
         * Use <a>ListAvailableSolutionStacks</a> to obtain a list of available solution stacks.
         * </p>
         * <p>
         * A solution stack name or a source configuration parameter must be specified, otherwise AWS Elastic Beanstalk
         * returns an <code>InvalidParameterValue</code> error.
         * </p>
         * <p>
         * If a solution stack name is not specified and the source configuration parameter is specified, AWS Elastic
         * Beanstalk uses the same solution stack as the source configuration template.
         * </p>
         * 
         * @param solutionStackName
         *        The name of the solution stack used by this configuration. The solution stack specifies the operating
         *        system, architecture, and application server for a configuration template. It determines the set of
         *        configuration options as well as the possible and default values.</p>
         *        <p>
         *        Use <a>ListAvailableSolutionStacks</a> to obtain a list of available solution stacks.
         *        </p>
         *        <p>
         *        A solution stack name or a source configuration parameter must be specified, otherwise AWS Elastic
         *        Beanstalk returns an <code>InvalidParameterValue</code> error.
         *        </p>
         *        <p>
         *        If a solution stack name is not specified and the source configuration parameter is specified, AWS
         *        Elastic Beanstalk uses the same solution stack as the source configuration template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder solutionStackName(String solutionStackName);

        /**
         * <p>
         * The ARN of the custom platform.
         * </p>
         * 
         * @param platformArn
         *        The ARN of the custom platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformArn(String platformArn);

        /**
         * <p>
         * If specified, AWS Elastic Beanstalk uses the configuration values from the specified configuration template
         * to create a new configuration.
         * </p>
         * <p>
         * Values specified in the <code>OptionSettings</code> parameter of this call overrides any values obtained from
         * the <code>SourceConfiguration</code>.
         * </p>
         * <p>
         * If no configuration template is found, returns an <code>InvalidParameterValue</code> error.
         * </p>
         * <p>
         * Constraint: If both the solution stack name parameter and the source configuration parameters are specified,
         * the solution stack of the source configuration template must match the specified solution stack name or else
         * AWS Elastic Beanstalk returns an <code>InvalidParameterCombination</code> error.
         * </p>
         * 
         * @param sourceConfiguration
         *        If specified, AWS Elastic Beanstalk uses the configuration values from the specified configuration
         *        template to create a new configuration.</p>
         *        <p>
         *        Values specified in the <code>OptionSettings</code> parameter of this call overrides any values
         *        obtained from the <code>SourceConfiguration</code>.
         *        </p>
         *        <p>
         *        If no configuration template is found, returns an <code>InvalidParameterValue</code> error.
         *        </p>
         *        <p>
         *        Constraint: If both the solution stack name parameter and the source configuration parameters are
         *        specified, the solution stack of the source configuration template must match the specified solution
         *        stack name or else AWS Elastic Beanstalk returns an <code>InvalidParameterCombination</code> error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceConfiguration(SourceConfiguration sourceConfiguration);

        /**
         * <p>
         * If specified, AWS Elastic Beanstalk uses the configuration values from the specified configuration template
         * to create a new configuration.
         * </p>
         * <p>
         * Values specified in the <code>OptionSettings</code> parameter of this call overrides any values obtained from
         * the <code>SourceConfiguration</code>.
         * </p>
         * <p>
         * If no configuration template is found, returns an <code>InvalidParameterValue</code> error.
         * </p>
         * <p>
         * Constraint: If both the solution stack name parameter and the source configuration parameters are specified,
         * the solution stack of the source configuration template must match the specified solution stack name or else
         * AWS Elastic Beanstalk returns an <code>InvalidParameterCombination</code> error.
         * </p>
         * This is a convenience that creates an instance of the {@link SourceConfiguration.Builder} avoiding the need
         * to create one manually via {@link SourceConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link SourceConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #sourceConfiguration(SourceConfiguration)}.
         * 
         * @param sourceConfiguration
         *        a consumer that will call methods on {@link SourceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceConfiguration(SourceConfiguration)
         */
        default Builder sourceConfiguration(Consumer<SourceConfiguration.Builder> sourceConfiguration) {
            return sourceConfiguration(SourceConfiguration.builder().applyMutation(sourceConfiguration).build());
        }

        /**
         * <p>
         * The ID of the environment used with this configuration template.
         * </p>
         * 
         * @param environmentId
         *        The ID of the environment used with this configuration template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentId(String environmentId);

        /**
         * <p>
         * Describes this configuration.
         * </p>
         * 
         * @param description
         *        Describes this configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * If specified, AWS Elastic Beanstalk sets the specified configuration option to the requested value. The new
         * value overrides the value obtained from the solution stack or the source configuration template.
         * </p>
         * 
         * @param optionSettings
         *        If specified, AWS Elastic Beanstalk sets the specified configuration option to the requested value.
         *        The new value overrides the value obtained from the solution stack or the source configuration
         *        template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optionSettings(Collection<ConfigurationOptionSetting> optionSettings);

        /**
         * <p>
         * If specified, AWS Elastic Beanstalk sets the specified configuration option to the requested value. The new
         * value overrides the value obtained from the solution stack or the source configuration template.
         * </p>
         * 
         * @param optionSettings
         *        If specified, AWS Elastic Beanstalk sets the specified configuration option to the requested value.
         *        The new value overrides the value obtained from the solution stack or the source configuration
         *        template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optionSettings(ConfigurationOptionSetting... optionSettings);

        /**
         * <p>
         * If specified, AWS Elastic Beanstalk sets the specified configuration option to the requested value. The new
         * value overrides the value obtained from the solution stack or the source configuration template.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ConfigurationOptionSetting>.Builder}
         * avoiding the need to create one manually via {@link List<ConfigurationOptionSetting>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ConfigurationOptionSetting>.Builder#build()} is called
         * immediately and its result is passed to {@link #optionSettings(List<ConfigurationOptionSetting>)}.
         * 
         * @param optionSettings
         *        a consumer that will call methods on {@link List<ConfigurationOptionSetting>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #optionSettings(List<ConfigurationOptionSetting>)
         */
        Builder optionSettings(Consumer<ConfigurationOptionSetting.Builder>... optionSettings);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ElasticBeanstalkRequest.BuilderImpl implements Builder {
        private String applicationName;

        private String templateName;

        private String solutionStackName;

        private String platformArn;

        private SourceConfiguration sourceConfiguration;

        private String environmentId;

        private String description;

        private List<ConfigurationOptionSetting> optionSettings = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateConfigurationTemplateRequest model) {
            super(model);
            applicationName(model.applicationName);
            templateName(model.templateName);
            solutionStackName(model.solutionStackName);
            platformArn(model.platformArn);
            sourceConfiguration(model.sourceConfiguration);
            environmentId(model.environmentId);
            description(model.description);
            optionSettings(model.optionSettings);
        }

        public final String getApplicationName() {
            return applicationName;
        }

        @Override
        public final Builder applicationName(String applicationName) {
            this.applicationName = applicationName;
            return this;
        }

        public final void setApplicationName(String applicationName) {
            this.applicationName = applicationName;
        }

        public final String getTemplateName() {
            return templateName;
        }

        @Override
        public final Builder templateName(String templateName) {
            this.templateName = templateName;
            return this;
        }

        public final void setTemplateName(String templateName) {
            this.templateName = templateName;
        }

        public final String getSolutionStackName() {
            return solutionStackName;
        }

        @Override
        public final Builder solutionStackName(String solutionStackName) {
            this.solutionStackName = solutionStackName;
            return this;
        }

        public final void setSolutionStackName(String solutionStackName) {
            this.solutionStackName = solutionStackName;
        }

        public final String getPlatformArn() {
            return platformArn;
        }

        @Override
        public final Builder platformArn(String platformArn) {
            this.platformArn = platformArn;
            return this;
        }

        public final void setPlatformArn(String platformArn) {
            this.platformArn = platformArn;
        }

        public final SourceConfiguration.Builder getSourceConfiguration() {
            return sourceConfiguration != null ? sourceConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder sourceConfiguration(SourceConfiguration sourceConfiguration) {
            this.sourceConfiguration = sourceConfiguration;
            return this;
        }

        public final void setSourceConfiguration(SourceConfiguration.BuilderImpl sourceConfiguration) {
            this.sourceConfiguration = sourceConfiguration != null ? sourceConfiguration.build() : null;
        }

        public final String getEnvironmentId() {
            return environmentId;
        }

        @Override
        public final Builder environmentId(String environmentId) {
            this.environmentId = environmentId;
            return this;
        }

        public final void setEnvironmentId(String environmentId) {
            this.environmentId = environmentId;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<ConfigurationOptionSetting.Builder> getOptionSettings() {
            return optionSettings != null ? optionSettings.stream().map(ConfigurationOptionSetting::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder optionSettings(Collection<ConfigurationOptionSetting> optionSettings) {
            this.optionSettings = ConfigurationOptionSettingsListCopier.copy(optionSettings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder optionSettings(ConfigurationOptionSetting... optionSettings) {
            optionSettings(Arrays.asList(optionSettings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder optionSettings(Consumer<ConfigurationOptionSetting.Builder>... optionSettings) {
            optionSettings(Stream.of(optionSettings).map(c -> ConfigurationOptionSetting.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setOptionSettings(Collection<ConfigurationOptionSetting.BuilderImpl> optionSettings) {
            this.optionSettings = ConfigurationOptionSettingsListCopier.copyFromBuilder(optionSettings);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateConfigurationTemplateRequest build() {
            return new CreateConfigurationTemplateRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
