/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticbeanstalk.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * CPU utilization and load average metrics for an Amazon EC2 instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SystemStatus implements SdkPojo, Serializable, ToCopyableBuilder<SystemStatus.Builder, SystemStatus> {
    private static final SdkField<CPUUtilization> CPU_UTILIZATION_FIELD = SdkField
            .<CPUUtilization> builder(MarshallingType.SDK_POJO).memberName("CPUUtilization")
            .getter(getter(SystemStatus::cpuUtilization)).setter(setter(Builder::cpuUtilization))
            .constructor(CPUUtilization::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CPUUtilization").build()).build();

    private static final SdkField<List<Double>> LOAD_AVERAGE_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("LoadAverage")
            .getter(getter(SystemStatus::loadAverage))
            .setter(setter(Builder::loadAverage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoadAverage").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CPU_UTILIZATION_FIELD,
            LOAD_AVERAGE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final CPUUtilization cpuUtilization;

    private final List<Double> loadAverage;

    private SystemStatus(BuilderImpl builder) {
        this.cpuUtilization = builder.cpuUtilization;
        this.loadAverage = builder.loadAverage;
    }

    /**
     * <p>
     * CPU utilization metrics for the instance.
     * </p>
     * 
     * @return CPU utilization metrics for the instance.
     */
    public final CPUUtilization cpuUtilization() {
        return cpuUtilization;
    }

    /**
     * For responses, this returns true if the service returned a value for the LoadAverage property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLoadAverage() {
        return loadAverage != null && !(loadAverage instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Load average in the last 1-minute, 5-minute, and 15-minute periods. For more information, see <a href=
     * "https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-metrics.html#health-enhanced-metrics-os"
     * >Operating System Metrics</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLoadAverage} method.
     * </p>
     * 
     * @return Load average in the last 1-minute, 5-minute, and 15-minute periods. For more information, see <a href=
     *         "https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-metrics.html#health-enhanced-metrics-os"
     *         >Operating System Metrics</a>.
     */
    public final List<Double> loadAverage() {
        return loadAverage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(cpuUtilization());
        hashCode = 31 * hashCode + Objects.hashCode(hasLoadAverage() ? loadAverage() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SystemStatus)) {
            return false;
        }
        SystemStatus other = (SystemStatus) obj;
        return Objects.equals(cpuUtilization(), other.cpuUtilization()) && hasLoadAverage() == other.hasLoadAverage()
                && Objects.equals(loadAverage(), other.loadAverage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SystemStatus").add("CPUUtilization", cpuUtilization())
                .add("LoadAverage", hasLoadAverage() ? loadAverage() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CPUUtilization":
            return Optional.ofNullable(clazz.cast(cpuUtilization()));
        case "LoadAverage":
            return Optional.ofNullable(clazz.cast(loadAverage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CPUUtilization", CPU_UTILIZATION_FIELD);
        map.put("LoadAverage", LOAD_AVERAGE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SystemStatus, T> g) {
        return obj -> g.apply((SystemStatus) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SystemStatus> {
        /**
         * <p>
         * CPU utilization metrics for the instance.
         * </p>
         * 
         * @param cpuUtilization
         *        CPU utilization metrics for the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuUtilization(CPUUtilization cpuUtilization);

        /**
         * <p>
         * CPU utilization metrics for the instance.
         * </p>
         * This is a convenience method that creates an instance of the {@link CPUUtilization.Builder} avoiding the need
         * to create one manually via {@link CPUUtilization#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CPUUtilization.Builder#build()} is called immediately and its
         * result is passed to {@link #cpuUtilization(CPUUtilization)}.
         * 
         * @param cpuUtilization
         *        a consumer that will call methods on {@link CPUUtilization.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cpuUtilization(CPUUtilization)
         */
        default Builder cpuUtilization(Consumer<CPUUtilization.Builder> cpuUtilization) {
            return cpuUtilization(CPUUtilization.builder().applyMutation(cpuUtilization).build());
        }

        /**
         * <p>
         * Load average in the last 1-minute, 5-minute, and 15-minute periods. For more information, see <a href=
         * "https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-metrics.html#health-enhanced-metrics-os"
         * >Operating System Metrics</a>.
         * </p>
         * 
         * @param loadAverage
         *        Load average in the last 1-minute, 5-minute, and 15-minute periods. For more information, see <a href=
         *        "https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-metrics.html#health-enhanced-metrics-os"
         *        >Operating System Metrics</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadAverage(Collection<Double> loadAverage);

        /**
         * <p>
         * Load average in the last 1-minute, 5-minute, and 15-minute periods. For more information, see <a href=
         * "https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-metrics.html#health-enhanced-metrics-os"
         * >Operating System Metrics</a>.
         * </p>
         * 
         * @param loadAverage
         *        Load average in the last 1-minute, 5-minute, and 15-minute periods. For more information, see <a href=
         *        "https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced-metrics.html#health-enhanced-metrics-os"
         *        >Operating System Metrics</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadAverage(Double... loadAverage);
    }

    static final class BuilderImpl implements Builder {
        private CPUUtilization cpuUtilization;

        private List<Double> loadAverage = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SystemStatus model) {
            cpuUtilization(model.cpuUtilization);
            loadAverage(model.loadAverage);
        }

        public final CPUUtilization.Builder getCpuUtilization() {
            return cpuUtilization != null ? cpuUtilization.toBuilder() : null;
        }

        public final void setCpuUtilization(CPUUtilization.BuilderImpl cpuUtilization) {
            this.cpuUtilization = cpuUtilization != null ? cpuUtilization.build() : null;
        }

        @Override
        public final Builder cpuUtilization(CPUUtilization cpuUtilization) {
            this.cpuUtilization = cpuUtilization;
            return this;
        }

        public final Collection<Double> getLoadAverage() {
            if (loadAverage instanceof SdkAutoConstructList) {
                return null;
            }
            return loadAverage;
        }

        public final void setLoadAverage(Collection<Double> loadAverage) {
            this.loadAverage = LoadAverageCopier.copy(loadAverage);
        }

        @Override
        public final Builder loadAverage(Collection<Double> loadAverage) {
            this.loadAverage = LoadAverageCopier.copy(loadAverage);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder loadAverage(Double... loadAverage) {
            loadAverage(Arrays.asList(loadAverage));
            return this;
        }

        @Override
        public SystemStatus build() {
            return new SystemStatus(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
