/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticbeanstalk.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Request to describe application versions.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeApplicationVersionsRequest extends ElasticBeanstalkRequest implements
        ToCopyableBuilder<DescribeApplicationVersionsRequest.Builder, DescribeApplicationVersionsRequest> {
    private static final SdkField<String> APPLICATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeApplicationVersionsRequest::applicationName)).setter(setter(Builder::applicationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationName").build()).build();

    private static final SdkField<List<String>> VERSION_LABELS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DescribeApplicationVersionsRequest::versionLabels))
            .setter(setter(Builder::versionLabels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionLabels").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MAX_RECORDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(DescribeApplicationVersionsRequest::maxRecords)).setter(setter(Builder::maxRecords))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxRecords").build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeApplicationVersionsRequest::nextToken)).setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextToken").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLICATION_NAME_FIELD,
            VERSION_LABELS_FIELD, MAX_RECORDS_FIELD, NEXT_TOKEN_FIELD));

    private final String applicationName;

    private final List<String> versionLabels;

    private final Integer maxRecords;

    private final String nextToken;

    private DescribeApplicationVersionsRequest(BuilderImpl builder) {
        super(builder);
        this.applicationName = builder.applicationName;
        this.versionLabels = builder.versionLabels;
        this.maxRecords = builder.maxRecords;
        this.nextToken = builder.nextToken;
    }

    /**
     * <p>
     * Specify an application name to show only application versions for that application.
     * </p>
     * 
     * @return Specify an application name to show only application versions for that application.
     */
    public String applicationName() {
        return applicationName;
    }

    /**
     * <p>
     * Specify a version label to show a specific application version.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Specify a version label to show a specific application version.
     */
    public List<String> versionLabels() {
        return versionLabels;
    }

    /**
     * <p>
     * For a paginated request. Specify a maximum number of application versions to include in each response.
     * </p>
     * <p>
     * If no <code>MaxRecords</code> is specified, all available application versions are retrieved in a single
     * response.
     * </p>
     * 
     * @return For a paginated request. Specify a maximum number of application versions to include in each
     *         response.</p>
     *         <p>
     *         If no <code>MaxRecords</code> is specified, all available application versions are retrieved in a single
     *         response.
     */
    public Integer maxRecords() {
        return maxRecords;
    }

    /**
     * <p>
     * For a paginated request. Specify a token from a previous response page to retrieve the next response page. All
     * other parameter values must be identical to the ones specified in the initial request.
     * </p>
     * <p>
     * If no <code>NextToken</code> is specified, the first page is retrieved.
     * </p>
     * 
     * @return For a paginated request. Specify a token from a previous response page to retrieve the next response
     *         page. All other parameter values must be identical to the ones specified in the initial request.</p>
     *         <p>
     *         If no <code>NextToken</code> is specified, the first page is retrieved.
     */
    public String nextToken() {
        return nextToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(applicationName());
        hashCode = 31 * hashCode + Objects.hashCode(versionLabels());
        hashCode = 31 * hashCode + Objects.hashCode(maxRecords());
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribeApplicationVersionsRequest)) {
            return false;
        }
        DescribeApplicationVersionsRequest other = (DescribeApplicationVersionsRequest) obj;
        return Objects.equals(applicationName(), other.applicationName())
                && Objects.equals(versionLabels(), other.versionLabels()) && Objects.equals(maxRecords(), other.maxRecords())
                && Objects.equals(nextToken(), other.nextToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DescribeApplicationVersionsRequest").add("ApplicationName", applicationName())
                .add("VersionLabels", versionLabels()).add("MaxRecords", maxRecords()).add("NextToken", nextToken()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApplicationName":
            return Optional.ofNullable(clazz.cast(applicationName()));
        case "VersionLabels":
            return Optional.ofNullable(clazz.cast(versionLabels()));
        case "MaxRecords":
            return Optional.ofNullable(clazz.cast(maxRecords()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribeApplicationVersionsRequest, T> g) {
        return obj -> g.apply((DescribeApplicationVersionsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ElasticBeanstalkRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, DescribeApplicationVersionsRequest> {
        /**
         * <p>
         * Specify an application name to show only application versions for that application.
         * </p>
         * 
         * @param applicationName
         *        Specify an application name to show only application versions for that application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationName(String applicationName);

        /**
         * <p>
         * Specify a version label to show a specific application version.
         * </p>
         * 
         * @param versionLabels
         *        Specify a version label to show a specific application version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionLabels(Collection<String> versionLabels);

        /**
         * <p>
         * Specify a version label to show a specific application version.
         * </p>
         * 
         * @param versionLabels
         *        Specify a version label to show a specific application version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionLabels(String... versionLabels);

        /**
         * <p>
         * For a paginated request. Specify a maximum number of application versions to include in each response.
         * </p>
         * <p>
         * If no <code>MaxRecords</code> is specified, all available application versions are retrieved in a single
         * response.
         * </p>
         * 
         * @param maxRecords
         *        For a paginated request. Specify a maximum number of application versions to include in each
         *        response.</p>
         *        <p>
         *        If no <code>MaxRecords</code> is specified, all available application versions are retrieved in a
         *        single response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxRecords(Integer maxRecords);

        /**
         * <p>
         * For a paginated request. Specify a token from a previous response page to retrieve the next response page.
         * All other parameter values must be identical to the ones specified in the initial request.
         * </p>
         * <p>
         * If no <code>NextToken</code> is specified, the first page is retrieved.
         * </p>
         * 
         * @param nextToken
         *        For a paginated request. Specify a token from a previous response page to retrieve the next response
         *        page. All other parameter values must be identical to the ones specified in the initial request.</p>
         *        <p>
         *        If no <code>NextToken</code> is specified, the first page is retrieved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ElasticBeanstalkRequest.BuilderImpl implements Builder {
        private String applicationName;

        private List<String> versionLabels = DefaultSdkAutoConstructList.getInstance();

        private Integer maxRecords;

        private String nextToken;

        private BuilderImpl() {
        }

        private BuilderImpl(DescribeApplicationVersionsRequest model) {
            super(model);
            applicationName(model.applicationName);
            versionLabels(model.versionLabels);
            maxRecords(model.maxRecords);
            nextToken(model.nextToken);
        }

        public final String getApplicationName() {
            return applicationName;
        }

        @Override
        public final Builder applicationName(String applicationName) {
            this.applicationName = applicationName;
            return this;
        }

        public final void setApplicationName(String applicationName) {
            this.applicationName = applicationName;
        }

        public final Collection<String> getVersionLabels() {
            return versionLabels;
        }

        @Override
        public final Builder versionLabels(Collection<String> versionLabels) {
            this.versionLabels = VersionLabelsListCopier.copy(versionLabels);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder versionLabels(String... versionLabels) {
            versionLabels(Arrays.asList(versionLabels));
            return this;
        }

        public final void setVersionLabels(Collection<String> versionLabels) {
            this.versionLabels = VersionLabelsListCopier.copy(versionLabels);
        }

        public final Integer getMaxRecords() {
            return maxRecords;
        }

        @Override
        public final Builder maxRecords(Integer maxRecords) {
            this.maxRecords = maxRecords;
            return this;
        }

        public final void setMaxRecords(Integer maxRecords) {
            this.maxRecords = maxRecords;
        }

        public final String getNextToken() {
            return nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DescribeApplicationVersionsRequest build() {
            return new DescribeApplicationVersionsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
