/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticbeanstalk.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Request to create or update a group of environments.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComposeEnvironmentsRequest extends ElasticBeanstalkRequest implements
        ToCopyableBuilder<ComposeEnvironmentsRequest.Builder, ComposeEnvironmentsRequest> {
    private static final SdkField<String> APPLICATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ComposeEnvironmentsRequest::applicationName)).setter(setter(Builder::applicationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationName").build()).build();

    private static final SdkField<String> GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ComposeEnvironmentsRequest::groupName)).setter(setter(Builder::groupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupName").build()).build();

    private static final SdkField<List<String>> VERSION_LABELS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ComposeEnvironmentsRequest::versionLabels))
            .setter(setter(Builder::versionLabels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionLabels").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLICATION_NAME_FIELD,
            GROUP_NAME_FIELD, VERSION_LABELS_FIELD));

    private final String applicationName;

    private final String groupName;

    private final List<String> versionLabels;

    private ComposeEnvironmentsRequest(BuilderImpl builder) {
        super(builder);
        this.applicationName = builder.applicationName;
        this.groupName = builder.groupName;
        this.versionLabels = builder.versionLabels;
    }

    /**
     * <p>
     * The name of the application to which the specified source bundles belong.
     * </p>
     * 
     * @return The name of the application to which the specified source bundles belong.
     */
    public String applicationName() {
        return applicationName;
    }

    /**
     * <p>
     * The name of the group to which the target environments belong. Specify a group name only if the environment name
     * defined in each target environment's manifest ends with a + (plus) character. See <a
     * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/environment-cfg-manifest.html">Environment Manifest
     * (env.yaml)</a> for details.
     * </p>
     * 
     * @return The name of the group to which the target environments belong. Specify a group name only if the
     *         environment name defined in each target environment's manifest ends with a + (plus) character. See <a
     *         href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/environment-cfg-manifest.html">Environment
     *         Manifest (env.yaml)</a> for details.
     */
    public String groupName() {
        return groupName;
    }

    /**
     * <p>
     * A list of version labels, specifying one or more application source bundles that belong to the target
     * application. Each source bundle must include an environment manifest that specifies the name of the environment
     * and the name of the solution stack to use, and optionally can specify environment links to create.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of version labels, specifying one or more application source bundles that belong to the target
     *         application. Each source bundle must include an environment manifest that specifies the name of the
     *         environment and the name of the solution stack to use, and optionally can specify environment links to
     *         create.
     */
    public List<String> versionLabels() {
        return versionLabels;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(applicationName());
        hashCode = 31 * hashCode + Objects.hashCode(groupName());
        hashCode = 31 * hashCode + Objects.hashCode(versionLabels());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComposeEnvironmentsRequest)) {
            return false;
        }
        ComposeEnvironmentsRequest other = (ComposeEnvironmentsRequest) obj;
        return Objects.equals(applicationName(), other.applicationName()) && Objects.equals(groupName(), other.groupName())
                && Objects.equals(versionLabels(), other.versionLabels());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ComposeEnvironmentsRequest").add("ApplicationName", applicationName())
                .add("GroupName", groupName()).add("VersionLabels", versionLabels()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApplicationName":
            return Optional.ofNullable(clazz.cast(applicationName()));
        case "GroupName":
            return Optional.ofNullable(clazz.cast(groupName()));
        case "VersionLabels":
            return Optional.ofNullable(clazz.cast(versionLabels()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ComposeEnvironmentsRequest, T> g) {
        return obj -> g.apply((ComposeEnvironmentsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ElasticBeanstalkRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, ComposeEnvironmentsRequest> {
        /**
         * <p>
         * The name of the application to which the specified source bundles belong.
         * </p>
         * 
         * @param applicationName
         *        The name of the application to which the specified source bundles belong.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationName(String applicationName);

        /**
         * <p>
         * The name of the group to which the target environments belong. Specify a group name only if the environment
         * name defined in each target environment's manifest ends with a + (plus) character. See <a
         * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/environment-cfg-manifest.html">Environment
         * Manifest (env.yaml)</a> for details.
         * </p>
         * 
         * @param groupName
         *        The name of the group to which the target environments belong. Specify a group name only if the
         *        environment name defined in each target environment's manifest ends with a + (plus) character. See <a
         *        href
         *        ="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/environment-cfg-manifest.html">Environment
         *        Manifest (env.yaml)</a> for details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupName(String groupName);

        /**
         * <p>
         * A list of version labels, specifying one or more application source bundles that belong to the target
         * application. Each source bundle must include an environment manifest that specifies the name of the
         * environment and the name of the solution stack to use, and optionally can specify environment links to
         * create.
         * </p>
         * 
         * @param versionLabels
         *        A list of version labels, specifying one or more application source bundles that belong to the target
         *        application. Each source bundle must include an environment manifest that specifies the name of the
         *        environment and the name of the solution stack to use, and optionally can specify environment links to
         *        create.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionLabels(Collection<String> versionLabels);

        /**
         * <p>
         * A list of version labels, specifying one or more application source bundles that belong to the target
         * application. Each source bundle must include an environment manifest that specifies the name of the
         * environment and the name of the solution stack to use, and optionally can specify environment links to
         * create.
         * </p>
         * 
         * @param versionLabels
         *        A list of version labels, specifying one or more application source bundles that belong to the target
         *        application. Each source bundle must include an environment manifest that specifies the name of the
         *        environment and the name of the solution stack to use, and optionally can specify environment links to
         *        create.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionLabels(String... versionLabels);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ElasticBeanstalkRequest.BuilderImpl implements Builder {
        private String applicationName;

        private String groupName;

        private List<String> versionLabels = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ComposeEnvironmentsRequest model) {
            super(model);
            applicationName(model.applicationName);
            groupName(model.groupName);
            versionLabels(model.versionLabels);
        }

        public final String getApplicationName() {
            return applicationName;
        }

        @Override
        public final Builder applicationName(String applicationName) {
            this.applicationName = applicationName;
            return this;
        }

        public final void setApplicationName(String applicationName) {
            this.applicationName = applicationName;
        }

        public final String getGroupName() {
            return groupName;
        }

        @Override
        public final Builder groupName(String groupName) {
            this.groupName = groupName;
            return this;
        }

        public final void setGroupName(String groupName) {
            this.groupName = groupName;
        }

        public final Collection<String> getVersionLabels() {
            return versionLabels;
        }

        @Override
        public final Builder versionLabels(Collection<String> versionLabels) {
            this.versionLabels = VersionLabelsCopier.copy(versionLabels);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder versionLabels(String... versionLabels) {
            versionLabels(Arrays.asList(versionLabels));
            return this;
        }

        public final void setVersionLabels(Collection<String> versionLabels) {
            this.versionLabels = VersionLabelsCopier.copy(versionLabels);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ComposeEnvironmentsRequest build() {
            return new ComposeEnvironmentsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
