/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticbeanstalk.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Request to update an environment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateEnvironmentRequest extends ElasticBeanstalkRequest implements
        ToCopyableBuilder<UpdateEnvironmentRequest.Builder, UpdateEnvironmentRequest> {
    private static final SdkField<String> APPLICATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateEnvironmentRequest::applicationName)).setter(setter(Builder::applicationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationName").build()).build();

    private static final SdkField<String> ENVIRONMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateEnvironmentRequest::environmentId)).setter(setter(Builder::environmentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentId").build()).build();

    private static final SdkField<String> ENVIRONMENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateEnvironmentRequest::environmentName)).setter(setter(Builder::environmentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentName").build()).build();

    private static final SdkField<String> GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateEnvironmentRequest::groupName)).setter(setter(Builder::groupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupName").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateEnvironmentRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<EnvironmentTier> TIER_FIELD = SdkField.<EnvironmentTier> builder(MarshallingType.SDK_POJO)
            .getter(getter(UpdateEnvironmentRequest::tier)).setter(setter(Builder::tier)).constructor(EnvironmentTier::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tier").build()).build();

    private static final SdkField<String> VERSION_LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateEnvironmentRequest::versionLabel)).setter(setter(Builder::versionLabel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionLabel").build()).build();

    private static final SdkField<String> TEMPLATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateEnvironmentRequest::templateName)).setter(setter(Builder::templateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateName").build()).build();

    private static final SdkField<String> SOLUTION_STACK_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateEnvironmentRequest::solutionStackName)).setter(setter(Builder::solutionStackName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SolutionStackName").build()).build();

    private static final SdkField<String> PLATFORM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UpdateEnvironmentRequest::platformArn)).setter(setter(Builder::platformArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlatformArn").build()).build();

    private static final SdkField<List<ConfigurationOptionSetting>> OPTION_SETTINGS_FIELD = SdkField
            .<List<ConfigurationOptionSetting>> builder(MarshallingType.LIST)
            .getter(getter(UpdateEnvironmentRequest::optionSettings))
            .setter(setter(Builder::optionSettings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OptionSettings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ConfigurationOptionSetting> builder(MarshallingType.SDK_POJO)
                                            .constructor(ConfigurationOptionSetting::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<OptionSpecification>> OPTIONS_TO_REMOVE_FIELD = SdkField
            .<List<OptionSpecification>> builder(MarshallingType.LIST)
            .getter(getter(UpdateEnvironmentRequest::optionsToRemove))
            .setter(setter(Builder::optionsToRemove))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OptionsToRemove").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<OptionSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(OptionSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLICATION_NAME_FIELD,
            ENVIRONMENT_ID_FIELD, ENVIRONMENT_NAME_FIELD, GROUP_NAME_FIELD, DESCRIPTION_FIELD, TIER_FIELD, VERSION_LABEL_FIELD,
            TEMPLATE_NAME_FIELD, SOLUTION_STACK_NAME_FIELD, PLATFORM_ARN_FIELD, OPTION_SETTINGS_FIELD, OPTIONS_TO_REMOVE_FIELD));

    private final String applicationName;

    private final String environmentId;

    private final String environmentName;

    private final String groupName;

    private final String description;

    private final EnvironmentTier tier;

    private final String versionLabel;

    private final String templateName;

    private final String solutionStackName;

    private final String platformArn;

    private final List<ConfigurationOptionSetting> optionSettings;

    private final List<OptionSpecification> optionsToRemove;

    private UpdateEnvironmentRequest(BuilderImpl builder) {
        super(builder);
        this.applicationName = builder.applicationName;
        this.environmentId = builder.environmentId;
        this.environmentName = builder.environmentName;
        this.groupName = builder.groupName;
        this.description = builder.description;
        this.tier = builder.tier;
        this.versionLabel = builder.versionLabel;
        this.templateName = builder.templateName;
        this.solutionStackName = builder.solutionStackName;
        this.platformArn = builder.platformArn;
        this.optionSettings = builder.optionSettings;
        this.optionsToRemove = builder.optionsToRemove;
    }

    /**
     * <p>
     * The name of the application with which the environment is associated.
     * </p>
     * 
     * @return The name of the application with which the environment is associated.
     */
    public String applicationName() {
        return applicationName;
    }

    /**
     * <p>
     * The ID of the environment to update.
     * </p>
     * <p>
     * If no environment with this ID exists, AWS Elastic Beanstalk returns an <code>InvalidParameterValue</code> error.
     * </p>
     * <p>
     * Condition: You must specify either this or an EnvironmentName, or both. If you do not specify either, AWS Elastic
     * Beanstalk returns <code>MissingRequiredParameter</code> error.
     * </p>
     * 
     * @return The ID of the environment to update.</p>
     *         <p>
     *         If no environment with this ID exists, AWS Elastic Beanstalk returns an
     *         <code>InvalidParameterValue</code> error.
     *         </p>
     *         <p>
     *         Condition: You must specify either this or an EnvironmentName, or both. If you do not specify either, AWS
     *         Elastic Beanstalk returns <code>MissingRequiredParameter</code> error.
     */
    public String environmentId() {
        return environmentId;
    }

    /**
     * <p>
     * The name of the environment to update. If no environment with this name exists, AWS Elastic Beanstalk returns an
     * <code>InvalidParameterValue</code> error.
     * </p>
     * <p>
     * Condition: You must specify either this or an EnvironmentId, or both. If you do not specify either, AWS Elastic
     * Beanstalk returns <code>MissingRequiredParameter</code> error.
     * </p>
     * 
     * @return The name of the environment to update. If no environment with this name exists, AWS Elastic Beanstalk
     *         returns an <code>InvalidParameterValue</code> error. </p>
     *         <p>
     *         Condition: You must specify either this or an EnvironmentId, or both. If you do not specify either, AWS
     *         Elastic Beanstalk returns <code>MissingRequiredParameter</code> error.
     */
    public String environmentName() {
        return environmentName;
    }

    /**
     * <p>
     * The name of the group to which the target environment belongs. Specify a group name only if the environment's
     * name is specified in an environment manifest and not with the environment name or environment ID parameters. See
     * <a href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/environment-cfg-manifest.html">Environment
     * Manifest (env.yaml)</a> for details.
     * </p>
     * 
     * @return The name of the group to which the target environment belongs. Specify a group name only if the
     *         environment's name is specified in an environment manifest and not with the environment name or
     *         environment ID parameters. See <a
     *         href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/environment-cfg-manifest.html">Environment
     *         Manifest (env.yaml)</a> for details.
     */
    public String groupName() {
        return groupName;
    }

    /**
     * <p>
     * If this parameter is specified, AWS Elastic Beanstalk updates the description of this environment.
     * </p>
     * 
     * @return If this parameter is specified, AWS Elastic Beanstalk updates the description of this environment.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * This specifies the tier to use to update the environment.
     * </p>
     * <p>
     * Condition: At this time, if you change the tier version, name, or type, AWS Elastic Beanstalk returns
     * <code>InvalidParameterValue</code> error.
     * </p>
     * 
     * @return This specifies the tier to use to update the environment.</p>
     *         <p>
     *         Condition: At this time, if you change the tier version, name, or type, AWS Elastic Beanstalk returns
     *         <code>InvalidParameterValue</code> error.
     */
    public EnvironmentTier tier() {
        return tier;
    }

    /**
     * <p>
     * If this parameter is specified, AWS Elastic Beanstalk deploys the named application version to the environment.
     * If no such application version is found, returns an <code>InvalidParameterValue</code> error.
     * </p>
     * 
     * @return If this parameter is specified, AWS Elastic Beanstalk deploys the named application version to the
     *         environment. If no such application version is found, returns an <code>InvalidParameterValue</code>
     *         error.
     */
    public String versionLabel() {
        return versionLabel;
    }

    /**
     * <p>
     * If this parameter is specified, AWS Elastic Beanstalk deploys this configuration template to the environment. If
     * no such configuration template is found, AWS Elastic Beanstalk returns an <code>InvalidParameterValue</code>
     * error.
     * </p>
     * 
     * @return If this parameter is specified, AWS Elastic Beanstalk deploys this configuration template to the
     *         environment. If no such configuration template is found, AWS Elastic Beanstalk returns an
     *         <code>InvalidParameterValue</code> error.
     */
    public String templateName() {
        return templateName;
    }

    /**
     * <p>
     * This specifies the platform version that the environment will run after the environment is updated.
     * </p>
     * 
     * @return This specifies the platform version that the environment will run after the environment is updated.
     */
    public String solutionStackName() {
        return solutionStackName;
    }

    /**
     * <p>
     * The ARN of the platform, if used.
     * </p>
     * 
     * @return The ARN of the platform, if used.
     */
    public String platformArn() {
        return platformArn;
    }

    /**
     * <p>
     * If specified, AWS Elastic Beanstalk updates the configuration set associated with the running environment and
     * sets the specified configuration options to the requested value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return If specified, AWS Elastic Beanstalk updates the configuration set associated with the running environment
     *         and sets the specified configuration options to the requested value.
     */
    public List<ConfigurationOptionSetting> optionSettings() {
        return optionSettings;
    }

    /**
     * <p>
     * A list of custom user-defined configuration options to remove from the configuration set for this environment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of custom user-defined configuration options to remove from the configuration set for this
     *         environment.
     */
    public List<OptionSpecification> optionsToRemove() {
        return optionsToRemove;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(applicationName());
        hashCode = 31 * hashCode + Objects.hashCode(environmentId());
        hashCode = 31 * hashCode + Objects.hashCode(environmentName());
        hashCode = 31 * hashCode + Objects.hashCode(groupName());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(tier());
        hashCode = 31 * hashCode + Objects.hashCode(versionLabel());
        hashCode = 31 * hashCode + Objects.hashCode(templateName());
        hashCode = 31 * hashCode + Objects.hashCode(solutionStackName());
        hashCode = 31 * hashCode + Objects.hashCode(platformArn());
        hashCode = 31 * hashCode + Objects.hashCode(optionSettings());
        hashCode = 31 * hashCode + Objects.hashCode(optionsToRemove());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateEnvironmentRequest)) {
            return false;
        }
        UpdateEnvironmentRequest other = (UpdateEnvironmentRequest) obj;
        return Objects.equals(applicationName(), other.applicationName())
                && Objects.equals(environmentId(), other.environmentId())
                && Objects.equals(environmentName(), other.environmentName()) && Objects.equals(groupName(), other.groupName())
                && Objects.equals(description(), other.description()) && Objects.equals(tier(), other.tier())
                && Objects.equals(versionLabel(), other.versionLabel()) && Objects.equals(templateName(), other.templateName())
                && Objects.equals(solutionStackName(), other.solutionStackName())
                && Objects.equals(platformArn(), other.platformArn()) && Objects.equals(optionSettings(), other.optionSettings())
                && Objects.equals(optionsToRemove(), other.optionsToRemove());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("UpdateEnvironmentRequest").add("ApplicationName", applicationName())
                .add("EnvironmentId", environmentId()).add("EnvironmentName", environmentName()).add("GroupName", groupName())
                .add("Description", description()).add("Tier", tier()).add("VersionLabel", versionLabel())
                .add("TemplateName", templateName()).add("SolutionStackName", solutionStackName())
                .add("PlatformArn", platformArn()).add("OptionSettings", optionSettings())
                .add("OptionsToRemove", optionsToRemove()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApplicationName":
            return Optional.ofNullable(clazz.cast(applicationName()));
        case "EnvironmentId":
            return Optional.ofNullable(clazz.cast(environmentId()));
        case "EnvironmentName":
            return Optional.ofNullable(clazz.cast(environmentName()));
        case "GroupName":
            return Optional.ofNullable(clazz.cast(groupName()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Tier":
            return Optional.ofNullable(clazz.cast(tier()));
        case "VersionLabel":
            return Optional.ofNullable(clazz.cast(versionLabel()));
        case "TemplateName":
            return Optional.ofNullable(clazz.cast(templateName()));
        case "SolutionStackName":
            return Optional.ofNullable(clazz.cast(solutionStackName()));
        case "PlatformArn":
            return Optional.ofNullable(clazz.cast(platformArn()));
        case "OptionSettings":
            return Optional.ofNullable(clazz.cast(optionSettings()));
        case "OptionsToRemove":
            return Optional.ofNullable(clazz.cast(optionsToRemove()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpdateEnvironmentRequest, T> g) {
        return obj -> g.apply((UpdateEnvironmentRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ElasticBeanstalkRequest.Builder, SdkPojo, CopyableBuilder<Builder, UpdateEnvironmentRequest> {
        /**
         * <p>
         * The name of the application with which the environment is associated.
         * </p>
         * 
         * @param applicationName
         *        The name of the application with which the environment is associated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationName(String applicationName);

        /**
         * <p>
         * The ID of the environment to update.
         * </p>
         * <p>
         * If no environment with this ID exists, AWS Elastic Beanstalk returns an <code>InvalidParameterValue</code>
         * error.
         * </p>
         * <p>
         * Condition: You must specify either this or an EnvironmentName, or both. If you do not specify either, AWS
         * Elastic Beanstalk returns <code>MissingRequiredParameter</code> error.
         * </p>
         * 
         * @param environmentId
         *        The ID of the environment to update.</p>
         *        <p>
         *        If no environment with this ID exists, AWS Elastic Beanstalk returns an
         *        <code>InvalidParameterValue</code> error.
         *        </p>
         *        <p>
         *        Condition: You must specify either this or an EnvironmentName, or both. If you do not specify either,
         *        AWS Elastic Beanstalk returns <code>MissingRequiredParameter</code> error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentId(String environmentId);

        /**
         * <p>
         * The name of the environment to update. If no environment with this name exists, AWS Elastic Beanstalk returns
         * an <code>InvalidParameterValue</code> error.
         * </p>
         * <p>
         * Condition: You must specify either this or an EnvironmentId, or both. If you do not specify either, AWS
         * Elastic Beanstalk returns <code>MissingRequiredParameter</code> error.
         * </p>
         * 
         * @param environmentName
         *        The name of the environment to update. If no environment with this name exists, AWS Elastic Beanstalk
         *        returns an <code>InvalidParameterValue</code> error. </p>
         *        <p>
         *        Condition: You must specify either this or an EnvironmentId, or both. If you do not specify either,
         *        AWS Elastic Beanstalk returns <code>MissingRequiredParameter</code> error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentName(String environmentName);

        /**
         * <p>
         * The name of the group to which the target environment belongs. Specify a group name only if the environment's
         * name is specified in an environment manifest and not with the environment name or environment ID parameters.
         * See <a
         * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/environment-cfg-manifest.html">Environment
         * Manifest (env.yaml)</a> for details.
         * </p>
         * 
         * @param groupName
         *        The name of the group to which the target environment belongs. Specify a group name only if the
         *        environment's name is specified in an environment manifest and not with the environment name or
         *        environment ID parameters. See <a
         *        href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/environment-cfg-manifest.html"
         *        >Environment Manifest (env.yaml)</a> for details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupName(String groupName);

        /**
         * <p>
         * If this parameter is specified, AWS Elastic Beanstalk updates the description of this environment.
         * </p>
         * 
         * @param description
         *        If this parameter is specified, AWS Elastic Beanstalk updates the description of this environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * This specifies the tier to use to update the environment.
         * </p>
         * <p>
         * Condition: At this time, if you change the tier version, name, or type, AWS Elastic Beanstalk returns
         * <code>InvalidParameterValue</code> error.
         * </p>
         * 
         * @param tier
         *        This specifies the tier to use to update the environment.</p>
         *        <p>
         *        Condition: At this time, if you change the tier version, name, or type, AWS Elastic Beanstalk returns
         *        <code>InvalidParameterValue</code> error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tier(EnvironmentTier tier);

        /**
         * <p>
         * This specifies the tier to use to update the environment.
         * </p>
         * <p>
         * Condition: At this time, if you change the tier version, name, or type, AWS Elastic Beanstalk returns
         * <code>InvalidParameterValue</code> error.
         * </p>
         * This is a convenience that creates an instance of the {@link EnvironmentTier.Builder} avoiding the need to
         * create one manually via {@link EnvironmentTier#builder()}.
         *
         * When the {@link Consumer} completes, {@link EnvironmentTier.Builder#build()} is called immediately and its
         * result is passed to {@link #tier(EnvironmentTier)}.
         * 
         * @param tier
         *        a consumer that will call methods on {@link EnvironmentTier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tier(EnvironmentTier)
         */
        default Builder tier(Consumer<EnvironmentTier.Builder> tier) {
            return tier(EnvironmentTier.builder().applyMutation(tier).build());
        }

        /**
         * <p>
         * If this parameter is specified, AWS Elastic Beanstalk deploys the named application version to the
         * environment. If no such application version is found, returns an <code>InvalidParameterValue</code> error.
         * </p>
         * 
         * @param versionLabel
         *        If this parameter is specified, AWS Elastic Beanstalk deploys the named application version to the
         *        environment. If no such application version is found, returns an <code>InvalidParameterValue</code>
         *        error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionLabel(String versionLabel);

        /**
         * <p>
         * If this parameter is specified, AWS Elastic Beanstalk deploys this configuration template to the environment.
         * If no such configuration template is found, AWS Elastic Beanstalk returns an
         * <code>InvalidParameterValue</code> error.
         * </p>
         * 
         * @param templateName
         *        If this parameter is specified, AWS Elastic Beanstalk deploys this configuration template to the
         *        environment. If no such configuration template is found, AWS Elastic Beanstalk returns an
         *        <code>InvalidParameterValue</code> error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateName(String templateName);

        /**
         * <p>
         * This specifies the platform version that the environment will run after the environment is updated.
         * </p>
         * 
         * @param solutionStackName
         *        This specifies the platform version that the environment will run after the environment is updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder solutionStackName(String solutionStackName);

        /**
         * <p>
         * The ARN of the platform, if used.
         * </p>
         * 
         * @param platformArn
         *        The ARN of the platform, if used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformArn(String platformArn);

        /**
         * <p>
         * If specified, AWS Elastic Beanstalk updates the configuration set associated with the running environment and
         * sets the specified configuration options to the requested value.
         * </p>
         * 
         * @param optionSettings
         *        If specified, AWS Elastic Beanstalk updates the configuration set associated with the running
         *        environment and sets the specified configuration options to the requested value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optionSettings(Collection<ConfigurationOptionSetting> optionSettings);

        /**
         * <p>
         * If specified, AWS Elastic Beanstalk updates the configuration set associated with the running environment and
         * sets the specified configuration options to the requested value.
         * </p>
         * 
         * @param optionSettings
         *        If specified, AWS Elastic Beanstalk updates the configuration set associated with the running
         *        environment and sets the specified configuration options to the requested value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optionSettings(ConfigurationOptionSetting... optionSettings);

        /**
         * <p>
         * If specified, AWS Elastic Beanstalk updates the configuration set associated with the running environment and
         * sets the specified configuration options to the requested value.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ConfigurationOptionSetting>.Builder}
         * avoiding the need to create one manually via {@link List<ConfigurationOptionSetting>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ConfigurationOptionSetting>.Builder#build()} is called
         * immediately and its result is passed to {@link #optionSettings(List<ConfigurationOptionSetting>)}.
         * 
         * @param optionSettings
         *        a consumer that will call methods on {@link List<ConfigurationOptionSetting>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #optionSettings(List<ConfigurationOptionSetting>)
         */
        Builder optionSettings(Consumer<ConfigurationOptionSetting.Builder>... optionSettings);

        /**
         * <p>
         * A list of custom user-defined configuration options to remove from the configuration set for this
         * environment.
         * </p>
         * 
         * @param optionsToRemove
         *        A list of custom user-defined configuration options to remove from the configuration set for this
         *        environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optionsToRemove(Collection<OptionSpecification> optionsToRemove);

        /**
         * <p>
         * A list of custom user-defined configuration options to remove from the configuration set for this
         * environment.
         * </p>
         * 
         * @param optionsToRemove
         *        A list of custom user-defined configuration options to remove from the configuration set for this
         *        environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optionsToRemove(OptionSpecification... optionsToRemove);

        /**
         * <p>
         * A list of custom user-defined configuration options to remove from the configuration set for this
         * environment.
         * </p>
         * This is a convenience that creates an instance of the {@link List<OptionSpecification>.Builder} avoiding the
         * need to create one manually via {@link List<OptionSpecification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<OptionSpecification>.Builder#build()} is called immediately
         * and its result is passed to {@link #optionsToRemove(List<OptionSpecification>)}.
         * 
         * @param optionsToRemove
         *        a consumer that will call methods on {@link List<OptionSpecification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #optionsToRemove(List<OptionSpecification>)
         */
        Builder optionsToRemove(Consumer<OptionSpecification.Builder>... optionsToRemove);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ElasticBeanstalkRequest.BuilderImpl implements Builder {
        private String applicationName;

        private String environmentId;

        private String environmentName;

        private String groupName;

        private String description;

        private EnvironmentTier tier;

        private String versionLabel;

        private String templateName;

        private String solutionStackName;

        private String platformArn;

        private List<ConfigurationOptionSetting> optionSettings = DefaultSdkAutoConstructList.getInstance();

        private List<OptionSpecification> optionsToRemove = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateEnvironmentRequest model) {
            super(model);
            applicationName(model.applicationName);
            environmentId(model.environmentId);
            environmentName(model.environmentName);
            groupName(model.groupName);
            description(model.description);
            tier(model.tier);
            versionLabel(model.versionLabel);
            templateName(model.templateName);
            solutionStackName(model.solutionStackName);
            platformArn(model.platformArn);
            optionSettings(model.optionSettings);
            optionsToRemove(model.optionsToRemove);
        }

        public final String getApplicationName() {
            return applicationName;
        }

        @Override
        public final Builder applicationName(String applicationName) {
            this.applicationName = applicationName;
            return this;
        }

        public final void setApplicationName(String applicationName) {
            this.applicationName = applicationName;
        }

        public final String getEnvironmentId() {
            return environmentId;
        }

        @Override
        public final Builder environmentId(String environmentId) {
            this.environmentId = environmentId;
            return this;
        }

        public final void setEnvironmentId(String environmentId) {
            this.environmentId = environmentId;
        }

        public final String getEnvironmentName() {
            return environmentName;
        }

        @Override
        public final Builder environmentName(String environmentName) {
            this.environmentName = environmentName;
            return this;
        }

        public final void setEnvironmentName(String environmentName) {
            this.environmentName = environmentName;
        }

        public final String getGroupName() {
            return groupName;
        }

        @Override
        public final Builder groupName(String groupName) {
            this.groupName = groupName;
            return this;
        }

        public final void setGroupName(String groupName) {
            this.groupName = groupName;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final EnvironmentTier.Builder getTier() {
            return tier != null ? tier.toBuilder() : null;
        }

        @Override
        public final Builder tier(EnvironmentTier tier) {
            this.tier = tier;
            return this;
        }

        public final void setTier(EnvironmentTier.BuilderImpl tier) {
            this.tier = tier != null ? tier.build() : null;
        }

        public final String getVersionLabel() {
            return versionLabel;
        }

        @Override
        public final Builder versionLabel(String versionLabel) {
            this.versionLabel = versionLabel;
            return this;
        }

        public final void setVersionLabel(String versionLabel) {
            this.versionLabel = versionLabel;
        }

        public final String getTemplateName() {
            return templateName;
        }

        @Override
        public final Builder templateName(String templateName) {
            this.templateName = templateName;
            return this;
        }

        public final void setTemplateName(String templateName) {
            this.templateName = templateName;
        }

        public final String getSolutionStackName() {
            return solutionStackName;
        }

        @Override
        public final Builder solutionStackName(String solutionStackName) {
            this.solutionStackName = solutionStackName;
            return this;
        }

        public final void setSolutionStackName(String solutionStackName) {
            this.solutionStackName = solutionStackName;
        }

        public final String getPlatformArn() {
            return platformArn;
        }

        @Override
        public final Builder platformArn(String platformArn) {
            this.platformArn = platformArn;
            return this;
        }

        public final void setPlatformArn(String platformArn) {
            this.platformArn = platformArn;
        }

        public final Collection<ConfigurationOptionSetting.Builder> getOptionSettings() {
            return optionSettings != null ? optionSettings.stream().map(ConfigurationOptionSetting::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder optionSettings(Collection<ConfigurationOptionSetting> optionSettings) {
            this.optionSettings = ConfigurationOptionSettingsListCopier.copy(optionSettings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder optionSettings(ConfigurationOptionSetting... optionSettings) {
            optionSettings(Arrays.asList(optionSettings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder optionSettings(Consumer<ConfigurationOptionSetting.Builder>... optionSettings) {
            optionSettings(Stream.of(optionSettings).map(c -> ConfigurationOptionSetting.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setOptionSettings(Collection<ConfigurationOptionSetting.BuilderImpl> optionSettings) {
            this.optionSettings = ConfigurationOptionSettingsListCopier.copyFromBuilder(optionSettings);
        }

        public final Collection<OptionSpecification.Builder> getOptionsToRemove() {
            return optionsToRemove != null ? optionsToRemove.stream().map(OptionSpecification::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder optionsToRemove(Collection<OptionSpecification> optionsToRemove) {
            this.optionsToRemove = OptionsSpecifierListCopier.copy(optionsToRemove);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder optionsToRemove(OptionSpecification... optionsToRemove) {
            optionsToRemove(Arrays.asList(optionsToRemove));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder optionsToRemove(Consumer<OptionSpecification.Builder>... optionsToRemove) {
            optionsToRemove(Stream.of(optionsToRemove).map(c -> OptionSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setOptionsToRemove(Collection<OptionSpecification.BuilderImpl> optionsToRemove) {
            this.optionsToRemove = OptionsSpecifierListCopier.copyFromBuilder(optionsToRemove);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateEnvironmentRequest build() {
            return new UpdateEnvironmentRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
