/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticinference;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.elasticinference.model.BadRequestException;
import software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorOfferingsRequest;
import software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorOfferingsResponse;
import software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorTypesRequest;
import software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorTypesResponse;
import software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsRequest;
import software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsResponse;
import software.amazon.awssdk.services.elasticinference.model.ElasticInferenceException;
import software.amazon.awssdk.services.elasticinference.model.InternalServerException;
import software.amazon.awssdk.services.elasticinference.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.elasticinference.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.elasticinference.model.ResourceNotFoundException;
import software.amazon.awssdk.services.elasticinference.model.TagResourceRequest;
import software.amazon.awssdk.services.elasticinference.model.TagResourceResponse;
import software.amazon.awssdk.services.elasticinference.model.UntagResourceRequest;
import software.amazon.awssdk.services.elasticinference.model.UntagResourceResponse;
import software.amazon.awssdk.services.elasticinference.paginators.DescribeAcceleratorsIterable;

/**
 * Service client for accessing Amazon Elastic Inference. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * Elastic Inference public APIs.
 * </p>
 * <p>
 * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI), and
 * will help current customers migrate their workloads to options that offer better price and performance. After April
 * 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon SageMaker, Amazon
 * ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past 30-day period are
 * considered current customers and will be able to continue using the service.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface ElasticInferenceClient extends AwsClient {
    String SERVICE_NAME = "elastic-inference";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "api.elastic-inference";

    /**
     * <p>
     * Describes the locations in which a given accelerator type or set of types is present in a given region.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     *
     * @param describeAcceleratorOfferingsRequest
     * @return Result of the DescribeAcceleratorOfferings operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.DescribeAcceleratorOfferings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/DescribeAcceleratorOfferings"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeAcceleratorOfferingsResponse describeAcceleratorOfferings(
            DescribeAcceleratorOfferingsRequest describeAcceleratorOfferingsRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ElasticInferenceException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes the locations in which a given accelerator type or set of types is present in a given region.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeAcceleratorOfferingsRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeAcceleratorOfferingsRequest#builder()}
     * </p>
     *
     * @param describeAcceleratorOfferingsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorOfferingsRequest.Builder}
     *        to create a request.
     * @return Result of the DescribeAcceleratorOfferings operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.DescribeAcceleratorOfferings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/DescribeAcceleratorOfferings"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeAcceleratorOfferingsResponse describeAcceleratorOfferings(
            Consumer<DescribeAcceleratorOfferingsRequest.Builder> describeAcceleratorOfferingsRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, ElasticInferenceException {
        return describeAcceleratorOfferings(DescribeAcceleratorOfferingsRequest.builder()
                .applyMutation(describeAcceleratorOfferingsRequest).build());
    }

    /**
     * <p>
     * Describes the accelerator types available in a given region, as well as their characteristics, such as memory and
     * throughput.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     *
     * @param describeAcceleratorTypesRequest
     * @return Result of the DescribeAcceleratorTypes operation returned by the service.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.DescribeAcceleratorTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/DescribeAcceleratorTypes"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeAcceleratorTypesResponse describeAcceleratorTypes(
            DescribeAcceleratorTypesRequest describeAcceleratorTypesRequest) throws InternalServerException, AwsServiceException,
            SdkClientException, ElasticInferenceException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes the accelerator types available in a given region, as well as their characteristics, such as memory and
     * throughput.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeAcceleratorTypesRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeAcceleratorTypesRequest#builder()}
     * </p>
     *
     * @param describeAcceleratorTypesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorTypesRequest.Builder} to
     *        create a request.
     * @return Result of the DescribeAcceleratorTypes operation returned by the service.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.DescribeAcceleratorTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/DescribeAcceleratorTypes"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeAcceleratorTypesResponse describeAcceleratorTypes(
            Consumer<DescribeAcceleratorTypesRequest.Builder> describeAcceleratorTypesRequest) throws InternalServerException,
            AwsServiceException, SdkClientException, ElasticInferenceException {
        return describeAcceleratorTypes(DescribeAcceleratorTypesRequest.builder().applyMutation(describeAcceleratorTypesRequest)
                .build());
    }

    /**
     * <p>
     * Describes information over a provided set of accelerators belonging to an account.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     *
     * @param describeAcceleratorsRequest
     * @return Result of the DescribeAccelerators operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.DescribeAccelerators
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/DescribeAccelerators"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeAcceleratorsResponse describeAccelerators(DescribeAcceleratorsRequest describeAcceleratorsRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, ElasticInferenceException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes information over a provided set of accelerators belonging to an account.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeAcceleratorsRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeAcceleratorsRequest#builder()}
     * </p>
     *
     * @param describeAcceleratorsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsRequest.Builder} to
     *        create a request.
     * @return Result of the DescribeAccelerators operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.DescribeAccelerators
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/DescribeAccelerators"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeAcceleratorsResponse describeAccelerators(
            Consumer<DescribeAcceleratorsRequest.Builder> describeAcceleratorsRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ElasticInferenceException {
        return describeAccelerators(DescribeAcceleratorsRequest.builder().applyMutation(describeAcceleratorsRequest).build());
    }

    /**
     * <p>
     * Describes information over a provided set of accelerators belonging to an account.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAccelerators(software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticinference.paginators.DescribeAcceleratorsIterable responses = client.describeAcceleratorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.elasticinference.paginators.DescribeAcceleratorsIterable responses = client
     *             .describeAcceleratorsPaginator(request);
     *     for (software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticinference.paginators.DescribeAcceleratorsIterable responses = client.describeAcceleratorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAccelerators(software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAcceleratorsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.DescribeAccelerators
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/DescribeAccelerators"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeAcceleratorsIterable describeAcceleratorsPaginator(DescribeAcceleratorsRequest describeAcceleratorsRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, ElasticInferenceException {
        return new DescribeAcceleratorsIterable(this, describeAcceleratorsRequest);
    }

    /**
     * <p>
     * Describes information over a provided set of accelerators belonging to an account.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAccelerators(software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticinference.paginators.DescribeAcceleratorsIterable responses = client.describeAcceleratorsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.elasticinference.paginators.DescribeAcceleratorsIterable responses = client
     *             .describeAcceleratorsPaginator(request);
     *     for (software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.elasticinference.paginators.DescribeAcceleratorsIterable responses = client.describeAcceleratorsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAccelerators(software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeAcceleratorsRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeAcceleratorsRequest#builder()}
     * </p>
     *
     * @param describeAcceleratorsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.elasticinference.model.DescribeAcceleratorsRequest.Builder} to
     *        create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.DescribeAccelerators
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/DescribeAccelerators"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeAcceleratorsIterable describeAcceleratorsPaginator(
            Consumer<DescribeAcceleratorsRequest.Builder> describeAcceleratorsRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ElasticInferenceException {
        return describeAcceleratorsPaginator(DescribeAcceleratorsRequest.builder().applyMutation(describeAcceleratorsRequest)
                .build());
    }

    /**
     * <p>
     * Returns all tags of an Elastic Inference Accelerator.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, ElasticInferenceException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns all tags of an Elastic Inference Accelerator.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.elasticinference.model.ListTagsForResourceRequest.Builder} to
     *        create a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ElasticInferenceException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Adds the specified tags to an Elastic Inference Accelerator.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ElasticInferenceException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds the specified tags to an Elastic Inference Accelerator.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.elasticinference.model.TagResourceRequest.Builder} to create a
     *        request.
     * @return Result of the TagResource operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ElasticInferenceException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes the specified tags from an Elastic Inference Accelerator.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ElasticInferenceException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes the specified tags from an Elastic Inference Accelerator.
     * </p>
     * <p>
     * February 15, 2023: Starting April 15, 2023, AWS will not onboard new customers to Amazon Elastic Inference (EI),
     * and will help current customers migrate their workloads to options that offer better price and performance. After
     * April 15, 2023, new customers will not be able to launch instances with Amazon EI accelerators in Amazon
     * SageMaker, Amazon ECS, or Amazon EC2. However, customers who have used Amazon EI at least once during the past
     * 30-day period are considered current customers and will be able to continue using the service.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.elasticinference.model.UntagResourceRequest.Builder} to create a
     *        request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws BadRequestException
     *         Raised when a malformed input has been provided to the API.
     * @throws ResourceNotFoundException
     *         Raised when the requested resource cannot be found.
     * @throws InternalServerException
     *         Raised when an unexpected error occurred during request processing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ElasticInferenceException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ElasticInferenceClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elastic-inference-2017-07-25/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, ElasticInferenceException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * Create a {@link ElasticInferenceClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static ElasticInferenceClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link ElasticInferenceClient}.
     */
    static ElasticInferenceClientBuilder builder() {
        return new DefaultElasticInferenceClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default ElasticInferenceServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
