/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticloadbalancingv2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a target group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TargetGroup implements SdkPojo, Serializable, ToCopyableBuilder<TargetGroup.Builder, TargetGroup> {
    private static final SdkField<String> TARGET_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetGroupArn").getter(getter(TargetGroup::targetGroupArn)).setter(setter(Builder::targetGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetGroupArn").build()).build();

    private static final SdkField<String> TARGET_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetGroupName").getter(getter(TargetGroup::targetGroupName)).setter(setter(Builder::targetGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetGroupName").build()).build();

    private static final SdkField<String> PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Protocol").getter(getter(TargetGroup::protocolAsString)).setter(setter(Builder::protocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocol").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(TargetGroup::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(TargetGroup::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<String> HEALTH_CHECK_PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HealthCheckProtocol").getter(getter(TargetGroup::healthCheckProtocolAsString))
            .setter(setter(Builder::healthCheckProtocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthCheckProtocol").build())
            .build();

    private static final SdkField<String> HEALTH_CHECK_PORT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HealthCheckPort").getter(getter(TargetGroup::healthCheckPort)).setter(setter(Builder::healthCheckPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthCheckPort").build()).build();

    private static final SdkField<Boolean> HEALTH_CHECK_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("HealthCheckEnabled").getter(getter(TargetGroup::healthCheckEnabled))
            .setter(setter(Builder::healthCheckEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthCheckEnabled").build())
            .build();

    private static final SdkField<Integer> HEALTH_CHECK_INTERVAL_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("HealthCheckIntervalSeconds")
            .getter(getter(TargetGroup::healthCheckIntervalSeconds))
            .setter(setter(Builder::healthCheckIntervalSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthCheckIntervalSeconds").build())
            .build();

    private static final SdkField<Integer> HEALTH_CHECK_TIMEOUT_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("HealthCheckTimeoutSeconds")
            .getter(getter(TargetGroup::healthCheckTimeoutSeconds)).setter(setter(Builder::healthCheckTimeoutSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthCheckTimeoutSeconds").build())
            .build();

    private static final SdkField<Integer> HEALTHY_THRESHOLD_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("HealthyThresholdCount").getter(getter(TargetGroup::healthyThresholdCount))
            .setter(setter(Builder::healthyThresholdCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthyThresholdCount").build())
            .build();

    private static final SdkField<Integer> UNHEALTHY_THRESHOLD_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("UnhealthyThresholdCount").getter(getter(TargetGroup::unhealthyThresholdCount))
            .setter(setter(Builder::unhealthyThresholdCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UnhealthyThresholdCount").build())
            .build();

    private static final SdkField<String> HEALTH_CHECK_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HealthCheckPath").getter(getter(TargetGroup::healthCheckPath)).setter(setter(Builder::healthCheckPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthCheckPath").build()).build();

    private static final SdkField<Matcher> MATCHER_FIELD = SdkField.<Matcher> builder(MarshallingType.SDK_POJO)
            .memberName("Matcher").getter(getter(TargetGroup::matcher)).setter(setter(Builder::matcher))
            .constructor(Matcher::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Matcher").build()).build();

    private static final SdkField<List<String>> LOAD_BALANCER_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LoadBalancerArns")
            .getter(getter(TargetGroup::loadBalancerArns))
            .setter(setter(Builder::loadBalancerArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoadBalancerArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TARGET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetType").getter(getter(TargetGroup::targetTypeAsString)).setter(setter(Builder::targetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetType").build()).build();

    private static final SdkField<String> PROTOCOL_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProtocolVersion").getter(getter(TargetGroup::protocolVersion)).setter(setter(Builder::protocolVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProtocolVersion").build()).build();

    private static final SdkField<String> IP_ADDRESS_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddressType").getter(getter(TargetGroup::ipAddressTypeAsString))
            .setter(setter(Builder::ipAddressType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddressType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TARGET_GROUP_ARN_FIELD,
            TARGET_GROUP_NAME_FIELD, PROTOCOL_FIELD, PORT_FIELD, VPC_ID_FIELD, HEALTH_CHECK_PROTOCOL_FIELD,
            HEALTH_CHECK_PORT_FIELD, HEALTH_CHECK_ENABLED_FIELD, HEALTH_CHECK_INTERVAL_SECONDS_FIELD,
            HEALTH_CHECK_TIMEOUT_SECONDS_FIELD, HEALTHY_THRESHOLD_COUNT_FIELD, UNHEALTHY_THRESHOLD_COUNT_FIELD,
            HEALTH_CHECK_PATH_FIELD, MATCHER_FIELD, LOAD_BALANCER_ARNS_FIELD, TARGET_TYPE_FIELD, PROTOCOL_VERSION_FIELD,
            IP_ADDRESS_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String targetGroupArn;

    private final String targetGroupName;

    private final String protocol;

    private final Integer port;

    private final String vpcId;

    private final String healthCheckProtocol;

    private final String healthCheckPort;

    private final Boolean healthCheckEnabled;

    private final Integer healthCheckIntervalSeconds;

    private final Integer healthCheckTimeoutSeconds;

    private final Integer healthyThresholdCount;

    private final Integer unhealthyThresholdCount;

    private final String healthCheckPath;

    private final Matcher matcher;

    private final List<String> loadBalancerArns;

    private final String targetType;

    private final String protocolVersion;

    private final String ipAddressType;

    private TargetGroup(BuilderImpl builder) {
        this.targetGroupArn = builder.targetGroupArn;
        this.targetGroupName = builder.targetGroupName;
        this.protocol = builder.protocol;
        this.port = builder.port;
        this.vpcId = builder.vpcId;
        this.healthCheckProtocol = builder.healthCheckProtocol;
        this.healthCheckPort = builder.healthCheckPort;
        this.healthCheckEnabled = builder.healthCheckEnabled;
        this.healthCheckIntervalSeconds = builder.healthCheckIntervalSeconds;
        this.healthCheckTimeoutSeconds = builder.healthCheckTimeoutSeconds;
        this.healthyThresholdCount = builder.healthyThresholdCount;
        this.unhealthyThresholdCount = builder.unhealthyThresholdCount;
        this.healthCheckPath = builder.healthCheckPath;
        this.matcher = builder.matcher;
        this.loadBalancerArns = builder.loadBalancerArns;
        this.targetType = builder.targetType;
        this.protocolVersion = builder.protocolVersion;
        this.ipAddressType = builder.ipAddressType;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the target group.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the target group.
     */
    public final String targetGroupArn() {
        return targetGroupArn;
    }

    /**
     * <p>
     * The name of the target group.
     * </p>
     * 
     * @return The name of the target group.
     */
    public final String targetGroupName() {
        return targetGroupName;
    }

    /**
     * <p>
     * The protocol to use for routing traffic to the targets.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link ProtocolEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol to use for routing traffic to the targets.
     * @see ProtocolEnum
     */
    public final ProtocolEnum protocol() {
        return ProtocolEnum.fromValue(protocol);
    }

    /**
     * <p>
     * The protocol to use for routing traffic to the targets.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link ProtocolEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol to use for routing traffic to the targets.
     * @see ProtocolEnum
     */
    public final String protocolAsString() {
        return protocol;
    }

    /**
     * <p>
     * The port on which the targets are listening. Not used if the target is a Lambda function.
     * </p>
     * 
     * @return The port on which the targets are listening. Not used if the target is a Lambda function.
     */
    public final Integer port() {
        return port;
    }

    /**
     * <p>
     * The ID of the VPC for the targets.
     * </p>
     * 
     * @return The ID of the VPC for the targets.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The protocol to use to connect with the target. The GENEVE, TLS, UDP, and TCP_UDP protocols are not supported for
     * health checks.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #healthCheckProtocol} will return {@link ProtocolEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by
     * the service is available from {@link #healthCheckProtocolAsString}.
     * </p>
     * 
     * @return The protocol to use to connect with the target. The GENEVE, TLS, UDP, and TCP_UDP protocols are not
     *         supported for health checks.
     * @see ProtocolEnum
     */
    public final ProtocolEnum healthCheckProtocol() {
        return ProtocolEnum.fromValue(healthCheckProtocol);
    }

    /**
     * <p>
     * The protocol to use to connect with the target. The GENEVE, TLS, UDP, and TCP_UDP protocols are not supported for
     * health checks.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #healthCheckProtocol} will return {@link ProtocolEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by
     * the service is available from {@link #healthCheckProtocolAsString}.
     * </p>
     * 
     * @return The protocol to use to connect with the target. The GENEVE, TLS, UDP, and TCP_UDP protocols are not
     *         supported for health checks.
     * @see ProtocolEnum
     */
    public final String healthCheckProtocolAsString() {
        return healthCheckProtocol;
    }

    /**
     * <p>
     * The port to use to connect with the target.
     * </p>
     * 
     * @return The port to use to connect with the target.
     */
    public final String healthCheckPort() {
        return healthCheckPort;
    }

    /**
     * <p>
     * Indicates whether health checks are enabled.
     * </p>
     * 
     * @return Indicates whether health checks are enabled.
     */
    public final Boolean healthCheckEnabled() {
        return healthCheckEnabled;
    }

    /**
     * <p>
     * The approximate amount of time, in seconds, between health checks of an individual target.
     * </p>
     * 
     * @return The approximate amount of time, in seconds, between health checks of an individual target.
     */
    public final Integer healthCheckIntervalSeconds() {
        return healthCheckIntervalSeconds;
    }

    /**
     * <p>
     * The amount of time, in seconds, during which no response means a failed health check.
     * </p>
     * 
     * @return The amount of time, in seconds, during which no response means a failed health check.
     */
    public final Integer healthCheckTimeoutSeconds() {
        return healthCheckTimeoutSeconds;
    }

    /**
     * <p>
     * The number of consecutive health checks successes required before considering an unhealthy target healthy.
     * </p>
     * 
     * @return The number of consecutive health checks successes required before considering an unhealthy target
     *         healthy.
     */
    public final Integer healthyThresholdCount() {
        return healthyThresholdCount;
    }

    /**
     * <p>
     * The number of consecutive health check failures required before considering the target unhealthy.
     * </p>
     * 
     * @return The number of consecutive health check failures required before considering the target unhealthy.
     */
    public final Integer unhealthyThresholdCount() {
        return unhealthyThresholdCount;
    }

    /**
     * <p>
     * The destination for health checks on the targets.
     * </p>
     * 
     * @return The destination for health checks on the targets.
     */
    public final String healthCheckPath() {
        return healthCheckPath;
    }

    /**
     * <p>
     * The HTTP or gRPC codes to use when checking for a successful response from a target.
     * </p>
     * 
     * @return The HTTP or gRPC codes to use when checking for a successful response from a target.
     */
    public final Matcher matcher() {
        return matcher;
    }

    /**
     * For responses, this returns true if the service returned a value for the LoadBalancerArns property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLoadBalancerArns() {
        return loadBalancerArns != null && !(loadBalancerArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Names (ARN) of the load balancers that route traffic to this target group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLoadBalancerArns} method.
     * </p>
     * 
     * @return The Amazon Resource Names (ARN) of the load balancers that route traffic to this target group.
     */
    public final List<String> loadBalancerArns() {
        return loadBalancerArns;
    }

    /**
     * <p>
     * The type of target that you must specify when registering targets with this target group. The possible values are
     * <code>instance</code> (register targets by instance ID), <code>ip</code> (register targets by IP address),
     * <code>lambda</code> (register a single Lambda function as a target), or <code>alb</code> (register a single
     * Application Load Balancer as a target).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetType} will
     * return {@link TargetTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #targetTypeAsString}.
     * </p>
     * 
     * @return The type of target that you must specify when registering targets with this target group. The possible
     *         values are <code>instance</code> (register targets by instance ID), <code>ip</code> (register targets by
     *         IP address), <code>lambda</code> (register a single Lambda function as a target), or <code>alb</code>
     *         (register a single Application Load Balancer as a target).
     * @see TargetTypeEnum
     */
    public final TargetTypeEnum targetType() {
        return TargetTypeEnum.fromValue(targetType);
    }

    /**
     * <p>
     * The type of target that you must specify when registering targets with this target group. The possible values are
     * <code>instance</code> (register targets by instance ID), <code>ip</code> (register targets by IP address),
     * <code>lambda</code> (register a single Lambda function as a target), or <code>alb</code> (register a single
     * Application Load Balancer as a target).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetType} will
     * return {@link TargetTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #targetTypeAsString}.
     * </p>
     * 
     * @return The type of target that you must specify when registering targets with this target group. The possible
     *         values are <code>instance</code> (register targets by instance ID), <code>ip</code> (register targets by
     *         IP address), <code>lambda</code> (register a single Lambda function as a target), or <code>alb</code>
     *         (register a single Application Load Balancer as a target).
     * @see TargetTypeEnum
     */
    public final String targetTypeAsString() {
        return targetType;
    }

    /**
     * <p>
     * [HTTP/HTTPS protocol] The protocol version. The possible values are <code>GRPC</code>, <code>HTTP1</code>, and
     * <code>HTTP2</code>.
     * </p>
     * 
     * @return [HTTP/HTTPS protocol] The protocol version. The possible values are <code>GRPC</code>, <code>HTTP1</code>
     *         , and <code>HTTP2</code>.
     */
    public final String protocolVersion() {
        return protocolVersion;
    }

    /**
     * <p>
     * The type of IP address used for this target group. The possible values are <code>ipv4</code> and
     * <code>ipv6</code>. This is an optional parameter. If not specified, the IP address type defaults to
     * <code>ipv4</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link TargetGroupIpAddressTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The type of IP address used for this target group. The possible values are <code>ipv4</code> and
     *         <code>ipv6</code>. This is an optional parameter. If not specified, the IP address type defaults to
     *         <code>ipv4</code>.
     * @see TargetGroupIpAddressTypeEnum
     */
    public final TargetGroupIpAddressTypeEnum ipAddressType() {
        return TargetGroupIpAddressTypeEnum.fromValue(ipAddressType);
    }

    /**
     * <p>
     * The type of IP address used for this target group. The possible values are <code>ipv4</code> and
     * <code>ipv6</code>. This is an optional parameter. If not specified, the IP address type defaults to
     * <code>ipv4</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link TargetGroupIpAddressTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The type of IP address used for this target group. The possible values are <code>ipv4</code> and
     *         <code>ipv6</code>. This is an optional parameter. If not specified, the IP address type defaults to
     *         <code>ipv4</code>.
     * @see TargetGroupIpAddressTypeEnum
     */
    public final String ipAddressTypeAsString() {
        return ipAddressType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(targetGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(targetGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(protocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheckProtocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheckPort());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheckEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheckIntervalSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheckTimeoutSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(healthyThresholdCount());
        hashCode = 31 * hashCode + Objects.hashCode(unhealthyThresholdCount());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheckPath());
        hashCode = 31 * hashCode + Objects.hashCode(matcher());
        hashCode = 31 * hashCode + Objects.hashCode(hasLoadBalancerArns() ? loadBalancerArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(targetTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(protocolVersion());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddressTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TargetGroup)) {
            return false;
        }
        TargetGroup other = (TargetGroup) obj;
        return Objects.equals(targetGroupArn(), other.targetGroupArn())
                && Objects.equals(targetGroupName(), other.targetGroupName())
                && Objects.equals(protocolAsString(), other.protocolAsString()) && Objects.equals(port(), other.port())
                && Objects.equals(vpcId(), other.vpcId())
                && Objects.equals(healthCheckProtocolAsString(), other.healthCheckProtocolAsString())
                && Objects.equals(healthCheckPort(), other.healthCheckPort())
                && Objects.equals(healthCheckEnabled(), other.healthCheckEnabled())
                && Objects.equals(healthCheckIntervalSeconds(), other.healthCheckIntervalSeconds())
                && Objects.equals(healthCheckTimeoutSeconds(), other.healthCheckTimeoutSeconds())
                && Objects.equals(healthyThresholdCount(), other.healthyThresholdCount())
                && Objects.equals(unhealthyThresholdCount(), other.unhealthyThresholdCount())
                && Objects.equals(healthCheckPath(), other.healthCheckPath()) && Objects.equals(matcher(), other.matcher())
                && hasLoadBalancerArns() == other.hasLoadBalancerArns()
                && Objects.equals(loadBalancerArns(), other.loadBalancerArns())
                && Objects.equals(targetTypeAsString(), other.targetTypeAsString())
                && Objects.equals(protocolVersion(), other.protocolVersion())
                && Objects.equals(ipAddressTypeAsString(), other.ipAddressTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TargetGroup").add("TargetGroupArn", targetGroupArn()).add("TargetGroupName", targetGroupName())
                .add("Protocol", protocolAsString()).add("Port", port()).add("VpcId", vpcId())
                .add("HealthCheckProtocol", healthCheckProtocolAsString()).add("HealthCheckPort", healthCheckPort())
                .add("HealthCheckEnabled", healthCheckEnabled()).add("HealthCheckIntervalSeconds", healthCheckIntervalSeconds())
                .add("HealthCheckTimeoutSeconds", healthCheckTimeoutSeconds())
                .add("HealthyThresholdCount", healthyThresholdCount()).add("UnhealthyThresholdCount", unhealthyThresholdCount())
                .add("HealthCheckPath", healthCheckPath()).add("Matcher", matcher())
                .add("LoadBalancerArns", hasLoadBalancerArns() ? loadBalancerArns() : null)
                .add("TargetType", targetTypeAsString()).add("ProtocolVersion", protocolVersion())
                .add("IpAddressType", ipAddressTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TargetGroupArn":
            return Optional.ofNullable(clazz.cast(targetGroupArn()));
        case "TargetGroupName":
            return Optional.ofNullable(clazz.cast(targetGroupName()));
        case "Protocol":
            return Optional.ofNullable(clazz.cast(protocolAsString()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "HealthCheckProtocol":
            return Optional.ofNullable(clazz.cast(healthCheckProtocolAsString()));
        case "HealthCheckPort":
            return Optional.ofNullable(clazz.cast(healthCheckPort()));
        case "HealthCheckEnabled":
            return Optional.ofNullable(clazz.cast(healthCheckEnabled()));
        case "HealthCheckIntervalSeconds":
            return Optional.ofNullable(clazz.cast(healthCheckIntervalSeconds()));
        case "HealthCheckTimeoutSeconds":
            return Optional.ofNullable(clazz.cast(healthCheckTimeoutSeconds()));
        case "HealthyThresholdCount":
            return Optional.ofNullable(clazz.cast(healthyThresholdCount()));
        case "UnhealthyThresholdCount":
            return Optional.ofNullable(clazz.cast(unhealthyThresholdCount()));
        case "HealthCheckPath":
            return Optional.ofNullable(clazz.cast(healthCheckPath()));
        case "Matcher":
            return Optional.ofNullable(clazz.cast(matcher()));
        case "LoadBalancerArns":
            return Optional.ofNullable(clazz.cast(loadBalancerArns()));
        case "TargetType":
            return Optional.ofNullable(clazz.cast(targetTypeAsString()));
        case "ProtocolVersion":
            return Optional.ofNullable(clazz.cast(protocolVersion()));
        case "IpAddressType":
            return Optional.ofNullable(clazz.cast(ipAddressTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TargetGroup, T> g) {
        return obj -> g.apply((TargetGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TargetGroup> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the target group.
         * </p>
         * 
         * @param targetGroupArn
         *        The Amazon Resource Name (ARN) of the target group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetGroupArn(String targetGroupArn);

        /**
         * <p>
         * The name of the target group.
         * </p>
         * 
         * @param targetGroupName
         *        The name of the target group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetGroupName(String targetGroupName);

        /**
         * <p>
         * The protocol to use for routing traffic to the targets.
         * </p>
         * 
         * @param protocol
         *        The protocol to use for routing traffic to the targets.
         * @see ProtocolEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtocolEnum
         */
        Builder protocol(String protocol);

        /**
         * <p>
         * The protocol to use for routing traffic to the targets.
         * </p>
         * 
         * @param protocol
         *        The protocol to use for routing traffic to the targets.
         * @see ProtocolEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtocolEnum
         */
        Builder protocol(ProtocolEnum protocol);

        /**
         * <p>
         * The port on which the targets are listening. Not used if the target is a Lambda function.
         * </p>
         * 
         * @param port
         *        The port on which the targets are listening. Not used if the target is a Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * The ID of the VPC for the targets.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC for the targets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The protocol to use to connect with the target. The GENEVE, TLS, UDP, and TCP_UDP protocols are not supported
         * for health checks.
         * </p>
         * 
         * @param healthCheckProtocol
         *        The protocol to use to connect with the target. The GENEVE, TLS, UDP, and TCP_UDP protocols are not
         *        supported for health checks.
         * @see ProtocolEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtocolEnum
         */
        Builder healthCheckProtocol(String healthCheckProtocol);

        /**
         * <p>
         * The protocol to use to connect with the target. The GENEVE, TLS, UDP, and TCP_UDP protocols are not supported
         * for health checks.
         * </p>
         * 
         * @param healthCheckProtocol
         *        The protocol to use to connect with the target. The GENEVE, TLS, UDP, and TCP_UDP protocols are not
         *        supported for health checks.
         * @see ProtocolEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtocolEnum
         */
        Builder healthCheckProtocol(ProtocolEnum healthCheckProtocol);

        /**
         * <p>
         * The port to use to connect with the target.
         * </p>
         * 
         * @param healthCheckPort
         *        The port to use to connect with the target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheckPort(String healthCheckPort);

        /**
         * <p>
         * Indicates whether health checks are enabled.
         * </p>
         * 
         * @param healthCheckEnabled
         *        Indicates whether health checks are enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheckEnabled(Boolean healthCheckEnabled);

        /**
         * <p>
         * The approximate amount of time, in seconds, between health checks of an individual target.
         * </p>
         * 
         * @param healthCheckIntervalSeconds
         *        The approximate amount of time, in seconds, between health checks of an individual target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheckIntervalSeconds(Integer healthCheckIntervalSeconds);

        /**
         * <p>
         * The amount of time, in seconds, during which no response means a failed health check.
         * </p>
         * 
         * @param healthCheckTimeoutSeconds
         *        The amount of time, in seconds, during which no response means a failed health check.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheckTimeoutSeconds(Integer healthCheckTimeoutSeconds);

        /**
         * <p>
         * The number of consecutive health checks successes required before considering an unhealthy target healthy.
         * </p>
         * 
         * @param healthyThresholdCount
         *        The number of consecutive health checks successes required before considering an unhealthy target
         *        healthy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthyThresholdCount(Integer healthyThresholdCount);

        /**
         * <p>
         * The number of consecutive health check failures required before considering the target unhealthy.
         * </p>
         * 
         * @param unhealthyThresholdCount
         *        The number of consecutive health check failures required before considering the target unhealthy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unhealthyThresholdCount(Integer unhealthyThresholdCount);

        /**
         * <p>
         * The destination for health checks on the targets.
         * </p>
         * 
         * @param healthCheckPath
         *        The destination for health checks on the targets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheckPath(String healthCheckPath);

        /**
         * <p>
         * The HTTP or gRPC codes to use when checking for a successful response from a target.
         * </p>
         * 
         * @param matcher
         *        The HTTP or gRPC codes to use when checking for a successful response from a target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matcher(Matcher matcher);

        /**
         * <p>
         * The HTTP or gRPC codes to use when checking for a successful response from a target.
         * </p>
         * This is a convenience that creates an instance of the {@link Matcher.Builder} avoiding the need to create one
         * manually via {@link Matcher#builder()}.
         *
         * When the {@link Consumer} completes, {@link Matcher.Builder#build()} is called immediately and its result is
         * passed to {@link #matcher(Matcher)}.
         * 
         * @param matcher
         *        a consumer that will call methods on {@link Matcher.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #matcher(Matcher)
         */
        default Builder matcher(Consumer<Matcher.Builder> matcher) {
            return matcher(Matcher.builder().applyMutation(matcher).build());
        }

        /**
         * <p>
         * The Amazon Resource Names (ARN) of the load balancers that route traffic to this target group.
         * </p>
         * 
         * @param loadBalancerArns
         *        The Amazon Resource Names (ARN) of the load balancers that route traffic to this target group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancerArns(Collection<String> loadBalancerArns);

        /**
         * <p>
         * The Amazon Resource Names (ARN) of the load balancers that route traffic to this target group.
         * </p>
         * 
         * @param loadBalancerArns
         *        The Amazon Resource Names (ARN) of the load balancers that route traffic to this target group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancerArns(String... loadBalancerArns);

        /**
         * <p>
         * The type of target that you must specify when registering targets with this target group. The possible values
         * are <code>instance</code> (register targets by instance ID), <code>ip</code> (register targets by IP
         * address), <code>lambda</code> (register a single Lambda function as a target), or <code>alb</code> (register
         * a single Application Load Balancer as a target).
         * </p>
         * 
         * @param targetType
         *        The type of target that you must specify when registering targets with this target group. The possible
         *        values are <code>instance</code> (register targets by instance ID), <code>ip</code> (register targets
         *        by IP address), <code>lambda</code> (register a single Lambda function as a target), or
         *        <code>alb</code> (register a single Application Load Balancer as a target).
         * @see TargetTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetTypeEnum
         */
        Builder targetType(String targetType);

        /**
         * <p>
         * The type of target that you must specify when registering targets with this target group. The possible values
         * are <code>instance</code> (register targets by instance ID), <code>ip</code> (register targets by IP
         * address), <code>lambda</code> (register a single Lambda function as a target), or <code>alb</code> (register
         * a single Application Load Balancer as a target).
         * </p>
         * 
         * @param targetType
         *        The type of target that you must specify when registering targets with this target group. The possible
         *        values are <code>instance</code> (register targets by instance ID), <code>ip</code> (register targets
         *        by IP address), <code>lambda</code> (register a single Lambda function as a target), or
         *        <code>alb</code> (register a single Application Load Balancer as a target).
         * @see TargetTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetTypeEnum
         */
        Builder targetType(TargetTypeEnum targetType);

        /**
         * <p>
         * [HTTP/HTTPS protocol] The protocol version. The possible values are <code>GRPC</code>, <code>HTTP1</code>,
         * and <code>HTTP2</code>.
         * </p>
         * 
         * @param protocolVersion
         *        [HTTP/HTTPS protocol] The protocol version. The possible values are <code>GRPC</code>,
         *        <code>HTTP1</code>, and <code>HTTP2</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocolVersion(String protocolVersion);

        /**
         * <p>
         * The type of IP address used for this target group. The possible values are <code>ipv4</code> and
         * <code>ipv6</code>. This is an optional parameter. If not specified, the IP address type defaults to
         * <code>ipv4</code>.
         * </p>
         * 
         * @param ipAddressType
         *        The type of IP address used for this target group. The possible values are <code>ipv4</code> and
         *        <code>ipv6</code>. This is an optional parameter. If not specified, the IP address type defaults to
         *        <code>ipv4</code>.
         * @see TargetGroupIpAddressTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetGroupIpAddressTypeEnum
         */
        Builder ipAddressType(String ipAddressType);

        /**
         * <p>
         * The type of IP address used for this target group. The possible values are <code>ipv4</code> and
         * <code>ipv6</code>. This is an optional parameter. If not specified, the IP address type defaults to
         * <code>ipv4</code>.
         * </p>
         * 
         * @param ipAddressType
         *        The type of IP address used for this target group. The possible values are <code>ipv4</code> and
         *        <code>ipv6</code>. This is an optional parameter. If not specified, the IP address type defaults to
         *        <code>ipv4</code>.
         * @see TargetGroupIpAddressTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetGroupIpAddressTypeEnum
         */
        Builder ipAddressType(TargetGroupIpAddressTypeEnum ipAddressType);
    }

    static final class BuilderImpl implements Builder {
        private String targetGroupArn;

        private String targetGroupName;

        private String protocol;

        private Integer port;

        private String vpcId;

        private String healthCheckProtocol;

        private String healthCheckPort;

        private Boolean healthCheckEnabled;

        private Integer healthCheckIntervalSeconds;

        private Integer healthCheckTimeoutSeconds;

        private Integer healthyThresholdCount;

        private Integer unhealthyThresholdCount;

        private String healthCheckPath;

        private Matcher matcher;

        private List<String> loadBalancerArns = DefaultSdkAutoConstructList.getInstance();

        private String targetType;

        private String protocolVersion;

        private String ipAddressType;

        private BuilderImpl() {
        }

        private BuilderImpl(TargetGroup model) {
            targetGroupArn(model.targetGroupArn);
            targetGroupName(model.targetGroupName);
            protocol(model.protocol);
            port(model.port);
            vpcId(model.vpcId);
            healthCheckProtocol(model.healthCheckProtocol);
            healthCheckPort(model.healthCheckPort);
            healthCheckEnabled(model.healthCheckEnabled);
            healthCheckIntervalSeconds(model.healthCheckIntervalSeconds);
            healthCheckTimeoutSeconds(model.healthCheckTimeoutSeconds);
            healthyThresholdCount(model.healthyThresholdCount);
            unhealthyThresholdCount(model.unhealthyThresholdCount);
            healthCheckPath(model.healthCheckPath);
            matcher(model.matcher);
            loadBalancerArns(model.loadBalancerArns);
            targetType(model.targetType);
            protocolVersion(model.protocolVersion);
            ipAddressType(model.ipAddressType);
        }

        public final String getTargetGroupArn() {
            return targetGroupArn;
        }

        public final void setTargetGroupArn(String targetGroupArn) {
            this.targetGroupArn = targetGroupArn;
        }

        @Override
        @Transient
        public final Builder targetGroupArn(String targetGroupArn) {
            this.targetGroupArn = targetGroupArn;
            return this;
        }

        public final String getTargetGroupName() {
            return targetGroupName;
        }

        public final void setTargetGroupName(String targetGroupName) {
            this.targetGroupName = targetGroupName;
        }

        @Override
        @Transient
        public final Builder targetGroupName(String targetGroupName) {
            this.targetGroupName = targetGroupName;
            return this;
        }

        public final String getProtocol() {
            return protocol;
        }

        public final void setProtocol(String protocol) {
            this.protocol = protocol;
        }

        @Override
        @Transient
        public final Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        @Override
        @Transient
        public final Builder protocol(ProtocolEnum protocol) {
            this.protocol(protocol == null ? null : protocol.toString());
            return this;
        }

        public final Integer getPort() {
            return port;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        @Override
        @Transient
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        @Transient
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final String getHealthCheckProtocol() {
            return healthCheckProtocol;
        }

        public final void setHealthCheckProtocol(String healthCheckProtocol) {
            this.healthCheckProtocol = healthCheckProtocol;
        }

        @Override
        @Transient
        public final Builder healthCheckProtocol(String healthCheckProtocol) {
            this.healthCheckProtocol = healthCheckProtocol;
            return this;
        }

        @Override
        @Transient
        public final Builder healthCheckProtocol(ProtocolEnum healthCheckProtocol) {
            this.healthCheckProtocol(healthCheckProtocol == null ? null : healthCheckProtocol.toString());
            return this;
        }

        public final String getHealthCheckPort() {
            return healthCheckPort;
        }

        public final void setHealthCheckPort(String healthCheckPort) {
            this.healthCheckPort = healthCheckPort;
        }

        @Override
        @Transient
        public final Builder healthCheckPort(String healthCheckPort) {
            this.healthCheckPort = healthCheckPort;
            return this;
        }

        public final Boolean getHealthCheckEnabled() {
            return healthCheckEnabled;
        }

        public final void setHealthCheckEnabled(Boolean healthCheckEnabled) {
            this.healthCheckEnabled = healthCheckEnabled;
        }

        @Override
        @Transient
        public final Builder healthCheckEnabled(Boolean healthCheckEnabled) {
            this.healthCheckEnabled = healthCheckEnabled;
            return this;
        }

        public final Integer getHealthCheckIntervalSeconds() {
            return healthCheckIntervalSeconds;
        }

        public final void setHealthCheckIntervalSeconds(Integer healthCheckIntervalSeconds) {
            this.healthCheckIntervalSeconds = healthCheckIntervalSeconds;
        }

        @Override
        @Transient
        public final Builder healthCheckIntervalSeconds(Integer healthCheckIntervalSeconds) {
            this.healthCheckIntervalSeconds = healthCheckIntervalSeconds;
            return this;
        }

        public final Integer getHealthCheckTimeoutSeconds() {
            return healthCheckTimeoutSeconds;
        }

        public final void setHealthCheckTimeoutSeconds(Integer healthCheckTimeoutSeconds) {
            this.healthCheckTimeoutSeconds = healthCheckTimeoutSeconds;
        }

        @Override
        @Transient
        public final Builder healthCheckTimeoutSeconds(Integer healthCheckTimeoutSeconds) {
            this.healthCheckTimeoutSeconds = healthCheckTimeoutSeconds;
            return this;
        }

        public final Integer getHealthyThresholdCount() {
            return healthyThresholdCount;
        }

        public final void setHealthyThresholdCount(Integer healthyThresholdCount) {
            this.healthyThresholdCount = healthyThresholdCount;
        }

        @Override
        @Transient
        public final Builder healthyThresholdCount(Integer healthyThresholdCount) {
            this.healthyThresholdCount = healthyThresholdCount;
            return this;
        }

        public final Integer getUnhealthyThresholdCount() {
            return unhealthyThresholdCount;
        }

        public final void setUnhealthyThresholdCount(Integer unhealthyThresholdCount) {
            this.unhealthyThresholdCount = unhealthyThresholdCount;
        }

        @Override
        @Transient
        public final Builder unhealthyThresholdCount(Integer unhealthyThresholdCount) {
            this.unhealthyThresholdCount = unhealthyThresholdCount;
            return this;
        }

        public final String getHealthCheckPath() {
            return healthCheckPath;
        }

        public final void setHealthCheckPath(String healthCheckPath) {
            this.healthCheckPath = healthCheckPath;
        }

        @Override
        @Transient
        public final Builder healthCheckPath(String healthCheckPath) {
            this.healthCheckPath = healthCheckPath;
            return this;
        }

        public final Matcher.Builder getMatcher() {
            return matcher != null ? matcher.toBuilder() : null;
        }

        public final void setMatcher(Matcher.BuilderImpl matcher) {
            this.matcher = matcher != null ? matcher.build() : null;
        }

        @Override
        @Transient
        public final Builder matcher(Matcher matcher) {
            this.matcher = matcher;
            return this;
        }

        public final Collection<String> getLoadBalancerArns() {
            if (loadBalancerArns instanceof SdkAutoConstructList) {
                return null;
            }
            return loadBalancerArns;
        }

        public final void setLoadBalancerArns(Collection<String> loadBalancerArns) {
            this.loadBalancerArns = LoadBalancerArnsCopier.copy(loadBalancerArns);
        }

        @Override
        @Transient
        public final Builder loadBalancerArns(Collection<String> loadBalancerArns) {
            this.loadBalancerArns = LoadBalancerArnsCopier.copy(loadBalancerArns);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder loadBalancerArns(String... loadBalancerArns) {
            loadBalancerArns(Arrays.asList(loadBalancerArns));
            return this;
        }

        public final String getTargetType() {
            return targetType;
        }

        public final void setTargetType(String targetType) {
            this.targetType = targetType;
        }

        @Override
        @Transient
        public final Builder targetType(String targetType) {
            this.targetType = targetType;
            return this;
        }

        @Override
        @Transient
        public final Builder targetType(TargetTypeEnum targetType) {
            this.targetType(targetType == null ? null : targetType.toString());
            return this;
        }

        public final String getProtocolVersion() {
            return protocolVersion;
        }

        public final void setProtocolVersion(String protocolVersion) {
            this.protocolVersion = protocolVersion;
        }

        @Override
        @Transient
        public final Builder protocolVersion(String protocolVersion) {
            this.protocolVersion = protocolVersion;
            return this;
        }

        public final String getIpAddressType() {
            return ipAddressType;
        }

        public final void setIpAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
        }

        @Override
        @Transient
        public final Builder ipAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
            return this;
        }

        @Override
        @Transient
        public final Builder ipAddressType(TargetGroupIpAddressTypeEnum ipAddressType) {
            this.ipAddressType(ipAddressType == null ? null : ipAddressType.toString());
            return this;
        }

        @Override
        public TargetGroup build() {
            return new TargetGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
