/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticloadbalancingv2.waiters;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.elasticloadbalancingv2.ElasticLoadBalancingV2Client;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.DescribeLoadBalancersRequest;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.DescribeLoadBalancersResponse;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.DescribeTargetHealthRequest;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.DescribeTargetHealthResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
public interface ElasticLoadBalancingV2Waiter extends SdkAutoCloseable {
    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeLoadBalancers} API until the desired condition
     * {@code LoadBalancerAvailable} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeLoadBalancersRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancerAvailable(
            DescribeLoadBalancersRequest describeLoadBalancersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeLoadBalancers} API until the desired condition
     * {@code LoadBalancerAvailable} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeLoadBalancersRequest#builder()}
     *
     * @param describeLoadBalancersRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancerAvailable(
            Consumer<DescribeLoadBalancersRequest.Builder> describeLoadBalancersRequest) {
        return waitUntilLoadBalancerAvailable(DescribeLoadBalancersRequest.builder().applyMutation(describeLoadBalancersRequest)
                .build());
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeLoadBalancers} API until the desired condition
     * {@code LoadBalancerAvailable} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeLoadBalancersRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancerAvailable(
            DescribeLoadBalancersRequest describeLoadBalancersRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeLoadBalancers} API until the desired condition
     * {@code LoadBalancerAvailable} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeLoadBalancersRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancerAvailable(
            Consumer<DescribeLoadBalancersRequest.Builder> describeLoadBalancersRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilLoadBalancerAvailable(DescribeLoadBalancersRequest.builder().applyMutation(describeLoadBalancersRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeLoadBalancers} API until the desired condition
     * {@code LoadBalancerExists} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeLoadBalancersRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancerExists(
            DescribeLoadBalancersRequest describeLoadBalancersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeLoadBalancers} API until the desired condition
     * {@code LoadBalancerExists} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeLoadBalancersRequest#builder()}
     *
     * @param describeLoadBalancersRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancerExists(
            Consumer<DescribeLoadBalancersRequest.Builder> describeLoadBalancersRequest) {
        return waitUntilLoadBalancerExists(DescribeLoadBalancersRequest.builder().applyMutation(describeLoadBalancersRequest)
                .build());
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeLoadBalancers} API until the desired condition
     * {@code LoadBalancerExists} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeLoadBalancersRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancerExists(
            DescribeLoadBalancersRequest describeLoadBalancersRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeLoadBalancers} API until the desired condition
     * {@code LoadBalancerExists} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeLoadBalancersRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancerExists(
            Consumer<DescribeLoadBalancersRequest.Builder> describeLoadBalancersRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilLoadBalancerExists(DescribeLoadBalancersRequest.builder().applyMutation(describeLoadBalancersRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeLoadBalancers} API until the desired condition
     * {@code LoadBalancersDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeLoadBalancersRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancersDeleted(
            DescribeLoadBalancersRequest describeLoadBalancersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeLoadBalancers} API until the desired condition
     * {@code LoadBalancersDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeLoadBalancersRequest#builder()}
     *
     * @param describeLoadBalancersRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancersDeleted(
            Consumer<DescribeLoadBalancersRequest.Builder> describeLoadBalancersRequest) {
        return waitUntilLoadBalancersDeleted(DescribeLoadBalancersRequest.builder().applyMutation(describeLoadBalancersRequest)
                .build());
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeLoadBalancers} API until the desired condition
     * {@code LoadBalancersDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeLoadBalancersRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancersDeleted(
            DescribeLoadBalancersRequest describeLoadBalancersRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeLoadBalancers} API until the desired condition
     * {@code LoadBalancersDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeLoadBalancersRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeLoadBalancersResponse> waitUntilLoadBalancersDeleted(
            Consumer<DescribeLoadBalancersRequest.Builder> describeLoadBalancersRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilLoadBalancersDeleted(DescribeLoadBalancersRequest.builder().applyMutation(describeLoadBalancersRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeTargetHealth} API until the desired condition
     * {@code TargetDeregistered} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeTargetHealthRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeTargetHealthResponse> waitUntilTargetDeregistered(
            DescribeTargetHealthRequest describeTargetHealthRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeTargetHealth} API until the desired condition
     * {@code TargetDeregistered} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeTargetHealthRequest#builder()}
     *
     * @param describeTargetHealthRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeTargetHealthResponse> waitUntilTargetDeregistered(
            Consumer<DescribeTargetHealthRequest.Builder> describeTargetHealthRequest) {
        return waitUntilTargetDeregistered(DescribeTargetHealthRequest.builder().applyMutation(describeTargetHealthRequest)
                .build());
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeTargetHealth} API until the desired condition
     * {@code TargetDeregistered} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeTargetHealthRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeTargetHealthResponse> waitUntilTargetDeregistered(
            DescribeTargetHealthRequest describeTargetHealthRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeTargetHealth} API until the desired condition
     * {@code TargetDeregistered} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeTargetHealthRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeTargetHealthResponse> waitUntilTargetDeregistered(
            Consumer<DescribeTargetHealthRequest.Builder> describeTargetHealthRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilTargetDeregistered(DescribeTargetHealthRequest.builder().applyMutation(describeTargetHealthRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeTargetHealth} API until the desired condition
     * {@code TargetInService} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeTargetHealthRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeTargetHealthResponse> waitUntilTargetInService(
            DescribeTargetHealthRequest describeTargetHealthRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeTargetHealth} API until the desired condition
     * {@code TargetInService} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeTargetHealthRequest#builder()}
     *
     * @param describeTargetHealthRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeTargetHealthResponse> waitUntilTargetInService(
            Consumer<DescribeTargetHealthRequest.Builder> describeTargetHealthRequest) {
        return waitUntilTargetInService(DescribeTargetHealthRequest.builder().applyMutation(describeTargetHealthRequest).build());
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeTargetHealth} API until the desired condition
     * {@code TargetInService} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeTargetHealthRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeTargetHealthResponse> waitUntilTargetInService(
            DescribeTargetHealthRequest describeTargetHealthRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElasticLoadBalancingV2Client#describeTargetHealth} API until the desired condition
     * {@code TargetInService} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeTargetHealthRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeTargetHealthResponse> waitUntilTargetInService(
            Consumer<DescribeTargetHealthRequest.Builder> describeTargetHealthRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilTargetInService(DescribeTargetHealthRequest.builder().applyMutation(describeTargetHealthRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link ElasticLoadBalancingV2Waiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultElasticLoadBalancingV2Waiter.builder();
    }

    /**
     * Create an instance of {@link ElasticLoadBalancingV2Waiter} with the default configuration.
     * <p>
     * <b>A default {@link ElasticLoadBalancingV2Client} will be created to poll resources. It is recommended to share a
     * single instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link ElasticLoadBalancingV2Waiter}
     */
    static ElasticLoadBalancingV2Waiter create() {
        return DefaultElasticLoadBalancingV2Waiter.builder().build();
    }

    interface Builder {
        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link ElasticLoadBalancingV2Client} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(ElasticLoadBalancingV2Client client);

        /**
         * Builds an instance of {@link ElasticLoadBalancingV2Waiter} based on the configurations supplied to this
         * builder
         *
         * @return An initialized {@link ElasticLoadBalancingV2Waiter}
         */
        ElasticLoadBalancingV2Waiter build();
    }
}
