/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticloadbalancingv2.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.services.elasticloadbalancingv2.ElasticLoadBalancingV2AsyncClient;
import software.amazon.awssdk.services.elasticloadbalancingv2.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.DescribeLoadBalancersRequest;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.DescribeLoadBalancersResponse;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.DescribeTargetHealthRequest;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.DescribeTargetHealthResponse;
import software.amazon.awssdk.services.elasticloadbalancingv2.model.ElasticLoadBalancingV2Request;
import software.amazon.awssdk.services.elasticloadbalancingv2.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultElasticLoadBalancingV2AsyncWaiter implements ElasticLoadBalancingV2AsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final ElasticLoadBalancingV2AsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeLoadBalancersResponse> loadBalancerExistsWaiter;

    private final AsyncWaiter<DescribeLoadBalancersResponse> loadBalancerAvailableWaiter;

    private final AsyncWaiter<DescribeLoadBalancersResponse> loadBalancersDeletedWaiter;

    private final AsyncWaiter<DescribeTargetHealthResponse> targetInServiceWaiter;

    private final AsyncWaiter<DescribeTargetHealthResponse> targetDeregisteredWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultElasticLoadBalancingV2AsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = ElasticLoadBalancingV2AsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.loadBalancerExistsWaiter = AsyncWaiter.builder(DescribeLoadBalancersResponse.class)
                .acceptors(loadBalancerExistsWaiterAcceptors())
                .overrideConfiguration(loadBalancerExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.loadBalancerAvailableWaiter = AsyncWaiter.builder(DescribeLoadBalancersResponse.class)
                .acceptors(loadBalancerAvailableWaiterAcceptors())
                .overrideConfiguration(loadBalancerAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.loadBalancersDeletedWaiter = AsyncWaiter.builder(DescribeLoadBalancersResponse.class)
                .acceptors(loadBalancersDeletedWaiterAcceptors())
                .overrideConfiguration(loadBalancersDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.targetInServiceWaiter = AsyncWaiter.builder(DescribeTargetHealthResponse.class)
                .acceptors(targetInServiceWaiterAcceptors())
                .overrideConfiguration(targetInServiceWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.targetDeregisteredWaiter = AsyncWaiter.builder(DescribeTargetHealthResponse.class)
                .acceptors(targetDeregisteredWaiterAcceptors())
                .overrideConfiguration(targetDeregisteredWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeLoadBalancersResponse>> waitUntilLoadBalancerAvailable(
            DescribeLoadBalancersRequest describeLoadBalancersRequest) {
        return loadBalancerAvailableWaiter.runAsync(() -> client
                .describeLoadBalancers(applyWaitersUserAgent(describeLoadBalancersRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeLoadBalancersResponse>> waitUntilLoadBalancerAvailable(
            DescribeLoadBalancersRequest describeLoadBalancersRequest, WaiterOverrideConfiguration overrideConfig) {
        return loadBalancerAvailableWaiter.runAsync(
                () -> client.describeLoadBalancers(applyWaitersUserAgent(describeLoadBalancersRequest)),
                loadBalancerAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeLoadBalancersResponse>> waitUntilLoadBalancerExists(
            DescribeLoadBalancersRequest describeLoadBalancersRequest) {
        return loadBalancerExistsWaiter.runAsync(() -> client
                .describeLoadBalancers(applyWaitersUserAgent(describeLoadBalancersRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeLoadBalancersResponse>> waitUntilLoadBalancerExists(
            DescribeLoadBalancersRequest describeLoadBalancersRequest, WaiterOverrideConfiguration overrideConfig) {
        return loadBalancerExistsWaiter.runAsync(
                () -> client.describeLoadBalancers(applyWaitersUserAgent(describeLoadBalancersRequest)),
                loadBalancerExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeLoadBalancersResponse>> waitUntilLoadBalancersDeleted(
            DescribeLoadBalancersRequest describeLoadBalancersRequest) {
        return loadBalancersDeletedWaiter.runAsync(() -> client
                .describeLoadBalancers(applyWaitersUserAgent(describeLoadBalancersRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeLoadBalancersResponse>> waitUntilLoadBalancersDeleted(
            DescribeLoadBalancersRequest describeLoadBalancersRequest, WaiterOverrideConfiguration overrideConfig) {
        return loadBalancersDeletedWaiter.runAsync(
                () -> client.describeLoadBalancers(applyWaitersUserAgent(describeLoadBalancersRequest)),
                loadBalancersDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTargetHealthResponse>> waitUntilTargetDeregistered(
            DescribeTargetHealthRequest describeTargetHealthRequest) {
        return targetDeregisteredWaiter.runAsync(() -> client
                .describeTargetHealth(applyWaitersUserAgent(describeTargetHealthRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTargetHealthResponse>> waitUntilTargetDeregistered(
            DescribeTargetHealthRequest describeTargetHealthRequest, WaiterOverrideConfiguration overrideConfig) {
        return targetDeregisteredWaiter.runAsync(
                () -> client.describeTargetHealth(applyWaitersUserAgent(describeTargetHealthRequest)),
                targetDeregisteredWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTargetHealthResponse>> waitUntilTargetInService(
            DescribeTargetHealthRequest describeTargetHealthRequest) {
        return targetInServiceWaiter.runAsync(() -> client
                .describeTargetHealth(applyWaitersUserAgent(describeTargetHealthRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTargetHealthResponse>> waitUntilTargetInService(
            DescribeTargetHealthRequest describeTargetHealthRequest, WaiterOverrideConfiguration overrideConfig) {
        return targetInServiceWaiter.runAsync(
                () -> client.describeTargetHealth(applyWaitersUserAgent(describeTargetHealthRequest)),
                targetInServiceWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeLoadBalancersResponse>> loadBalancerExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeLoadBalancersResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "LoadBalancerNotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeLoadBalancersResponse>> loadBalancerAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeLoadBalancersResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("LoadBalancers").flatten().field("State").field("Code").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "active"));
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("LoadBalancers").flatten().field("State").field("Code").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "provisioning"));
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "LoadBalancerNotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeLoadBalancersResponse>> loadBalancersDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeLoadBalancersResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("LoadBalancers").flatten().field("State").field("Code").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "active"));
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "LoadBalancerNotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTargetHealthResponse>> targetInServiceWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTargetHealthResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("TargetHealthDescriptions").flatten().field("TargetHealth").field("State")
                    .values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "healthy"));
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "InvalidInstance")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTargetHealthResponse>> targetDeregisteredWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTargetHealthResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "InvalidTarget")));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("TargetHealthDescriptions").flatten().field("TargetHealth").field("State")
                    .values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "unused"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration loadBalancerExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration loadBalancerAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration loadBalancersDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration targetInServiceWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration targetDeregisteredWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static ElasticLoadBalancingV2AsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends ElasticLoadBalancingV2Request> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements ElasticLoadBalancingV2AsyncWaiter.Builder {
        private ElasticLoadBalancingV2AsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public ElasticLoadBalancingV2AsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public ElasticLoadBalancingV2AsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public ElasticLoadBalancingV2AsyncWaiter.Builder client(ElasticLoadBalancingV2AsyncClient client) {
            this.client = client;
            return this;
        }

        public ElasticLoadBalancingV2AsyncWaiter build() {
            return new DefaultElasticLoadBalancingV2AsyncWaiter(this);
        }
    }
}
