/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A description of the Amazon EC2 instance on which the cluster (job flow) runs. A valid JobFlowInstancesConfig must
 * contain either InstanceGroups or InstanceFleets. They cannot be used together. You may also have MasterInstanceType,
 * SlaveInstanceType, and InstanceCount (all three must be present), but we don't recommend this configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobFlowInstancesConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<JobFlowInstancesConfig.Builder, JobFlowInstancesConfig> {
    private static final SdkField<String> MASTER_INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MasterInstanceType").getter(getter(JobFlowInstancesConfig::masterInstanceType))
            .setter(setter(Builder::masterInstanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterInstanceType").build())
            .build();

    private static final SdkField<String> SLAVE_INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SlaveInstanceType").getter(getter(JobFlowInstancesConfig::slaveInstanceType))
            .setter(setter(Builder::slaveInstanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SlaveInstanceType").build()).build();

    private static final SdkField<Integer> INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstanceCount").getter(getter(JobFlowInstancesConfig::instanceCount))
            .setter(setter(Builder::instanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceCount").build()).build();

    private static final SdkField<List<InstanceGroupConfig>> INSTANCE_GROUPS_FIELD = SdkField
            .<List<InstanceGroupConfig>> builder(MarshallingType.LIST)
            .memberName("InstanceGroups")
            .getter(getter(JobFlowInstancesConfig::instanceGroups))
            .setter(setter(Builder::instanceGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InstanceGroupConfig> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceGroupConfig::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<InstanceFleetConfig>> INSTANCE_FLEETS_FIELD = SdkField
            .<List<InstanceFleetConfig>> builder(MarshallingType.LIST)
            .memberName("InstanceFleets")
            .getter(getter(JobFlowInstancesConfig::instanceFleets))
            .setter(setter(Builder::instanceFleets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceFleets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InstanceFleetConfig> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceFleetConfig::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> EC2_KEY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Ec2KeyName").getter(getter(JobFlowInstancesConfig::ec2KeyName)).setter(setter(Builder::ec2KeyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ec2KeyName").build()).build();

    private static final SdkField<PlacementType> PLACEMENT_FIELD = SdkField.<PlacementType> builder(MarshallingType.SDK_POJO)
            .memberName("Placement").getter(getter(JobFlowInstancesConfig::placement)).setter(setter(Builder::placement))
            .constructor(PlacementType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Placement").build()).build();

    private static final SdkField<Boolean> KEEP_JOB_FLOW_ALIVE_WHEN_NO_STEPS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("KeepJobFlowAliveWhenNoSteps")
            .getter(getter(JobFlowInstancesConfig::keepJobFlowAliveWhenNoSteps))
            .setter(setter(Builder::keepJobFlowAliveWhenNoSteps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeepJobFlowAliveWhenNoSteps")
                    .build()).build();

    private static final SdkField<Boolean> TERMINATION_PROTECTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("TerminationProtected").getter(getter(JobFlowInstancesConfig::terminationProtected))
            .setter(setter(Builder::terminationProtected))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TerminationProtected").build())
            .build();

    private static final SdkField<String> HADOOP_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HadoopVersion").getter(getter(JobFlowInstancesConfig::hadoopVersion))
            .setter(setter(Builder::hadoopVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HadoopVersion").build()).build();

    private static final SdkField<String> EC2_SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Ec2SubnetId").getter(getter(JobFlowInstancesConfig::ec2SubnetId)).setter(setter(Builder::ec2SubnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ec2SubnetId").build()).build();

    private static final SdkField<List<String>> EC2_SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Ec2SubnetIds")
            .getter(getter(JobFlowInstancesConfig::ec2SubnetIds))
            .setter(setter(Builder::ec2SubnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ec2SubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> EMR_MANAGED_MASTER_SECURITY_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EmrManagedMasterSecurityGroup")
            .getter(getter(JobFlowInstancesConfig::emrManagedMasterSecurityGroup))
            .setter(setter(Builder::emrManagedMasterSecurityGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EmrManagedMasterSecurityGroup")
                    .build()).build();

    private static final SdkField<String> EMR_MANAGED_SLAVE_SECURITY_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EmrManagedSlaveSecurityGroup")
            .getter(getter(JobFlowInstancesConfig::emrManagedSlaveSecurityGroup))
            .setter(setter(Builder::emrManagedSlaveSecurityGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EmrManagedSlaveSecurityGroup")
                    .build()).build();

    private static final SdkField<String> SERVICE_ACCESS_SECURITY_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ServiceAccessSecurityGroup")
            .getter(getter(JobFlowInstancesConfig::serviceAccessSecurityGroup))
            .setter(setter(Builder::serviceAccessSecurityGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceAccessSecurityGroup").build())
            .build();

    private static final SdkField<List<String>> ADDITIONAL_MASTER_SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AdditionalMasterSecurityGroups")
            .getter(getter(JobFlowInstancesConfig::additionalMasterSecurityGroups))
            .setter(setter(Builder::additionalMasterSecurityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalMasterSecurityGroups")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ADDITIONAL_SLAVE_SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AdditionalSlaveSecurityGroups")
            .getter(getter(JobFlowInstancesConfig::additionalSlaveSecurityGroups))
            .setter(setter(Builder::additionalSlaveSecurityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalSlaveSecurityGroups")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MASTER_INSTANCE_TYPE_FIELD,
            SLAVE_INSTANCE_TYPE_FIELD, INSTANCE_COUNT_FIELD, INSTANCE_GROUPS_FIELD, INSTANCE_FLEETS_FIELD, EC2_KEY_NAME_FIELD,
            PLACEMENT_FIELD, KEEP_JOB_FLOW_ALIVE_WHEN_NO_STEPS_FIELD, TERMINATION_PROTECTED_FIELD, HADOOP_VERSION_FIELD,
            EC2_SUBNET_ID_FIELD, EC2_SUBNET_IDS_FIELD, EMR_MANAGED_MASTER_SECURITY_GROUP_FIELD,
            EMR_MANAGED_SLAVE_SECURITY_GROUP_FIELD, SERVICE_ACCESS_SECURITY_GROUP_FIELD, ADDITIONAL_MASTER_SECURITY_GROUPS_FIELD,
            ADDITIONAL_SLAVE_SECURITY_GROUPS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String masterInstanceType;

    private final String slaveInstanceType;

    private final Integer instanceCount;

    private final List<InstanceGroupConfig> instanceGroups;

    private final List<InstanceFleetConfig> instanceFleets;

    private final String ec2KeyName;

    private final PlacementType placement;

    private final Boolean keepJobFlowAliveWhenNoSteps;

    private final Boolean terminationProtected;

    private final String hadoopVersion;

    private final String ec2SubnetId;

    private final List<String> ec2SubnetIds;

    private final String emrManagedMasterSecurityGroup;

    private final String emrManagedSlaveSecurityGroup;

    private final String serviceAccessSecurityGroup;

    private final List<String> additionalMasterSecurityGroups;

    private final List<String> additionalSlaveSecurityGroups;

    private JobFlowInstancesConfig(BuilderImpl builder) {
        this.masterInstanceType = builder.masterInstanceType;
        this.slaveInstanceType = builder.slaveInstanceType;
        this.instanceCount = builder.instanceCount;
        this.instanceGroups = builder.instanceGroups;
        this.instanceFleets = builder.instanceFleets;
        this.ec2KeyName = builder.ec2KeyName;
        this.placement = builder.placement;
        this.keepJobFlowAliveWhenNoSteps = builder.keepJobFlowAliveWhenNoSteps;
        this.terminationProtected = builder.terminationProtected;
        this.hadoopVersion = builder.hadoopVersion;
        this.ec2SubnetId = builder.ec2SubnetId;
        this.ec2SubnetIds = builder.ec2SubnetIds;
        this.emrManagedMasterSecurityGroup = builder.emrManagedMasterSecurityGroup;
        this.emrManagedSlaveSecurityGroup = builder.emrManagedSlaveSecurityGroup;
        this.serviceAccessSecurityGroup = builder.serviceAccessSecurityGroup;
        this.additionalMasterSecurityGroups = builder.additionalMasterSecurityGroups;
        this.additionalSlaveSecurityGroups = builder.additionalSlaveSecurityGroups;
    }

    /**
     * <p>
     * The EC2 instance type of the master node.
     * </p>
     * 
     * @return The EC2 instance type of the master node.
     */
    public final String masterInstanceType() {
        return masterInstanceType;
    }

    /**
     * <p>
     * The EC2 instance type of the core and task nodes.
     * </p>
     * 
     * @return The EC2 instance type of the core and task nodes.
     */
    public final String slaveInstanceType() {
        return slaveInstanceType;
    }

    /**
     * <p>
     * The number of EC2 instances in the cluster.
     * </p>
     * 
     * @return The number of EC2 instances in the cluster.
     */
    public final Integer instanceCount() {
        return instanceCount;
    }

    /**
     * Returns true if the InstanceGroups property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasInstanceGroups() {
        return instanceGroups != null && !(instanceGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Configuration for the instance groups in a cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasInstanceGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Configuration for the instance groups in a cluster.
     */
    public final List<InstanceGroupConfig> instanceGroups() {
        return instanceGroups;
    }

    /**
     * Returns true if the InstanceFleets property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasInstanceFleets() {
        return instanceFleets != null && !(instanceFleets instanceof SdkAutoConstructList);
    }

    /**
     * <note>
     * <p>
     * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x
     * versions.
     * </p>
     * </note>
     * <p>
     * Describes the EC2 instances and instance configurations for clusters that use the instance fleet configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasInstanceFleets()} to see if a value was sent in this field.
     * </p>
     * 
     * @return <p>
     *         The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding
     *         5.0.x versions.
     *         </p>
     *         </note>
     *         <p>
     *         Describes the EC2 instances and instance configurations for clusters that use the instance fleet
     *         configuration.
     */
    public final List<InstanceFleetConfig> instanceFleets() {
        return instanceFleets;
    }

    /**
     * <p>
     * The name of the EC2 key pair that can be used to connect to the master node using SSH as the user called
     * "hadoop."
     * </p>
     * 
     * @return The name of the EC2 key pair that can be used to connect to the master node using SSH as the user called
     *         "hadoop."
     */
    public final String ec2KeyName() {
        return ec2KeyName;
    }

    /**
     * <p>
     * The Availability Zone in which the cluster runs.
     * </p>
     * 
     * @return The Availability Zone in which the cluster runs.
     */
    public final PlacementType placement() {
        return placement;
    }

    /**
     * <p>
     * Specifies whether the cluster should remain available after completing all steps.
     * </p>
     * 
     * @return Specifies whether the cluster should remain available after completing all steps.
     */
    public final Boolean keepJobFlowAliveWhenNoSteps() {
        return keepJobFlowAliveWhenNoSteps;
    }

    /**
     * <p>
     * Specifies whether to lock the cluster to prevent the Amazon EC2 instances from being terminated by API call, user
     * intervention, or in the event of a job-flow error.
     * </p>
     * 
     * @return Specifies whether to lock the cluster to prevent the Amazon EC2 instances from being terminated by API
     *         call, user intervention, or in the event of a job-flow error.
     */
    public final Boolean terminationProtected() {
        return terminationProtected;
    }

    /**
     * <p>
     * Applies only to Amazon EMR release versions earlier than 4.0. The Hadoop version for the cluster. Valid inputs
     * are "0.18" (no longer maintained), "0.20" (no longer maintained), "0.20.205" (no longer maintained), "1.0.3",
     * "2.2.0", or "2.4.0". If you do not set this value, the default of 0.18 is used, unless the
     * <code>AmiVersion</code> parameter is set in the RunJobFlow call, in which case the default version of Hadoop for
     * that AMI version is used.
     * </p>
     * 
     * @return Applies only to Amazon EMR release versions earlier than 4.0. The Hadoop version for the cluster. Valid
     *         inputs are "0.18" (no longer maintained), "0.20" (no longer maintained), "0.20.205" (no longer
     *         maintained), "1.0.3", "2.2.0", or "2.4.0". If you do not set this value, the default of 0.18 is used,
     *         unless the <code>AmiVersion</code> parameter is set in the RunJobFlow call, in which case the default
     *         version of Hadoop for that AMI version is used.
     */
    public final String hadoopVersion() {
        return hadoopVersion;
    }

    /**
     * <p>
     * Applies to clusters that use the uniform instance group configuration. To launch the cluster in Amazon Virtual
     * Private Cloud (Amazon VPC), set this parameter to the identifier of the Amazon VPC subnet where you want the
     * cluster to launch. If you do not specify this value and your account supports EC2-Classic, the cluster launches
     * in EC2-Classic.
     * </p>
     * 
     * @return Applies to clusters that use the uniform instance group configuration. To launch the cluster in Amazon
     *         Virtual Private Cloud (Amazon VPC), set this parameter to the identifier of the Amazon VPC subnet where
     *         you want the cluster to launch. If you do not specify this value and your account supports EC2-Classic,
     *         the cluster launches in EC2-Classic.
     */
    public final String ec2SubnetId() {
        return ec2SubnetId;
    }

    /**
     * Returns true if the Ec2SubnetIds property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasEc2SubnetIds() {
        return ec2SubnetIds != null && !(ec2SubnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Applies to clusters that use the instance fleet configuration. When multiple EC2 subnet IDs are specified, Amazon
     * EMR evaluates them and launches instances in the optimal subnet.
     * </p>
     * <note>
     * <p>
     * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x
     * versions.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEc2SubnetIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Applies to clusters that use the instance fleet configuration. When multiple EC2 subnet IDs are
     *         specified, Amazon EMR evaluates them and launches instances in the optimal subnet.</p> <note>
     *         <p>
     *         The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding
     *         5.0.x versions.
     *         </p>
     */
    public final List<String> ec2SubnetIds() {
        return ec2SubnetIds;
    }

    /**
     * <p>
     * The identifier of the Amazon EC2 security group for the master node.
     * </p>
     * 
     * @return The identifier of the Amazon EC2 security group for the master node.
     */
    public final String emrManagedMasterSecurityGroup() {
        return emrManagedMasterSecurityGroup;
    }

    /**
     * <p>
     * The identifier of the Amazon EC2 security group for the core and task nodes.
     * </p>
     * 
     * @return The identifier of the Amazon EC2 security group for the core and task nodes.
     */
    public final String emrManagedSlaveSecurityGroup() {
        return emrManagedSlaveSecurityGroup;
    }

    /**
     * <p>
     * The identifier of the Amazon EC2 security group for the Amazon EMR service to access clusters in VPC private
     * subnets.
     * </p>
     * 
     * @return The identifier of the Amazon EC2 security group for the Amazon EMR service to access clusters in VPC
     *         private subnets.
     */
    public final String serviceAccessSecurityGroup() {
        return serviceAccessSecurityGroup;
    }

    /**
     * Returns true if the AdditionalMasterSecurityGroups property was specified by the sender (it may be empty), or
     * false if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender
     * is the AWS service.
     */
    public final boolean hasAdditionalMasterSecurityGroups() {
        return additionalMasterSecurityGroups != null && !(additionalMasterSecurityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of additional Amazon EC2 security group IDs for the master node.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAdditionalMasterSecurityGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of additional Amazon EC2 security group IDs for the master node.
     */
    public final List<String> additionalMasterSecurityGroups() {
        return additionalMasterSecurityGroups;
    }

    /**
     * Returns true if the AdditionalSlaveSecurityGroups property was specified by the sender (it may be empty), or
     * false if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender
     * is the AWS service.
     */
    public final boolean hasAdditionalSlaveSecurityGroups() {
        return additionalSlaveSecurityGroups != null && !(additionalSlaveSecurityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of additional Amazon EC2 security group IDs for the core and task nodes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAdditionalSlaveSecurityGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of additional Amazon EC2 security group IDs for the core and task nodes.
     */
    public final List<String> additionalSlaveSecurityGroups() {
        return additionalSlaveSecurityGroups;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(masterInstanceType());
        hashCode = 31 * hashCode + Objects.hashCode(slaveInstanceType());
        hashCode = 31 * hashCode + Objects.hashCode(instanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceGroups() ? instanceGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceFleets() ? instanceFleets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ec2KeyName());
        hashCode = 31 * hashCode + Objects.hashCode(placement());
        hashCode = 31 * hashCode + Objects.hashCode(keepJobFlowAliveWhenNoSteps());
        hashCode = 31 * hashCode + Objects.hashCode(terminationProtected());
        hashCode = 31 * hashCode + Objects.hashCode(hadoopVersion());
        hashCode = 31 * hashCode + Objects.hashCode(ec2SubnetId());
        hashCode = 31 * hashCode + Objects.hashCode(hasEc2SubnetIds() ? ec2SubnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(emrManagedMasterSecurityGroup());
        hashCode = 31 * hashCode + Objects.hashCode(emrManagedSlaveSecurityGroup());
        hashCode = 31 * hashCode + Objects.hashCode(serviceAccessSecurityGroup());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasAdditionalMasterSecurityGroups() ? additionalMasterSecurityGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAdditionalSlaveSecurityGroups() ? additionalSlaveSecurityGroups() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobFlowInstancesConfig)) {
            return false;
        }
        JobFlowInstancesConfig other = (JobFlowInstancesConfig) obj;
        return Objects.equals(masterInstanceType(), other.masterInstanceType())
                && Objects.equals(slaveInstanceType(), other.slaveInstanceType())
                && Objects.equals(instanceCount(), other.instanceCount()) && hasInstanceGroups() == other.hasInstanceGroups()
                && Objects.equals(instanceGroups(), other.instanceGroups()) && hasInstanceFleets() == other.hasInstanceFleets()
                && Objects.equals(instanceFleets(), other.instanceFleets()) && Objects.equals(ec2KeyName(), other.ec2KeyName())
                && Objects.equals(placement(), other.placement())
                && Objects.equals(keepJobFlowAliveWhenNoSteps(), other.keepJobFlowAliveWhenNoSteps())
                && Objects.equals(terminationProtected(), other.terminationProtected())
                && Objects.equals(hadoopVersion(), other.hadoopVersion()) && Objects.equals(ec2SubnetId(), other.ec2SubnetId())
                && hasEc2SubnetIds() == other.hasEc2SubnetIds() && Objects.equals(ec2SubnetIds(), other.ec2SubnetIds())
                && Objects.equals(emrManagedMasterSecurityGroup(), other.emrManagedMasterSecurityGroup())
                && Objects.equals(emrManagedSlaveSecurityGroup(), other.emrManagedSlaveSecurityGroup())
                && Objects.equals(serviceAccessSecurityGroup(), other.serviceAccessSecurityGroup())
                && hasAdditionalMasterSecurityGroups() == other.hasAdditionalMasterSecurityGroups()
                && Objects.equals(additionalMasterSecurityGroups(), other.additionalMasterSecurityGroups())
                && hasAdditionalSlaveSecurityGroups() == other.hasAdditionalSlaveSecurityGroups()
                && Objects.equals(additionalSlaveSecurityGroups(), other.additionalSlaveSecurityGroups());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("JobFlowInstancesConfig")
                .add("MasterInstanceType", masterInstanceType())
                .add("SlaveInstanceType", slaveInstanceType())
                .add("InstanceCount", instanceCount())
                .add("InstanceGroups", hasInstanceGroups() ? instanceGroups() : null)
                .add("InstanceFleets", hasInstanceFleets() ? instanceFleets() : null)
                .add("Ec2KeyName", ec2KeyName())
                .add("Placement", placement())
                .add("KeepJobFlowAliveWhenNoSteps", keepJobFlowAliveWhenNoSteps())
                .add("TerminationProtected", terminationProtected())
                .add("HadoopVersion", hadoopVersion())
                .add("Ec2SubnetId", ec2SubnetId())
                .add("Ec2SubnetIds", hasEc2SubnetIds() ? ec2SubnetIds() : null)
                .add("EmrManagedMasterSecurityGroup", emrManagedMasterSecurityGroup())
                .add("EmrManagedSlaveSecurityGroup", emrManagedSlaveSecurityGroup())
                .add("ServiceAccessSecurityGroup", serviceAccessSecurityGroup())
                .add("AdditionalMasterSecurityGroups",
                        hasAdditionalMasterSecurityGroups() ? additionalMasterSecurityGroups() : null)
                .add("AdditionalSlaveSecurityGroups", hasAdditionalSlaveSecurityGroups() ? additionalSlaveSecurityGroups() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MasterInstanceType":
            return Optional.ofNullable(clazz.cast(masterInstanceType()));
        case "SlaveInstanceType":
            return Optional.ofNullable(clazz.cast(slaveInstanceType()));
        case "InstanceCount":
            return Optional.ofNullable(clazz.cast(instanceCount()));
        case "InstanceGroups":
            return Optional.ofNullable(clazz.cast(instanceGroups()));
        case "InstanceFleets":
            return Optional.ofNullable(clazz.cast(instanceFleets()));
        case "Ec2KeyName":
            return Optional.ofNullable(clazz.cast(ec2KeyName()));
        case "Placement":
            return Optional.ofNullable(clazz.cast(placement()));
        case "KeepJobFlowAliveWhenNoSteps":
            return Optional.ofNullable(clazz.cast(keepJobFlowAliveWhenNoSteps()));
        case "TerminationProtected":
            return Optional.ofNullable(clazz.cast(terminationProtected()));
        case "HadoopVersion":
            return Optional.ofNullable(clazz.cast(hadoopVersion()));
        case "Ec2SubnetId":
            return Optional.ofNullable(clazz.cast(ec2SubnetId()));
        case "Ec2SubnetIds":
            return Optional.ofNullable(clazz.cast(ec2SubnetIds()));
        case "EmrManagedMasterSecurityGroup":
            return Optional.ofNullable(clazz.cast(emrManagedMasterSecurityGroup()));
        case "EmrManagedSlaveSecurityGroup":
            return Optional.ofNullable(clazz.cast(emrManagedSlaveSecurityGroup()));
        case "ServiceAccessSecurityGroup":
            return Optional.ofNullable(clazz.cast(serviceAccessSecurityGroup()));
        case "AdditionalMasterSecurityGroups":
            return Optional.ofNullable(clazz.cast(additionalMasterSecurityGroups()));
        case "AdditionalSlaveSecurityGroups":
            return Optional.ofNullable(clazz.cast(additionalSlaveSecurityGroups()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JobFlowInstancesConfig, T> g) {
        return obj -> g.apply((JobFlowInstancesConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobFlowInstancesConfig> {
        /**
         * <p>
         * The EC2 instance type of the master node.
         * </p>
         * 
         * @param masterInstanceType
         *        The EC2 instance type of the master node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterInstanceType(String masterInstanceType);

        /**
         * <p>
         * The EC2 instance type of the core and task nodes.
         * </p>
         * 
         * @param slaveInstanceType
         *        The EC2 instance type of the core and task nodes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slaveInstanceType(String slaveInstanceType);

        /**
         * <p>
         * The number of EC2 instances in the cluster.
         * </p>
         * 
         * @param instanceCount
         *        The number of EC2 instances in the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceCount(Integer instanceCount);

        /**
         * <p>
         * Configuration for the instance groups in a cluster.
         * </p>
         * 
         * @param instanceGroups
         *        Configuration for the instance groups in a cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGroups(Collection<InstanceGroupConfig> instanceGroups);

        /**
         * <p>
         * Configuration for the instance groups in a cluster.
         * </p>
         * 
         * @param instanceGroups
         *        Configuration for the instance groups in a cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGroups(InstanceGroupConfig... instanceGroups);

        /**
         * <p>
         * Configuration for the instance groups in a cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link List<InstanceGroupConfig>.Builder} avoiding the
         * need to create one manually via {@link List<InstanceGroupConfig>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<InstanceGroupConfig>.Builder#build()} is called immediately
         * and its result is passed to {@link #instanceGroups(List<InstanceGroupConfig>)}.
         * 
         * @param instanceGroups
         *        a consumer that will call methods on {@link List<InstanceGroupConfig>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceGroups(List<InstanceGroupConfig>)
         */
        Builder instanceGroups(Consumer<InstanceGroupConfig.Builder>... instanceGroups);

        /**
         * <note>
         * <p>
         * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x
         * versions.
         * </p>
         * </note>
         * <p>
         * Describes the EC2 instances and instance configurations for clusters that use the instance fleet
         * configuration.
         * </p>
         * 
         * @param instanceFleets
         *        <p>
         *        The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding
         *        5.0.x versions.
         *        </p>
         *        </note>
         *        <p>
         *        Describes the EC2 instances and instance configurations for clusters that use the instance fleet
         *        configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceFleets(Collection<InstanceFleetConfig> instanceFleets);

        /**
         * <note>
         * <p>
         * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x
         * versions.
         * </p>
         * </note>
         * <p>
         * Describes the EC2 instances and instance configurations for clusters that use the instance fleet
         * configuration.
         * </p>
         * 
         * @param instanceFleets
         *        <p>
         *        The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding
         *        5.0.x versions.
         *        </p>
         *        </note>
         *        <p>
         *        Describes the EC2 instances and instance configurations for clusters that use the instance fleet
         *        configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceFleets(InstanceFleetConfig... instanceFleets);

        /**
         * <note>
         * <p>
         * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x
         * versions.
         * </p>
         * </note>
         * <p>
         * Describes the EC2 instances and instance configurations for clusters that use the instance fleet
         * configuration.
         * </p>
         * This is a convenience that creates an instance of the {@link List<InstanceFleetConfig>.Builder} avoiding the
         * need to create one manually via {@link List<InstanceFleetConfig>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<InstanceFleetConfig>.Builder#build()} is called immediately
         * and its result is passed to {@link #instanceFleets(List<InstanceFleetConfig>)}.
         * 
         * @param instanceFleets
         *        a consumer that will call methods on {@link List<InstanceFleetConfig>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceFleets(List<InstanceFleetConfig>)
         */
        Builder instanceFleets(Consumer<InstanceFleetConfig.Builder>... instanceFleets);

        /**
         * <p>
         * The name of the EC2 key pair that can be used to connect to the master node using SSH as the user called
         * "hadoop."
         * </p>
         * 
         * @param ec2KeyName
         *        The name of the EC2 key pair that can be used to connect to the master node using SSH as the user
         *        called "hadoop."
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2KeyName(String ec2KeyName);

        /**
         * <p>
         * The Availability Zone in which the cluster runs.
         * </p>
         * 
         * @param placement
         *        The Availability Zone in which the cluster runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placement(PlacementType placement);

        /**
         * <p>
         * The Availability Zone in which the cluster runs.
         * </p>
         * This is a convenience that creates an instance of the {@link PlacementType.Builder} avoiding the need to
         * create one manually via {@link PlacementType#builder()}.
         *
         * When the {@link Consumer} completes, {@link PlacementType.Builder#build()} is called immediately and its
         * result is passed to {@link #placement(PlacementType)}.
         * 
         * @param placement
         *        a consumer that will call methods on {@link PlacementType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #placement(PlacementType)
         */
        default Builder placement(Consumer<PlacementType.Builder> placement) {
            return placement(PlacementType.builder().applyMutation(placement).build());
        }

        /**
         * <p>
         * Specifies whether the cluster should remain available after completing all steps.
         * </p>
         * 
         * @param keepJobFlowAliveWhenNoSteps
         *        Specifies whether the cluster should remain available after completing all steps.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keepJobFlowAliveWhenNoSteps(Boolean keepJobFlowAliveWhenNoSteps);

        /**
         * <p>
         * Specifies whether to lock the cluster to prevent the Amazon EC2 instances from being terminated by API call,
         * user intervention, or in the event of a job-flow error.
         * </p>
         * 
         * @param terminationProtected
         *        Specifies whether to lock the cluster to prevent the Amazon EC2 instances from being terminated by API
         *        call, user intervention, or in the event of a job-flow error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminationProtected(Boolean terminationProtected);

        /**
         * <p>
         * Applies only to Amazon EMR release versions earlier than 4.0. The Hadoop version for the cluster. Valid
         * inputs are "0.18" (no longer maintained), "0.20" (no longer maintained), "0.20.205" (no longer maintained),
         * "1.0.3", "2.2.0", or "2.4.0". If you do not set this value, the default of 0.18 is used, unless the
         * <code>AmiVersion</code> parameter is set in the RunJobFlow call, in which case the default version of Hadoop
         * for that AMI version is used.
         * </p>
         * 
         * @param hadoopVersion
         *        Applies only to Amazon EMR release versions earlier than 4.0. The Hadoop version for the cluster.
         *        Valid inputs are "0.18" (no longer maintained), "0.20" (no longer maintained), "0.20.205" (no longer
         *        maintained), "1.0.3", "2.2.0", or "2.4.0". If you do not set this value, the default of 0.18 is used,
         *        unless the <code>AmiVersion</code> parameter is set in the RunJobFlow call, in which case the default
         *        version of Hadoop for that AMI version is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hadoopVersion(String hadoopVersion);

        /**
         * <p>
         * Applies to clusters that use the uniform instance group configuration. To launch the cluster in Amazon
         * Virtual Private Cloud (Amazon VPC), set this parameter to the identifier of the Amazon VPC subnet where you
         * want the cluster to launch. If you do not specify this value and your account supports EC2-Classic, the
         * cluster launches in EC2-Classic.
         * </p>
         * 
         * @param ec2SubnetId
         *        Applies to clusters that use the uniform instance group configuration. To launch the cluster in Amazon
         *        Virtual Private Cloud (Amazon VPC), set this parameter to the identifier of the Amazon VPC subnet
         *        where you want the cluster to launch. If you do not specify this value and your account supports
         *        EC2-Classic, the cluster launches in EC2-Classic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2SubnetId(String ec2SubnetId);

        /**
         * <p>
         * Applies to clusters that use the instance fleet configuration. When multiple EC2 subnet IDs are specified,
         * Amazon EMR evaluates them and launches instances in the optimal subnet.
         * </p>
         * <note>
         * <p>
         * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x
         * versions.
         * </p>
         * </note>
         * 
         * @param ec2SubnetIds
         *        Applies to clusters that use the instance fleet configuration. When multiple EC2 subnet IDs are
         *        specified, Amazon EMR evaluates them and launches instances in the optimal subnet.</p> <note>
         *        <p>
         *        The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding
         *        5.0.x versions.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2SubnetIds(Collection<String> ec2SubnetIds);

        /**
         * <p>
         * Applies to clusters that use the instance fleet configuration. When multiple EC2 subnet IDs are specified,
         * Amazon EMR evaluates them and launches instances in the optimal subnet.
         * </p>
         * <note>
         * <p>
         * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x
         * versions.
         * </p>
         * </note>
         * 
         * @param ec2SubnetIds
         *        Applies to clusters that use the instance fleet configuration. When multiple EC2 subnet IDs are
         *        specified, Amazon EMR evaluates them and launches instances in the optimal subnet.</p> <note>
         *        <p>
         *        The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding
         *        5.0.x versions.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2SubnetIds(String... ec2SubnetIds);

        /**
         * <p>
         * The identifier of the Amazon EC2 security group for the master node.
         * </p>
         * 
         * @param emrManagedMasterSecurityGroup
         *        The identifier of the Amazon EC2 security group for the master node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emrManagedMasterSecurityGroup(String emrManagedMasterSecurityGroup);

        /**
         * <p>
         * The identifier of the Amazon EC2 security group for the core and task nodes.
         * </p>
         * 
         * @param emrManagedSlaveSecurityGroup
         *        The identifier of the Amazon EC2 security group for the core and task nodes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emrManagedSlaveSecurityGroup(String emrManagedSlaveSecurityGroup);

        /**
         * <p>
         * The identifier of the Amazon EC2 security group for the Amazon EMR service to access clusters in VPC private
         * subnets.
         * </p>
         * 
         * @param serviceAccessSecurityGroup
         *        The identifier of the Amazon EC2 security group for the Amazon EMR service to access clusters in VPC
         *        private subnets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceAccessSecurityGroup(String serviceAccessSecurityGroup);

        /**
         * <p>
         * A list of additional Amazon EC2 security group IDs for the master node.
         * </p>
         * 
         * @param additionalMasterSecurityGroups
         *        A list of additional Amazon EC2 security group IDs for the master node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalMasterSecurityGroups(Collection<String> additionalMasterSecurityGroups);

        /**
         * <p>
         * A list of additional Amazon EC2 security group IDs for the master node.
         * </p>
         * 
         * @param additionalMasterSecurityGroups
         *        A list of additional Amazon EC2 security group IDs for the master node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalMasterSecurityGroups(String... additionalMasterSecurityGroups);

        /**
         * <p>
         * A list of additional Amazon EC2 security group IDs for the core and task nodes.
         * </p>
         * 
         * @param additionalSlaveSecurityGroups
         *        A list of additional Amazon EC2 security group IDs for the core and task nodes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalSlaveSecurityGroups(Collection<String> additionalSlaveSecurityGroups);

        /**
         * <p>
         * A list of additional Amazon EC2 security group IDs for the core and task nodes.
         * </p>
         * 
         * @param additionalSlaveSecurityGroups
         *        A list of additional Amazon EC2 security group IDs for the core and task nodes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalSlaveSecurityGroups(String... additionalSlaveSecurityGroups);
    }

    static final class BuilderImpl implements Builder {
        private String masterInstanceType;

        private String slaveInstanceType;

        private Integer instanceCount;

        private List<InstanceGroupConfig> instanceGroups = DefaultSdkAutoConstructList.getInstance();

        private List<InstanceFleetConfig> instanceFleets = DefaultSdkAutoConstructList.getInstance();

        private String ec2KeyName;

        private PlacementType placement;

        private Boolean keepJobFlowAliveWhenNoSteps;

        private Boolean terminationProtected;

        private String hadoopVersion;

        private String ec2SubnetId;

        private List<String> ec2SubnetIds = DefaultSdkAutoConstructList.getInstance();

        private String emrManagedMasterSecurityGroup;

        private String emrManagedSlaveSecurityGroup;

        private String serviceAccessSecurityGroup;

        private List<String> additionalMasterSecurityGroups = DefaultSdkAutoConstructList.getInstance();

        private List<String> additionalSlaveSecurityGroups = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(JobFlowInstancesConfig model) {
            masterInstanceType(model.masterInstanceType);
            slaveInstanceType(model.slaveInstanceType);
            instanceCount(model.instanceCount);
            instanceGroups(model.instanceGroups);
            instanceFleets(model.instanceFleets);
            ec2KeyName(model.ec2KeyName);
            placement(model.placement);
            keepJobFlowAliveWhenNoSteps(model.keepJobFlowAliveWhenNoSteps);
            terminationProtected(model.terminationProtected);
            hadoopVersion(model.hadoopVersion);
            ec2SubnetId(model.ec2SubnetId);
            ec2SubnetIds(model.ec2SubnetIds);
            emrManagedMasterSecurityGroup(model.emrManagedMasterSecurityGroup);
            emrManagedSlaveSecurityGroup(model.emrManagedSlaveSecurityGroup);
            serviceAccessSecurityGroup(model.serviceAccessSecurityGroup);
            additionalMasterSecurityGroups(model.additionalMasterSecurityGroups);
            additionalSlaveSecurityGroups(model.additionalSlaveSecurityGroups);
        }

        public final String getMasterInstanceType() {
            return masterInstanceType;
        }

        @Override
        public final Builder masterInstanceType(String masterInstanceType) {
            this.masterInstanceType = masterInstanceType;
            return this;
        }

        public final void setMasterInstanceType(String masterInstanceType) {
            this.masterInstanceType = masterInstanceType;
        }

        public final String getSlaveInstanceType() {
            return slaveInstanceType;
        }

        @Override
        public final Builder slaveInstanceType(String slaveInstanceType) {
            this.slaveInstanceType = slaveInstanceType;
            return this;
        }

        public final void setSlaveInstanceType(String slaveInstanceType) {
            this.slaveInstanceType = slaveInstanceType;
        }

        public final Integer getInstanceCount() {
            return instanceCount;
        }

        @Override
        public final Builder instanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
            return this;
        }

        public final void setInstanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
        }

        public final List<InstanceGroupConfig.Builder> getInstanceGroups() {
            List<InstanceGroupConfig.Builder> result = InstanceGroupConfigListCopier.copyToBuilder(this.instanceGroups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder instanceGroups(Collection<InstanceGroupConfig> instanceGroups) {
            this.instanceGroups = InstanceGroupConfigListCopier.copy(instanceGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceGroups(InstanceGroupConfig... instanceGroups) {
            instanceGroups(Arrays.asList(instanceGroups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceGroups(Consumer<InstanceGroupConfig.Builder>... instanceGroups) {
            instanceGroups(Stream.of(instanceGroups).map(c -> InstanceGroupConfig.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setInstanceGroups(Collection<InstanceGroupConfig.BuilderImpl> instanceGroups) {
            this.instanceGroups = InstanceGroupConfigListCopier.copyFromBuilder(instanceGroups);
        }

        public final List<InstanceFleetConfig.Builder> getInstanceFleets() {
            List<InstanceFleetConfig.Builder> result = InstanceFleetConfigListCopier.copyToBuilder(this.instanceFleets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder instanceFleets(Collection<InstanceFleetConfig> instanceFleets) {
            this.instanceFleets = InstanceFleetConfigListCopier.copy(instanceFleets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceFleets(InstanceFleetConfig... instanceFleets) {
            instanceFleets(Arrays.asList(instanceFleets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceFleets(Consumer<InstanceFleetConfig.Builder>... instanceFleets) {
            instanceFleets(Stream.of(instanceFleets).map(c -> InstanceFleetConfig.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setInstanceFleets(Collection<InstanceFleetConfig.BuilderImpl> instanceFleets) {
            this.instanceFleets = InstanceFleetConfigListCopier.copyFromBuilder(instanceFleets);
        }

        public final String getEc2KeyName() {
            return ec2KeyName;
        }

        @Override
        public final Builder ec2KeyName(String ec2KeyName) {
            this.ec2KeyName = ec2KeyName;
            return this;
        }

        public final void setEc2KeyName(String ec2KeyName) {
            this.ec2KeyName = ec2KeyName;
        }

        public final PlacementType.Builder getPlacement() {
            return placement != null ? placement.toBuilder() : null;
        }

        @Override
        public final Builder placement(PlacementType placement) {
            this.placement = placement;
            return this;
        }

        public final void setPlacement(PlacementType.BuilderImpl placement) {
            this.placement = placement != null ? placement.build() : null;
        }

        public final Boolean getKeepJobFlowAliveWhenNoSteps() {
            return keepJobFlowAliveWhenNoSteps;
        }

        @Override
        public final Builder keepJobFlowAliveWhenNoSteps(Boolean keepJobFlowAliveWhenNoSteps) {
            this.keepJobFlowAliveWhenNoSteps = keepJobFlowAliveWhenNoSteps;
            return this;
        }

        public final void setKeepJobFlowAliveWhenNoSteps(Boolean keepJobFlowAliveWhenNoSteps) {
            this.keepJobFlowAliveWhenNoSteps = keepJobFlowAliveWhenNoSteps;
        }

        public final Boolean getTerminationProtected() {
            return terminationProtected;
        }

        @Override
        public final Builder terminationProtected(Boolean terminationProtected) {
            this.terminationProtected = terminationProtected;
            return this;
        }

        public final void setTerminationProtected(Boolean terminationProtected) {
            this.terminationProtected = terminationProtected;
        }

        public final String getHadoopVersion() {
            return hadoopVersion;
        }

        @Override
        public final Builder hadoopVersion(String hadoopVersion) {
            this.hadoopVersion = hadoopVersion;
            return this;
        }

        public final void setHadoopVersion(String hadoopVersion) {
            this.hadoopVersion = hadoopVersion;
        }

        public final String getEc2SubnetId() {
            return ec2SubnetId;
        }

        @Override
        public final Builder ec2SubnetId(String ec2SubnetId) {
            this.ec2SubnetId = ec2SubnetId;
            return this;
        }

        public final void setEc2SubnetId(String ec2SubnetId) {
            this.ec2SubnetId = ec2SubnetId;
        }

        public final Collection<String> getEc2SubnetIds() {
            if (ec2SubnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return ec2SubnetIds;
        }

        @Override
        public final Builder ec2SubnetIds(Collection<String> ec2SubnetIds) {
            this.ec2SubnetIds = XmlStringMaxLen256ListCopier.copy(ec2SubnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ec2SubnetIds(String... ec2SubnetIds) {
            ec2SubnetIds(Arrays.asList(ec2SubnetIds));
            return this;
        }

        public final void setEc2SubnetIds(Collection<String> ec2SubnetIds) {
            this.ec2SubnetIds = XmlStringMaxLen256ListCopier.copy(ec2SubnetIds);
        }

        public final String getEmrManagedMasterSecurityGroup() {
            return emrManagedMasterSecurityGroup;
        }

        @Override
        public final Builder emrManagedMasterSecurityGroup(String emrManagedMasterSecurityGroup) {
            this.emrManagedMasterSecurityGroup = emrManagedMasterSecurityGroup;
            return this;
        }

        public final void setEmrManagedMasterSecurityGroup(String emrManagedMasterSecurityGroup) {
            this.emrManagedMasterSecurityGroup = emrManagedMasterSecurityGroup;
        }

        public final String getEmrManagedSlaveSecurityGroup() {
            return emrManagedSlaveSecurityGroup;
        }

        @Override
        public final Builder emrManagedSlaveSecurityGroup(String emrManagedSlaveSecurityGroup) {
            this.emrManagedSlaveSecurityGroup = emrManagedSlaveSecurityGroup;
            return this;
        }

        public final void setEmrManagedSlaveSecurityGroup(String emrManagedSlaveSecurityGroup) {
            this.emrManagedSlaveSecurityGroup = emrManagedSlaveSecurityGroup;
        }

        public final String getServiceAccessSecurityGroup() {
            return serviceAccessSecurityGroup;
        }

        @Override
        public final Builder serviceAccessSecurityGroup(String serviceAccessSecurityGroup) {
            this.serviceAccessSecurityGroup = serviceAccessSecurityGroup;
            return this;
        }

        public final void setServiceAccessSecurityGroup(String serviceAccessSecurityGroup) {
            this.serviceAccessSecurityGroup = serviceAccessSecurityGroup;
        }

        public final Collection<String> getAdditionalMasterSecurityGroups() {
            if (additionalMasterSecurityGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return additionalMasterSecurityGroups;
        }

        @Override
        public final Builder additionalMasterSecurityGroups(Collection<String> additionalMasterSecurityGroups) {
            this.additionalMasterSecurityGroups = SecurityGroupsListCopier.copy(additionalMasterSecurityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder additionalMasterSecurityGroups(String... additionalMasterSecurityGroups) {
            additionalMasterSecurityGroups(Arrays.asList(additionalMasterSecurityGroups));
            return this;
        }

        public final void setAdditionalMasterSecurityGroups(Collection<String> additionalMasterSecurityGroups) {
            this.additionalMasterSecurityGroups = SecurityGroupsListCopier.copy(additionalMasterSecurityGroups);
        }

        public final Collection<String> getAdditionalSlaveSecurityGroups() {
            if (additionalSlaveSecurityGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return additionalSlaveSecurityGroups;
        }

        @Override
        public final Builder additionalSlaveSecurityGroups(Collection<String> additionalSlaveSecurityGroups) {
            this.additionalSlaveSecurityGroups = SecurityGroupsListCopier.copy(additionalSlaveSecurityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder additionalSlaveSecurityGroups(String... additionalSlaveSecurityGroups) {
            additionalSlaveSecurityGroups(Arrays.asList(additionalSlaveSecurityGroups));
            return this;
        }

        public final void setAdditionalSlaveSecurityGroups(Collection<String> additionalSlaveSecurityGroups) {
            this.additionalSlaveSecurityGroups = SecurityGroupsListCopier.copy(additionalSlaveSecurityGroups);
        }

        @Override
        public JobFlowInstancesConfig build() {
            return new JobFlowInstancesConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
