/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An entity describing an executable that runs on a cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Command implements SdkPojo, Serializable, ToCopyableBuilder<Command.Builder, Command> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Command::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> SCRIPT_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScriptPath").getter(getter(Command::scriptPath)).setter(setter(Builder::scriptPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScriptPath").build()).build();

    private static final SdkField<List<String>> ARGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Args")
            .getter(getter(Command::args))
            .setter(setter(Builder::args))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Args").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, SCRIPT_PATH_FIELD,
            ARGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String scriptPath;

    private final List<String> args;

    private Command(BuilderImpl builder) {
        this.name = builder.name;
        this.scriptPath = builder.scriptPath;
        this.args = builder.args;
    }

    /**
     * <p>
     * The name of the command.
     * </p>
     * 
     * @return The name of the command.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon S3 location of the command script.
     * </p>
     * 
     * @return The Amazon S3 location of the command script.
     */
    public final String scriptPath() {
        return scriptPath;
    }

    /**
     * For responses, this returns true if the service returned a value for the Args property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasArgs() {
        return args != null && !(args instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Arguments for Amazon EMR to pass to the command for execution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasArgs} method.
     * </p>
     * 
     * @return Arguments for Amazon EMR to pass to the command for execution.
     */
    public final List<String> args() {
        return args;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(scriptPath());
        hashCode = 31 * hashCode + Objects.hashCode(hasArgs() ? args() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Command)) {
            return false;
        }
        Command other = (Command) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(scriptPath(), other.scriptPath())
                && hasArgs() == other.hasArgs() && Objects.equals(args(), other.args());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Command").add("Name", name()).add("ScriptPath", scriptPath())
                .add("Args", hasArgs() ? args() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ScriptPath":
            return Optional.ofNullable(clazz.cast(scriptPath()));
        case "Args":
            return Optional.ofNullable(clazz.cast(args()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Command, T> g) {
        return obj -> g.apply((Command) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Command> {
        /**
         * <p>
         * The name of the command.
         * </p>
         * 
         * @param name
         *        The name of the command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon S3 location of the command script.
         * </p>
         * 
         * @param scriptPath
         *        The Amazon S3 location of the command script.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scriptPath(String scriptPath);

        /**
         * <p>
         * Arguments for Amazon EMR to pass to the command for execution.
         * </p>
         * 
         * @param args
         *        Arguments for Amazon EMR to pass to the command for execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder args(Collection<String> args);

        /**
         * <p>
         * Arguments for Amazon EMR to pass to the command for execution.
         * </p>
         * 
         * @param args
         *        Arguments for Amazon EMR to pass to the command for execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder args(String... args);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String scriptPath;

        private List<String> args = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Command model) {
            name(model.name);
            scriptPath(model.scriptPath);
            args(model.args);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getScriptPath() {
            return scriptPath;
        }

        public final void setScriptPath(String scriptPath) {
            this.scriptPath = scriptPath;
        }

        @Override
        @Transient
        public final Builder scriptPath(String scriptPath) {
            this.scriptPath = scriptPath;
            return this;
        }

        public final Collection<String> getArgs() {
            if (args instanceof SdkAutoConstructList) {
                return null;
            }
            return args;
        }

        public final void setArgs(Collection<String> args) {
            this.args = StringListCopier.copy(args);
        }

        @Override
        @Transient
        public final Builder args(Collection<String> args) {
            this.args = StringListCopier.copy(args);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder args(String... args) {
            args(Arrays.asList(args));
            return this;
        }

        @Override
        public Command build() {
            return new Command(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
