/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.emr.internal.EmrServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.emr.model.AddInstanceFleetRequest;
import software.amazon.awssdk.services.emr.model.AddInstanceFleetResponse;
import software.amazon.awssdk.services.emr.model.AddInstanceGroupsRequest;
import software.amazon.awssdk.services.emr.model.AddInstanceGroupsResponse;
import software.amazon.awssdk.services.emr.model.AddJobFlowStepsRequest;
import software.amazon.awssdk.services.emr.model.AddJobFlowStepsResponse;
import software.amazon.awssdk.services.emr.model.AddTagsRequest;
import software.amazon.awssdk.services.emr.model.AddTagsResponse;
import software.amazon.awssdk.services.emr.model.CancelStepsRequest;
import software.amazon.awssdk.services.emr.model.CancelStepsResponse;
import software.amazon.awssdk.services.emr.model.CreateSecurityConfigurationRequest;
import software.amazon.awssdk.services.emr.model.CreateSecurityConfigurationResponse;
import software.amazon.awssdk.services.emr.model.CreateStudioRequest;
import software.amazon.awssdk.services.emr.model.CreateStudioResponse;
import software.amazon.awssdk.services.emr.model.CreateStudioSessionMappingRequest;
import software.amazon.awssdk.services.emr.model.CreateStudioSessionMappingResponse;
import software.amazon.awssdk.services.emr.model.DeleteSecurityConfigurationRequest;
import software.amazon.awssdk.services.emr.model.DeleteSecurityConfigurationResponse;
import software.amazon.awssdk.services.emr.model.DeleteStudioRequest;
import software.amazon.awssdk.services.emr.model.DeleteStudioResponse;
import software.amazon.awssdk.services.emr.model.DeleteStudioSessionMappingRequest;
import software.amazon.awssdk.services.emr.model.DeleteStudioSessionMappingResponse;
import software.amazon.awssdk.services.emr.model.DescribeClusterRequest;
import software.amazon.awssdk.services.emr.model.DescribeClusterResponse;
import software.amazon.awssdk.services.emr.model.DescribeNotebookExecutionRequest;
import software.amazon.awssdk.services.emr.model.DescribeNotebookExecutionResponse;
import software.amazon.awssdk.services.emr.model.DescribeReleaseLabelRequest;
import software.amazon.awssdk.services.emr.model.DescribeReleaseLabelResponse;
import software.amazon.awssdk.services.emr.model.DescribeSecurityConfigurationRequest;
import software.amazon.awssdk.services.emr.model.DescribeSecurityConfigurationResponse;
import software.amazon.awssdk.services.emr.model.DescribeStepRequest;
import software.amazon.awssdk.services.emr.model.DescribeStepResponse;
import software.amazon.awssdk.services.emr.model.DescribeStudioRequest;
import software.amazon.awssdk.services.emr.model.DescribeStudioResponse;
import software.amazon.awssdk.services.emr.model.EmrException;
import software.amazon.awssdk.services.emr.model.GetAutoTerminationPolicyRequest;
import software.amazon.awssdk.services.emr.model.GetAutoTerminationPolicyResponse;
import software.amazon.awssdk.services.emr.model.GetBlockPublicAccessConfigurationRequest;
import software.amazon.awssdk.services.emr.model.GetBlockPublicAccessConfigurationResponse;
import software.amazon.awssdk.services.emr.model.GetClusterSessionCredentialsRequest;
import software.amazon.awssdk.services.emr.model.GetClusterSessionCredentialsResponse;
import software.amazon.awssdk.services.emr.model.GetManagedScalingPolicyRequest;
import software.amazon.awssdk.services.emr.model.GetManagedScalingPolicyResponse;
import software.amazon.awssdk.services.emr.model.GetStudioSessionMappingRequest;
import software.amazon.awssdk.services.emr.model.GetStudioSessionMappingResponse;
import software.amazon.awssdk.services.emr.model.InternalServerErrorException;
import software.amazon.awssdk.services.emr.model.InternalServerException;
import software.amazon.awssdk.services.emr.model.InvalidRequestException;
import software.amazon.awssdk.services.emr.model.ListBootstrapActionsRequest;
import software.amazon.awssdk.services.emr.model.ListBootstrapActionsResponse;
import software.amazon.awssdk.services.emr.model.ListClustersRequest;
import software.amazon.awssdk.services.emr.model.ListClustersResponse;
import software.amazon.awssdk.services.emr.model.ListInstanceFleetsRequest;
import software.amazon.awssdk.services.emr.model.ListInstanceFleetsResponse;
import software.amazon.awssdk.services.emr.model.ListInstanceGroupsRequest;
import software.amazon.awssdk.services.emr.model.ListInstanceGroupsResponse;
import software.amazon.awssdk.services.emr.model.ListInstancesRequest;
import software.amazon.awssdk.services.emr.model.ListInstancesResponse;
import software.amazon.awssdk.services.emr.model.ListNotebookExecutionsRequest;
import software.amazon.awssdk.services.emr.model.ListNotebookExecutionsResponse;
import software.amazon.awssdk.services.emr.model.ListReleaseLabelsRequest;
import software.amazon.awssdk.services.emr.model.ListReleaseLabelsResponse;
import software.amazon.awssdk.services.emr.model.ListSecurityConfigurationsRequest;
import software.amazon.awssdk.services.emr.model.ListSecurityConfigurationsResponse;
import software.amazon.awssdk.services.emr.model.ListStepsRequest;
import software.amazon.awssdk.services.emr.model.ListStepsResponse;
import software.amazon.awssdk.services.emr.model.ListStudioSessionMappingsRequest;
import software.amazon.awssdk.services.emr.model.ListStudioSessionMappingsResponse;
import software.amazon.awssdk.services.emr.model.ListStudiosRequest;
import software.amazon.awssdk.services.emr.model.ListStudiosResponse;
import software.amazon.awssdk.services.emr.model.ListSupportedInstanceTypesRequest;
import software.amazon.awssdk.services.emr.model.ListSupportedInstanceTypesResponse;
import software.amazon.awssdk.services.emr.model.ModifyClusterRequest;
import software.amazon.awssdk.services.emr.model.ModifyClusterResponse;
import software.amazon.awssdk.services.emr.model.ModifyInstanceFleetRequest;
import software.amazon.awssdk.services.emr.model.ModifyInstanceFleetResponse;
import software.amazon.awssdk.services.emr.model.ModifyInstanceGroupsRequest;
import software.amazon.awssdk.services.emr.model.ModifyInstanceGroupsResponse;
import software.amazon.awssdk.services.emr.model.PutAutoScalingPolicyRequest;
import software.amazon.awssdk.services.emr.model.PutAutoScalingPolicyResponse;
import software.amazon.awssdk.services.emr.model.PutAutoTerminationPolicyRequest;
import software.amazon.awssdk.services.emr.model.PutAutoTerminationPolicyResponse;
import software.amazon.awssdk.services.emr.model.PutBlockPublicAccessConfigurationRequest;
import software.amazon.awssdk.services.emr.model.PutBlockPublicAccessConfigurationResponse;
import software.amazon.awssdk.services.emr.model.PutManagedScalingPolicyRequest;
import software.amazon.awssdk.services.emr.model.PutManagedScalingPolicyResponse;
import software.amazon.awssdk.services.emr.model.RemoveAutoScalingPolicyRequest;
import software.amazon.awssdk.services.emr.model.RemoveAutoScalingPolicyResponse;
import software.amazon.awssdk.services.emr.model.RemoveAutoTerminationPolicyRequest;
import software.amazon.awssdk.services.emr.model.RemoveAutoTerminationPolicyResponse;
import software.amazon.awssdk.services.emr.model.RemoveManagedScalingPolicyRequest;
import software.amazon.awssdk.services.emr.model.RemoveManagedScalingPolicyResponse;
import software.amazon.awssdk.services.emr.model.RemoveTagsRequest;
import software.amazon.awssdk.services.emr.model.RemoveTagsResponse;
import software.amazon.awssdk.services.emr.model.RunJobFlowRequest;
import software.amazon.awssdk.services.emr.model.RunJobFlowResponse;
import software.amazon.awssdk.services.emr.model.SetKeepJobFlowAliveWhenNoStepsRequest;
import software.amazon.awssdk.services.emr.model.SetKeepJobFlowAliveWhenNoStepsResponse;
import software.amazon.awssdk.services.emr.model.SetTerminationProtectionRequest;
import software.amazon.awssdk.services.emr.model.SetTerminationProtectionResponse;
import software.amazon.awssdk.services.emr.model.SetUnhealthyNodeReplacementRequest;
import software.amazon.awssdk.services.emr.model.SetUnhealthyNodeReplacementResponse;
import software.amazon.awssdk.services.emr.model.SetVisibleToAllUsersRequest;
import software.amazon.awssdk.services.emr.model.SetVisibleToAllUsersResponse;
import software.amazon.awssdk.services.emr.model.StartNotebookExecutionRequest;
import software.amazon.awssdk.services.emr.model.StartNotebookExecutionResponse;
import software.amazon.awssdk.services.emr.model.StopNotebookExecutionRequest;
import software.amazon.awssdk.services.emr.model.StopNotebookExecutionResponse;
import software.amazon.awssdk.services.emr.model.TerminateJobFlowsRequest;
import software.amazon.awssdk.services.emr.model.TerminateJobFlowsResponse;
import software.amazon.awssdk.services.emr.model.UpdateStudioRequest;
import software.amazon.awssdk.services.emr.model.UpdateStudioResponse;
import software.amazon.awssdk.services.emr.model.UpdateStudioSessionMappingRequest;
import software.amazon.awssdk.services.emr.model.UpdateStudioSessionMappingResponse;
import software.amazon.awssdk.services.emr.transform.AddInstanceFleetRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.AddInstanceGroupsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.AddJobFlowStepsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.AddTagsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.CancelStepsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.CreateSecurityConfigurationRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.CreateStudioRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.CreateStudioSessionMappingRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.DeleteSecurityConfigurationRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.DeleteStudioRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.DeleteStudioSessionMappingRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.DescribeClusterRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.DescribeNotebookExecutionRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.DescribeReleaseLabelRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.DescribeSecurityConfigurationRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.DescribeStepRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.DescribeStudioRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.GetAutoTerminationPolicyRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.GetBlockPublicAccessConfigurationRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.GetClusterSessionCredentialsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.GetManagedScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.GetStudioSessionMappingRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListBootstrapActionsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListClustersRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListInstanceFleetsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListInstanceGroupsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListInstancesRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListNotebookExecutionsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListReleaseLabelsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListSecurityConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListStepsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListStudioSessionMappingsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListStudiosRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ListSupportedInstanceTypesRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ModifyClusterRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ModifyInstanceFleetRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.ModifyInstanceGroupsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.PutAutoScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.PutAutoTerminationPolicyRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.PutBlockPublicAccessConfigurationRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.PutManagedScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.RemoveAutoScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.RemoveAutoTerminationPolicyRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.RemoveManagedScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.RemoveTagsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.RunJobFlowRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.SetKeepJobFlowAliveWhenNoStepsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.SetTerminationProtectionRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.SetUnhealthyNodeReplacementRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.SetVisibleToAllUsersRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.StartNotebookExecutionRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.StopNotebookExecutionRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.TerminateJobFlowsRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.UpdateStudioRequestMarshaller;
import software.amazon.awssdk.services.emr.transform.UpdateStudioSessionMappingRequestMarshaller;
import software.amazon.awssdk.services.emr.waiters.EmrWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link EmrClient}.
 *
 * @see EmrClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEmrClient implements EmrClient {
    private static final Logger log = Logger.loggerFor(DefaultEmrClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEmrClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Adds an instance fleet to a running cluster.
     * </p>
     * <note>
     * <p>
     * The instance fleet configuration is available only in Amazon EMR releases 4.8.0 and later, excluding 5.0.x.
     * </p>
     * </note>
     *
     * @param addInstanceFleetRequest
     * @return Result of the AddInstanceFleet operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.AddInstanceFleet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/AddInstanceFleet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AddInstanceFleetResponse addInstanceFleet(AddInstanceFleetRequest addInstanceFleetRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddInstanceFleetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddInstanceFleetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addInstanceFleetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addInstanceFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddInstanceFleet");

            return clientHandler.execute(new ClientExecutionParams<AddInstanceFleetRequest, AddInstanceFleetResponse>()
                    .withOperationName("AddInstanceFleet").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(addInstanceFleetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AddInstanceFleetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more instance groups to a running cluster.
     * </p>
     *
     * @param addInstanceGroupsRequest
     *        Input to an AddInstanceGroups call.
     * @return Result of the AddInstanceGroups operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.AddInstanceGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/AddInstanceGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AddInstanceGroupsResponse addInstanceGroups(AddInstanceGroupsRequest addInstanceGroupsRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddInstanceGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddInstanceGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addInstanceGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addInstanceGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddInstanceGroups");

            return clientHandler.execute(new ClientExecutionParams<AddInstanceGroupsRequest, AddInstanceGroupsResponse>()
                    .withOperationName("AddInstanceGroups").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(addInstanceGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AddInstanceGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * AddJobFlowSteps adds new steps to a running cluster. A maximum of 256 steps are allowed in each job flow.
     * </p>
     * <p>
     * If your cluster is long-running (such as a Hive data warehouse) or complex, you may require more than 256 steps
     * to process your data. You can bypass the 256-step limitation in various ways, including using SSH to connect to
     * the master node and submitting queries directly to the software running on the master node, such as Hive and
     * Hadoop.
     * </p>
     * <p>
     * A step specifies the location of a JAR file stored either on the master node of the cluster or in Amazon S3. Each
     * step is performed by the main function of the main class of the JAR file. The main class can be specified either
     * in the manifest of the JAR or by using the MainFunction parameter of the step.
     * </p>
     * <p>
     * Amazon EMR executes each step in the order listed. For a step to be considered complete, the main function must
     * exit with a zero exit code and all Hadoop jobs started while the step was running must have completed and run
     * successfully.
     * </p>
     * <p>
     * You can only add steps to a cluster that is in one of the following states: STARTING, BOOTSTRAPPING, RUNNING, or
     * WAITING.
     * </p>
     * <note>
     * <p>
     * The string values passed into <code>HadoopJarStep</code> object cannot exceed a total of 10240 characters.
     * </p>
     * </note>
     *
     * @param addJobFlowStepsRequest
     *        The input argument to the <a>AddJobFlowSteps</a> operation.
     * @return Result of the AddJobFlowSteps operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.AddJobFlowSteps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/AddJobFlowSteps"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AddJobFlowStepsResponse addJobFlowSteps(AddJobFlowStepsRequest addJobFlowStepsRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddJobFlowStepsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddJobFlowStepsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addJobFlowStepsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addJobFlowStepsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddJobFlowSteps");

            return clientHandler.execute(new ClientExecutionParams<AddJobFlowStepsRequest, AddJobFlowStepsResponse>()
                    .withOperationName("AddJobFlowSteps").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(addJobFlowStepsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AddJobFlowStepsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds tags to an Amazon EMR resource, such as a cluster or an Amazon EMR Studio. Tags make it easier to associate
     * resources in various ways, such as grouping clusters to track your Amazon EMR resource allocation costs. For more
     * information, see <a href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-tags.html">Tag
     * Clusters</a>.
     * </p>
     *
     * @param addTagsRequest
     *        This input identifies an Amazon EMR resource and a list of tags to attach.
     * @return Result of the AddTags operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.AddTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/AddTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AddTagsResponse addTags(AddTagsRequest addTagsRequest) throws InternalServerException, InvalidRequestException,
            AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddTags");

            return clientHandler.execute(new ClientExecutionParams<AddTagsRequest, AddTagsResponse>()
                    .withOperationName("AddTags").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(addTagsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AddTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Cancels a pending step or steps in a running cluster. Available only in Amazon EMR versions 4.8.0 and later,
     * excluding version 5.0.0. A maximum of 256 steps are allowed in each CancelSteps request. CancelSteps is
     * idempotent but asynchronous; it does not guarantee that a step will be canceled, even if the request is
     * successfully submitted. When you use Amazon EMR releases 5.28.0 and later, you can cancel steps that are in a
     * <code>PENDING</code> or <code>RUNNING</code> state. In earlier versions of Amazon EMR, you can only cancel steps
     * that are in a <code>PENDING</code> state.
     * </p>
     *
     * @param cancelStepsRequest
     *        The input argument to the <a>CancelSteps</a> operation.
     * @return Result of the CancelSteps operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.CancelSteps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/CancelSteps" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CancelStepsResponse cancelSteps(CancelStepsRequest cancelStepsRequest) throws InternalServerErrorException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelStepsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelStepsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelStepsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelStepsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelSteps");

            return clientHandler.execute(new ClientExecutionParams<CancelStepsRequest, CancelStepsResponse>()
                    .withOperationName("CancelSteps").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(cancelStepsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelStepsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a security configuration, which is stored in the service and can be specified when a cluster is created.
     * </p>
     *
     * @param createSecurityConfigurationRequest
     * @return Result of the CreateSecurityConfiguration operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.CreateSecurityConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/CreateSecurityConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateSecurityConfigurationResponse createSecurityConfiguration(
            CreateSecurityConfigurationRequest createSecurityConfigurationRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSecurityConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSecurityConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSecurityConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSecurityConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSecurityConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateSecurityConfigurationRequest, CreateSecurityConfigurationResponse>()
                            .withOperationName("CreateSecurityConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createSecurityConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateSecurityConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Amazon EMR Studio.
     * </p>
     *
     * @param createStudioRequest
     * @return Result of the CreateStudio operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.CreateStudio
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/CreateStudio" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateStudioResponse createStudio(CreateStudioRequest createStudioRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateStudioResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateStudioResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStudioRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStudioRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStudio");

            return clientHandler.execute(new ClientExecutionParams<CreateStudioRequest, CreateStudioResponse>()
                    .withOperationName("CreateStudio").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createStudioRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateStudioRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Maps a user or group to the Amazon EMR Studio specified by <code>StudioId</code>, and applies a session policy to
     * refine Studio permissions for that user or group. Use <code>CreateStudioSessionMapping</code> to assign users to
     * a Studio when you use IAM Identity Center authentication. For instructions on how to assign users to a Studio
     * when you use IAM authentication, see <a href=
     * "https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio-manage-users.html#emr-studio-assign-users-groups"
     * >Assign a user or group to your EMR Studio</a>.
     * </p>
     *
     * @param createStudioSessionMappingRequest
     * @return Result of the CreateStudioSessionMapping operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.CreateStudioSessionMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/CreateStudioSessionMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateStudioSessionMappingResponse createStudioSessionMapping(
            CreateStudioSessionMappingRequest createStudioSessionMappingRequest) throws InternalServerErrorException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateStudioSessionMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateStudioSessionMappingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStudioSessionMappingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStudioSessionMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStudioSessionMapping");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateStudioSessionMappingRequest, CreateStudioSessionMappingResponse>()
                            .withOperationName("CreateStudioSessionMapping").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createStudioSessionMappingRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateStudioSessionMappingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a security configuration.
     * </p>
     *
     * @param deleteSecurityConfigurationRequest
     * @return Result of the DeleteSecurityConfiguration operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.DeleteSecurityConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/DeleteSecurityConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteSecurityConfigurationResponse deleteSecurityConfiguration(
            DeleteSecurityConfigurationRequest deleteSecurityConfigurationRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSecurityConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSecurityConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSecurityConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSecurityConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSecurityConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteSecurityConfigurationRequest, DeleteSecurityConfigurationResponse>()
                            .withOperationName("DeleteSecurityConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteSecurityConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteSecurityConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes an Amazon EMR Studio from the Studio metadata store.
     * </p>
     *
     * @param deleteStudioRequest
     * @return Result of the DeleteStudio operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.DeleteStudio
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/DeleteStudio" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteStudioResponse deleteStudio(DeleteStudioRequest deleteStudioRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteStudioResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteStudioResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStudioRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStudioRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStudio");

            return clientHandler.execute(new ClientExecutionParams<DeleteStudioRequest, DeleteStudioResponse>()
                    .withOperationName("DeleteStudio").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteStudioRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteStudioRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a user or group from an Amazon EMR Studio.
     * </p>
     *
     * @param deleteStudioSessionMappingRequest
     * @return Result of the DeleteStudioSessionMapping operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.DeleteStudioSessionMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/DeleteStudioSessionMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteStudioSessionMappingResponse deleteStudioSessionMapping(
            DeleteStudioSessionMappingRequest deleteStudioSessionMappingRequest) throws InternalServerErrorException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteStudioSessionMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteStudioSessionMappingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStudioSessionMappingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStudioSessionMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStudioSessionMapping");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteStudioSessionMappingRequest, DeleteStudioSessionMappingResponse>()
                            .withOperationName("DeleteStudioSessionMapping").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteStudioSessionMappingRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteStudioSessionMappingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides cluster-level details including status, hardware and software configuration, VPC settings, and so on.
     * </p>
     *
     * @param describeClusterRequest
     *        This input determines which cluster to describe.
     * @return Result of the DescribeCluster operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.DescribeCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/DescribeCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeClusterResponse describeCluster(DescribeClusterRequest describeClusterRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCluster");

            return clientHandler.execute(new ClientExecutionParams<DescribeClusterRequest, DescribeClusterResponse>()
                    .withOperationName("DescribeCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides details of a notebook execution.
     * </p>
     *
     * @param describeNotebookExecutionRequest
     * @return Result of the DescribeNotebookExecution operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.DescribeNotebookExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/DescribeNotebookExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeNotebookExecutionResponse describeNotebookExecution(
            DescribeNotebookExecutionRequest describeNotebookExecutionRequest) throws InternalServerErrorException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeNotebookExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeNotebookExecutionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeNotebookExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeNotebookExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNotebookExecution");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeNotebookExecutionRequest, DescribeNotebookExecutionResponse>()
                            .withOperationName("DescribeNotebookExecution").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeNotebookExecutionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeNotebookExecutionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides Amazon EMR release label details, such as the releases available the Region where the API request is
     * run, and the available applications for a specific Amazon EMR release label. Can also list Amazon EMR releases
     * that support a specified version of Spark.
     * </p>
     *
     * @param describeReleaseLabelRequest
     * @return Result of the DescribeReleaseLabel operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.DescribeReleaseLabel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/DescribeReleaseLabel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeReleaseLabelResponse describeReleaseLabel(DescribeReleaseLabelRequest describeReleaseLabelRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeReleaseLabelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeReleaseLabelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeReleaseLabelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeReleaseLabelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeReleaseLabel");

            return clientHandler.execute(new ClientExecutionParams<DescribeReleaseLabelRequest, DescribeReleaseLabelResponse>()
                    .withOperationName("DescribeReleaseLabel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeReleaseLabelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeReleaseLabelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides the details of a security configuration by returning the configuration JSON.
     * </p>
     *
     * @param describeSecurityConfigurationRequest
     * @return Result of the DescribeSecurityConfiguration operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.DescribeSecurityConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/DescribeSecurityConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeSecurityConfigurationResponse describeSecurityConfiguration(
            DescribeSecurityConfigurationRequest describeSecurityConfigurationRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSecurityConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSecurityConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSecurityConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeSecurityConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSecurityConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeSecurityConfigurationRequest, DescribeSecurityConfigurationResponse>()
                            .withOperationName("DescribeSecurityConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeSecurityConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeSecurityConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides more detail about the cluster step.
     * </p>
     *
     * @param describeStepRequest
     *        This input determines which step to describe.
     * @return Result of the DescribeStep operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.DescribeStep
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/DescribeStep" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeStepResponse describeStep(DescribeStepRequest describeStepRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStepResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeStepResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeStepRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStepRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStep");

            return clientHandler.execute(new ClientExecutionParams<DescribeStepRequest, DescribeStepResponse>()
                    .withOperationName("DescribeStep").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeStepRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeStepRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details for the specified Amazon EMR Studio including ID, Name, VPC, Studio access URL, and so on.
     * </p>
     *
     * @param describeStudioRequest
     * @return Result of the DescribeStudio operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.DescribeStudio
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/DescribeStudio"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeStudioResponse describeStudio(DescribeStudioRequest describeStudioRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStudioResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeStudioResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeStudioRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStudioRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStudio");

            return clientHandler.execute(new ClientExecutionParams<DescribeStudioRequest, DescribeStudioResponse>()
                    .withOperationName("DescribeStudio").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeStudioRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeStudioRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the auto-termination policy for an Amazon EMR cluster.
     * </p>
     *
     * @param getAutoTerminationPolicyRequest
     * @return Result of the GetAutoTerminationPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.GetAutoTerminationPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/GetAutoTerminationPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAutoTerminationPolicyResponse getAutoTerminationPolicy(
            GetAutoTerminationPolicyRequest getAutoTerminationPolicyRequest) throws AwsServiceException, SdkClientException,
            EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAutoTerminationPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAutoTerminationPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAutoTerminationPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAutoTerminationPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAutoTerminationPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<GetAutoTerminationPolicyRequest, GetAutoTerminationPolicyResponse>()
                            .withOperationName("GetAutoTerminationPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getAutoTerminationPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetAutoTerminationPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the Amazon EMR block public access configuration for your Amazon Web Services account in the current
     * Region. For more information see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/configure-block-public-access.html">Configure Block
     * Public Access for Amazon EMR</a> in the <i>Amazon EMR Management Guide</i>.
     * </p>
     *
     * @param getBlockPublicAccessConfigurationRequest
     * @return Result of the GetBlockPublicAccessConfiguration operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.GetBlockPublicAccessConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/GetBlockPublicAccessConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetBlockPublicAccessConfigurationResponse getBlockPublicAccessConfiguration(
            GetBlockPublicAccessConfigurationRequest getBlockPublicAccessConfigurationRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBlockPublicAccessConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBlockPublicAccessConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBlockPublicAccessConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getBlockPublicAccessConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBlockPublicAccessConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetBlockPublicAccessConfigurationRequest, GetBlockPublicAccessConfigurationResponse>()
                            .withOperationName("GetBlockPublicAccessConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getBlockPublicAccessConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetBlockPublicAccessConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides temporary, HTTP basic credentials that are associated with a given runtime IAM role and used by a
     * cluster with fine-grained access control activated. You can use these credentials to connect to cluster endpoints
     * that support username and password authentication.
     * </p>
     *
     * @param getClusterSessionCredentialsRequest
     * @return Result of the GetClusterSessionCredentials operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.GetClusterSessionCredentials
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/GetClusterSessionCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetClusterSessionCredentialsResponse getClusterSessionCredentials(
            GetClusterSessionCredentialsRequest getClusterSessionCredentialsRequest) throws InternalServerErrorException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetClusterSessionCredentialsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetClusterSessionCredentialsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getClusterSessionCredentialsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getClusterSessionCredentialsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetClusterSessionCredentials");

            return clientHandler
                    .execute(new ClientExecutionParams<GetClusterSessionCredentialsRequest, GetClusterSessionCredentialsResponse>()
                            .withOperationName("GetClusterSessionCredentials").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getClusterSessionCredentialsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetClusterSessionCredentialsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Fetches the attached managed scaling policy for an Amazon EMR cluster.
     * </p>
     *
     * @param getManagedScalingPolicyRequest
     * @return Result of the GetManagedScalingPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.GetManagedScalingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/GetManagedScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetManagedScalingPolicyResponse getManagedScalingPolicy(GetManagedScalingPolicyRequest getManagedScalingPolicyRequest)
            throws AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetManagedScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetManagedScalingPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getManagedScalingPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getManagedScalingPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetManagedScalingPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<GetManagedScalingPolicyRequest, GetManagedScalingPolicyResponse>()
                            .withOperationName("GetManagedScalingPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getManagedScalingPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetManagedScalingPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Fetches mapping details for the specified Amazon EMR Studio and identity (user or group).
     * </p>
     *
     * @param getStudioSessionMappingRequest
     * @return Result of the GetStudioSessionMapping operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.GetStudioSessionMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/GetStudioSessionMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetStudioSessionMappingResponse getStudioSessionMapping(GetStudioSessionMappingRequest getStudioSessionMappingRequest)
            throws InternalServerErrorException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetStudioSessionMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetStudioSessionMappingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStudioSessionMappingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStudioSessionMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStudioSessionMapping");

            return clientHandler
                    .execute(new ClientExecutionParams<GetStudioSessionMappingRequest, GetStudioSessionMappingResponse>()
                            .withOperationName("GetStudioSessionMapping").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getStudioSessionMappingRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetStudioSessionMappingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the bootstrap actions associated with a cluster.
     * </p>
     *
     * @param listBootstrapActionsRequest
     *        This input determines which bootstrap actions to retrieve.
     * @return Result of the ListBootstrapActions operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListBootstrapActions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListBootstrapActions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListBootstrapActionsResponse listBootstrapActions(ListBootstrapActionsRequest listBootstrapActionsRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBootstrapActionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListBootstrapActionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBootstrapActionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBootstrapActionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBootstrapActions");

            return clientHandler.execute(new ClientExecutionParams<ListBootstrapActionsRequest, ListBootstrapActionsResponse>()
                    .withOperationName("ListBootstrapActions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listBootstrapActionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBootstrapActionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides the status of all clusters visible to this Amazon Web Services account. Allows you to filter the list of
     * clusters based on certain criteria; for example, filtering by cluster creation date and time or by status. This
     * call returns a maximum of 50 clusters in unsorted order per call, but returns a marker to track the paging of the
     * cluster list across multiple ListClusters calls.
     * </p>
     *
     * @param listClustersRequest
     *        This input determines how the ListClusters action filters the list of clusters that it returns.
     * @return Result of the ListClusters operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListClusters" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListClustersResponse listClusters(ListClustersRequest listClustersRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListClustersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listClustersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListClusters");

            return clientHandler.execute(new ClientExecutionParams<ListClustersRequest, ListClustersResponse>()
                    .withOperationName("ListClusters").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listClustersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListClustersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all available details about the instance fleets in a cluster.
     * </p>
     * <note>
     * <p>
     * The instance fleet configuration is available only in Amazon EMR releases 4.8.0 and later, excluding 5.0.x
     * versions.
     * </p>
     * </note>
     *
     * @param listInstanceFleetsRequest
     * @return Result of the ListInstanceFleets operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListInstanceFleets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListInstanceFleets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInstanceFleetsResponse listInstanceFleets(ListInstanceFleetsRequest listInstanceFleetsRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInstanceFleetsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListInstanceFleetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInstanceFleetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInstanceFleetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInstanceFleets");

            return clientHandler.execute(new ClientExecutionParams<ListInstanceFleetsRequest, ListInstanceFleetsResponse>()
                    .withOperationName("ListInstanceFleets").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listInstanceFleetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInstanceFleetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides all available details about the instance groups in a cluster.
     * </p>
     *
     * @param listInstanceGroupsRequest
     *        This input determines which instance groups to retrieve.
     * @return Result of the ListInstanceGroups operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListInstanceGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListInstanceGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInstanceGroupsResponse listInstanceGroups(ListInstanceGroupsRequest listInstanceGroupsRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInstanceGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListInstanceGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInstanceGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInstanceGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInstanceGroups");

            return clientHandler.execute(new ClientExecutionParams<ListInstanceGroupsRequest, ListInstanceGroupsResponse>()
                    .withOperationName("ListInstanceGroups").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listInstanceGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInstanceGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information for all active Amazon EC2 instances and Amazon EC2 instances terminated in the last 30 days,
     * up to a maximum of 2,000. Amazon EC2 instances in any of the following states are considered active:
     * AWAITING_FULFILLMENT, PROVISIONING, BOOTSTRAPPING, RUNNING.
     * </p>
     *
     * @param listInstancesRequest
     *        This input determines which instances to list.
     * @return Result of the ListInstances operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInstancesResponse listInstances(ListInstancesRequest listInstancesRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInstancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInstancesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInstances");

            return clientHandler.execute(new ClientExecutionParams<ListInstancesRequest, ListInstancesResponse>()
                    .withOperationName("ListInstances").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listInstancesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides summaries of all notebook executions. You can filter the list based on multiple criteria such as status,
     * time range, and editor id. Returns a maximum of 50 notebook executions and a marker to track the paging of a
     * longer notebook execution list across multiple <code>ListNotebookExecutions</code> calls.
     * </p>
     *
     * @param listNotebookExecutionsRequest
     * @return Result of the ListNotebookExecutions operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListNotebookExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListNotebookExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNotebookExecutionsResponse listNotebookExecutions(ListNotebookExecutionsRequest listNotebookExecutionsRequest)
            throws InternalServerErrorException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNotebookExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListNotebookExecutionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNotebookExecutionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNotebookExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNotebookExecutions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListNotebookExecutionsRequest, ListNotebookExecutionsResponse>()
                            .withOperationName("ListNotebookExecutions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listNotebookExecutionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListNotebookExecutionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves release labels of Amazon EMR services in the Region where the API is called.
     * </p>
     *
     * @param listReleaseLabelsRequest
     * @return Result of the ListReleaseLabels operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListReleaseLabels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListReleaseLabels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReleaseLabelsResponse listReleaseLabels(ListReleaseLabelsRequest listReleaseLabelsRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListReleaseLabelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListReleaseLabelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listReleaseLabelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReleaseLabelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReleaseLabels");

            return clientHandler.execute(new ClientExecutionParams<ListReleaseLabelsRequest, ListReleaseLabelsResponse>()
                    .withOperationName("ListReleaseLabels").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listReleaseLabelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListReleaseLabelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the security configurations visible to this account, providing their creation dates and times, and
     * their names. This call returns a maximum of 50 clusters per call, but returns a marker to track the paging of the
     * cluster list across multiple ListSecurityConfigurations calls.
     * </p>
     *
     * @param listSecurityConfigurationsRequest
     * @return Result of the ListSecurityConfigurations operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListSecurityConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListSecurityConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSecurityConfigurationsResponse listSecurityConfigurations(
            ListSecurityConfigurationsRequest listSecurityConfigurationsRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSecurityConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSecurityConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSecurityConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSecurityConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSecurityConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListSecurityConfigurationsRequest, ListSecurityConfigurationsResponse>()
                            .withOperationName("ListSecurityConfigurations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listSecurityConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListSecurityConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of steps for the cluster in reverse order unless you specify <code>stepIds</code> with the
     * request or filter by <code>StepStates</code>. You can specify a maximum of 10 <code>stepIDs</code>. The CLI
     * automatically paginates results to return a list greater than 50 steps. To return more than 50 steps using the
     * CLI, specify a <code>Marker</code>, which is a pagination token that indicates the next set of steps to retrieve.
     * </p>
     *
     * @param listStepsRequest
     *        This input determines which steps to list.
     * @return Result of the ListSteps operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListSteps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListSteps" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListStepsResponse listSteps(ListStepsRequest listStepsRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStepsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStepsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStepsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStepsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSteps");

            return clientHandler.execute(new ClientExecutionParams<ListStepsRequest, ListStepsResponse>()
                    .withOperationName("ListSteps").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listStepsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStepsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all user or group session mappings for the Amazon EMR Studio specified by <code>StudioId</code>
     * .
     * </p>
     *
     * @param listStudioSessionMappingsRequest
     * @return Result of the ListStudioSessionMappings operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListStudioSessionMappings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListStudioSessionMappings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListStudioSessionMappingsResponse listStudioSessionMappings(
            ListStudioSessionMappingsRequest listStudioSessionMappingsRequest) throws InternalServerErrorException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStudioSessionMappingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListStudioSessionMappingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStudioSessionMappingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStudioSessionMappingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStudioSessionMappings");

            return clientHandler
                    .execute(new ClientExecutionParams<ListStudioSessionMappingsRequest, ListStudioSessionMappingsResponse>()
                            .withOperationName("ListStudioSessionMappings").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listStudioSessionMappingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListStudioSessionMappingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all Amazon EMR Studios associated with the Amazon Web Services account. The list includes
     * details such as ID, Studio Access URL, and creation time for each Studio.
     * </p>
     *
     * @param listStudiosRequest
     * @return Result of the ListStudios operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListStudios
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListStudios" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListStudiosResponse listStudios(ListStudiosRequest listStudiosRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStudiosResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStudiosResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStudiosRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStudiosRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStudios");

            return clientHandler.execute(new ClientExecutionParams<ListStudiosRequest, ListStudiosResponse>()
                    .withOperationName("ListStudios").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listStudiosRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStudiosRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A list of the instance types that Amazon EMR supports. You can filter the list by Amazon Web Services Region and
     * Amazon EMR release.
     * </p>
     *
     * @param listSupportedInstanceTypesRequest
     * @return Result of the ListSupportedInstanceTypes operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ListSupportedInstanceTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ListSupportedInstanceTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSupportedInstanceTypesResponse listSupportedInstanceTypes(
            ListSupportedInstanceTypesRequest listSupportedInstanceTypesRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSupportedInstanceTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSupportedInstanceTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSupportedInstanceTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSupportedInstanceTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSupportedInstanceTypes");

            return clientHandler
                    .execute(new ClientExecutionParams<ListSupportedInstanceTypesRequest, ListSupportedInstanceTypesResponse>()
                            .withOperationName("ListSupportedInstanceTypes").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listSupportedInstanceTypesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListSupportedInstanceTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies the number of steps that can be executed concurrently for the cluster specified using ClusterID.
     * </p>
     *
     * @param modifyClusterRequest
     * @return Result of the ModifyCluster operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ModifyCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ModifyCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ModifyClusterResponse modifyCluster(ModifyClusterRequest modifyClusterRequest) throws InternalServerErrorException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ModifyClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(modifyClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyCluster");

            return clientHandler.execute(new ClientExecutionParams<ModifyClusterRequest, ModifyClusterResponse>()
                    .withOperationName("ModifyCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(modifyClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ModifyClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies the target On-Demand and target Spot capacities for the instance fleet with the specified
     * InstanceFleetID within the cluster specified using ClusterID. The call either succeeds or fails atomically.
     * </p>
     * <note>
     * <p>
     * The instance fleet configuration is available only in Amazon EMR releases 4.8.0 and later, excluding 5.0.x
     * versions.
     * </p>
     * </note>
     *
     * @param modifyInstanceFleetRequest
     * @return Result of the ModifyInstanceFleet operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ModifyInstanceFleet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ModifyInstanceFleet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ModifyInstanceFleetResponse modifyInstanceFleet(ModifyInstanceFleetRequest modifyInstanceFleetRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyInstanceFleetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ModifyInstanceFleetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(modifyInstanceFleetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyInstanceFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyInstanceFleet");

            return clientHandler.execute(new ClientExecutionParams<ModifyInstanceFleetRequest, ModifyInstanceFleetResponse>()
                    .withOperationName("ModifyInstanceFleet").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(modifyInstanceFleetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ModifyInstanceFleetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * ModifyInstanceGroups modifies the number of nodes and configuration settings of an instance group. The input
     * parameters include the new target instance count for the group and the instance group ID. The call will either
     * succeed or fail atomically.
     * </p>
     *
     * @param modifyInstanceGroupsRequest
     *        Change the size of some instance groups.
     * @return Result of the ModifyInstanceGroups operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.ModifyInstanceGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ModifyInstanceGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ModifyInstanceGroupsResponse modifyInstanceGroups(ModifyInstanceGroupsRequest modifyInstanceGroupsRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyInstanceGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ModifyInstanceGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(modifyInstanceGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyInstanceGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyInstanceGroups");

            return clientHandler.execute(new ClientExecutionParams<ModifyInstanceGroupsRequest, ModifyInstanceGroupsResponse>()
                    .withOperationName("ModifyInstanceGroups").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(modifyInstanceGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ModifyInstanceGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates an automatic scaling policy for a core instance group or task instance group in an Amazon EMR
     * cluster. The automatic scaling policy defines how an instance group dynamically adds and terminates Amazon EC2
     * instances in response to the value of a CloudWatch metric.
     * </p>
     *
     * @param putAutoScalingPolicyRequest
     * @return Result of the PutAutoScalingPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.PutAutoScalingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/PutAutoScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutAutoScalingPolicyResponse putAutoScalingPolicy(PutAutoScalingPolicyRequest putAutoScalingPolicyRequest)
            throws AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAutoScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutAutoScalingPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAutoScalingPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAutoScalingPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAutoScalingPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutAutoScalingPolicyRequest, PutAutoScalingPolicyResponse>()
                    .withOperationName("PutAutoScalingPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putAutoScalingPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutAutoScalingPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * Auto-termination is supported in Amazon EMR releases 5.30.0 and 6.1.0 and later. For more information, see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-auto-termination-policy.html">Using an
     * auto-termination policy</a>.
     * </p>
     * </note>
     * <p>
     * Creates or updates an auto-termination policy for an Amazon EMR cluster. An auto-termination policy defines the
     * amount of idle time in seconds after which a cluster automatically terminates. For alternative cluster
     * termination options, see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-termination.html">Control cluster
     * termination</a>.
     * </p>
     *
     * @param putAutoTerminationPolicyRequest
     * @return Result of the PutAutoTerminationPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.PutAutoTerminationPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/PutAutoTerminationPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutAutoTerminationPolicyResponse putAutoTerminationPolicy(
            PutAutoTerminationPolicyRequest putAutoTerminationPolicyRequest) throws AwsServiceException, SdkClientException,
            EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAutoTerminationPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutAutoTerminationPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAutoTerminationPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAutoTerminationPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAutoTerminationPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<PutAutoTerminationPolicyRequest, PutAutoTerminationPolicyResponse>()
                            .withOperationName("PutAutoTerminationPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putAutoTerminationPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutAutoTerminationPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates an Amazon EMR block public access configuration for your Amazon Web Services account in the
     * current Region. For more information see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/configure-block-public-access.html">Configure Block
     * Public Access for Amazon EMR</a> in the <i>Amazon EMR Management Guide</i>.
     * </p>
     *
     * @param putBlockPublicAccessConfigurationRequest
     * @return Result of the PutBlockPublicAccessConfiguration operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.PutBlockPublicAccessConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/PutBlockPublicAccessConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutBlockPublicAccessConfigurationResponse putBlockPublicAccessConfiguration(
            PutBlockPublicAccessConfigurationRequest putBlockPublicAccessConfigurationRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutBlockPublicAccessConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutBlockPublicAccessConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putBlockPublicAccessConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putBlockPublicAccessConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutBlockPublicAccessConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<PutBlockPublicAccessConfigurationRequest, PutBlockPublicAccessConfigurationResponse>()
                            .withOperationName("PutBlockPublicAccessConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putBlockPublicAccessConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutBlockPublicAccessConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates a managed scaling policy for an Amazon EMR cluster. The managed scaling policy defines the
     * limits for resources, such as Amazon EC2 instances that can be added or terminated from a cluster. The policy
     * only applies to the core and task nodes. The master node cannot be scaled after initial configuration.
     * </p>
     *
     * @param putManagedScalingPolicyRequest
     * @return Result of the PutManagedScalingPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.PutManagedScalingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/PutManagedScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutManagedScalingPolicyResponse putManagedScalingPolicy(PutManagedScalingPolicyRequest putManagedScalingPolicyRequest)
            throws AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutManagedScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutManagedScalingPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putManagedScalingPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putManagedScalingPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutManagedScalingPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<PutManagedScalingPolicyRequest, PutManagedScalingPolicyResponse>()
                            .withOperationName("PutManagedScalingPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putManagedScalingPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutManagedScalingPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes an automatic scaling policy from a specified instance group within an Amazon EMR cluster.
     * </p>
     *
     * @param removeAutoScalingPolicyRequest
     * @return Result of the RemoveAutoScalingPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.RemoveAutoScalingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/RemoveAutoScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveAutoScalingPolicyResponse removeAutoScalingPolicy(RemoveAutoScalingPolicyRequest removeAutoScalingPolicyRequest)
            throws AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveAutoScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveAutoScalingPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removeAutoScalingPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeAutoScalingPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveAutoScalingPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveAutoScalingPolicyRequest, RemoveAutoScalingPolicyResponse>()
                            .withOperationName("RemoveAutoScalingPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(removeAutoScalingPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveAutoScalingPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes an auto-termination policy from an Amazon EMR cluster.
     * </p>
     *
     * @param removeAutoTerminationPolicyRequest
     * @return Result of the RemoveAutoTerminationPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.RemoveAutoTerminationPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/RemoveAutoTerminationPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveAutoTerminationPolicyResponse removeAutoTerminationPolicy(
            RemoveAutoTerminationPolicyRequest removeAutoTerminationPolicyRequest) throws AwsServiceException,
            SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveAutoTerminationPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveAutoTerminationPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removeAutoTerminationPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeAutoTerminationPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveAutoTerminationPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveAutoTerminationPolicyRequest, RemoveAutoTerminationPolicyResponse>()
                            .withOperationName("RemoveAutoTerminationPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(removeAutoTerminationPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveAutoTerminationPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a managed scaling policy from a specified Amazon EMR cluster.
     * </p>
     *
     * @param removeManagedScalingPolicyRequest
     * @return Result of the RemoveManagedScalingPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.RemoveManagedScalingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/RemoveManagedScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveManagedScalingPolicyResponse removeManagedScalingPolicy(
            RemoveManagedScalingPolicyRequest removeManagedScalingPolicyRequest) throws AwsServiceException, SdkClientException,
            EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveManagedScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveManagedScalingPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removeManagedScalingPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeManagedScalingPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveManagedScalingPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveManagedScalingPolicyRequest, RemoveManagedScalingPolicyResponse>()
                            .withOperationName("RemoveManagedScalingPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(removeManagedScalingPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveManagedScalingPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from an Amazon EMR resource, such as a cluster or Amazon EMR Studio. Tags make it easier to
     * associate resources in various ways, such as grouping clusters to track your Amazon EMR resource allocation
     * costs. For more information, see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-tags.html">Tag Clusters</a>.
     * </p>
     * <p>
     * The following example removes the stack tag with value Prod from a cluster:
     * </p>
     *
     * @param removeTagsRequest
     *        This input identifies an Amazon EMR resource and a list of tags to remove.
     * @return Result of the RemoveTags operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.RemoveTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/RemoveTags" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RemoveTagsResponse removeTags(RemoveTagsRequest removeTagsRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RemoveTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removeTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveTags");

            return clientHandler.execute(new ClientExecutionParams<RemoveTagsRequest, RemoveTagsResponse>()
                    .withOperationName("RemoveTags").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(removeTagsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RemoveTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * RunJobFlow creates and starts running a new cluster (job flow). The cluster runs the steps specified. After the
     * steps complete, the cluster stops and the HDFS partition is lost. To prevent loss of data, configure the last
     * step of the job flow to store results in Amazon S3. If the <a>JobFlowInstancesConfig</a>
     * <code>KeepJobFlowAliveWhenNoSteps</code> parameter is set to <code>TRUE</code>, the cluster transitions to the
     * WAITING state rather than shutting down after the steps have completed.
     * </p>
     * <p>
     * For additional protection, you can set the <a>JobFlowInstancesConfig</a> <code>TerminationProtected</code>
     * parameter to <code>TRUE</code> to lock the cluster and prevent it from being terminated by API call, user
     * intervention, or in the event of a job flow error.
     * </p>
     * <p>
     * A maximum of 256 steps are allowed in each job flow.
     * </p>
     * <p>
     * If your cluster is long-running (such as a Hive data warehouse) or complex, you may require more than 256 steps
     * to process your data. You can bypass the 256-step limitation in various ways, including using the SSH shell to
     * connect to the master node and submitting queries directly to the software running on the master node, such as
     * Hive and Hadoop.
     * </p>
     * <p>
     * For long-running clusters, we recommend that you periodically store your results.
     * </p>
     * <note>
     * <p>
     * The instance fleets configuration is available only in Amazon EMR releases 4.8.0 and later, excluding 5.0.x
     * versions. The RunJobFlow request can contain InstanceFleets parameters or InstanceGroups parameters, but not
     * both.
     * </p>
     * </note>
     *
     * @param runJobFlowRequest
     *        Input to the <a>RunJobFlow</a> operation.
     * @return Result of the RunJobFlow operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.RunJobFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/RunJobFlow" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RunJobFlowResponse runJobFlow(RunJobFlowRequest runJobFlowRequest) throws InternalServerErrorException,
            AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RunJobFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RunJobFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(runJobFlowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, runJobFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RunJobFlow");

            return clientHandler.execute(new ClientExecutionParams<RunJobFlowRequest, RunJobFlowResponse>()
                    .withOperationName("RunJobFlow").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(runJobFlowRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RunJobFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You can use the <code>SetKeepJobFlowAliveWhenNoSteps</code> to configure a cluster (job flow) to terminate after
     * the step execution, i.e., all your steps are executed. If you want a transient cluster that shuts down after the
     * last of the current executing steps are completed, you can configure <code>SetKeepJobFlowAliveWhenNoSteps</code>
     * to false. If you want a long running cluster, configure <code>SetKeepJobFlowAliveWhenNoSteps</code> to true.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/UsingEMR_TerminationProtection.html">Managing
     * Cluster Termination</a> in the <i>Amazon EMR Management Guide</i>.
     * </p>
     *
     * @param setKeepJobFlowAliveWhenNoStepsRequest
     * @return Result of the SetKeepJobFlowAliveWhenNoSteps operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.SetKeepJobFlowAliveWhenNoSteps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/SetKeepJobFlowAliveWhenNoSteps"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetKeepJobFlowAliveWhenNoStepsResponse setKeepJobFlowAliveWhenNoSteps(
            SetKeepJobFlowAliveWhenNoStepsRequest setKeepJobFlowAliveWhenNoStepsRequest) throws InternalServerErrorException,
            AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetKeepJobFlowAliveWhenNoStepsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetKeepJobFlowAliveWhenNoStepsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setKeepJobFlowAliveWhenNoStepsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                setKeepJobFlowAliveWhenNoStepsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetKeepJobFlowAliveWhenNoSteps");

            return clientHandler
                    .execute(new ClientExecutionParams<SetKeepJobFlowAliveWhenNoStepsRequest, SetKeepJobFlowAliveWhenNoStepsResponse>()
                            .withOperationName("SetKeepJobFlowAliveWhenNoSteps").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(setKeepJobFlowAliveWhenNoStepsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SetKeepJobFlowAliveWhenNoStepsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * SetTerminationProtection locks a cluster (job flow) so the Amazon EC2 instances in the cluster cannot be
     * terminated by user intervention, an API call, or in the event of a job-flow error. The cluster still terminates
     * upon successful completion of the job flow. Calling <code>SetTerminationProtection</code> on a cluster is similar
     * to calling the Amazon EC2 <code>DisableAPITermination</code> API on all Amazon EC2 instances in a cluster.
     * </p>
     * <p>
     * <code>SetTerminationProtection</code> is used to prevent accidental termination of a cluster and to ensure that
     * in the event of an error, the instances persist so that you can recover any data stored in their ephemeral
     * instance storage.
     * </p>
     * <p>
     * To terminate a cluster that has been locked by setting <code>SetTerminationProtection</code> to <code>true</code>
     * , you must first unlock the job flow by a subsequent call to <code>SetTerminationProtection</code> in which you
     * set the value to <code>false</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/UsingEMR_TerminationProtection.html">Managing
     * Cluster Termination</a> in the <i>Amazon EMR Management Guide</i>.
     * </p>
     *
     * @param setTerminationProtectionRequest
     *        The input argument to the <a>TerminationProtection</a> operation.
     * @return Result of the SetTerminationProtection operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.SetTerminationProtection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/SetTerminationProtection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetTerminationProtectionResponse setTerminationProtection(
            SetTerminationProtectionRequest setTerminationProtectionRequest) throws InternalServerErrorException,
            AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetTerminationProtectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetTerminationProtectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setTerminationProtectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setTerminationProtectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetTerminationProtection");

            return clientHandler
                    .execute(new ClientExecutionParams<SetTerminationProtectionRequest, SetTerminationProtectionResponse>()
                            .withOperationName("SetTerminationProtection").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(setTerminationProtectionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SetTerminationProtectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Specify whether to enable unhealthy node replacement, which lets Amazon EMR gracefully replace core nodes on a
     * cluster if any nodes become unhealthy. For example, a node becomes unhealthy if disk usage is above 90%. If
     * unhealthy node replacement is on and <code>TerminationProtected</code> are off, Amazon EMR immediately terminates
     * the unhealthy core nodes. To use unhealthy node replacement and retain unhealthy core nodes, use to turn on
     * termination protection. In such cases, Amazon EMR adds the unhealthy nodes to a denylist, reducing job
     * interruptions and failures.
     * </p>
     * <p>
     * If unhealthy node replacement is on, Amazon EMR notifies YARN and other applications on the cluster to stop
     * scheduling tasks with these nodes, moves the data, and then terminates the nodes.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-node-replacement.html">graceful node
     * replacement</a> in the <i>Amazon EMR Management Guide</i>.
     * </p>
     *
     * @param setUnhealthyNodeReplacementRequest
     * @return Result of the SetUnhealthyNodeReplacement operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.SetUnhealthyNodeReplacement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/SetUnhealthyNodeReplacement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetUnhealthyNodeReplacementResponse setUnhealthyNodeReplacement(
            SetUnhealthyNodeReplacementRequest setUnhealthyNodeReplacementRequest) throws InternalServerErrorException,
            AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetUnhealthyNodeReplacementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetUnhealthyNodeReplacementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setUnhealthyNodeReplacementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setUnhealthyNodeReplacementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetUnhealthyNodeReplacement");

            return clientHandler
                    .execute(new ClientExecutionParams<SetUnhealthyNodeReplacementRequest, SetUnhealthyNodeReplacementResponse>()
                            .withOperationName("SetUnhealthyNodeReplacement").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(setUnhealthyNodeReplacementRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SetUnhealthyNodeReplacementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <important>
     * <p>
     * The SetVisibleToAllUsers parameter is no longer supported. Your cluster may be visible to all users in your
     * account. To restrict cluster access using an IAM policy, see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-IAM.html">Identity and Access
     * Management for Amazon EMR</a>.
     * </p>
     * </important>
     * <p>
     * Sets the <a>Cluster&#36VisibleToAllUsers</a> value for an Amazon EMR cluster. When <code>true</code>, IAM
     * principals in the Amazon Web Services account can perform Amazon EMR cluster actions that their IAM policies
     * allow. When <code>false</code>, only the IAM principal that created the cluster and the Amazon Web Services
     * account root user can perform Amazon EMR actions on the cluster, regardless of IAM permissions policies attached
     * to other IAM principals.
     * </p>
     * <p>
     * This action works on running clusters. When you create a cluster, use the
     * <a>RunJobFlowInput&#36VisibleToAllUsers</a> parameter.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/emr/latest/ManagementGuide/security_IAM_emr-with-IAM.html#security_set_visible_to_all_users"
     * >Understanding the Amazon EMR Cluster VisibleToAllUsers Setting</a> in the <i>Amazon EMR Management Guide</i>.
     * </p>
     *
     * @param setVisibleToAllUsersRequest
     *        The input to the SetVisibleToAllUsers action.
     * @return Result of the SetVisibleToAllUsers operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.SetVisibleToAllUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/SetVisibleToAllUsers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetVisibleToAllUsersResponse setVisibleToAllUsers(SetVisibleToAllUsersRequest setVisibleToAllUsersRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetVisibleToAllUsersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetVisibleToAllUsersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setVisibleToAllUsersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setVisibleToAllUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetVisibleToAllUsers");

            return clientHandler.execute(new ClientExecutionParams<SetVisibleToAllUsersRequest, SetVisibleToAllUsersResponse>()
                    .withOperationName("SetVisibleToAllUsers").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(setVisibleToAllUsersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SetVisibleToAllUsersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a notebook execution.
     * </p>
     *
     * @param startNotebookExecutionRequest
     * @return Result of the StartNotebookExecution operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.StartNotebookExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/StartNotebookExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartNotebookExecutionResponse startNotebookExecution(StartNotebookExecutionRequest startNotebookExecutionRequest)
            throws InternalServerException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartNotebookExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartNotebookExecutionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startNotebookExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startNotebookExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartNotebookExecution");

            return clientHandler
                    .execute(new ClientExecutionParams<StartNotebookExecutionRequest, StartNotebookExecutionResponse>()
                            .withOperationName("StartNotebookExecution").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startNotebookExecutionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartNotebookExecutionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops a notebook execution.
     * </p>
     *
     * @param stopNotebookExecutionRequest
     * @return Result of the StopNotebookExecution operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.StopNotebookExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/StopNotebookExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopNotebookExecutionResponse stopNotebookExecution(StopNotebookExecutionRequest stopNotebookExecutionRequest)
            throws InternalServerErrorException, InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopNotebookExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopNotebookExecutionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopNotebookExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopNotebookExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopNotebookExecution");

            return clientHandler.execute(new ClientExecutionParams<StopNotebookExecutionRequest, StopNotebookExecutionResponse>()
                    .withOperationName("StopNotebookExecution").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(stopNotebookExecutionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopNotebookExecutionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * TerminateJobFlows shuts a list of clusters (job flows) down. When a job flow is shut down, any step not yet
     * completed is canceled and the Amazon EC2 instances on which the cluster is running are stopped. Any log files not
     * already saved are uploaded to Amazon S3 if a LogUri was specified when the cluster was created.
     * </p>
     * <p>
     * The maximum number of clusters allowed is 10. The call to <code>TerminateJobFlows</code> is asynchronous.
     * Depending on the configuration of the cluster, it may take up to 1-5 minutes for the cluster to completely
     * terminate and release allocated resources, such as Amazon EC2 instances.
     * </p>
     *
     * @param terminateJobFlowsRequest
     *        Input to the <a>TerminateJobFlows</a> operation.
     * @return Result of the TerminateJobFlows operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.TerminateJobFlows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/TerminateJobFlows"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TerminateJobFlowsResponse terminateJobFlows(TerminateJobFlowsRequest terminateJobFlowsRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TerminateJobFlowsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TerminateJobFlowsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(terminateJobFlowsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, terminateJobFlowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TerminateJobFlows");

            return clientHandler.execute(new ClientExecutionParams<TerminateJobFlowsRequest, TerminateJobFlowsResponse>()
                    .withOperationName("TerminateJobFlows").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(terminateJobFlowsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TerminateJobFlowsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an Amazon EMR Studio configuration, including attributes such as name, description, and subnets.
     * </p>
     *
     * @param updateStudioRequest
     * @return Result of the UpdateStudio operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon EMR service.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.UpdateStudio
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/UpdateStudio" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateStudioResponse updateStudio(UpdateStudioRequest updateStudioRequest) throws InternalServerException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateStudioResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateStudioResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateStudioRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStudioRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStudio");

            return clientHandler.execute(new ClientExecutionParams<UpdateStudioRequest, UpdateStudioResponse>()
                    .withOperationName("UpdateStudio").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateStudioRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateStudioRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the session policy attached to the user or group for the specified Amazon EMR Studio.
     * </p>
     *
     * @param updateStudioSessionMappingRequest
     * @return Result of the UpdateStudioSessionMapping operation returned by the service.
     * @throws InternalServerErrorException
     *         Indicates that an error occurred while processing the request and that the request was not completed.
     * @throws InvalidRequestException
     *         This exception occurs when there is something wrong with user input.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrClient.UpdateStudioSessionMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/UpdateStudioSessionMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateStudioSessionMappingResponse updateStudioSessionMapping(
            UpdateStudioSessionMappingRequest updateStudioSessionMappingRequest) throws InternalServerErrorException,
            InvalidRequestException, AwsServiceException, SdkClientException, EmrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateStudioSessionMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateStudioSessionMappingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateStudioSessionMappingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStudioSessionMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStudioSessionMapping");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateStudioSessionMappingRequest, UpdateStudioSessionMappingResponse>()
                            .withOperationName("UpdateStudioSessionMapping").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateStudioSessionMappingRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateStudioSessionMappingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link EmrWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link EmrWaiter}
     */
    @Override
    public EmrWaiter waiter() {
        return EmrWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        EmrServiceClientConfigurationBuilder serviceConfigBuilder = new EmrServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EmrException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(400).build());
    }

    @Override
    public final EmrServiceClientConfiguration serviceClientConfiguration() {
        return new EmrServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
