/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The resize specification for On-Demand Instances in the instance fleet, which contains the resize timeout period.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OnDemandResizingSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<OnDemandResizingSpecification.Builder, OnDemandResizingSpecification> {
    private static final SdkField<Integer> TIMEOUT_DURATION_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TimeoutDurationMinutes").getter(getter(OnDemandResizingSpecification::timeoutDurationMinutes))
            .setter(setter(Builder::timeoutDurationMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeoutDurationMinutes").build())
            .build();

    private static final SdkField<String> ALLOCATION_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AllocationStrategy").getter(getter(OnDemandResizingSpecification::allocationStrategyAsString))
            .setter(setter(Builder::allocationStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationStrategy").build())
            .build();

    private static final SdkField<OnDemandCapacityReservationOptions> CAPACITY_RESERVATION_OPTIONS_FIELD = SdkField
            .<OnDemandCapacityReservationOptions> builder(MarshallingType.SDK_POJO)
            .memberName("CapacityReservationOptions")
            .getter(getter(OnDemandResizingSpecification::capacityReservationOptions))
            .setter(setter(Builder::capacityReservationOptions))
            .constructor(OnDemandCapacityReservationOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityReservationOptions").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            TIMEOUT_DURATION_MINUTES_FIELD, ALLOCATION_STRATEGY_FIELD, CAPACITY_RESERVATION_OPTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer timeoutDurationMinutes;

    private final String allocationStrategy;

    private final OnDemandCapacityReservationOptions capacityReservationOptions;

    private OnDemandResizingSpecification(BuilderImpl builder) {
        this.timeoutDurationMinutes = builder.timeoutDurationMinutes;
        this.allocationStrategy = builder.allocationStrategy;
        this.capacityReservationOptions = builder.capacityReservationOptions;
    }

    /**
     * <p>
     * On-Demand resize timeout in minutes. If On-Demand Instances are not provisioned within this time, the resize
     * workflow stops. The minimum value is 5 minutes, and the maximum value is 10,080 minutes (7 days). The timeout
     * applies to all resize workflows on the Instance Fleet. The resize could be triggered by Amazon EMR Managed
     * Scaling or by the customer (via Amazon EMR Console, Amazon EMR CLI modify-instance-fleet or Amazon EMR SDK
     * ModifyInstanceFleet API) or by Amazon EMR due to Amazon EC2 Spot Reclamation.
     * </p>
     * 
     * @return On-Demand resize timeout in minutes. If On-Demand Instances are not provisioned within this time, the
     *         resize workflow stops. The minimum value is 5 minutes, and the maximum value is 10,080 minutes (7 days).
     *         The timeout applies to all resize workflows on the Instance Fleet. The resize could be triggered by
     *         Amazon EMR Managed Scaling or by the customer (via Amazon EMR Console, Amazon EMR CLI
     *         modify-instance-fleet or Amazon EMR SDK ModifyInstanceFleet API) or by Amazon EMR due to Amazon EC2 Spot
     *         Reclamation.
     */
    public final Integer timeoutDurationMinutes() {
        return timeoutDurationMinutes;
    }

    /**
     * <p>
     * Specifies the allocation strategy to use to launch On-Demand instances during a resize. The default is
     * <code>lowest-price</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allocationStrategy} will return {@link OnDemandProvisioningAllocationStrategy#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #allocationStrategyAsString}.
     * </p>
     * 
     * @return Specifies the allocation strategy to use to launch On-Demand instances during a resize. The default is
     *         <code>lowest-price</code>.
     * @see OnDemandProvisioningAllocationStrategy
     */
    public final OnDemandProvisioningAllocationStrategy allocationStrategy() {
        return OnDemandProvisioningAllocationStrategy.fromValue(allocationStrategy);
    }

    /**
     * <p>
     * Specifies the allocation strategy to use to launch On-Demand instances during a resize. The default is
     * <code>lowest-price</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allocationStrategy} will return {@link OnDemandProvisioningAllocationStrategy#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #allocationStrategyAsString}.
     * </p>
     * 
     * @return Specifies the allocation strategy to use to launch On-Demand instances during a resize. The default is
     *         <code>lowest-price</code>.
     * @see OnDemandProvisioningAllocationStrategy
     */
    public final String allocationStrategyAsString() {
        return allocationStrategy;
    }

    /**
     * Returns the value of the CapacityReservationOptions property for this object.
     * 
     * @return The value of the CapacityReservationOptions property for this object.
     */
    public final OnDemandCapacityReservationOptions capacityReservationOptions() {
        return capacityReservationOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(timeoutDurationMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(allocationStrategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(capacityReservationOptions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OnDemandResizingSpecification)) {
            return false;
        }
        OnDemandResizingSpecification other = (OnDemandResizingSpecification) obj;
        return Objects.equals(timeoutDurationMinutes(), other.timeoutDurationMinutes())
                && Objects.equals(allocationStrategyAsString(), other.allocationStrategyAsString())
                && Objects.equals(capacityReservationOptions(), other.capacityReservationOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OnDemandResizingSpecification").add("TimeoutDurationMinutes", timeoutDurationMinutes())
                .add("AllocationStrategy", allocationStrategyAsString())
                .add("CapacityReservationOptions", capacityReservationOptions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TimeoutDurationMinutes":
            return Optional.ofNullable(clazz.cast(timeoutDurationMinutes()));
        case "AllocationStrategy":
            return Optional.ofNullable(clazz.cast(allocationStrategyAsString()));
        case "CapacityReservationOptions":
            return Optional.ofNullable(clazz.cast(capacityReservationOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OnDemandResizingSpecification, T> g) {
        return obj -> g.apply((OnDemandResizingSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OnDemandResizingSpecification> {
        /**
         * <p>
         * On-Demand resize timeout in minutes. If On-Demand Instances are not provisioned within this time, the resize
         * workflow stops. The minimum value is 5 minutes, and the maximum value is 10,080 minutes (7 days). The timeout
         * applies to all resize workflows on the Instance Fleet. The resize could be triggered by Amazon EMR Managed
         * Scaling or by the customer (via Amazon EMR Console, Amazon EMR CLI modify-instance-fleet or Amazon EMR SDK
         * ModifyInstanceFleet API) or by Amazon EMR due to Amazon EC2 Spot Reclamation.
         * </p>
         * 
         * @param timeoutDurationMinutes
         *        On-Demand resize timeout in minutes. If On-Demand Instances are not provisioned within this time, the
         *        resize workflow stops. The minimum value is 5 minutes, and the maximum value is 10,080 minutes (7
         *        days). The timeout applies to all resize workflows on the Instance Fleet. The resize could be
         *        triggered by Amazon EMR Managed Scaling or by the customer (via Amazon EMR Console, Amazon EMR CLI
         *        modify-instance-fleet or Amazon EMR SDK ModifyInstanceFleet API) or by Amazon EMR due to Amazon EC2
         *        Spot Reclamation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutDurationMinutes(Integer timeoutDurationMinutes);

        /**
         * <p>
         * Specifies the allocation strategy to use to launch On-Demand instances during a resize. The default is
         * <code>lowest-price</code>.
         * </p>
         * 
         * @param allocationStrategy
         *        Specifies the allocation strategy to use to launch On-Demand instances during a resize. The default is
         *        <code>lowest-price</code>.
         * @see OnDemandProvisioningAllocationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OnDemandProvisioningAllocationStrategy
         */
        Builder allocationStrategy(String allocationStrategy);

        /**
         * <p>
         * Specifies the allocation strategy to use to launch On-Demand instances during a resize. The default is
         * <code>lowest-price</code>.
         * </p>
         * 
         * @param allocationStrategy
         *        Specifies the allocation strategy to use to launch On-Demand instances during a resize. The default is
         *        <code>lowest-price</code>.
         * @see OnDemandProvisioningAllocationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OnDemandProvisioningAllocationStrategy
         */
        Builder allocationStrategy(OnDemandProvisioningAllocationStrategy allocationStrategy);

        /**
         * Sets the value of the CapacityReservationOptions property for this object.
         *
         * @param capacityReservationOptions
         *        The new value for the CapacityReservationOptions property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityReservationOptions(OnDemandCapacityReservationOptions capacityReservationOptions);

        /**
         * Sets the value of the CapacityReservationOptions property for this object.
         *
         * This is a convenience method that creates an instance of the
         * {@link OnDemandCapacityReservationOptions.Builder} avoiding the need to create one manually via
         * {@link OnDemandCapacityReservationOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OnDemandCapacityReservationOptions.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #capacityReservationOptions(OnDemandCapacityReservationOptions)}.
         * 
         * @param capacityReservationOptions
         *        a consumer that will call methods on {@link OnDemandCapacityReservationOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #capacityReservationOptions(OnDemandCapacityReservationOptions)
         */
        default Builder capacityReservationOptions(Consumer<OnDemandCapacityReservationOptions.Builder> capacityReservationOptions) {
            return capacityReservationOptions(OnDemandCapacityReservationOptions.builder()
                    .applyMutation(capacityReservationOptions).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Integer timeoutDurationMinutes;

        private String allocationStrategy;

        private OnDemandCapacityReservationOptions capacityReservationOptions;

        private BuilderImpl() {
        }

        private BuilderImpl(OnDemandResizingSpecification model) {
            timeoutDurationMinutes(model.timeoutDurationMinutes);
            allocationStrategy(model.allocationStrategy);
            capacityReservationOptions(model.capacityReservationOptions);
        }

        public final Integer getTimeoutDurationMinutes() {
            return timeoutDurationMinutes;
        }

        public final void setTimeoutDurationMinutes(Integer timeoutDurationMinutes) {
            this.timeoutDurationMinutes = timeoutDurationMinutes;
        }

        @Override
        public final Builder timeoutDurationMinutes(Integer timeoutDurationMinutes) {
            this.timeoutDurationMinutes = timeoutDurationMinutes;
            return this;
        }

        public final String getAllocationStrategy() {
            return allocationStrategy;
        }

        public final void setAllocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
        }

        @Override
        public final Builder allocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
            return this;
        }

        @Override
        public final Builder allocationStrategy(OnDemandProvisioningAllocationStrategy allocationStrategy) {
            this.allocationStrategy(allocationStrategy == null ? null : allocationStrategy.toString());
            return this;
        }

        public final OnDemandCapacityReservationOptions.Builder getCapacityReservationOptions() {
            return capacityReservationOptions != null ? capacityReservationOptions.toBuilder() : null;
        }

        public final void setCapacityReservationOptions(OnDemandCapacityReservationOptions.BuilderImpl capacityReservationOptions) {
            this.capacityReservationOptions = capacityReservationOptions != null ? capacityReservationOptions.build() : null;
        }

        @Override
        public final Builder capacityReservationOptions(OnDemandCapacityReservationOptions capacityReservationOptions) {
            this.capacityReservationOptions = capacityReservationOptions;
            return this;
        }

        @Override
        public OnDemandResizingSpecification build() {
            return new OnDemandResizingSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
