/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The resize specification for Spot Instances in the instance fleet, which contains the resize timeout period.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SpotResizingSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<SpotResizingSpecification.Builder, SpotResizingSpecification> {
    private static final SdkField<Integer> TIMEOUT_DURATION_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TimeoutDurationMinutes").getter(getter(SpotResizingSpecification::timeoutDurationMinutes))
            .setter(setter(Builder::timeoutDurationMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeoutDurationMinutes").build())
            .build();

    private static final SdkField<String> ALLOCATION_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AllocationStrategy").getter(getter(SpotResizingSpecification::allocationStrategyAsString))
            .setter(setter(Builder::allocationStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationStrategy").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            TIMEOUT_DURATION_MINUTES_FIELD, ALLOCATION_STRATEGY_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer timeoutDurationMinutes;

    private final String allocationStrategy;

    private SpotResizingSpecification(BuilderImpl builder) {
        this.timeoutDurationMinutes = builder.timeoutDurationMinutes;
        this.allocationStrategy = builder.allocationStrategy;
    }

    /**
     * <p>
     * Spot resize timeout in minutes. If Spot Instances are not provisioned within this time, the resize workflow will
     * stop provisioning of Spot instances. Minimum value is 5 minutes and maximum value is 10,080 minutes (7 days). The
     * timeout applies to all resize workflows on the Instance Fleet. The resize could be triggered by Amazon EMR
     * Managed Scaling or by the customer (via Amazon EMR Console, Amazon EMR CLI modify-instance-fleet or Amazon EMR
     * SDK ModifyInstanceFleet API) or by Amazon EMR due to Amazon EC2 Spot Reclamation.
     * </p>
     * 
     * @return Spot resize timeout in minutes. If Spot Instances are not provisioned within this time, the resize
     *         workflow will stop provisioning of Spot instances. Minimum value is 5 minutes and maximum value is 10,080
     *         minutes (7 days). The timeout applies to all resize workflows on the Instance Fleet. The resize could be
     *         triggered by Amazon EMR Managed Scaling or by the customer (via Amazon EMR Console, Amazon EMR CLI
     *         modify-instance-fleet or Amazon EMR SDK ModifyInstanceFleet API) or by Amazon EMR due to Amazon EC2 Spot
     *         Reclamation.
     */
    public final Integer timeoutDurationMinutes() {
        return timeoutDurationMinutes;
    }

    /**
     * <p>
     * Specifies the allocation strategy to use to launch Spot instances during a resize. If you run Amazon EMR releases
     * 6.9.0 or higher, the default is <code>price-capacity-optimized</code>. If you run Amazon EMR releases 6.8.0 or
     * lower, the default is <code>capacity-optimized</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allocationStrategy} will return {@link SpotProvisioningAllocationStrategy#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #allocationStrategyAsString}.
     * </p>
     * 
     * @return Specifies the allocation strategy to use to launch Spot instances during a resize. If you run Amazon EMR
     *         releases 6.9.0 or higher, the default is <code>price-capacity-optimized</code>. If you run Amazon EMR
     *         releases 6.8.0 or lower, the default is <code>capacity-optimized</code>.
     * @see SpotProvisioningAllocationStrategy
     */
    public final SpotProvisioningAllocationStrategy allocationStrategy() {
        return SpotProvisioningAllocationStrategy.fromValue(allocationStrategy);
    }

    /**
     * <p>
     * Specifies the allocation strategy to use to launch Spot instances during a resize. If you run Amazon EMR releases
     * 6.9.0 or higher, the default is <code>price-capacity-optimized</code>. If you run Amazon EMR releases 6.8.0 or
     * lower, the default is <code>capacity-optimized</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allocationStrategy} will return {@link SpotProvisioningAllocationStrategy#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #allocationStrategyAsString}.
     * </p>
     * 
     * @return Specifies the allocation strategy to use to launch Spot instances during a resize. If you run Amazon EMR
     *         releases 6.9.0 or higher, the default is <code>price-capacity-optimized</code>. If you run Amazon EMR
     *         releases 6.8.0 or lower, the default is <code>capacity-optimized</code>.
     * @see SpotProvisioningAllocationStrategy
     */
    public final String allocationStrategyAsString() {
        return allocationStrategy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(timeoutDurationMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(allocationStrategyAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SpotResizingSpecification)) {
            return false;
        }
        SpotResizingSpecification other = (SpotResizingSpecification) obj;
        return Objects.equals(timeoutDurationMinutes(), other.timeoutDurationMinutes())
                && Objects.equals(allocationStrategyAsString(), other.allocationStrategyAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SpotResizingSpecification").add("TimeoutDurationMinutes", timeoutDurationMinutes())
                .add("AllocationStrategy", allocationStrategyAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TimeoutDurationMinutes":
            return Optional.ofNullable(clazz.cast(timeoutDurationMinutes()));
        case "AllocationStrategy":
            return Optional.ofNullable(clazz.cast(allocationStrategyAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SpotResizingSpecification, T> g) {
        return obj -> g.apply((SpotResizingSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SpotResizingSpecification> {
        /**
         * <p>
         * Spot resize timeout in minutes. If Spot Instances are not provisioned within this time, the resize workflow
         * will stop provisioning of Spot instances. Minimum value is 5 minutes and maximum value is 10,080 minutes (7
         * days). The timeout applies to all resize workflows on the Instance Fleet. The resize could be triggered by
         * Amazon EMR Managed Scaling or by the customer (via Amazon EMR Console, Amazon EMR CLI modify-instance-fleet
         * or Amazon EMR SDK ModifyInstanceFleet API) or by Amazon EMR due to Amazon EC2 Spot Reclamation.
         * </p>
         * 
         * @param timeoutDurationMinutes
         *        Spot resize timeout in minutes. If Spot Instances are not provisioned within this time, the resize
         *        workflow will stop provisioning of Spot instances. Minimum value is 5 minutes and maximum value is
         *        10,080 minutes (7 days). The timeout applies to all resize workflows on the Instance Fleet. The resize
         *        could be triggered by Amazon EMR Managed Scaling or by the customer (via Amazon EMR Console, Amazon
         *        EMR CLI modify-instance-fleet or Amazon EMR SDK ModifyInstanceFleet API) or by Amazon EMR due to
         *        Amazon EC2 Spot Reclamation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutDurationMinutes(Integer timeoutDurationMinutes);

        /**
         * <p>
         * Specifies the allocation strategy to use to launch Spot instances during a resize. If you run Amazon EMR
         * releases 6.9.0 or higher, the default is <code>price-capacity-optimized</code>. If you run Amazon EMR
         * releases 6.8.0 or lower, the default is <code>capacity-optimized</code>.
         * </p>
         * 
         * @param allocationStrategy
         *        Specifies the allocation strategy to use to launch Spot instances during a resize. If you run Amazon
         *        EMR releases 6.9.0 or higher, the default is <code>price-capacity-optimized</code>. If you run Amazon
         *        EMR releases 6.8.0 or lower, the default is <code>capacity-optimized</code>.
         * @see SpotProvisioningAllocationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotProvisioningAllocationStrategy
         */
        Builder allocationStrategy(String allocationStrategy);

        /**
         * <p>
         * Specifies the allocation strategy to use to launch Spot instances during a resize. If you run Amazon EMR
         * releases 6.9.0 or higher, the default is <code>price-capacity-optimized</code>. If you run Amazon EMR
         * releases 6.8.0 or lower, the default is <code>capacity-optimized</code>.
         * </p>
         * 
         * @param allocationStrategy
         *        Specifies the allocation strategy to use to launch Spot instances during a resize. If you run Amazon
         *        EMR releases 6.9.0 or higher, the default is <code>price-capacity-optimized</code>. If you run Amazon
         *        EMR releases 6.8.0 or lower, the default is <code>capacity-optimized</code>.
         * @see SpotProvisioningAllocationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotProvisioningAllocationStrategy
         */
        Builder allocationStrategy(SpotProvisioningAllocationStrategy allocationStrategy);
    }

    static final class BuilderImpl implements Builder {
        private Integer timeoutDurationMinutes;

        private String allocationStrategy;

        private BuilderImpl() {
        }

        private BuilderImpl(SpotResizingSpecification model) {
            timeoutDurationMinutes(model.timeoutDurationMinutes);
            allocationStrategy(model.allocationStrategy);
        }

        public final Integer getTimeoutDurationMinutes() {
            return timeoutDurationMinutes;
        }

        public final void setTimeoutDurationMinutes(Integer timeoutDurationMinutes) {
            this.timeoutDurationMinutes = timeoutDurationMinutes;
        }

        @Override
        public final Builder timeoutDurationMinutes(Integer timeoutDurationMinutes) {
            this.timeoutDurationMinutes = timeoutDurationMinutes;
            return this;
        }

        public final String getAllocationStrategy() {
            return allocationStrategy;
        }

        public final void setAllocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
        }

        @Override
        public final Builder allocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
            return this;
        }

        @Override
        public final Builder allocationStrategy(SpotProvisioningAllocationStrategy allocationStrategy) {
            this.allocationStrategy(allocationStrategy == null ? null : allocationStrategy.toString());
            return this;
        }

        @Override
        public SpotResizingSpecification build() {
            return new SpotResizingSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
