/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster. The automatic
 * scaling policy defines how an instance group dynamically adds and terminates Amazon EC2 instances in response to the
 * value of a CloudWatch metric. See <a>PutAutoScalingPolicy</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoScalingPolicyDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoScalingPolicyDescription.Builder, AutoScalingPolicyDescription> {
    private static final SdkField<AutoScalingPolicyStatus> STATUS_FIELD = SdkField
            .<AutoScalingPolicyStatus> builder(MarshallingType.SDK_POJO).memberName("Status")
            .getter(getter(AutoScalingPolicyDescription::status)).setter(setter(Builder::status))
            .constructor(AutoScalingPolicyStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<ScalingConstraints> CONSTRAINTS_FIELD = SdkField
            .<ScalingConstraints> builder(MarshallingType.SDK_POJO).memberName("Constraints")
            .getter(getter(AutoScalingPolicyDescription::constraints)).setter(setter(Builder::constraints))
            .constructor(ScalingConstraints::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Constraints").build()).build();

    private static final SdkField<List<ScalingRule>> RULES_FIELD = SdkField
            .<List<ScalingRule>> builder(MarshallingType.LIST)
            .memberName("Rules")
            .getter(getter(AutoScalingPolicyDescription::rules))
            .setter(setter(Builder::rules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Rules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ScalingRule> builder(MarshallingType.SDK_POJO)
                                            .constructor(ScalingRule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            CONSTRAINTS_FIELD, RULES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final AutoScalingPolicyStatus status;

    private final ScalingConstraints constraints;

    private final List<ScalingRule> rules;

    private AutoScalingPolicyDescription(BuilderImpl builder) {
        this.status = builder.status;
        this.constraints = builder.constraints;
        this.rules = builder.rules;
    }

    /**
     * <p>
     * The status of an automatic scaling policy.
     * </p>
     * 
     * @return The status of an automatic scaling policy.
     */
    public final AutoScalingPolicyStatus status() {
        return status;
    }

    /**
     * <p>
     * The upper and lower Amazon EC2 instance limits for an automatic scaling policy. Automatic scaling activity will
     * not cause an instance group to grow above or below these limits.
     * </p>
     * 
     * @return The upper and lower Amazon EC2 instance limits for an automatic scaling policy. Automatic scaling
     *         activity will not cause an instance group to grow above or below these limits.
     */
    public final ScalingConstraints constraints() {
        return constraints;
    }

    /**
     * For responses, this returns true if the service returned a value for the Rules property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasRules() {
        return rules != null && !(rules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The scale-in and scale-out rules that comprise the automatic scaling policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRules} method.
     * </p>
     * 
     * @return The scale-in and scale-out rules that comprise the automatic scaling policy.
     */
    public final List<ScalingRule> rules() {
        return rules;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(constraints());
        hashCode = 31 * hashCode + Objects.hashCode(hasRules() ? rules() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoScalingPolicyDescription)) {
            return false;
        }
        AutoScalingPolicyDescription other = (AutoScalingPolicyDescription) obj;
        return Objects.equals(status(), other.status()) && Objects.equals(constraints(), other.constraints())
                && hasRules() == other.hasRules() && Objects.equals(rules(), other.rules());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AutoScalingPolicyDescription").add("Status", status()).add("Constraints", constraints())
                .add("Rules", hasRules() ? rules() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Constraints":
            return Optional.ofNullable(clazz.cast(constraints()));
        case "Rules":
            return Optional.ofNullable(clazz.cast(rules()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Status", STATUS_FIELD);
        map.put("Constraints", CONSTRAINTS_FIELD);
        map.put("Rules", RULES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AutoScalingPolicyDescription, T> g) {
        return obj -> g.apply((AutoScalingPolicyDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoScalingPolicyDescription> {
        /**
         * <p>
         * The status of an automatic scaling policy.
         * </p>
         * 
         * @param status
         *        The status of an automatic scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(AutoScalingPolicyStatus status);

        /**
         * <p>
         * The status of an automatic scaling policy.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoScalingPolicyStatus.Builder} avoiding
         * the need to create one manually via {@link AutoScalingPolicyStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoScalingPolicyStatus.Builder#build()} is called immediately
         * and its result is passed to {@link #status(AutoScalingPolicyStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link AutoScalingPolicyStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(AutoScalingPolicyStatus)
         */
        default Builder status(Consumer<AutoScalingPolicyStatus.Builder> status) {
            return status(AutoScalingPolicyStatus.builder().applyMutation(status).build());
        }

        /**
         * <p>
         * The upper and lower Amazon EC2 instance limits for an automatic scaling policy. Automatic scaling activity
         * will not cause an instance group to grow above or below these limits.
         * </p>
         * 
         * @param constraints
         *        The upper and lower Amazon EC2 instance limits for an automatic scaling policy. Automatic scaling
         *        activity will not cause an instance group to grow above or below these limits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder constraints(ScalingConstraints constraints);

        /**
         * <p>
         * The upper and lower Amazon EC2 instance limits for an automatic scaling policy. Automatic scaling activity
         * will not cause an instance group to grow above or below these limits.
         * </p>
         * This is a convenience method that creates an instance of the {@link ScalingConstraints.Builder} avoiding the
         * need to create one manually via {@link ScalingConstraints#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ScalingConstraints.Builder#build()} is called immediately and its
         * result is passed to {@link #constraints(ScalingConstraints)}.
         * 
         * @param constraints
         *        a consumer that will call methods on {@link ScalingConstraints.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #constraints(ScalingConstraints)
         */
        default Builder constraints(Consumer<ScalingConstraints.Builder> constraints) {
            return constraints(ScalingConstraints.builder().applyMutation(constraints).build());
        }

        /**
         * <p>
         * The scale-in and scale-out rules that comprise the automatic scaling policy.
         * </p>
         * 
         * @param rules
         *        The scale-in and scale-out rules that comprise the automatic scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(Collection<ScalingRule> rules);

        /**
         * <p>
         * The scale-in and scale-out rules that comprise the automatic scaling policy.
         * </p>
         * 
         * @param rules
         *        The scale-in and scale-out rules that comprise the automatic scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(ScalingRule... rules);

        /**
         * <p>
         * The scale-in and scale-out rules that comprise the automatic scaling policy.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.emr.model.ScalingRule.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.emr.model.ScalingRule#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.emr.model.ScalingRule.Builder#build()} is called immediately and its
         * result is passed to {@link #rules(List<ScalingRule>)}.
         * 
         * @param rules
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.emr.model.ScalingRule.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rules(java.util.Collection<ScalingRule>)
         */
        Builder rules(Consumer<ScalingRule.Builder>... rules);
    }

    static final class BuilderImpl implements Builder {
        private AutoScalingPolicyStatus status;

        private ScalingConstraints constraints;

        private List<ScalingRule> rules = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AutoScalingPolicyDescription model) {
            status(model.status);
            constraints(model.constraints);
            rules(model.rules);
        }

        public final AutoScalingPolicyStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        public final void setStatus(AutoScalingPolicyStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        public final Builder status(AutoScalingPolicyStatus status) {
            this.status = status;
            return this;
        }

        public final ScalingConstraints.Builder getConstraints() {
            return constraints != null ? constraints.toBuilder() : null;
        }

        public final void setConstraints(ScalingConstraints.BuilderImpl constraints) {
            this.constraints = constraints != null ? constraints.build() : null;
        }

        @Override
        public final Builder constraints(ScalingConstraints constraints) {
            this.constraints = constraints;
            return this;
        }

        public final List<ScalingRule.Builder> getRules() {
            List<ScalingRule.Builder> result = ScalingRuleListCopier.copyToBuilder(this.rules);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRules(Collection<ScalingRule.BuilderImpl> rules) {
            this.rules = ScalingRuleListCopier.copyFromBuilder(rules);
        }

        @Override
        public final Builder rules(Collection<ScalingRule> rules) {
            this.rules = ScalingRuleListCopier.copy(rules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rules(ScalingRule... rules) {
            rules(Arrays.asList(rules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rules(Consumer<ScalingRule.Builder>... rules) {
            rules(Stream.of(rules).map(c -> ScalingRule.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public AutoScalingPolicyDescription build() {
            return new AutoScalingPolicyDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
