/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A cluster step consisting of a JAR file whose main function will be executed. The main function submits a job for
 * Hadoop to execute and waits for the job to finish or fail.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HadoopStepConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<HadoopStepConfig.Builder, HadoopStepConfig> {
    private static final SdkField<String> JAR_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Jar")
            .getter(getter(HadoopStepConfig::jar)).setter(setter(Builder::jar))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Jar").build()).build();

    private static final SdkField<Map<String, String>> PROPERTIES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Properties")
            .getter(getter(HadoopStepConfig::properties))
            .setter(setter(Builder::properties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Properties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> MAIN_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MainClass").getter(getter(HadoopStepConfig::mainClass)).setter(setter(Builder::mainClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MainClass").build()).build();

    private static final SdkField<List<String>> ARGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Args")
            .getter(getter(HadoopStepConfig::args))
            .setter(setter(Builder::args))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Args").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(JAR_FIELD, PROPERTIES_FIELD,
            MAIN_CLASS_FIELD, ARGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String jar;

    private final Map<String, String> properties;

    private final String mainClass;

    private final List<String> args;

    private HadoopStepConfig(BuilderImpl builder) {
        this.jar = builder.jar;
        this.properties = builder.properties;
        this.mainClass = builder.mainClass;
        this.args = builder.args;
    }

    /**
     * <p>
     * The path to the JAR file that runs during the step.
     * </p>
     * 
     * @return The path to the JAR file that runs during the step.
     */
    public final String jar() {
        return jar;
    }

    /**
     * For responses, this returns true if the service returned a value for the Properties property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasProperties() {
        return properties != null && !(properties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The list of Java properties that are set when the step runs. You can use these properties to pass key-value pairs
     * to your main function.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProperties} method.
     * </p>
     * 
     * @return The list of Java properties that are set when the step runs. You can use these properties to pass
     *         key-value pairs to your main function.
     */
    public final Map<String, String> properties() {
        return properties;
    }

    /**
     * <p>
     * The name of the main class in the specified Java file. If not specified, the JAR file should specify a main class
     * in its manifest file.
     * </p>
     * 
     * @return The name of the main class in the specified Java file. If not specified, the JAR file should specify a
     *         main class in its manifest file.
     */
    public final String mainClass() {
        return mainClass;
    }

    /**
     * For responses, this returns true if the service returned a value for the Args property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasArgs() {
        return args != null && !(args instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of command line arguments to pass to the JAR file's main function for execution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasArgs} method.
     * </p>
     * 
     * @return The list of command line arguments to pass to the JAR file's main function for execution.
     */
    public final List<String> args() {
        return args;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(jar());
        hashCode = 31 * hashCode + Objects.hashCode(hasProperties() ? properties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(mainClass());
        hashCode = 31 * hashCode + Objects.hashCode(hasArgs() ? args() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HadoopStepConfig)) {
            return false;
        }
        HadoopStepConfig other = (HadoopStepConfig) obj;
        return Objects.equals(jar(), other.jar()) && hasProperties() == other.hasProperties()
                && Objects.equals(properties(), other.properties()) && Objects.equals(mainClass(), other.mainClass())
                && hasArgs() == other.hasArgs() && Objects.equals(args(), other.args());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HadoopStepConfig").add("Jar", jar()).add("Properties", hasProperties() ? properties() : null)
                .add("MainClass", mainClass()).add("Args", hasArgs() ? args() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Jar":
            return Optional.ofNullable(clazz.cast(jar()));
        case "Properties":
            return Optional.ofNullable(clazz.cast(properties()));
        case "MainClass":
            return Optional.ofNullable(clazz.cast(mainClass()));
        case "Args":
            return Optional.ofNullable(clazz.cast(args()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Jar", JAR_FIELD);
        map.put("Properties", PROPERTIES_FIELD);
        map.put("MainClass", MAIN_CLASS_FIELD);
        map.put("Args", ARGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<HadoopStepConfig, T> g) {
        return obj -> g.apply((HadoopStepConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HadoopStepConfig> {
        /**
         * <p>
         * The path to the JAR file that runs during the step.
         * </p>
         * 
         * @param jar
         *        The path to the JAR file that runs during the step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jar(String jar);

        /**
         * <p>
         * The list of Java properties that are set when the step runs. You can use these properties to pass key-value
         * pairs to your main function.
         * </p>
         * 
         * @param properties
         *        The list of Java properties that are set when the step runs. You can use these properties to pass
         *        key-value pairs to your main function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder properties(Map<String, String> properties);

        /**
         * <p>
         * The name of the main class in the specified Java file. If not specified, the JAR file should specify a main
         * class in its manifest file.
         * </p>
         * 
         * @param mainClass
         *        The name of the main class in the specified Java file. If not specified, the JAR file should specify a
         *        main class in its manifest file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mainClass(String mainClass);

        /**
         * <p>
         * The list of command line arguments to pass to the JAR file's main function for execution.
         * </p>
         * 
         * @param args
         *        The list of command line arguments to pass to the JAR file's main function for execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder args(Collection<String> args);

        /**
         * <p>
         * The list of command line arguments to pass to the JAR file's main function for execution.
         * </p>
         * 
         * @param args
         *        The list of command line arguments to pass to the JAR file's main function for execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder args(String... args);
    }

    static final class BuilderImpl implements Builder {
        private String jar;

        private Map<String, String> properties = DefaultSdkAutoConstructMap.getInstance();

        private String mainClass;

        private List<String> args = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(HadoopStepConfig model) {
            jar(model.jar);
            properties(model.properties);
            mainClass(model.mainClass);
            args(model.args);
        }

        public final String getJar() {
            return jar;
        }

        public final void setJar(String jar) {
            this.jar = jar;
        }

        @Override
        public final Builder jar(String jar) {
            this.jar = jar;
            return this;
        }

        public final Map<String, String> getProperties() {
            if (properties instanceof SdkAutoConstructMap) {
                return null;
            }
            return properties;
        }

        public final void setProperties(Map<String, String> properties) {
            this.properties = StringMapCopier.copy(properties);
        }

        @Override
        public final Builder properties(Map<String, String> properties) {
            this.properties = StringMapCopier.copy(properties);
            return this;
        }

        public final String getMainClass() {
            return mainClass;
        }

        public final void setMainClass(String mainClass) {
            this.mainClass = mainClass;
        }

        @Override
        public final Builder mainClass(String mainClass) {
            this.mainClass = mainClass;
            return this;
        }

        public final Collection<String> getArgs() {
            if (args instanceof SdkAutoConstructList) {
                return null;
            }
            return args;
        }

        public final void setArgs(Collection<String> args) {
            this.args = StringListCopier.copy(args);
        }

        @Override
        public final Builder args(Collection<String> args) {
            this.args = StringListCopier.copy(args);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder args(String... args) {
            args(Arrays.asList(args));
            return this;
        }

        @Override
        public HadoopStepConfig build() {
            return new HadoopStepConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
