/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an instance fleet, which is a group of Amazon EC2 instances that host a particular node type (master, core,
 * or task) in an Amazon EMR cluster. Instance fleets can consist of a mix of instance types and On-Demand and Spot
 * Instances, which are provisioned to meet a defined target capacity.
 * </p>
 * <note>
 * <p>
 * The instance fleet configuration is available only in Amazon EMR releases 4.8.0 and later, excluding 5.0.x versions.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceFleet implements SdkPojo, Serializable, ToCopyableBuilder<InstanceFleet.Builder, InstanceFleet> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(InstanceFleet::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(InstanceFleet::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<InstanceFleetStatus> STATUS_FIELD = SdkField
            .<InstanceFleetStatus> builder(MarshallingType.SDK_POJO).memberName("Status").getter(getter(InstanceFleet::status))
            .setter(setter(Builder::status)).constructor(InstanceFleetStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> INSTANCE_FLEET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceFleetType").getter(getter(InstanceFleet::instanceFleetTypeAsString))
            .setter(setter(Builder::instanceFleetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceFleetType").build()).build();

    private static final SdkField<Integer> TARGET_ON_DEMAND_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TargetOnDemandCapacity").getter(getter(InstanceFleet::targetOnDemandCapacity))
            .setter(setter(Builder::targetOnDemandCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetOnDemandCapacity").build())
            .build();

    private static final SdkField<Integer> TARGET_SPOT_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TargetSpotCapacity").getter(getter(InstanceFleet::targetSpotCapacity))
            .setter(setter(Builder::targetSpotCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetSpotCapacity").build())
            .build();

    private static final SdkField<Integer> PROVISIONED_ON_DEMAND_CAPACITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("ProvisionedOnDemandCapacity")
            .getter(getter(InstanceFleet::provisionedOnDemandCapacity))
            .setter(setter(Builder::provisionedOnDemandCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProvisionedOnDemandCapacity")
                    .build()).build();

    private static final SdkField<Integer> PROVISIONED_SPOT_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ProvisionedSpotCapacity").getter(getter(InstanceFleet::provisionedSpotCapacity))
            .setter(setter(Builder::provisionedSpotCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProvisionedSpotCapacity").build())
            .build();

    private static final SdkField<List<InstanceTypeSpecification>> INSTANCE_TYPE_SPECIFICATIONS_FIELD = SdkField
            .<List<InstanceTypeSpecification>> builder(MarshallingType.LIST)
            .memberName("InstanceTypeSpecifications")
            .getter(getter(InstanceFleet::instanceTypeSpecifications))
            .setter(setter(Builder::instanceTypeSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceTypeSpecifications").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InstanceTypeSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceTypeSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<InstanceFleetProvisioningSpecifications> LAUNCH_SPECIFICATIONS_FIELD = SdkField
            .<InstanceFleetProvisioningSpecifications> builder(MarshallingType.SDK_POJO).memberName("LaunchSpecifications")
            .getter(getter(InstanceFleet::launchSpecifications)).setter(setter(Builder::launchSpecifications))
            .constructor(InstanceFleetProvisioningSpecifications::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchSpecifications").build())
            .build();

    private static final SdkField<InstanceFleetResizingSpecifications> RESIZE_SPECIFICATIONS_FIELD = SdkField
            .<InstanceFleetResizingSpecifications> builder(MarshallingType.SDK_POJO).memberName("ResizeSpecifications")
            .getter(getter(InstanceFleet::resizeSpecifications)).setter(setter(Builder::resizeSpecifications))
            .constructor(InstanceFleetResizingSpecifications::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResizeSpecifications").build())
            .build();

    private static final SdkField<String> CONTEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Context")
            .getter(getter(InstanceFleet::context)).setter(setter(Builder::context))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Context").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, NAME_FIELD,
            STATUS_FIELD, INSTANCE_FLEET_TYPE_FIELD, TARGET_ON_DEMAND_CAPACITY_FIELD, TARGET_SPOT_CAPACITY_FIELD,
            PROVISIONED_ON_DEMAND_CAPACITY_FIELD, PROVISIONED_SPOT_CAPACITY_FIELD, INSTANCE_TYPE_SPECIFICATIONS_FIELD,
            LAUNCH_SPECIFICATIONS_FIELD, RESIZE_SPECIFICATIONS_FIELD, CONTEXT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String name;

    private final InstanceFleetStatus status;

    private final String instanceFleetType;

    private final Integer targetOnDemandCapacity;

    private final Integer targetSpotCapacity;

    private final Integer provisionedOnDemandCapacity;

    private final Integer provisionedSpotCapacity;

    private final List<InstanceTypeSpecification> instanceTypeSpecifications;

    private final InstanceFleetProvisioningSpecifications launchSpecifications;

    private final InstanceFleetResizingSpecifications resizeSpecifications;

    private final String context;

    private InstanceFleet(BuilderImpl builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.status = builder.status;
        this.instanceFleetType = builder.instanceFleetType;
        this.targetOnDemandCapacity = builder.targetOnDemandCapacity;
        this.targetSpotCapacity = builder.targetSpotCapacity;
        this.provisionedOnDemandCapacity = builder.provisionedOnDemandCapacity;
        this.provisionedSpotCapacity = builder.provisionedSpotCapacity;
        this.instanceTypeSpecifications = builder.instanceTypeSpecifications;
        this.launchSpecifications = builder.launchSpecifications;
        this.resizeSpecifications = builder.resizeSpecifications;
        this.context = builder.context;
    }

    /**
     * <p>
     * The unique identifier of the instance fleet.
     * </p>
     * 
     * @return The unique identifier of the instance fleet.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * A friendly name for the instance fleet.
     * </p>
     * 
     * @return A friendly name for the instance fleet.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The current status of the instance fleet.
     * </p>
     * 
     * @return The current status of the instance fleet.
     */
    public final InstanceFleetStatus status() {
        return status;
    }

    /**
     * <p>
     * The node type that the instance fleet hosts. Valid values are MASTER, CORE, or TASK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceFleetType}
     * will return {@link InstanceFleetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceFleetTypeAsString}.
     * </p>
     * 
     * @return The node type that the instance fleet hosts. Valid values are MASTER, CORE, or TASK.
     * @see InstanceFleetType
     */
    public final InstanceFleetType instanceFleetType() {
        return InstanceFleetType.fromValue(instanceFleetType);
    }

    /**
     * <p>
     * The node type that the instance fleet hosts. Valid values are MASTER, CORE, or TASK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceFleetType}
     * will return {@link InstanceFleetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceFleetTypeAsString}.
     * </p>
     * 
     * @return The node type that the instance fleet hosts. Valid values are MASTER, CORE, or TASK.
     * @see InstanceFleetType
     */
    public final String instanceFleetTypeAsString() {
        return instanceFleetType;
    }

    /**
     * <p>
     * The target capacity of On-Demand units for the instance fleet, which determines how many On-Demand Instances to
     * provision. When the instance fleet launches, Amazon EMR tries to provision On-Demand Instances as specified by
     * <a>InstanceTypeConfig</a>. Each instance configuration has a specified <code>WeightedCapacity</code>. When an
     * On-Demand Instance is provisioned, the <code>WeightedCapacity</code> units count toward the target capacity.
     * Amazon EMR provisions instances until the target capacity is totally fulfilled, even if this results in an
     * overage. For example, if there are 2 units remaining to fulfill capacity, and Amazon EMR can only provision an
     * instance with a <code>WeightedCapacity</code> of 5 units, the instance is provisioned, and the target capacity is
     * exceeded by 3 units. You can use <a>InstanceFleet$ProvisionedOnDemandCapacity</a> to determine the Spot capacity
     * units that have been provisioned for the instance fleet.
     * </p>
     * <note>
     * <p>
     * If not specified or set to 0, only Spot Instances are provisioned for the instance fleet using
     * <code>TargetSpotCapacity</code>. At least one of <code>TargetSpotCapacity</code> and
     * <code>TargetOnDemandCapacity</code> should be greater than 0. For a master instance fleet, only one of
     * <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> can be specified, and its value must be
     * 1.
     * </p>
     * </note>
     * 
     * @return The target capacity of On-Demand units for the instance fleet, which determines how many On-Demand
     *         Instances to provision. When the instance fleet launches, Amazon EMR tries to provision On-Demand
     *         Instances as specified by <a>InstanceTypeConfig</a>. Each instance configuration has a specified
     *         <code>WeightedCapacity</code>. When an On-Demand Instance is provisioned, the
     *         <code>WeightedCapacity</code> units count toward the target capacity. Amazon EMR provisions instances
     *         until the target capacity is totally fulfilled, even if this results in an overage. For example, if there
     *         are 2 units remaining to fulfill capacity, and Amazon EMR can only provision an instance with a
     *         <code>WeightedCapacity</code> of 5 units, the instance is provisioned, and the target capacity is
     *         exceeded by 3 units. You can use <a>InstanceFleet$ProvisionedOnDemandCapacity</a> to determine the Spot
     *         capacity units that have been provisioned for the instance fleet.</p> <note>
     *         <p>
     *         If not specified or set to 0, only Spot Instances are provisioned for the instance fleet using
     *         <code>TargetSpotCapacity</code>. At least one of <code>TargetSpotCapacity</code> and
     *         <code>TargetOnDemandCapacity</code> should be greater than 0. For a master instance fleet, only one of
     *         <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> can be specified, and its value
     *         must be 1.
     *         </p>
     */
    public final Integer targetOnDemandCapacity() {
        return targetOnDemandCapacity;
    }

    /**
     * <p>
     * The target capacity of Spot units for the instance fleet, which determines how many Spot Instances to provision.
     * When the instance fleet launches, Amazon EMR tries to provision Spot Instances as specified by
     * <a>InstanceTypeConfig</a>. Each instance configuration has a specified <code>WeightedCapacity</code>. When a Spot
     * instance is provisioned, the <code>WeightedCapacity</code> units count toward the target capacity. Amazon EMR
     * provisions instances until the target capacity is totally fulfilled, even if this results in an overage. For
     * example, if there are 2 units remaining to fulfill capacity, and Amazon EMR can only provision an instance with a
     * <code>WeightedCapacity</code> of 5 units, the instance is provisioned, and the target capacity is exceeded by 3
     * units. You can use <a>InstanceFleet$ProvisionedSpotCapacity</a> to determine the Spot capacity units that have
     * been provisioned for the instance fleet.
     * </p>
     * <note>
     * <p>
     * If not specified or set to 0, only On-Demand Instances are provisioned for the instance fleet. At least one of
     * <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> should be greater than 0. For a master
     * instance fleet, only one of <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> can be
     * specified, and its value must be 1.
     * </p>
     * </note>
     * 
     * @return The target capacity of Spot units for the instance fleet, which determines how many Spot Instances to
     *         provision. When the instance fleet launches, Amazon EMR tries to provision Spot Instances as specified by
     *         <a>InstanceTypeConfig</a>. Each instance configuration has a specified <code>WeightedCapacity</code>.
     *         When a Spot instance is provisioned, the <code>WeightedCapacity</code> units count toward the target
     *         capacity. Amazon EMR provisions instances until the target capacity is totally fulfilled, even if this
     *         results in an overage. For example, if there are 2 units remaining to fulfill capacity, and Amazon EMR
     *         can only provision an instance with a <code>WeightedCapacity</code> of 5 units, the instance is
     *         provisioned, and the target capacity is exceeded by 3 units. You can use
     *         <a>InstanceFleet$ProvisionedSpotCapacity</a> to determine the Spot capacity units that have been
     *         provisioned for the instance fleet.</p> <note>
     *         <p>
     *         If not specified or set to 0, only On-Demand Instances are provisioned for the instance fleet. At least
     *         one of <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> should be greater than 0.
     *         For a master instance fleet, only one of <code>TargetSpotCapacity</code> and
     *         <code>TargetOnDemandCapacity</code> can be specified, and its value must be 1.
     *         </p>
     */
    public final Integer targetSpotCapacity() {
        return targetSpotCapacity;
    }

    /**
     * <p>
     * The number of On-Demand units that have been provisioned for the instance fleet to fulfill
     * <code>TargetOnDemandCapacity</code>. This provisioned capacity might be less than or greater than
     * <code>TargetOnDemandCapacity</code>.
     * </p>
     * 
     * @return The number of On-Demand units that have been provisioned for the instance fleet to fulfill
     *         <code>TargetOnDemandCapacity</code>. This provisioned capacity might be less than or greater than
     *         <code>TargetOnDemandCapacity</code>.
     */
    public final Integer provisionedOnDemandCapacity() {
        return provisionedOnDemandCapacity;
    }

    /**
     * <p>
     * The number of Spot units that have been provisioned for this instance fleet to fulfill
     * <code>TargetSpotCapacity</code>. This provisioned capacity might be less than or greater than
     * <code>TargetSpotCapacity</code>.
     * </p>
     * 
     * @return The number of Spot units that have been provisioned for this instance fleet to fulfill
     *         <code>TargetSpotCapacity</code>. This provisioned capacity might be less than or greater than
     *         <code>TargetSpotCapacity</code>.
     */
    public final Integer provisionedSpotCapacity() {
        return provisionedSpotCapacity;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceTypeSpecifications property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasInstanceTypeSpecifications() {
        return instanceTypeSpecifications != null && !(instanceTypeSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of specifications for the instance types that comprise an instance fleet.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceTypeSpecifications} method.
     * </p>
     * 
     * @return An array of specifications for the instance types that comprise an instance fleet.
     */
    public final List<InstanceTypeSpecification> instanceTypeSpecifications() {
        return instanceTypeSpecifications;
    }

    /**
     * <p>
     * Describes the launch specification for an instance fleet.
     * </p>
     * 
     * @return Describes the launch specification for an instance fleet.
     */
    public final InstanceFleetProvisioningSpecifications launchSpecifications() {
        return launchSpecifications;
    }

    /**
     * <p>
     * The resize specification for the instance fleet.
     * </p>
     * 
     * @return The resize specification for the instance fleet.
     */
    public final InstanceFleetResizingSpecifications resizeSpecifications() {
        return resizeSpecifications;
    }

    /**
     * <p>
     * Reserved.
     * </p>
     * 
     * @return Reserved.
     */
    public final String context() {
        return context;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(instanceFleetTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetOnDemandCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(targetSpotCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(provisionedOnDemandCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(provisionedSpotCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceTypeSpecifications() ? instanceTypeSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(launchSpecifications());
        hashCode = 31 * hashCode + Objects.hashCode(resizeSpecifications());
        hashCode = 31 * hashCode + Objects.hashCode(context());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceFleet)) {
            return false;
        }
        InstanceFleet other = (InstanceFleet) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(name(), other.name())
                && Objects.equals(status(), other.status())
                && Objects.equals(instanceFleetTypeAsString(), other.instanceFleetTypeAsString())
                && Objects.equals(targetOnDemandCapacity(), other.targetOnDemandCapacity())
                && Objects.equals(targetSpotCapacity(), other.targetSpotCapacity())
                && Objects.equals(provisionedOnDemandCapacity(), other.provisionedOnDemandCapacity())
                && Objects.equals(provisionedSpotCapacity(), other.provisionedSpotCapacity())
                && hasInstanceTypeSpecifications() == other.hasInstanceTypeSpecifications()
                && Objects.equals(instanceTypeSpecifications(), other.instanceTypeSpecifications())
                && Objects.equals(launchSpecifications(), other.launchSpecifications())
                && Objects.equals(resizeSpecifications(), other.resizeSpecifications())
                && Objects.equals(context(), other.context());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceFleet").add("Id", id()).add("Name", name()).add("Status", status())
                .add("InstanceFleetType", instanceFleetTypeAsString()).add("TargetOnDemandCapacity", targetOnDemandCapacity())
                .add("TargetSpotCapacity", targetSpotCapacity())
                .add("ProvisionedOnDemandCapacity", provisionedOnDemandCapacity())
                .add("ProvisionedSpotCapacity", provisionedSpotCapacity())
                .add("InstanceTypeSpecifications", hasInstanceTypeSpecifications() ? instanceTypeSpecifications() : null)
                .add("LaunchSpecifications", launchSpecifications()).add("ResizeSpecifications", resizeSpecifications())
                .add("Context", context()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "InstanceFleetType":
            return Optional.ofNullable(clazz.cast(instanceFleetTypeAsString()));
        case "TargetOnDemandCapacity":
            return Optional.ofNullable(clazz.cast(targetOnDemandCapacity()));
        case "TargetSpotCapacity":
            return Optional.ofNullable(clazz.cast(targetSpotCapacity()));
        case "ProvisionedOnDemandCapacity":
            return Optional.ofNullable(clazz.cast(provisionedOnDemandCapacity()));
        case "ProvisionedSpotCapacity":
            return Optional.ofNullable(clazz.cast(provisionedSpotCapacity()));
        case "InstanceTypeSpecifications":
            return Optional.ofNullable(clazz.cast(instanceTypeSpecifications()));
        case "LaunchSpecifications":
            return Optional.ofNullable(clazz.cast(launchSpecifications()));
        case "ResizeSpecifications":
            return Optional.ofNullable(clazz.cast(resizeSpecifications()));
        case "Context":
            return Optional.ofNullable(clazz.cast(context()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Id", ID_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("InstanceFleetType", INSTANCE_FLEET_TYPE_FIELD);
        map.put("TargetOnDemandCapacity", TARGET_ON_DEMAND_CAPACITY_FIELD);
        map.put("TargetSpotCapacity", TARGET_SPOT_CAPACITY_FIELD);
        map.put("ProvisionedOnDemandCapacity", PROVISIONED_ON_DEMAND_CAPACITY_FIELD);
        map.put("ProvisionedSpotCapacity", PROVISIONED_SPOT_CAPACITY_FIELD);
        map.put("InstanceTypeSpecifications", INSTANCE_TYPE_SPECIFICATIONS_FIELD);
        map.put("LaunchSpecifications", LAUNCH_SPECIFICATIONS_FIELD);
        map.put("ResizeSpecifications", RESIZE_SPECIFICATIONS_FIELD);
        map.put("Context", CONTEXT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InstanceFleet, T> g) {
        return obj -> g.apply((InstanceFleet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceFleet> {
        /**
         * <p>
         * The unique identifier of the instance fleet.
         * </p>
         * 
         * @param id
         *        The unique identifier of the instance fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * A friendly name for the instance fleet.
         * </p>
         * 
         * @param name
         *        A friendly name for the instance fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The current status of the instance fleet.
         * </p>
         * 
         * @param status
         *        The current status of the instance fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(InstanceFleetStatus status);

        /**
         * <p>
         * The current status of the instance fleet.
         * </p>
         * This is a convenience method that creates an instance of the {@link InstanceFleetStatus.Builder} avoiding the
         * need to create one manually via {@link InstanceFleetStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceFleetStatus.Builder#build()} is called immediately and
         * its result is passed to {@link #status(InstanceFleetStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link InstanceFleetStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(InstanceFleetStatus)
         */
        default Builder status(Consumer<InstanceFleetStatus.Builder> status) {
            return status(InstanceFleetStatus.builder().applyMutation(status).build());
        }

        /**
         * <p>
         * The node type that the instance fleet hosts. Valid values are MASTER, CORE, or TASK.
         * </p>
         * 
         * @param instanceFleetType
         *        The node type that the instance fleet hosts. Valid values are MASTER, CORE, or TASK.
         * @see InstanceFleetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceFleetType
         */
        Builder instanceFleetType(String instanceFleetType);

        /**
         * <p>
         * The node type that the instance fleet hosts. Valid values are MASTER, CORE, or TASK.
         * </p>
         * 
         * @param instanceFleetType
         *        The node type that the instance fleet hosts. Valid values are MASTER, CORE, or TASK.
         * @see InstanceFleetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceFleetType
         */
        Builder instanceFleetType(InstanceFleetType instanceFleetType);

        /**
         * <p>
         * The target capacity of On-Demand units for the instance fleet, which determines how many On-Demand Instances
         * to provision. When the instance fleet launches, Amazon EMR tries to provision On-Demand Instances as
         * specified by <a>InstanceTypeConfig</a>. Each instance configuration has a specified
         * <code>WeightedCapacity</code>. When an On-Demand Instance is provisioned, the <code>WeightedCapacity</code>
         * units count toward the target capacity. Amazon EMR provisions instances until the target capacity is totally
         * fulfilled, even if this results in an overage. For example, if there are 2 units remaining to fulfill
         * capacity, and Amazon EMR can only provision an instance with a <code>WeightedCapacity</code> of 5 units, the
         * instance is provisioned, and the target capacity is exceeded by 3 units. You can use
         * <a>InstanceFleet$ProvisionedOnDemandCapacity</a> to determine the Spot capacity units that have been
         * provisioned for the instance fleet.
         * </p>
         * <note>
         * <p>
         * If not specified or set to 0, only Spot Instances are provisioned for the instance fleet using
         * <code>TargetSpotCapacity</code>. At least one of <code>TargetSpotCapacity</code> and
         * <code>TargetOnDemandCapacity</code> should be greater than 0. For a master instance fleet, only one of
         * <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> can be specified, and its value must
         * be 1.
         * </p>
         * </note>
         * 
         * @param targetOnDemandCapacity
         *        The target capacity of On-Demand units for the instance fleet, which determines how many On-Demand
         *        Instances to provision. When the instance fleet launches, Amazon EMR tries to provision On-Demand
         *        Instances as specified by <a>InstanceTypeConfig</a>. Each instance configuration has a specified
         *        <code>WeightedCapacity</code>. When an On-Demand Instance is provisioned, the
         *        <code>WeightedCapacity</code> units count toward the target capacity. Amazon EMR provisions instances
         *        until the target capacity is totally fulfilled, even if this results in an overage. For example, if
         *        there are 2 units remaining to fulfill capacity, and Amazon EMR can only provision an instance with a
         *        <code>WeightedCapacity</code> of 5 units, the instance is provisioned, and the target capacity is
         *        exceeded by 3 units. You can use <a>InstanceFleet$ProvisionedOnDemandCapacity</a> to determine the
         *        Spot capacity units that have been provisioned for the instance fleet.</p> <note>
         *        <p>
         *        If not specified or set to 0, only Spot Instances are provisioned for the instance fleet using
         *        <code>TargetSpotCapacity</code>. At least one of <code>TargetSpotCapacity</code> and
         *        <code>TargetOnDemandCapacity</code> should be greater than 0. For a master instance fleet, only one of
         *        <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> can be specified, and its
         *        value must be 1.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetOnDemandCapacity(Integer targetOnDemandCapacity);

        /**
         * <p>
         * The target capacity of Spot units for the instance fleet, which determines how many Spot Instances to
         * provision. When the instance fleet launches, Amazon EMR tries to provision Spot Instances as specified by
         * <a>InstanceTypeConfig</a>. Each instance configuration has a specified <code>WeightedCapacity</code>. When a
         * Spot instance is provisioned, the <code>WeightedCapacity</code> units count toward the target capacity.
         * Amazon EMR provisions instances until the target capacity is totally fulfilled, even if this results in an
         * overage. For example, if there are 2 units remaining to fulfill capacity, and Amazon EMR can only provision
         * an instance with a <code>WeightedCapacity</code> of 5 units, the instance is provisioned, and the target
         * capacity is exceeded by 3 units. You can use <a>InstanceFleet$ProvisionedSpotCapacity</a> to determine the
         * Spot capacity units that have been provisioned for the instance fleet.
         * </p>
         * <note>
         * <p>
         * If not specified or set to 0, only On-Demand Instances are provisioned for the instance fleet. At least one
         * of <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> should be greater than 0. For a
         * master instance fleet, only one of <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code>
         * can be specified, and its value must be 1.
         * </p>
         * </note>
         * 
         * @param targetSpotCapacity
         *        The target capacity of Spot units for the instance fleet, which determines how many Spot Instances to
         *        provision. When the instance fleet launches, Amazon EMR tries to provision Spot Instances as specified
         *        by <a>InstanceTypeConfig</a>. Each instance configuration has a specified
         *        <code>WeightedCapacity</code>. When a Spot instance is provisioned, the <code>WeightedCapacity</code>
         *        units count toward the target capacity. Amazon EMR provisions instances until the target capacity is
         *        totally fulfilled, even if this results in an overage. For example, if there are 2 units remaining to
         *        fulfill capacity, and Amazon EMR can only provision an instance with a <code>WeightedCapacity</code>
         *        of 5 units, the instance is provisioned, and the target capacity is exceeded by 3 units. You can use
         *        <a>InstanceFleet$ProvisionedSpotCapacity</a> to determine the Spot capacity units that have been
         *        provisioned for the instance fleet.</p> <note>
         *        <p>
         *        If not specified or set to 0, only On-Demand Instances are provisioned for the instance fleet. At
         *        least one of <code>TargetSpotCapacity</code> and <code>TargetOnDemandCapacity</code> should be greater
         *        than 0. For a master instance fleet, only one of <code>TargetSpotCapacity</code> and
         *        <code>TargetOnDemandCapacity</code> can be specified, and its value must be 1.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetSpotCapacity(Integer targetSpotCapacity);

        /**
         * <p>
         * The number of On-Demand units that have been provisioned for the instance fleet to fulfill
         * <code>TargetOnDemandCapacity</code>. This provisioned capacity might be less than or greater than
         * <code>TargetOnDemandCapacity</code>.
         * </p>
         * 
         * @param provisionedOnDemandCapacity
         *        The number of On-Demand units that have been provisioned for the instance fleet to fulfill
         *        <code>TargetOnDemandCapacity</code>. This provisioned capacity might be less than or greater than
         *        <code>TargetOnDemandCapacity</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedOnDemandCapacity(Integer provisionedOnDemandCapacity);

        /**
         * <p>
         * The number of Spot units that have been provisioned for this instance fleet to fulfill
         * <code>TargetSpotCapacity</code>. This provisioned capacity might be less than or greater than
         * <code>TargetSpotCapacity</code>.
         * </p>
         * 
         * @param provisionedSpotCapacity
         *        The number of Spot units that have been provisioned for this instance fleet to fulfill
         *        <code>TargetSpotCapacity</code>. This provisioned capacity might be less than or greater than
         *        <code>TargetSpotCapacity</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedSpotCapacity(Integer provisionedSpotCapacity);

        /**
         * <p>
         * An array of specifications for the instance types that comprise an instance fleet.
         * </p>
         * 
         * @param instanceTypeSpecifications
         *        An array of specifications for the instance types that comprise an instance fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypeSpecifications(Collection<InstanceTypeSpecification> instanceTypeSpecifications);

        /**
         * <p>
         * An array of specifications for the instance types that comprise an instance fleet.
         * </p>
         * 
         * @param instanceTypeSpecifications
         *        An array of specifications for the instance types that comprise an instance fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypeSpecifications(InstanceTypeSpecification... instanceTypeSpecifications);

        /**
         * <p>
         * An array of specifications for the instance types that comprise an instance fleet.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.emr.model.InstanceTypeSpecification.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.emr.model.InstanceTypeSpecification#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.emr.model.InstanceTypeSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #instanceTypeSpecifications(List<InstanceTypeSpecification>)}.
         * 
         * @param instanceTypeSpecifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.emr.model.InstanceTypeSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceTypeSpecifications(java.util.Collection<InstanceTypeSpecification>)
         */
        Builder instanceTypeSpecifications(Consumer<InstanceTypeSpecification.Builder>... instanceTypeSpecifications);

        /**
         * <p>
         * Describes the launch specification for an instance fleet.
         * </p>
         * 
         * @param launchSpecifications
         *        Describes the launch specification for an instance fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchSpecifications(InstanceFleetProvisioningSpecifications launchSpecifications);

        /**
         * <p>
         * Describes the launch specification for an instance fleet.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link InstanceFleetProvisioningSpecifications.Builder} avoiding the need to create one manually via
         * {@link InstanceFleetProvisioningSpecifications#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceFleetProvisioningSpecifications.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #launchSpecifications(InstanceFleetProvisioningSpecifications)}.
         * 
         * @param launchSpecifications
         *        a consumer that will call methods on {@link InstanceFleetProvisioningSpecifications.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #launchSpecifications(InstanceFleetProvisioningSpecifications)
         */
        default Builder launchSpecifications(Consumer<InstanceFleetProvisioningSpecifications.Builder> launchSpecifications) {
            return launchSpecifications(InstanceFleetProvisioningSpecifications.builder().applyMutation(launchSpecifications)
                    .build());
        }

        /**
         * <p>
         * The resize specification for the instance fleet.
         * </p>
         * 
         * @param resizeSpecifications
         *        The resize specification for the instance fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resizeSpecifications(InstanceFleetResizingSpecifications resizeSpecifications);

        /**
         * <p>
         * The resize specification for the instance fleet.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link InstanceFleetResizingSpecifications.Builder} avoiding the need to create one manually via
         * {@link InstanceFleetResizingSpecifications#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceFleetResizingSpecifications.Builder#build()} is called
         * immediately and its result is passed to {@link #resizeSpecifications(InstanceFleetResizingSpecifications)}.
         * 
         * @param resizeSpecifications
         *        a consumer that will call methods on {@link InstanceFleetResizingSpecifications.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resizeSpecifications(InstanceFleetResizingSpecifications)
         */
        default Builder resizeSpecifications(Consumer<InstanceFleetResizingSpecifications.Builder> resizeSpecifications) {
            return resizeSpecifications(InstanceFleetResizingSpecifications.builder().applyMutation(resizeSpecifications).build());
        }

        /**
         * <p>
         * Reserved.
         * </p>
         * 
         * @param context
         *        Reserved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder context(String context);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String name;

        private InstanceFleetStatus status;

        private String instanceFleetType;

        private Integer targetOnDemandCapacity;

        private Integer targetSpotCapacity;

        private Integer provisionedOnDemandCapacity;

        private Integer provisionedSpotCapacity;

        private List<InstanceTypeSpecification> instanceTypeSpecifications = DefaultSdkAutoConstructList.getInstance();

        private InstanceFleetProvisioningSpecifications launchSpecifications;

        private InstanceFleetResizingSpecifications resizeSpecifications;

        private String context;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceFleet model) {
            id(model.id);
            name(model.name);
            status(model.status);
            instanceFleetType(model.instanceFleetType);
            targetOnDemandCapacity(model.targetOnDemandCapacity);
            targetSpotCapacity(model.targetSpotCapacity);
            provisionedOnDemandCapacity(model.provisionedOnDemandCapacity);
            provisionedSpotCapacity(model.provisionedSpotCapacity);
            instanceTypeSpecifications(model.instanceTypeSpecifications);
            launchSpecifications(model.launchSpecifications);
            resizeSpecifications(model.resizeSpecifications);
            context(model.context);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final InstanceFleetStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        public final void setStatus(InstanceFleetStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        public final Builder status(InstanceFleetStatus status) {
            this.status = status;
            return this;
        }

        public final String getInstanceFleetType() {
            return instanceFleetType;
        }

        public final void setInstanceFleetType(String instanceFleetType) {
            this.instanceFleetType = instanceFleetType;
        }

        @Override
        public final Builder instanceFleetType(String instanceFleetType) {
            this.instanceFleetType = instanceFleetType;
            return this;
        }

        @Override
        public final Builder instanceFleetType(InstanceFleetType instanceFleetType) {
            this.instanceFleetType(instanceFleetType == null ? null : instanceFleetType.toString());
            return this;
        }

        public final Integer getTargetOnDemandCapacity() {
            return targetOnDemandCapacity;
        }

        public final void setTargetOnDemandCapacity(Integer targetOnDemandCapacity) {
            this.targetOnDemandCapacity = targetOnDemandCapacity;
        }

        @Override
        public final Builder targetOnDemandCapacity(Integer targetOnDemandCapacity) {
            this.targetOnDemandCapacity = targetOnDemandCapacity;
            return this;
        }

        public final Integer getTargetSpotCapacity() {
            return targetSpotCapacity;
        }

        public final void setTargetSpotCapacity(Integer targetSpotCapacity) {
            this.targetSpotCapacity = targetSpotCapacity;
        }

        @Override
        public final Builder targetSpotCapacity(Integer targetSpotCapacity) {
            this.targetSpotCapacity = targetSpotCapacity;
            return this;
        }

        public final Integer getProvisionedOnDemandCapacity() {
            return provisionedOnDemandCapacity;
        }

        public final void setProvisionedOnDemandCapacity(Integer provisionedOnDemandCapacity) {
            this.provisionedOnDemandCapacity = provisionedOnDemandCapacity;
        }

        @Override
        public final Builder provisionedOnDemandCapacity(Integer provisionedOnDemandCapacity) {
            this.provisionedOnDemandCapacity = provisionedOnDemandCapacity;
            return this;
        }

        public final Integer getProvisionedSpotCapacity() {
            return provisionedSpotCapacity;
        }

        public final void setProvisionedSpotCapacity(Integer provisionedSpotCapacity) {
            this.provisionedSpotCapacity = provisionedSpotCapacity;
        }

        @Override
        public final Builder provisionedSpotCapacity(Integer provisionedSpotCapacity) {
            this.provisionedSpotCapacity = provisionedSpotCapacity;
            return this;
        }

        public final List<InstanceTypeSpecification.Builder> getInstanceTypeSpecifications() {
            List<InstanceTypeSpecification.Builder> result = InstanceTypeSpecificationListCopier
                    .copyToBuilder(this.instanceTypeSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInstanceTypeSpecifications(
                Collection<InstanceTypeSpecification.BuilderImpl> instanceTypeSpecifications) {
            this.instanceTypeSpecifications = InstanceTypeSpecificationListCopier.copyFromBuilder(instanceTypeSpecifications);
        }

        @Override
        public final Builder instanceTypeSpecifications(Collection<InstanceTypeSpecification> instanceTypeSpecifications) {
            this.instanceTypeSpecifications = InstanceTypeSpecificationListCopier.copy(instanceTypeSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceTypeSpecifications(InstanceTypeSpecification... instanceTypeSpecifications) {
            instanceTypeSpecifications(Arrays.asList(instanceTypeSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceTypeSpecifications(Consumer<InstanceTypeSpecification.Builder>... instanceTypeSpecifications) {
            instanceTypeSpecifications(Stream.of(instanceTypeSpecifications)
                    .map(c -> InstanceTypeSpecification.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final InstanceFleetProvisioningSpecifications.Builder getLaunchSpecifications() {
            return launchSpecifications != null ? launchSpecifications.toBuilder() : null;
        }

        public final void setLaunchSpecifications(InstanceFleetProvisioningSpecifications.BuilderImpl launchSpecifications) {
            this.launchSpecifications = launchSpecifications != null ? launchSpecifications.build() : null;
        }

        @Override
        public final Builder launchSpecifications(InstanceFleetProvisioningSpecifications launchSpecifications) {
            this.launchSpecifications = launchSpecifications;
            return this;
        }

        public final InstanceFleetResizingSpecifications.Builder getResizeSpecifications() {
            return resizeSpecifications != null ? resizeSpecifications.toBuilder() : null;
        }

        public final void setResizeSpecifications(InstanceFleetResizingSpecifications.BuilderImpl resizeSpecifications) {
            this.resizeSpecifications = resizeSpecifications != null ? resizeSpecifications.build() : null;
        }

        @Override
        public final Builder resizeSpecifications(InstanceFleetResizingSpecifications resizeSpecifications) {
            this.resizeSpecifications = resizeSpecifications;
            return this;
        }

        public final String getContext() {
            return context;
        }

        public final void setContext(String context) {
            this.context = context;
        }

        @Override
        public final Builder context(String context) {
            this.context = context;
            return this;
        }

        @Override
        public InstanceFleet build() {
            return new InstanceFleet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
