/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Managed scaling policy for an Amazon EMR cluster. The policy specifies the limits for resources that can be added or
 * terminated from a cluster. The policy only applies to the core and task nodes. The master node cannot be scaled after
 * initial configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ManagedScalingPolicy implements SdkPojo, Serializable,
        ToCopyableBuilder<ManagedScalingPolicy.Builder, ManagedScalingPolicy> {
    private static final SdkField<ComputeLimits> COMPUTE_LIMITS_FIELD = SdkField
            .<ComputeLimits> builder(MarshallingType.SDK_POJO).memberName("ComputeLimits")
            .getter(getter(ManagedScalingPolicy::computeLimits)).setter(setter(Builder::computeLimits))
            .constructor(ComputeLimits::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComputeLimits").build()).build();

    private static final SdkField<Integer> UTILIZATION_PERFORMANCE_INDEX_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("UtilizationPerformanceIndex")
            .getter(getter(ManagedScalingPolicy::utilizationPerformanceIndex))
            .setter(setter(Builder::utilizationPerformanceIndex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UtilizationPerformanceIndex")
                    .build()).build();

    private static final SdkField<String> SCALING_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScalingStrategy").getter(getter(ManagedScalingPolicy::scalingStrategyAsString))
            .setter(setter(Builder::scalingStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingStrategy").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMPUTE_LIMITS_FIELD,
            UTILIZATION_PERFORMANCE_INDEX_FIELD, SCALING_STRATEGY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final ComputeLimits computeLimits;

    private final Integer utilizationPerformanceIndex;

    private final String scalingStrategy;

    private ManagedScalingPolicy(BuilderImpl builder) {
        this.computeLimits = builder.computeLimits;
        this.utilizationPerformanceIndex = builder.utilizationPerformanceIndex;
        this.scalingStrategy = builder.scalingStrategy;
    }

    /**
     * <p>
     * The Amazon EC2 unit limits for a managed scaling policy. The managed scaling activity of a cluster is not allowed
     * to go above or below these limits. The limit only applies to the core and task nodes. The master node cannot be
     * scaled after initial configuration.
     * </p>
     * 
     * @return The Amazon EC2 unit limits for a managed scaling policy. The managed scaling activity of a cluster is not
     *         allowed to go above or below these limits. The limit only applies to the core and task nodes. The master
     *         node cannot be scaled after initial configuration.
     */
    public final ComputeLimits computeLimits() {
        return computeLimits;
    }

    /**
     * <p>
     * An integer value that represents an advanced scaling strategy. Setting a higher value optimizes for performance.
     * Setting a lower value optimizes for resource conservation. Setting the value to 50 balances performance and
     * resource conservation. Possible values are 1, 25, 50, 75, and 100.
     * </p>
     * 
     * @return An integer value that represents an advanced scaling strategy. Setting a higher value optimizes for
     *         performance. Setting a lower value optimizes for resource conservation. Setting the value to 50 balances
     *         performance and resource conservation. Possible values are 1, 25, 50, 75, and 100.
     */
    public final Integer utilizationPerformanceIndex() {
        return utilizationPerformanceIndex;
    }

    /**
     * <p>
     * Determines whether a custom scaling utilization performance index can be set. Possible values include
     * <i>ADVANCED</i> or <i>DEFAULT</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalingStrategy}
     * will return {@link ScalingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalingStrategyAsString}.
     * </p>
     * 
     * @return Determines whether a custom scaling utilization performance index can be set. Possible values include
     *         <i>ADVANCED</i> or <i>DEFAULT</i>.
     * @see ScalingStrategy
     */
    public final ScalingStrategy scalingStrategy() {
        return ScalingStrategy.fromValue(scalingStrategy);
    }

    /**
     * <p>
     * Determines whether a custom scaling utilization performance index can be set. Possible values include
     * <i>ADVANCED</i> or <i>DEFAULT</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalingStrategy}
     * will return {@link ScalingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalingStrategyAsString}.
     * </p>
     * 
     * @return Determines whether a custom scaling utilization performance index can be set. Possible values include
     *         <i>ADVANCED</i> or <i>DEFAULT</i>.
     * @see ScalingStrategy
     */
    public final String scalingStrategyAsString() {
        return scalingStrategy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(computeLimits());
        hashCode = 31 * hashCode + Objects.hashCode(utilizationPerformanceIndex());
        hashCode = 31 * hashCode + Objects.hashCode(scalingStrategyAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ManagedScalingPolicy)) {
            return false;
        }
        ManagedScalingPolicy other = (ManagedScalingPolicy) obj;
        return Objects.equals(computeLimits(), other.computeLimits())
                && Objects.equals(utilizationPerformanceIndex(), other.utilizationPerformanceIndex())
                && Objects.equals(scalingStrategyAsString(), other.scalingStrategyAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ManagedScalingPolicy").add("ComputeLimits", computeLimits())
                .add("UtilizationPerformanceIndex", utilizationPerformanceIndex())
                .add("ScalingStrategy", scalingStrategyAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ComputeLimits":
            return Optional.ofNullable(clazz.cast(computeLimits()));
        case "UtilizationPerformanceIndex":
            return Optional.ofNullable(clazz.cast(utilizationPerformanceIndex()));
        case "ScalingStrategy":
            return Optional.ofNullable(clazz.cast(scalingStrategyAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ComputeLimits", COMPUTE_LIMITS_FIELD);
        map.put("UtilizationPerformanceIndex", UTILIZATION_PERFORMANCE_INDEX_FIELD);
        map.put("ScalingStrategy", SCALING_STRATEGY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ManagedScalingPolicy, T> g) {
        return obj -> g.apply((ManagedScalingPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ManagedScalingPolicy> {
        /**
         * <p>
         * The Amazon EC2 unit limits for a managed scaling policy. The managed scaling activity of a cluster is not
         * allowed to go above or below these limits. The limit only applies to the core and task nodes. The master node
         * cannot be scaled after initial configuration.
         * </p>
         * 
         * @param computeLimits
         *        The Amazon EC2 unit limits for a managed scaling policy. The managed scaling activity of a cluster is
         *        not allowed to go above or below these limits. The limit only applies to the core and task nodes. The
         *        master node cannot be scaled after initial configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computeLimits(ComputeLimits computeLimits);

        /**
         * <p>
         * The Amazon EC2 unit limits for a managed scaling policy. The managed scaling activity of a cluster is not
         * allowed to go above or below these limits. The limit only applies to the core and task nodes. The master node
         * cannot be scaled after initial configuration.
         * </p>
         * This is a convenience method that creates an instance of the {@link ComputeLimits.Builder} avoiding the need
         * to create one manually via {@link ComputeLimits#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ComputeLimits.Builder#build()} is called immediately and its
         * result is passed to {@link #computeLimits(ComputeLimits)}.
         * 
         * @param computeLimits
         *        a consumer that will call methods on {@link ComputeLimits.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #computeLimits(ComputeLimits)
         */
        default Builder computeLimits(Consumer<ComputeLimits.Builder> computeLimits) {
            return computeLimits(ComputeLimits.builder().applyMutation(computeLimits).build());
        }

        /**
         * <p>
         * An integer value that represents an advanced scaling strategy. Setting a higher value optimizes for
         * performance. Setting a lower value optimizes for resource conservation. Setting the value to 50 balances
         * performance and resource conservation. Possible values are 1, 25, 50, 75, and 100.
         * </p>
         * 
         * @param utilizationPerformanceIndex
         *        An integer value that represents an advanced scaling strategy. Setting a higher value optimizes for
         *        performance. Setting a lower value optimizes for resource conservation. Setting the value to 50
         *        balances performance and resource conservation. Possible values are 1, 25, 50, 75, and 100.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationPerformanceIndex(Integer utilizationPerformanceIndex);

        /**
         * <p>
         * Determines whether a custom scaling utilization performance index can be set. Possible values include
         * <i>ADVANCED</i> or <i>DEFAULT</i>.
         * </p>
         * 
         * @param scalingStrategy
         *        Determines whether a custom scaling utilization performance index can be set. Possible values include
         *        <i>ADVANCED</i> or <i>DEFAULT</i>.
         * @see ScalingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingStrategy
         */
        Builder scalingStrategy(String scalingStrategy);

        /**
         * <p>
         * Determines whether a custom scaling utilization performance index can be set. Possible values include
         * <i>ADVANCED</i> or <i>DEFAULT</i>.
         * </p>
         * 
         * @param scalingStrategy
         *        Determines whether a custom scaling utilization performance index can be set. Possible values include
         *        <i>ADVANCED</i> or <i>DEFAULT</i>.
         * @see ScalingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingStrategy
         */
        Builder scalingStrategy(ScalingStrategy scalingStrategy);
    }

    static final class BuilderImpl implements Builder {
        private ComputeLimits computeLimits;

        private Integer utilizationPerformanceIndex;

        private String scalingStrategy;

        private BuilderImpl() {
        }

        private BuilderImpl(ManagedScalingPolicy model) {
            computeLimits(model.computeLimits);
            utilizationPerformanceIndex(model.utilizationPerformanceIndex);
            scalingStrategy(model.scalingStrategy);
        }

        public final ComputeLimits.Builder getComputeLimits() {
            return computeLimits != null ? computeLimits.toBuilder() : null;
        }

        public final void setComputeLimits(ComputeLimits.BuilderImpl computeLimits) {
            this.computeLimits = computeLimits != null ? computeLimits.build() : null;
        }

        @Override
        public final Builder computeLimits(ComputeLimits computeLimits) {
            this.computeLimits = computeLimits;
            return this;
        }

        public final Integer getUtilizationPerformanceIndex() {
            return utilizationPerformanceIndex;
        }

        public final void setUtilizationPerformanceIndex(Integer utilizationPerformanceIndex) {
            this.utilizationPerformanceIndex = utilizationPerformanceIndex;
        }

        @Override
        public final Builder utilizationPerformanceIndex(Integer utilizationPerformanceIndex) {
            this.utilizationPerformanceIndex = utilizationPerformanceIndex;
            return this;
        }

        public final String getScalingStrategy() {
            return scalingStrategy;
        }

        public final void setScalingStrategy(String scalingStrategy) {
            this.scalingStrategy = scalingStrategy;
        }

        @Override
        public final Builder scalingStrategy(String scalingStrategy) {
            this.scalingStrategy = scalingStrategy;
            return this;
        }

        @Override
        public final Builder scalingStrategy(ScalingStrategy scalingStrategy) {
            this.scalingStrategy(scalingStrategy == null ? null : scalingStrategy.toString());
            return this;
        }

        @Override
        public ManagedScalingPolicy build() {
            return new ManagedScalingPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
