/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A notebook execution. An execution is a specific instance that an Amazon EMR Notebook is run using the
 * <code>StartNotebookExecution</code> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NotebookExecution implements SdkPojo, Serializable,
        ToCopyableBuilder<NotebookExecution.Builder, NotebookExecution> {
    private static final SdkField<String> NOTEBOOK_EXECUTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NotebookExecutionId").getter(getter(NotebookExecution::notebookExecutionId))
            .setter(setter(Builder::notebookExecutionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotebookExecutionId").build())
            .build();

    private static final SdkField<String> EDITOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EditorId").getter(getter(NotebookExecution::editorId)).setter(setter(Builder::editorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EditorId").build()).build();

    private static final SdkField<ExecutionEngineConfig> EXECUTION_ENGINE_FIELD = SdkField
            .<ExecutionEngineConfig> builder(MarshallingType.SDK_POJO).memberName("ExecutionEngine")
            .getter(getter(NotebookExecution::executionEngine)).setter(setter(Builder::executionEngine))
            .constructor(ExecutionEngineConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionEngine").build()).build();

    private static final SdkField<String> NOTEBOOK_EXECUTION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NotebookExecutionName").getter(getter(NotebookExecution::notebookExecutionName))
            .setter(setter(Builder::notebookExecutionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotebookExecutionName").build())
            .build();

    private static final SdkField<String> NOTEBOOK_PARAMS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NotebookParams").getter(getter(NotebookExecution::notebookParams))
            .setter(setter(Builder::notebookParams))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotebookParams").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(NotebookExecution::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(NotebookExecution::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(NotebookExecution::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(NotebookExecution::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> OUTPUT_NOTEBOOK_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputNotebookURI").getter(getter(NotebookExecution::outputNotebookURI))
            .setter(setter(Builder::outputNotebookURI))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputNotebookURI").build()).build();

    private static final SdkField<String> LAST_STATE_CHANGE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LastStateChangeReason").getter(getter(NotebookExecution::lastStateChangeReason))
            .setter(setter(Builder::lastStateChangeReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastStateChangeReason").build())
            .build();

    private static final SdkField<String> NOTEBOOK_INSTANCE_SECURITY_GROUP_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NotebookInstanceSecurityGroupId")
            .getter(getter(NotebookExecution::notebookInstanceSecurityGroupId))
            .setter(setter(Builder::notebookInstanceSecurityGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotebookInstanceSecurityGroupId")
                    .build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(NotebookExecution::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<NotebookS3LocationForOutput> NOTEBOOK_S3_LOCATION_FIELD = SdkField
            .<NotebookS3LocationForOutput> builder(MarshallingType.SDK_POJO).memberName("NotebookS3Location")
            .getter(getter(NotebookExecution::notebookS3Location)).setter(setter(Builder::notebookS3Location))
            .constructor(NotebookS3LocationForOutput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotebookS3Location").build())
            .build();

    private static final SdkField<OutputNotebookS3LocationForOutput> OUTPUT_NOTEBOOK_S3_LOCATION_FIELD = SdkField
            .<OutputNotebookS3LocationForOutput> builder(MarshallingType.SDK_POJO).memberName("OutputNotebookS3Location")
            .getter(getter(NotebookExecution::outputNotebookS3Location)).setter(setter(Builder::outputNotebookS3Location))
            .constructor(OutputNotebookS3LocationForOutput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputNotebookS3Location").build())
            .build();

    private static final SdkField<String> OUTPUT_NOTEBOOK_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutputNotebookFormat").getter(getter(NotebookExecution::outputNotebookFormatAsString))
            .setter(setter(Builder::outputNotebookFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputNotebookFormat").build())
            .build();

    private static final SdkField<Map<String, String>> ENVIRONMENT_VARIABLES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("EnvironmentVariables")
            .getter(getter(NotebookExecution::environmentVariables))
            .setter(setter(Builder::environmentVariables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentVariables").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NOTEBOOK_EXECUTION_ID_FIELD,
            EDITOR_ID_FIELD, EXECUTION_ENGINE_FIELD, NOTEBOOK_EXECUTION_NAME_FIELD, NOTEBOOK_PARAMS_FIELD, STATUS_FIELD,
            START_TIME_FIELD, END_TIME_FIELD, ARN_FIELD, OUTPUT_NOTEBOOK_URI_FIELD, LAST_STATE_CHANGE_REASON_FIELD,
            NOTEBOOK_INSTANCE_SECURITY_GROUP_ID_FIELD, TAGS_FIELD, NOTEBOOK_S3_LOCATION_FIELD, OUTPUT_NOTEBOOK_S3_LOCATION_FIELD,
            OUTPUT_NOTEBOOK_FORMAT_FIELD, ENVIRONMENT_VARIABLES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String notebookExecutionId;

    private final String editorId;

    private final ExecutionEngineConfig executionEngine;

    private final String notebookExecutionName;

    private final String notebookParams;

    private final String status;

    private final Instant startTime;

    private final Instant endTime;

    private final String arn;

    private final String outputNotebookURI;

    private final String lastStateChangeReason;

    private final String notebookInstanceSecurityGroupId;

    private final List<Tag> tags;

    private final NotebookS3LocationForOutput notebookS3Location;

    private final OutputNotebookS3LocationForOutput outputNotebookS3Location;

    private final String outputNotebookFormat;

    private final Map<String, String> environmentVariables;

    private NotebookExecution(BuilderImpl builder) {
        this.notebookExecutionId = builder.notebookExecutionId;
        this.editorId = builder.editorId;
        this.executionEngine = builder.executionEngine;
        this.notebookExecutionName = builder.notebookExecutionName;
        this.notebookParams = builder.notebookParams;
        this.status = builder.status;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.arn = builder.arn;
        this.outputNotebookURI = builder.outputNotebookURI;
        this.lastStateChangeReason = builder.lastStateChangeReason;
        this.notebookInstanceSecurityGroupId = builder.notebookInstanceSecurityGroupId;
        this.tags = builder.tags;
        this.notebookS3Location = builder.notebookS3Location;
        this.outputNotebookS3Location = builder.outputNotebookS3Location;
        this.outputNotebookFormat = builder.outputNotebookFormat;
        this.environmentVariables = builder.environmentVariables;
    }

    /**
     * <p>
     * The unique identifier of a notebook execution.
     * </p>
     * 
     * @return The unique identifier of a notebook execution.
     */
    public final String notebookExecutionId() {
        return notebookExecutionId;
    }

    /**
     * <p>
     * The unique identifier of the Amazon EMR Notebook that is used for the notebook execution.
     * </p>
     * 
     * @return The unique identifier of the Amazon EMR Notebook that is used for the notebook execution.
     */
    public final String editorId() {
        return editorId;
    }

    /**
     * <p>
     * The execution engine, such as an Amazon EMR cluster, used to run the Amazon EMR notebook and perform the notebook
     * execution.
     * </p>
     * 
     * @return The execution engine, such as an Amazon EMR cluster, used to run the Amazon EMR notebook and perform the
     *         notebook execution.
     */
    public final ExecutionEngineConfig executionEngine() {
        return executionEngine;
    }

    /**
     * <p>
     * A name for the notebook execution.
     * </p>
     * 
     * @return A name for the notebook execution.
     */
    public final String notebookExecutionName() {
        return notebookExecutionName;
    }

    /**
     * <p>
     * Input parameters in JSON format passed to the Amazon EMR Notebook at runtime for execution.
     * </p>
     * 
     * @return Input parameters in JSON format passed to the Amazon EMR Notebook at runtime for execution.
     */
    public final String notebookParams() {
        return notebookParams;
    }

    /**
     * <p>
     * The status of the notebook execution.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>START_PENDING</code> indicates that the cluster has received the execution request but execution has not
     * begun.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STARTING</code> indicates that the execution is starting on the cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RUNNING</code> indicates that the execution is being processed by the cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FINISHING</code> indicates that execution processing is in the final stages.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FINISHED</code> indicates that the execution has completed without error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILING</code> indicates that the execution is failing and will not finish successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> indicates that the execution failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOP_PENDING</code> indicates that the cluster has received a <code>StopNotebookExecution</code> request
     * and the stop is pending.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPING</code> indicates that the cluster is in the process of stopping the execution as a result of a
     * <code>StopNotebookExecution</code> request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPED</code> indicates that the execution stopped because of a <code>StopNotebookExecution</code>
     * request.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link NotebookExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the notebook execution.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>START_PENDING</code> indicates that the cluster has received the execution request but execution
     *         has not begun.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STARTING</code> indicates that the execution is starting on the cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RUNNING</code> indicates that the execution is being processed by the cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FINISHING</code> indicates that execution processing is in the final stages.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FINISHED</code> indicates that the execution has completed without error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILING</code> indicates that the execution is failing and will not finish successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> indicates that the execution failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOP_PENDING</code> indicates that the cluster has received a <code>StopNotebookExecution</code>
     *         request and the stop is pending.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPING</code> indicates that the cluster is in the process of stopping the execution as a result
     *         of a <code>StopNotebookExecution</code> request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPED</code> indicates that the execution stopped because of a <code>StopNotebookExecution</code>
     *         request.
     *         </p>
     *         </li>
     * @see NotebookExecutionStatus
     */
    public final NotebookExecutionStatus status() {
        return NotebookExecutionStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the notebook execution.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>START_PENDING</code> indicates that the cluster has received the execution request but execution has not
     * begun.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STARTING</code> indicates that the execution is starting on the cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RUNNING</code> indicates that the execution is being processed by the cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FINISHING</code> indicates that execution processing is in the final stages.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FINISHED</code> indicates that the execution has completed without error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILING</code> indicates that the execution is failing and will not finish successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> indicates that the execution failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOP_PENDING</code> indicates that the cluster has received a <code>StopNotebookExecution</code> request
     * and the stop is pending.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPING</code> indicates that the cluster is in the process of stopping the execution as a result of a
     * <code>StopNotebookExecution</code> request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPED</code> indicates that the execution stopped because of a <code>StopNotebookExecution</code>
     * request.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link NotebookExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the notebook execution.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>START_PENDING</code> indicates that the cluster has received the execution request but execution
     *         has not begun.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STARTING</code> indicates that the execution is starting on the cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RUNNING</code> indicates that the execution is being processed by the cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FINISHING</code> indicates that execution processing is in the final stages.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FINISHED</code> indicates that the execution has completed without error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILING</code> indicates that the execution is failing and will not finish successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> indicates that the execution failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOP_PENDING</code> indicates that the cluster has received a <code>StopNotebookExecution</code>
     *         request and the stop is pending.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPING</code> indicates that the cluster is in the process of stopping the execution as a result
     *         of a <code>StopNotebookExecution</code> request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPED</code> indicates that the execution stopped because of a <code>StopNotebookExecution</code>
     *         request.
     *         </p>
     *         </li>
     * @see NotebookExecutionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The timestamp when notebook execution started.
     * </p>
     * 
     * @return The timestamp when notebook execution started.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The timestamp when notebook execution ended.
     * </p>
     * 
     * @return The timestamp when notebook execution ended.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the notebook execution.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the notebook execution.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The location of the notebook execution's output file in Amazon S3.
     * </p>
     * 
     * @return The location of the notebook execution's output file in Amazon S3.
     */
    public final String outputNotebookURI() {
        return outputNotebookURI;
    }

    /**
     * <p>
     * The reason for the latest status change of the notebook execution.
     * </p>
     * 
     * @return The reason for the latest status change of the notebook execution.
     */
    public final String lastStateChangeReason() {
        return lastStateChangeReason;
    }

    /**
     * <p>
     * The unique identifier of the Amazon EC2 security group associated with the Amazon EMR Notebook instance. For more
     * information see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-security-groups.html"
     * >Specifying Amazon EC2 Security Groups for Amazon EMR Notebooks</a> in the <i>Amazon EMR Management Guide</i>.
     * </p>
     * 
     * @return The unique identifier of the Amazon EC2 security group associated with the Amazon EMR Notebook instance.
     *         For more information see <a href=
     *         "https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-security-groups.html"
     *         >Specifying Amazon EC2 Security Groups for Amazon EMR Notebooks</a> in the <i>Amazon EMR Management
     *         Guide</i>.
     */
    public final String notebookInstanceSecurityGroupId() {
        return notebookInstanceSecurityGroupId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags associated with a notebook execution. Tags are user-defined key-value pairs that consist of a
     * required key string with a maximum of 128 characters and an optional value string with a maximum of 256
     * characters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags associated with a notebook execution. Tags are user-defined key-value pairs that consist
     *         of a required key string with a maximum of 128 characters and an optional value string with a maximum of
     *         256 characters.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The Amazon S3 location that stores the notebook execution input.
     * </p>
     * 
     * @return The Amazon S3 location that stores the notebook execution input.
     */
    public final NotebookS3LocationForOutput notebookS3Location() {
        return notebookS3Location;
    }

    /**
     * <p>
     * The Amazon S3 location for the notebook execution output.
     * </p>
     * 
     * @return The Amazon S3 location for the notebook execution output.
     */
    public final OutputNotebookS3LocationForOutput outputNotebookS3Location() {
        return outputNotebookS3Location;
    }

    /**
     * <p>
     * The output format for the notebook execution.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #outputNotebookFormat} will return {@link OutputNotebookFormat#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #outputNotebookFormatAsString}.
     * </p>
     * 
     * @return The output format for the notebook execution.
     * @see OutputNotebookFormat
     */
    public final OutputNotebookFormat outputNotebookFormat() {
        return OutputNotebookFormat.fromValue(outputNotebookFormat);
    }

    /**
     * <p>
     * The output format for the notebook execution.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #outputNotebookFormat} will return {@link OutputNotebookFormat#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #outputNotebookFormatAsString}.
     * </p>
     * 
     * @return The output format for the notebook execution.
     * @see OutputNotebookFormat
     */
    public final String outputNotebookFormatAsString() {
        return outputNotebookFormat;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnvironmentVariables property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironmentVariables() {
        return environmentVariables != null && !(environmentVariables instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The environment variables associated with the notebook execution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironmentVariables} method.
     * </p>
     * 
     * @return The environment variables associated with the notebook execution.
     */
    public final Map<String, String> environmentVariables() {
        return environmentVariables;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(notebookExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(editorId());
        hashCode = 31 * hashCode + Objects.hashCode(executionEngine());
        hashCode = 31 * hashCode + Objects.hashCode(notebookExecutionName());
        hashCode = 31 * hashCode + Objects.hashCode(notebookParams());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(outputNotebookURI());
        hashCode = 31 * hashCode + Objects.hashCode(lastStateChangeReason());
        hashCode = 31 * hashCode + Objects.hashCode(notebookInstanceSecurityGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(notebookS3Location());
        hashCode = 31 * hashCode + Objects.hashCode(outputNotebookS3Location());
        hashCode = 31 * hashCode + Objects.hashCode(outputNotebookFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironmentVariables() ? environmentVariables() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NotebookExecution)) {
            return false;
        }
        NotebookExecution other = (NotebookExecution) obj;
        return Objects.equals(notebookExecutionId(), other.notebookExecutionId()) && Objects.equals(editorId(), other.editorId())
                && Objects.equals(executionEngine(), other.executionEngine())
                && Objects.equals(notebookExecutionName(), other.notebookExecutionName())
                && Objects.equals(notebookParams(), other.notebookParams())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(arn(), other.arn())
                && Objects.equals(outputNotebookURI(), other.outputNotebookURI())
                && Objects.equals(lastStateChangeReason(), other.lastStateChangeReason())
                && Objects.equals(notebookInstanceSecurityGroupId(), other.notebookInstanceSecurityGroupId())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(notebookS3Location(), other.notebookS3Location())
                && Objects.equals(outputNotebookS3Location(), other.outputNotebookS3Location())
                && Objects.equals(outputNotebookFormatAsString(), other.outputNotebookFormatAsString())
                && hasEnvironmentVariables() == other.hasEnvironmentVariables()
                && Objects.equals(environmentVariables(), other.environmentVariables());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NotebookExecution").add("NotebookExecutionId", notebookExecutionId())
                .add("EditorId", editorId()).add("ExecutionEngine", executionEngine())
                .add("NotebookExecutionName", notebookExecutionName()).add("NotebookParams", notebookParams())
                .add("Status", statusAsString()).add("StartTime", startTime()).add("EndTime", endTime()).add("Arn", arn())
                .add("OutputNotebookURI", outputNotebookURI()).add("LastStateChangeReason", lastStateChangeReason())
                .add("NotebookInstanceSecurityGroupId", notebookInstanceSecurityGroupId()).add("Tags", hasTags() ? tags() : null)
                .add("NotebookS3Location", notebookS3Location()).add("OutputNotebookS3Location", outputNotebookS3Location())
                .add("OutputNotebookFormat", outputNotebookFormatAsString())
                .add("EnvironmentVariables", hasEnvironmentVariables() ? environmentVariables() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NotebookExecutionId":
            return Optional.ofNullable(clazz.cast(notebookExecutionId()));
        case "EditorId":
            return Optional.ofNullable(clazz.cast(editorId()));
        case "ExecutionEngine":
            return Optional.ofNullable(clazz.cast(executionEngine()));
        case "NotebookExecutionName":
            return Optional.ofNullable(clazz.cast(notebookExecutionName()));
        case "NotebookParams":
            return Optional.ofNullable(clazz.cast(notebookParams()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "OutputNotebookURI":
            return Optional.ofNullable(clazz.cast(outputNotebookURI()));
        case "LastStateChangeReason":
            return Optional.ofNullable(clazz.cast(lastStateChangeReason()));
        case "NotebookInstanceSecurityGroupId":
            return Optional.ofNullable(clazz.cast(notebookInstanceSecurityGroupId()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "NotebookS3Location":
            return Optional.ofNullable(clazz.cast(notebookS3Location()));
        case "OutputNotebookS3Location":
            return Optional.ofNullable(clazz.cast(outputNotebookS3Location()));
        case "OutputNotebookFormat":
            return Optional.ofNullable(clazz.cast(outputNotebookFormatAsString()));
        case "EnvironmentVariables":
            return Optional.ofNullable(clazz.cast(environmentVariables()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("NotebookExecutionId", NOTEBOOK_EXECUTION_ID_FIELD);
        map.put("EditorId", EDITOR_ID_FIELD);
        map.put("ExecutionEngine", EXECUTION_ENGINE_FIELD);
        map.put("NotebookExecutionName", NOTEBOOK_EXECUTION_NAME_FIELD);
        map.put("NotebookParams", NOTEBOOK_PARAMS_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("StartTime", START_TIME_FIELD);
        map.put("EndTime", END_TIME_FIELD);
        map.put("Arn", ARN_FIELD);
        map.put("OutputNotebookURI", OUTPUT_NOTEBOOK_URI_FIELD);
        map.put("LastStateChangeReason", LAST_STATE_CHANGE_REASON_FIELD);
        map.put("NotebookInstanceSecurityGroupId", NOTEBOOK_INSTANCE_SECURITY_GROUP_ID_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("NotebookS3Location", NOTEBOOK_S3_LOCATION_FIELD);
        map.put("OutputNotebookS3Location", OUTPUT_NOTEBOOK_S3_LOCATION_FIELD);
        map.put("OutputNotebookFormat", OUTPUT_NOTEBOOK_FORMAT_FIELD);
        map.put("EnvironmentVariables", ENVIRONMENT_VARIABLES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<NotebookExecution, T> g) {
        return obj -> g.apply((NotebookExecution) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NotebookExecution> {
        /**
         * <p>
         * The unique identifier of a notebook execution.
         * </p>
         * 
         * @param notebookExecutionId
         *        The unique identifier of a notebook execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notebookExecutionId(String notebookExecutionId);

        /**
         * <p>
         * The unique identifier of the Amazon EMR Notebook that is used for the notebook execution.
         * </p>
         * 
         * @param editorId
         *        The unique identifier of the Amazon EMR Notebook that is used for the notebook execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder editorId(String editorId);

        /**
         * <p>
         * The execution engine, such as an Amazon EMR cluster, used to run the Amazon EMR notebook and perform the
         * notebook execution.
         * </p>
         * 
         * @param executionEngine
         *        The execution engine, such as an Amazon EMR cluster, used to run the Amazon EMR notebook and perform
         *        the notebook execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionEngine(ExecutionEngineConfig executionEngine);

        /**
         * <p>
         * The execution engine, such as an Amazon EMR cluster, used to run the Amazon EMR notebook and perform the
         * notebook execution.
         * </p>
         * This is a convenience method that creates an instance of the {@link ExecutionEngineConfig.Builder} avoiding
         * the need to create one manually via {@link ExecutionEngineConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ExecutionEngineConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #executionEngine(ExecutionEngineConfig)}.
         * 
         * @param executionEngine
         *        a consumer that will call methods on {@link ExecutionEngineConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #executionEngine(ExecutionEngineConfig)
         */
        default Builder executionEngine(Consumer<ExecutionEngineConfig.Builder> executionEngine) {
            return executionEngine(ExecutionEngineConfig.builder().applyMutation(executionEngine).build());
        }

        /**
         * <p>
         * A name for the notebook execution.
         * </p>
         * 
         * @param notebookExecutionName
         *        A name for the notebook execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notebookExecutionName(String notebookExecutionName);

        /**
         * <p>
         * Input parameters in JSON format passed to the Amazon EMR Notebook at runtime for execution.
         * </p>
         * 
         * @param notebookParams
         *        Input parameters in JSON format passed to the Amazon EMR Notebook at runtime for execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notebookParams(String notebookParams);

        /**
         * <p>
         * The status of the notebook execution.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>START_PENDING</code> indicates that the cluster has received the execution request but execution has
         * not begun.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STARTING</code> indicates that the execution is starting on the cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RUNNING</code> indicates that the execution is being processed by the cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FINISHING</code> indicates that execution processing is in the final stages.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FINISHED</code> indicates that the execution has completed without error.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILING</code> indicates that the execution is failing and will not finish successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> indicates that the execution failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOP_PENDING</code> indicates that the cluster has received a <code>StopNotebookExecution</code>
         * request and the stop is pending.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPING</code> indicates that the cluster is in the process of stopping the execution as a result of a
         * <code>StopNotebookExecution</code> request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPED</code> indicates that the execution stopped because of a <code>StopNotebookExecution</code>
         * request.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the notebook execution.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>START_PENDING</code> indicates that the cluster has received the execution request but execution
         *        has not begun.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STARTING</code> indicates that the execution is starting on the cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RUNNING</code> indicates that the execution is being processed by the cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FINISHING</code> indicates that execution processing is in the final stages.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FINISHED</code> indicates that the execution has completed without error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILING</code> indicates that the execution is failing and will not finish successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> indicates that the execution failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOP_PENDING</code> indicates that the cluster has received a <code>StopNotebookExecution</code>
         *        request and the stop is pending.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPING</code> indicates that the cluster is in the process of stopping the execution as a
         *        result of a <code>StopNotebookExecution</code> request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPED</code> indicates that the execution stopped because of a
         *        <code>StopNotebookExecution</code> request.
         *        </p>
         *        </li>
         * @see NotebookExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotebookExecutionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the notebook execution.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>START_PENDING</code> indicates that the cluster has received the execution request but execution has
         * not begun.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STARTING</code> indicates that the execution is starting on the cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RUNNING</code> indicates that the execution is being processed by the cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FINISHING</code> indicates that execution processing is in the final stages.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FINISHED</code> indicates that the execution has completed without error.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILING</code> indicates that the execution is failing and will not finish successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> indicates that the execution failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOP_PENDING</code> indicates that the cluster has received a <code>StopNotebookExecution</code>
         * request and the stop is pending.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPING</code> indicates that the cluster is in the process of stopping the execution as a result of a
         * <code>StopNotebookExecution</code> request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPED</code> indicates that the execution stopped because of a <code>StopNotebookExecution</code>
         * request.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the notebook execution.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>START_PENDING</code> indicates that the cluster has received the execution request but execution
         *        has not begun.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STARTING</code> indicates that the execution is starting on the cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RUNNING</code> indicates that the execution is being processed by the cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FINISHING</code> indicates that execution processing is in the final stages.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FINISHED</code> indicates that the execution has completed without error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILING</code> indicates that the execution is failing and will not finish successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> indicates that the execution failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOP_PENDING</code> indicates that the cluster has received a <code>StopNotebookExecution</code>
         *        request and the stop is pending.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPING</code> indicates that the cluster is in the process of stopping the execution as a
         *        result of a <code>StopNotebookExecution</code> request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPED</code> indicates that the execution stopped because of a
         *        <code>StopNotebookExecution</code> request.
         *        </p>
         *        </li>
         * @see NotebookExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NotebookExecutionStatus
         */
        Builder status(NotebookExecutionStatus status);

        /**
         * <p>
         * The timestamp when notebook execution started.
         * </p>
         * 
         * @param startTime
         *        The timestamp when notebook execution started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The timestamp when notebook execution ended.
         * </p>
         * 
         * @param endTime
         *        The timestamp when notebook execution ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the notebook execution.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the notebook execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The location of the notebook execution's output file in Amazon S3.
         * </p>
         * 
         * @param outputNotebookURI
         *        The location of the notebook execution's output file in Amazon S3.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputNotebookURI(String outputNotebookURI);

        /**
         * <p>
         * The reason for the latest status change of the notebook execution.
         * </p>
         * 
         * @param lastStateChangeReason
         *        The reason for the latest status change of the notebook execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastStateChangeReason(String lastStateChangeReason);

        /**
         * <p>
         * The unique identifier of the Amazon EC2 security group associated with the Amazon EMR Notebook instance. For
         * more information see <a
         * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-security-groups.html"
         * >Specifying Amazon EC2 Security Groups for Amazon EMR Notebooks</a> in the <i>Amazon EMR Management
         * Guide</i>.
         * </p>
         * 
         * @param notebookInstanceSecurityGroupId
         *        The unique identifier of the Amazon EC2 security group associated with the Amazon EMR Notebook
         *        instance. For more information see <a href=
         *        "https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-security-groups.html"
         *        >Specifying Amazon EC2 Security Groups for Amazon EMR Notebooks</a> in the <i>Amazon EMR Management
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notebookInstanceSecurityGroupId(String notebookInstanceSecurityGroupId);

        /**
         * <p>
         * A list of tags associated with a notebook execution. Tags are user-defined key-value pairs that consist of a
         * required key string with a maximum of 128 characters and an optional value string with a maximum of 256
         * characters.
         * </p>
         * 
         * @param tags
         *        A list of tags associated with a notebook execution. Tags are user-defined key-value pairs that
         *        consist of a required key string with a maximum of 128 characters and an optional value string with a
         *        maximum of 256 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags associated with a notebook execution. Tags are user-defined key-value pairs that consist of a
         * required key string with a maximum of 128 characters and an optional value string with a maximum of 256
         * characters.
         * </p>
         * 
         * @param tags
         *        A list of tags associated with a notebook execution. Tags are user-defined key-value pairs that
         *        consist of a required key string with a maximum of 128 characters and an optional value string with a
         *        maximum of 256 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags associated with a notebook execution. Tags are user-defined key-value pairs that consist of a
         * required key string with a maximum of 128 characters and an optional value string with a maximum of 256
         * characters.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.emr.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.emr.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.emr.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.emr.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The Amazon S3 location that stores the notebook execution input.
         * </p>
         * 
         * @param notebookS3Location
         *        The Amazon S3 location that stores the notebook execution input.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notebookS3Location(NotebookS3LocationForOutput notebookS3Location);

        /**
         * <p>
         * The Amazon S3 location that stores the notebook execution input.
         * </p>
         * This is a convenience method that creates an instance of the {@link NotebookS3LocationForOutput.Builder}
         * avoiding the need to create one manually via {@link NotebookS3LocationForOutput#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NotebookS3LocationForOutput.Builder#build()} is called
         * immediately and its result is passed to {@link #notebookS3Location(NotebookS3LocationForOutput)}.
         * 
         * @param notebookS3Location
         *        a consumer that will call methods on {@link NotebookS3LocationForOutput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notebookS3Location(NotebookS3LocationForOutput)
         */
        default Builder notebookS3Location(Consumer<NotebookS3LocationForOutput.Builder> notebookS3Location) {
            return notebookS3Location(NotebookS3LocationForOutput.builder().applyMutation(notebookS3Location).build());
        }

        /**
         * <p>
         * The Amazon S3 location for the notebook execution output.
         * </p>
         * 
         * @param outputNotebookS3Location
         *        The Amazon S3 location for the notebook execution output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputNotebookS3Location(OutputNotebookS3LocationForOutput outputNotebookS3Location);

        /**
         * <p>
         * The Amazon S3 location for the notebook execution output.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link OutputNotebookS3LocationForOutput.Builder} avoiding the need to create one manually via
         * {@link OutputNotebookS3LocationForOutput#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OutputNotebookS3LocationForOutput.Builder#build()} is called
         * immediately and its result is passed to {@link #outputNotebookS3Location(OutputNotebookS3LocationForOutput)}.
         * 
         * @param outputNotebookS3Location
         *        a consumer that will call methods on {@link OutputNotebookS3LocationForOutput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputNotebookS3Location(OutputNotebookS3LocationForOutput)
         */
        default Builder outputNotebookS3Location(Consumer<OutputNotebookS3LocationForOutput.Builder> outputNotebookS3Location) {
            return outputNotebookS3Location(OutputNotebookS3LocationForOutput.builder().applyMutation(outputNotebookS3Location)
                    .build());
        }

        /**
         * <p>
         * The output format for the notebook execution.
         * </p>
         * 
         * @param outputNotebookFormat
         *        The output format for the notebook execution.
         * @see OutputNotebookFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OutputNotebookFormat
         */
        Builder outputNotebookFormat(String outputNotebookFormat);

        /**
         * <p>
         * The output format for the notebook execution.
         * </p>
         * 
         * @param outputNotebookFormat
         *        The output format for the notebook execution.
         * @see OutputNotebookFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OutputNotebookFormat
         */
        Builder outputNotebookFormat(OutputNotebookFormat outputNotebookFormat);

        /**
         * <p>
         * The environment variables associated with the notebook execution.
         * </p>
         * 
         * @param environmentVariables
         *        The environment variables associated with the notebook execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentVariables(Map<String, String> environmentVariables);
    }

    static final class BuilderImpl implements Builder {
        private String notebookExecutionId;

        private String editorId;

        private ExecutionEngineConfig executionEngine;

        private String notebookExecutionName;

        private String notebookParams;

        private String status;

        private Instant startTime;

        private Instant endTime;

        private String arn;

        private String outputNotebookURI;

        private String lastStateChangeReason;

        private String notebookInstanceSecurityGroupId;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private NotebookS3LocationForOutput notebookS3Location;

        private OutputNotebookS3LocationForOutput outputNotebookS3Location;

        private String outputNotebookFormat;

        private Map<String, String> environmentVariables = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NotebookExecution model) {
            notebookExecutionId(model.notebookExecutionId);
            editorId(model.editorId);
            executionEngine(model.executionEngine);
            notebookExecutionName(model.notebookExecutionName);
            notebookParams(model.notebookParams);
            status(model.status);
            startTime(model.startTime);
            endTime(model.endTime);
            arn(model.arn);
            outputNotebookURI(model.outputNotebookURI);
            lastStateChangeReason(model.lastStateChangeReason);
            notebookInstanceSecurityGroupId(model.notebookInstanceSecurityGroupId);
            tags(model.tags);
            notebookS3Location(model.notebookS3Location);
            outputNotebookS3Location(model.outputNotebookS3Location);
            outputNotebookFormat(model.outputNotebookFormat);
            environmentVariables(model.environmentVariables);
        }

        public final String getNotebookExecutionId() {
            return notebookExecutionId;
        }

        public final void setNotebookExecutionId(String notebookExecutionId) {
            this.notebookExecutionId = notebookExecutionId;
        }

        @Override
        public final Builder notebookExecutionId(String notebookExecutionId) {
            this.notebookExecutionId = notebookExecutionId;
            return this;
        }

        public final String getEditorId() {
            return editorId;
        }

        public final void setEditorId(String editorId) {
            this.editorId = editorId;
        }

        @Override
        public final Builder editorId(String editorId) {
            this.editorId = editorId;
            return this;
        }

        public final ExecutionEngineConfig.Builder getExecutionEngine() {
            return executionEngine != null ? executionEngine.toBuilder() : null;
        }

        public final void setExecutionEngine(ExecutionEngineConfig.BuilderImpl executionEngine) {
            this.executionEngine = executionEngine != null ? executionEngine.build() : null;
        }

        @Override
        public final Builder executionEngine(ExecutionEngineConfig executionEngine) {
            this.executionEngine = executionEngine;
            return this;
        }

        public final String getNotebookExecutionName() {
            return notebookExecutionName;
        }

        public final void setNotebookExecutionName(String notebookExecutionName) {
            this.notebookExecutionName = notebookExecutionName;
        }

        @Override
        public final Builder notebookExecutionName(String notebookExecutionName) {
            this.notebookExecutionName = notebookExecutionName;
            return this;
        }

        public final String getNotebookParams() {
            return notebookParams;
        }

        public final void setNotebookParams(String notebookParams) {
            this.notebookParams = notebookParams;
        }

        @Override
        public final Builder notebookParams(String notebookParams) {
            this.notebookParams = notebookParams;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(NotebookExecutionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getOutputNotebookURI() {
            return outputNotebookURI;
        }

        public final void setOutputNotebookURI(String outputNotebookURI) {
            this.outputNotebookURI = outputNotebookURI;
        }

        @Override
        public final Builder outputNotebookURI(String outputNotebookURI) {
            this.outputNotebookURI = outputNotebookURI;
            return this;
        }

        public final String getLastStateChangeReason() {
            return lastStateChangeReason;
        }

        public final void setLastStateChangeReason(String lastStateChangeReason) {
            this.lastStateChangeReason = lastStateChangeReason;
        }

        @Override
        public final Builder lastStateChangeReason(String lastStateChangeReason) {
            this.lastStateChangeReason = lastStateChangeReason;
            return this;
        }

        public final String getNotebookInstanceSecurityGroupId() {
            return notebookInstanceSecurityGroupId;
        }

        public final void setNotebookInstanceSecurityGroupId(String notebookInstanceSecurityGroupId) {
            this.notebookInstanceSecurityGroupId = notebookInstanceSecurityGroupId;
        }

        @Override
        public final Builder notebookInstanceSecurityGroupId(String notebookInstanceSecurityGroupId) {
            this.notebookInstanceSecurityGroupId = notebookInstanceSecurityGroupId;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final NotebookS3LocationForOutput.Builder getNotebookS3Location() {
            return notebookS3Location != null ? notebookS3Location.toBuilder() : null;
        }

        public final void setNotebookS3Location(NotebookS3LocationForOutput.BuilderImpl notebookS3Location) {
            this.notebookS3Location = notebookS3Location != null ? notebookS3Location.build() : null;
        }

        @Override
        public final Builder notebookS3Location(NotebookS3LocationForOutput notebookS3Location) {
            this.notebookS3Location = notebookS3Location;
            return this;
        }

        public final OutputNotebookS3LocationForOutput.Builder getOutputNotebookS3Location() {
            return outputNotebookS3Location != null ? outputNotebookS3Location.toBuilder() : null;
        }

        public final void setOutputNotebookS3Location(OutputNotebookS3LocationForOutput.BuilderImpl outputNotebookS3Location) {
            this.outputNotebookS3Location = outputNotebookS3Location != null ? outputNotebookS3Location.build() : null;
        }

        @Override
        public final Builder outputNotebookS3Location(OutputNotebookS3LocationForOutput outputNotebookS3Location) {
            this.outputNotebookS3Location = outputNotebookS3Location;
            return this;
        }

        public final String getOutputNotebookFormat() {
            return outputNotebookFormat;
        }

        public final void setOutputNotebookFormat(String outputNotebookFormat) {
            this.outputNotebookFormat = outputNotebookFormat;
        }

        @Override
        public final Builder outputNotebookFormat(String outputNotebookFormat) {
            this.outputNotebookFormat = outputNotebookFormat;
            return this;
        }

        @Override
        public final Builder outputNotebookFormat(OutputNotebookFormat outputNotebookFormat) {
            this.outputNotebookFormat(outputNotebookFormat == null ? null : outputNotebookFormat.toString());
            return this;
        }

        public final Map<String, String> getEnvironmentVariables() {
            if (environmentVariables instanceof SdkAutoConstructMap) {
                return null;
            }
            return environmentVariables;
        }

        public final void setEnvironmentVariables(Map<String, String> environmentVariables) {
            this.environmentVariables = EnvironmentVariablesMapCopier.copy(environmentVariables);
        }

        @Override
        public final Builder environmentVariables(Map<String, String> environmentVariables) {
            this.environmentVariables = EnvironmentVariablesMapCopier.copy(environmentVariables);
            return this;
        }

        @Override
        public NotebookExecution build() {
            return new NotebookExecution(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
