/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emrcontainers.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This entity represents the endpoint that is managed by Amazon EMR on EKS.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Endpoint implements SdkPojo, Serializable, ToCopyableBuilder<Endpoint.Builder, Endpoint> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(Endpoint::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Endpoint::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Endpoint::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> VIRTUAL_CLUSTER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("virtualClusterId").getter(getter(Endpoint::virtualClusterId)).setter(setter(Builder::virtualClusterId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("virtualClusterId").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(Endpoint::type)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(Endpoint::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> RELEASE_LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("releaseLabel").getter(getter(Endpoint::releaseLabel)).setter(setter(Builder::releaseLabel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("releaseLabel").build()).build();

    private static final SdkField<String> EXECUTION_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("executionRoleArn").getter(getter(Endpoint::executionRoleArn)).setter(setter(Builder::executionRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionRoleArn").build()).build();

    private static final SdkField<String> CERTIFICATE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("certificateArn").getter(getter(Endpoint::certificateArn)).setter(setter(Builder::certificateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateArn").build()).build();

    private static final SdkField<ConfigurationOverrides> CONFIGURATION_OVERRIDES_FIELD = SdkField
            .<ConfigurationOverrides> builder(MarshallingType.SDK_POJO).memberName("configurationOverrides")
            .getter(getter(Endpoint::configurationOverrides)).setter(setter(Builder::configurationOverrides))
            .constructor(ConfigurationOverrides::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configurationOverrides").build())
            .build();

    private static final SdkField<String> SERVER_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serverUrl").getter(getter(Endpoint::serverUrl)).setter(setter(Builder::serverUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serverUrl").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(Endpoint::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<String> SECURITY_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("securityGroup").getter(getter(Endpoint::securityGroup)).setter(setter(Builder::securityGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("securityGroup").build()).build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("subnetIds")
            .getter(getter(Endpoint::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Endpoint::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, NAME_FIELD,
            ARN_FIELD, VIRTUAL_CLUSTER_ID_FIELD, TYPE_FIELD, STATE_FIELD, RELEASE_LABEL_FIELD, EXECUTION_ROLE_ARN_FIELD,
            CERTIFICATE_ARN_FIELD, CONFIGURATION_OVERRIDES_FIELD, SERVER_URL_FIELD, CREATED_AT_FIELD, SECURITY_GROUP_FIELD,
            SUBNET_IDS_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String name;

    private final String arn;

    private final String virtualClusterId;

    private final String type;

    private final String state;

    private final String releaseLabel;

    private final String executionRoleArn;

    private final String certificateArn;

    private final ConfigurationOverrides configurationOverrides;

    private final String serverUrl;

    private final Instant createdAt;

    private final String securityGroup;

    private final List<String> subnetIds;

    private final Map<String, String> tags;

    private Endpoint(BuilderImpl builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.arn = builder.arn;
        this.virtualClusterId = builder.virtualClusterId;
        this.type = builder.type;
        this.state = builder.state;
        this.releaseLabel = builder.releaseLabel;
        this.executionRoleArn = builder.executionRoleArn;
        this.certificateArn = builder.certificateArn;
        this.configurationOverrides = builder.configurationOverrides;
        this.serverUrl = builder.serverUrl;
        this.createdAt = builder.createdAt;
        this.securityGroup = builder.securityGroup;
        this.subnetIds = builder.subnetIds;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ID of the endpoint.
     * </p>
     * 
     * @return The ID of the endpoint.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The name of the endpoint.
     * </p>
     * 
     * @return The name of the endpoint.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The ARN of the endpoint.
     * </p>
     * 
     * @return The ARN of the endpoint.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The ID of the endpoint's virtual cluster.
     * </p>
     * 
     * @return The ID of the endpoint's virtual cluster.
     */
    public String virtualClusterId() {
        return virtualClusterId;
    }

    /**
     * <p>
     * The type of the endpoint.
     * </p>
     * 
     * @return The type of the endpoint.
     */
    public String type() {
        return type;
    }

    /**
     * <p>
     * The state of the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link EndpointState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the endpoint.
     * @see EndpointState
     */
    public EndpointState state() {
        return EndpointState.fromValue(state);
    }

    /**
     * <p>
     * The state of the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link EndpointState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the endpoint.
     * @see EndpointState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The EMR release version to be used for the endpoint.
     * </p>
     * 
     * @return The EMR release version to be used for the endpoint.
     */
    public String releaseLabel() {
        return releaseLabel;
    }

    /**
     * <p>
     * The execution role ARN of the endpoint.
     * </p>
     * 
     * @return The execution role ARN of the endpoint.
     */
    public String executionRoleArn() {
        return executionRoleArn;
    }

    /**
     * <p>
     * The certificate ARN of the endpoint.
     * </p>
     * 
     * @return The certificate ARN of the endpoint.
     */
    public String certificateArn() {
        return certificateArn;
    }

    /**
     * <p>
     * The configuration settings that are used to override existing configurations for endpoints.
     * </p>
     * 
     * @return The configuration settings that are used to override existing configurations for endpoints.
     */
    public ConfigurationOverrides configurationOverrides() {
        return configurationOverrides;
    }

    /**
     * <p>
     * The server URL of the endpoint.
     * </p>
     * 
     * @return The server URL of the endpoint.
     */
    public String serverUrl() {
        return serverUrl;
    }

    /**
     * <p>
     * The date and time when the endpoint was created.
     * </p>
     * 
     * @return The date and time when the endpoint was created.
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The security group configuration of the endpoint.
     * </p>
     * 
     * @return The security group configuration of the endpoint.
     */
    public String securityGroup() {
        return securityGroup;
    }

    /**
     * Returns true if the SubnetIds property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The subnet IDs of the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubnetIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The subnet IDs of the endpoint.
     */
    public List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags of the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tags of the endpoint.
     */
    public Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(virtualClusterId());
        hashCode = 31 * hashCode + Objects.hashCode(type());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(releaseLabel());
        hashCode = 31 * hashCode + Objects.hashCode(executionRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(certificateArn());
        hashCode = 31 * hashCode + Objects.hashCode(configurationOverrides());
        hashCode = 31 * hashCode + Objects.hashCode(serverUrl());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(securityGroup());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetIds() ? subnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Endpoint)) {
            return false;
        }
        Endpoint other = (Endpoint) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(virtualClusterId(), other.virtualClusterId()) && Objects.equals(type(), other.type())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(releaseLabel(), other.releaseLabel())
                && Objects.equals(executionRoleArn(), other.executionRoleArn())
                && Objects.equals(certificateArn(), other.certificateArn())
                && Objects.equals(configurationOverrides(), other.configurationOverrides())
                && Objects.equals(serverUrl(), other.serverUrl()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(securityGroup(), other.securityGroup()) && hasSubnetIds() == other.hasSubnetIds()
                && Objects.equals(subnetIds(), other.subnetIds()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Endpoint").add("Id", id()).add("Name", name()).add("Arn", arn())
                .add("VirtualClusterId", virtualClusterId()).add("Type", type()).add("State", stateAsString())
                .add("ReleaseLabel", releaseLabel()).add("ExecutionRoleArn", executionRoleArn())
                .add("CertificateArn", certificateArn()).add("ConfigurationOverrides", configurationOverrides())
                .add("ServerUrl", serverUrl()).add("CreatedAt", createdAt()).add("SecurityGroup", securityGroup())
                .add("SubnetIds", hasSubnetIds() ? subnetIds() : null).add("Tags", hasTags() ? tags() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "virtualClusterId":
            return Optional.ofNullable(clazz.cast(virtualClusterId()));
        case "type":
            return Optional.ofNullable(clazz.cast(type()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "releaseLabel":
            return Optional.ofNullable(clazz.cast(releaseLabel()));
        case "executionRoleArn":
            return Optional.ofNullable(clazz.cast(executionRoleArn()));
        case "certificateArn":
            return Optional.ofNullable(clazz.cast(certificateArn()));
        case "configurationOverrides":
            return Optional.ofNullable(clazz.cast(configurationOverrides()));
        case "serverUrl":
            return Optional.ofNullable(clazz.cast(serverUrl()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "securityGroup":
            return Optional.ofNullable(clazz.cast(securityGroup()));
        case "subnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Endpoint, T> g) {
        return obj -> g.apply((Endpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Endpoint> {
        /**
         * <p>
         * The ID of the endpoint.
         * </p>
         * 
         * @param id
         *        The ID of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the endpoint.
         * </p>
         * 
         * @param name
         *        The name of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ARN of the endpoint.
         * </p>
         * 
         * @param arn
         *        The ARN of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The ID of the endpoint's virtual cluster.
         * </p>
         * 
         * @param virtualClusterId
         *        The ID of the endpoint's virtual cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder virtualClusterId(String virtualClusterId);

        /**
         * <p>
         * The type of the endpoint.
         * </p>
         * 
         * @param type
         *        The type of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder type(String type);

        /**
         * <p>
         * The state of the endpoint.
         * </p>
         * 
         * @param state
         *        The state of the endpoint.
         * @see EndpointState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the endpoint.
         * </p>
         * 
         * @param state
         *        The state of the endpoint.
         * @see EndpointState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointState
         */
        Builder state(EndpointState state);

        /**
         * <p>
         * The EMR release version to be used for the endpoint.
         * </p>
         * 
         * @param releaseLabel
         *        The EMR release version to be used for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder releaseLabel(String releaseLabel);

        /**
         * <p>
         * The execution role ARN of the endpoint.
         * </p>
         * 
         * @param executionRoleArn
         *        The execution role ARN of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRoleArn(String executionRoleArn);

        /**
         * <p>
         * The certificate ARN of the endpoint.
         * </p>
         * 
         * @param certificateArn
         *        The certificate ARN of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateArn(String certificateArn);

        /**
         * <p>
         * The configuration settings that are used to override existing configurations for endpoints.
         * </p>
         * 
         * @param configurationOverrides
         *        The configuration settings that are used to override existing configurations for endpoints.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationOverrides(ConfigurationOverrides configurationOverrides);

        /**
         * <p>
         * The configuration settings that are used to override existing configurations for endpoints.
         * </p>
         * This is a convenience that creates an instance of the {@link ConfigurationOverrides.Builder} avoiding the
         * need to create one manually via {@link ConfigurationOverrides#builder()}.
         *
         * When the {@link Consumer} completes, {@link ConfigurationOverrides.Builder#build()} is called immediately and
         * its result is passed to {@link #configurationOverrides(ConfigurationOverrides)}.
         * 
         * @param configurationOverrides
         *        a consumer that will call methods on {@link ConfigurationOverrides.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configurationOverrides(ConfigurationOverrides)
         */
        default Builder configurationOverrides(Consumer<ConfigurationOverrides.Builder> configurationOverrides) {
            return configurationOverrides(ConfigurationOverrides.builder().applyMutation(configurationOverrides).build());
        }

        /**
         * <p>
         * The server URL of the endpoint.
         * </p>
         * 
         * @param serverUrl
         *        The server URL of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverUrl(String serverUrl);

        /**
         * <p>
         * The date and time when the endpoint was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time when the endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The security group configuration of the endpoint.
         * </p>
         * 
         * @param securityGroup
         *        The security group configuration of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroup(String securityGroup);

        /**
         * <p>
         * The subnet IDs of the endpoint.
         * </p>
         * 
         * @param subnetIds
         *        The subnet IDs of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * <p>
         * The subnet IDs of the endpoint.
         * </p>
         * 
         * @param subnetIds
         *        The subnet IDs of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * <p>
         * The tags of the endpoint.
         * </p>
         * 
         * @param tags
         *        The tags of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String name;

        private String arn;

        private String virtualClusterId;

        private String type;

        private String state;

        private String releaseLabel;

        private String executionRoleArn;

        private String certificateArn;

        private ConfigurationOverrides configurationOverrides;

        private String serverUrl;

        private Instant createdAt;

        private String securityGroup;

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Endpoint model) {
            id(model.id);
            name(model.name);
            arn(model.arn);
            virtualClusterId(model.virtualClusterId);
            type(model.type);
            state(model.state);
            releaseLabel(model.releaseLabel);
            executionRoleArn(model.executionRoleArn);
            certificateArn(model.certificateArn);
            configurationOverrides(model.configurationOverrides);
            serverUrl(model.serverUrl);
            createdAt(model.createdAt);
            securityGroup(model.securityGroup);
            subnetIds(model.subnetIds);
            tags(model.tags);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getVirtualClusterId() {
            return virtualClusterId;
        }

        @Override
        public final Builder virtualClusterId(String virtualClusterId) {
            this.virtualClusterId = virtualClusterId;
            return this;
        }

        public final void setVirtualClusterId(String virtualClusterId) {
            this.virtualClusterId = virtualClusterId;
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(EndpointState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getReleaseLabel() {
            return releaseLabel;
        }

        @Override
        public final Builder releaseLabel(String releaseLabel) {
            this.releaseLabel = releaseLabel;
            return this;
        }

        public final void setReleaseLabel(String releaseLabel) {
            this.releaseLabel = releaseLabel;
        }

        public final String getExecutionRoleArn() {
            return executionRoleArn;
        }

        @Override
        public final Builder executionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
            return this;
        }

        public final void setExecutionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
        }

        public final String getCertificateArn() {
            return certificateArn;
        }

        @Override
        public final Builder certificateArn(String certificateArn) {
            this.certificateArn = certificateArn;
            return this;
        }

        public final void setCertificateArn(String certificateArn) {
            this.certificateArn = certificateArn;
        }

        public final ConfigurationOverrides.Builder getConfigurationOverrides() {
            return configurationOverrides != null ? configurationOverrides.toBuilder() : null;
        }

        @Override
        public final Builder configurationOverrides(ConfigurationOverrides configurationOverrides) {
            this.configurationOverrides = configurationOverrides;
            return this;
        }

        public final void setConfigurationOverrides(ConfigurationOverrides.BuilderImpl configurationOverrides) {
            this.configurationOverrides = configurationOverrides != null ? configurationOverrides.build() : null;
        }

        public final String getServerUrl() {
            return serverUrl;
        }

        @Override
        public final Builder serverUrl(String serverUrl) {
            this.serverUrl = serverUrl;
            return this;
        }

        public final void setServerUrl(String serverUrl) {
            this.serverUrl = serverUrl;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final String getSecurityGroup() {
            return securityGroup;
        }

        @Override
        public final Builder securityGroup(String securityGroup) {
            this.securityGroup = securityGroup;
            return this;
        }

        public final void setSecurityGroup(String securityGroup) {
            this.securityGroup = securityGroup;
        }

        public final Collection<String> getSubnetIds() {
            if (subnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetIds;
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public Endpoint build() {
            return new Endpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
