/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emrcontainers;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.emrcontainers.model.CancelJobRunRequest;
import software.amazon.awssdk.services.emrcontainers.model.CancelJobRunResponse;
import software.amazon.awssdk.services.emrcontainers.model.CreateManagedEndpointRequest;
import software.amazon.awssdk.services.emrcontainers.model.CreateManagedEndpointResponse;
import software.amazon.awssdk.services.emrcontainers.model.CreateVirtualClusterRequest;
import software.amazon.awssdk.services.emrcontainers.model.CreateVirtualClusterResponse;
import software.amazon.awssdk.services.emrcontainers.model.DeleteManagedEndpointRequest;
import software.amazon.awssdk.services.emrcontainers.model.DeleteManagedEndpointResponse;
import software.amazon.awssdk.services.emrcontainers.model.DeleteVirtualClusterRequest;
import software.amazon.awssdk.services.emrcontainers.model.DeleteVirtualClusterResponse;
import software.amazon.awssdk.services.emrcontainers.model.DescribeJobRunRequest;
import software.amazon.awssdk.services.emrcontainers.model.DescribeJobRunResponse;
import software.amazon.awssdk.services.emrcontainers.model.DescribeManagedEndpointRequest;
import software.amazon.awssdk.services.emrcontainers.model.DescribeManagedEndpointResponse;
import software.amazon.awssdk.services.emrcontainers.model.DescribeVirtualClusterRequest;
import software.amazon.awssdk.services.emrcontainers.model.DescribeVirtualClusterResponse;
import software.amazon.awssdk.services.emrcontainers.model.EmrContainersException;
import software.amazon.awssdk.services.emrcontainers.model.EmrContainersRequest;
import software.amazon.awssdk.services.emrcontainers.model.InternalServerException;
import software.amazon.awssdk.services.emrcontainers.model.ListJobRunsRequest;
import software.amazon.awssdk.services.emrcontainers.model.ListJobRunsResponse;
import software.amazon.awssdk.services.emrcontainers.model.ListManagedEndpointsRequest;
import software.amazon.awssdk.services.emrcontainers.model.ListManagedEndpointsResponse;
import software.amazon.awssdk.services.emrcontainers.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.emrcontainers.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.emrcontainers.model.ListVirtualClustersRequest;
import software.amazon.awssdk.services.emrcontainers.model.ListVirtualClustersResponse;
import software.amazon.awssdk.services.emrcontainers.model.ResourceNotFoundException;
import software.amazon.awssdk.services.emrcontainers.model.StartJobRunRequest;
import software.amazon.awssdk.services.emrcontainers.model.StartJobRunResponse;
import software.amazon.awssdk.services.emrcontainers.model.TagResourceRequest;
import software.amazon.awssdk.services.emrcontainers.model.TagResourceResponse;
import software.amazon.awssdk.services.emrcontainers.model.UntagResourceRequest;
import software.amazon.awssdk.services.emrcontainers.model.UntagResourceResponse;
import software.amazon.awssdk.services.emrcontainers.model.ValidationException;
import software.amazon.awssdk.services.emrcontainers.paginators.ListJobRunsPublisher;
import software.amazon.awssdk.services.emrcontainers.paginators.ListManagedEndpointsPublisher;
import software.amazon.awssdk.services.emrcontainers.paginators.ListVirtualClustersPublisher;
import software.amazon.awssdk.services.emrcontainers.transform.CancelJobRunRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.CreateManagedEndpointRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.CreateVirtualClusterRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.DeleteManagedEndpointRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.DeleteVirtualClusterRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.DescribeJobRunRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.DescribeManagedEndpointRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.DescribeVirtualClusterRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.ListJobRunsRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.ListManagedEndpointsRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.ListVirtualClustersRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.StartJobRunRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link EmrContainersAsyncClient}.
 *
 * @see EmrContainersAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEmrContainersAsyncClient implements EmrContainersAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultEmrContainersAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEmrContainersAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels a job run. A job run is a unit of work, such as a Spark jar, PySpark script, or SparkSQL query, that you
     * submit to Amazon EMR on EKS.
     * </p>
     *
     * @param cancelJobRunRequest
     * @return A Java Future containing the result of the CancelJobRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.CancelJobRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/CancelJobRun" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelJobRunResponse> cancelJobRun(CancelJobRunRequest cancelJobRunRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJobRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelJobRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelJobRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelJobRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelJobRunRequest, CancelJobRunResponse>()
                            .withOperationName("CancelJobRun").withMarshaller(new CancelJobRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelJobRunRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = cancelJobRunRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CancelJobRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a managed endpoint. A managed endpoint is a gateway that connects EMR Studio to Amazon EMR on EKS so that
     * EMR Studio can communicate with your virtual cluster.
     * </p>
     *
     * @param createManagedEndpointRequest
     * @return A Java Future containing the result of the CreateManagedEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.CreateManagedEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/CreateManagedEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateManagedEndpointResponse> createManagedEndpoint(
            CreateManagedEndpointRequest createManagedEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createManagedEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateManagedEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateManagedEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateManagedEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateManagedEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateManagedEndpointRequest, CreateManagedEndpointResponse>()
                            .withOperationName("CreateManagedEndpoint")
                            .withMarshaller(new CreateManagedEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createManagedEndpointRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createManagedEndpointRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateManagedEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a virtual cluster. Virtual cluster is a managed entity on Amazon EMR on EKS. You can create, describe,
     * list and delete virtual clusters. They do not consume any additional resource in your system. A single virtual
     * cluster maps to a single Kubernetes namespace. Given this relationship, you can model virtual clusters the same
     * way you model Kubernetes namespaces to meet your requirements.
     * </p>
     *
     * @param createVirtualClusterRequest
     * @return A Java Future containing the result of the CreateVirtualCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.CreateVirtualCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/CreateVirtualCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVirtualClusterResponse> createVirtualCluster(
            CreateVirtualClusterRequest createVirtualClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVirtualClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVirtualCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVirtualClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVirtualClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVirtualClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVirtualClusterRequest, CreateVirtualClusterResponse>()
                            .withOperationName("CreateVirtualCluster")
                            .withMarshaller(new CreateVirtualClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createVirtualClusterRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createVirtualClusterRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateVirtualClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a managed endpoint. A managed endpoint is a gateway that connects EMR Studio to Amazon EMR on EKS so that
     * EMR Studio can communicate with your virtual cluster.
     * </p>
     *
     * @param deleteManagedEndpointRequest
     * @return A Java Future containing the result of the DeleteManagedEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.DeleteManagedEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/DeleteManagedEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteManagedEndpointResponse> deleteManagedEndpoint(
            DeleteManagedEndpointRequest deleteManagedEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteManagedEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteManagedEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteManagedEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteManagedEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteManagedEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteManagedEndpointRequest, DeleteManagedEndpointResponse>()
                            .withOperationName("DeleteManagedEndpoint")
                            .withMarshaller(new DeleteManagedEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteManagedEndpointRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteManagedEndpointRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteManagedEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a virtual cluster. Virtual cluster is a managed entity on Amazon EMR on EKS. You can create, describe,
     * list and delete virtual clusters. They do not consume any additional resource in your system. A single virtual
     * cluster maps to a single Kubernetes namespace. Given this relationship, you can model virtual clusters the same
     * way you model Kubernetes namespaces to meet your requirements.
     * </p>
     *
     * @param deleteVirtualClusterRequest
     * @return A Java Future containing the result of the DeleteVirtualCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.DeleteVirtualCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/DeleteVirtualCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVirtualClusterResponse> deleteVirtualCluster(
            DeleteVirtualClusterRequest deleteVirtualClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVirtualClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVirtualCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVirtualClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVirtualClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVirtualClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVirtualClusterRequest, DeleteVirtualClusterResponse>()
                            .withOperationName("DeleteVirtualCluster")
                            .withMarshaller(new DeleteVirtualClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteVirtualClusterRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteVirtualClusterRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteVirtualClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays detailed information about a job run. A job run is a unit of work, such as a Spark jar, PySpark script,
     * or SparkSQL query, that you submit to Amazon EMR on EKS.
     * </p>
     *
     * @param describeJobRunRequest
     * @return A Java Future containing the result of the DescribeJobRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.DescribeJobRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/DescribeJobRun" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobRunResponse> describeJobRun(DescribeJobRunRequest describeJobRunRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobRunResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeJobRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJobRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobRunRequest, DescribeJobRunResponse>()
                            .withOperationName("DescribeJobRun")
                            .withMarshaller(new DescribeJobRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeJobRunRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeJobRunRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeJobRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays detailed information about a managed endpoint. A managed endpoint is a gateway that connects EMR Studio
     * to Amazon EMR on EKS so that EMR Studio can communicate with your virtual cluster.
     * </p>
     *
     * @param describeManagedEndpointRequest
     * @return A Java Future containing the result of the DescribeManagedEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.DescribeManagedEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/DescribeManagedEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeManagedEndpointResponse> describeManagedEndpoint(
            DescribeManagedEndpointRequest describeManagedEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeManagedEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeManagedEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeManagedEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeManagedEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeManagedEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeManagedEndpointRequest, DescribeManagedEndpointResponse>()
                            .withOperationName("DescribeManagedEndpoint")
                            .withMarshaller(new DescribeManagedEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeManagedEndpointRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeManagedEndpointRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeManagedEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays detailed information about a specified virtual cluster. Virtual cluster is a managed entity on Amazon
     * EMR on EKS. You can create, describe, list and delete virtual clusters. They do not consume any additional
     * resource in your system. A single virtual cluster maps to a single Kubernetes namespace. Given this relationship,
     * you can model virtual clusters the same way you model Kubernetes namespaces to meet your requirements.
     * </p>
     *
     * @param describeVirtualClusterRequest
     * @return A Java Future containing the result of the DescribeVirtualCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.DescribeVirtualCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/DescribeVirtualCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVirtualClusterResponse> describeVirtualCluster(
            DescribeVirtualClusterRequest describeVirtualClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVirtualClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVirtualCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVirtualClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVirtualClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVirtualClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVirtualClusterRequest, DescribeVirtualClusterResponse>()
                            .withOperationName("DescribeVirtualCluster")
                            .withMarshaller(new DescribeVirtualClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeVirtualClusterRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeVirtualClusterRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeVirtualClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists job runs based on a set of parameters. A job run is a unit of work, such as a Spark jar, PySpark script, or
     * SparkSQL query, that you submit to Amazon EMR on EKS.
     * </p>
     *
     * @param listJobRunsRequest
     * @return A Java Future containing the result of the ListJobRuns operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.ListJobRuns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/ListJobRuns" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobRunsResponse> listJobRuns(ListJobRunsRequest listJobRunsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobRunsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobRuns");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobRunsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListJobRunsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJobRunsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobRunsRequest, ListJobRunsResponse>()
                            .withOperationName("ListJobRuns").withMarshaller(new ListJobRunsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listJobRunsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listJobRunsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListJobRunsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists job runs based on a set of parameters. A job run is a unit of work, such as a Spark jar, PySpark script, or
     * SparkSQL query, that you submit to Amazon EMR on EKS.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobRuns(software.amazon.awssdk.services.emrcontainers.model.ListJobRunsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emrcontainers.paginators.ListJobRunsPublisher publisher = client.listJobRunsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emrcontainers.paginators.ListJobRunsPublisher publisher = client.listJobRunsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.emrcontainers.model.ListJobRunsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.emrcontainers.model.ListJobRunsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobRuns(software.amazon.awssdk.services.emrcontainers.model.ListJobRunsRequest)} operation.</b>
     * </p>
     *
     * @param listJobRunsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.ListJobRuns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/ListJobRuns" target="_top">AWS
     *      API Documentation</a>
     */
    public ListJobRunsPublisher listJobRunsPaginator(ListJobRunsRequest listJobRunsRequest) {
        return new ListJobRunsPublisher(this, applyPaginatorUserAgent(listJobRunsRequest));
    }

    /**
     * <p>
     * Lists managed endpoints based on a set of parameters. A managed endpoint is a gateway that connects EMR Studio to
     * Amazon EMR on EKS so that EMR Studio can communicate with your virtual cluster.
     * </p>
     *
     * @param listManagedEndpointsRequest
     * @return A Java Future containing the result of the ListManagedEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.ListManagedEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/ListManagedEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListManagedEndpointsResponse> listManagedEndpoints(
            ListManagedEndpointsRequest listManagedEndpointsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listManagedEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListManagedEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListManagedEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListManagedEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListManagedEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListManagedEndpointsRequest, ListManagedEndpointsResponse>()
                            .withOperationName("ListManagedEndpoints")
                            .withMarshaller(new ListManagedEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listManagedEndpointsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listManagedEndpointsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListManagedEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists managed endpoints based on a set of parameters. A managed endpoint is a gateway that connects EMR Studio to
     * Amazon EMR on EKS so that EMR Studio can communicate with your virtual cluster.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listManagedEndpoints(software.amazon.awssdk.services.emrcontainers.model.ListManagedEndpointsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emrcontainers.paginators.ListManagedEndpointsPublisher publisher = client.listManagedEndpointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emrcontainers.paginators.ListManagedEndpointsPublisher publisher = client.listManagedEndpointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.emrcontainers.model.ListManagedEndpointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.emrcontainers.model.ListManagedEndpointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listManagedEndpoints(software.amazon.awssdk.services.emrcontainers.model.ListManagedEndpointsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listManagedEndpointsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.ListManagedEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/ListManagedEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    public ListManagedEndpointsPublisher listManagedEndpointsPaginator(ListManagedEndpointsRequest listManagedEndpointsRequest) {
        return new ListManagedEndpointsPublisher(this, applyPaginatorUserAgent(listManagedEndpointsRequest));
    }

    /**
     * <p>
     * Lists the tags assigned to the resources.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists information about the specified virtual cluster. Virtual cluster is a managed entity on Amazon EMR on EKS.
     * You can create, describe, list and delete virtual clusters. They do not consume any additional resource in your
     * system. A single virtual cluster maps to a single Kubernetes namespace. Given this relationship, you can model
     * virtual clusters the same way you model Kubernetes namespaces to meet your requirements.
     * </p>
     *
     * @param listVirtualClustersRequest
     * @return A Java Future containing the result of the ListVirtualClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.ListVirtualClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/ListVirtualClusters"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVirtualClustersResponse> listVirtualClusters(
            ListVirtualClustersRequest listVirtualClustersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVirtualClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVirtualClusters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVirtualClustersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVirtualClustersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVirtualClustersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVirtualClustersRequest, ListVirtualClustersResponse>()
                            .withOperationName("ListVirtualClusters")
                            .withMarshaller(new ListVirtualClustersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listVirtualClustersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listVirtualClustersRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListVirtualClustersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists information about the specified virtual cluster. Virtual cluster is a managed entity on Amazon EMR on EKS.
     * You can create, describe, list and delete virtual clusters. They do not consume any additional resource in your
     * system. A single virtual cluster maps to a single Kubernetes namespace. Given this relationship, you can model
     * virtual clusters the same way you model Kubernetes namespaces to meet your requirements.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listVirtualClusters(software.amazon.awssdk.services.emrcontainers.model.ListVirtualClustersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emrcontainers.paginators.ListVirtualClustersPublisher publisher = client.listVirtualClustersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.emrcontainers.paginators.ListVirtualClustersPublisher publisher = client.listVirtualClustersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.emrcontainers.model.ListVirtualClustersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.emrcontainers.model.ListVirtualClustersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listVirtualClusters(software.amazon.awssdk.services.emrcontainers.model.ListVirtualClustersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listVirtualClustersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.ListVirtualClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/ListVirtualClusters"
     *      target="_top">AWS API Documentation</a>
     */
    public ListVirtualClustersPublisher listVirtualClustersPaginator(ListVirtualClustersRequest listVirtualClustersRequest) {
        return new ListVirtualClustersPublisher(this, applyPaginatorUserAgent(listVirtualClustersRequest));
    }

    /**
     * <p>
     * Starts a job run. A job run is a unit of work, such as a Spark jar, PySpark script, or SparkSQL query, that you
     * submit to Amazon EMR on EKS.
     * </p>
     *
     * @param startJobRunRequest
     * @return A Java Future containing the result of the StartJobRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.StartJobRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/StartJobRun" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartJobRunResponse> startJobRun(StartJobRunRequest startJobRunRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startJobRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartJobRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartJobRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartJobRunResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartJobRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartJobRunRequest, StartJobRunResponse>()
                            .withOperationName("StartJobRun").withMarshaller(new StartJobRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startJobRunRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startJobRunRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StartJobRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns tags to resources. A tag is a label that you assign to an AWS resource. Each tag consists of a key and an
     * optional value, both of which you define. Tags enable you to categorize your AWS resources by attributes such as
     * purpose, owner, or environment. When you have many resources of the same type, you can quickly identify a
     * specific resource based on the tags you've assigned to it. For example, you can define a set of tags for your
     * Amazon EMR on EKS clusters to help you track each cluster's owner and stack level. We recommend that you devise a
     * consistent set of tag keys for each resource type. You can then search and filter the resources based on the tags
     * that you add.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from resources.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException This is an internal server exception.</li>
     *         <li>ValidationException There are invalid parameters in the client request.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrContainersException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrContainersAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EmrContainersException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends EmrContainersRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
