/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emrcontainers.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This entity represents the endpoint that is managed by Amazon EMR on EKS.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Endpoint implements SdkPojo, Serializable, ToCopyableBuilder<Endpoint.Builder, Endpoint> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(Endpoint::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Endpoint::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Endpoint::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> VIRTUAL_CLUSTER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("virtualClusterId").getter(getter(Endpoint::virtualClusterId)).setter(setter(Builder::virtualClusterId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("virtualClusterId").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(Endpoint::type)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(Endpoint::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> RELEASE_LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("releaseLabel").getter(getter(Endpoint::releaseLabel)).setter(setter(Builder::releaseLabel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("releaseLabel").build()).build();

    private static final SdkField<String> EXECUTION_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("executionRoleArn").getter(getter(Endpoint::executionRoleArn)).setter(setter(Builder::executionRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionRoleArn").build()).build();

    private static final SdkField<String> CERTIFICATE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("certificateArn").getter(getter(Endpoint::certificateArn)).setter(setter(Builder::certificateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateArn").build()).build();

    private static final SdkField<Certificate> CERTIFICATE_AUTHORITY_FIELD = SdkField
            .<Certificate> builder(MarshallingType.SDK_POJO).memberName("certificateAuthority")
            .getter(getter(Endpoint::certificateAuthority)).setter(setter(Builder::certificateAuthority))
            .constructor(Certificate::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateAuthority").build())
            .build();

    private static final SdkField<ConfigurationOverrides> CONFIGURATION_OVERRIDES_FIELD = SdkField
            .<ConfigurationOverrides> builder(MarshallingType.SDK_POJO).memberName("configurationOverrides")
            .getter(getter(Endpoint::configurationOverrides)).setter(setter(Builder::configurationOverrides))
            .constructor(ConfigurationOverrides::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configurationOverrides").build())
            .build();

    private static final SdkField<String> SERVER_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serverUrl").getter(getter(Endpoint::serverUrl)).setter(setter(Builder::serverUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serverUrl").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(Endpoint::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> SECURITY_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("securityGroup").getter(getter(Endpoint::securityGroup)).setter(setter(Builder::securityGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("securityGroup").build()).build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("subnetIds")
            .getter(getter(Endpoint::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATE_DETAILS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("stateDetails").getter(getter(Endpoint::stateDetails)).setter(setter(Builder::stateDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stateDetails").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("failureReason").getter(getter(Endpoint::failureReasonAsString)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failureReason").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Endpoint::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, NAME_FIELD,
            ARN_FIELD, VIRTUAL_CLUSTER_ID_FIELD, TYPE_FIELD, STATE_FIELD, RELEASE_LABEL_FIELD, EXECUTION_ROLE_ARN_FIELD,
            CERTIFICATE_ARN_FIELD, CERTIFICATE_AUTHORITY_FIELD, CONFIGURATION_OVERRIDES_FIELD, SERVER_URL_FIELD,
            CREATED_AT_FIELD, SECURITY_GROUP_FIELD, SUBNET_IDS_FIELD, STATE_DETAILS_FIELD, FAILURE_REASON_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String name;

    private final String arn;

    private final String virtualClusterId;

    private final String type;

    private final String state;

    private final String releaseLabel;

    private final String executionRoleArn;

    private final String certificateArn;

    private final Certificate certificateAuthority;

    private final ConfigurationOverrides configurationOverrides;

    private final String serverUrl;

    private final Instant createdAt;

    private final String securityGroup;

    private final List<String> subnetIds;

    private final String stateDetails;

    private final String failureReason;

    private final Map<String, String> tags;

    private Endpoint(BuilderImpl builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.arn = builder.arn;
        this.virtualClusterId = builder.virtualClusterId;
        this.type = builder.type;
        this.state = builder.state;
        this.releaseLabel = builder.releaseLabel;
        this.executionRoleArn = builder.executionRoleArn;
        this.certificateArn = builder.certificateArn;
        this.certificateAuthority = builder.certificateAuthority;
        this.configurationOverrides = builder.configurationOverrides;
        this.serverUrl = builder.serverUrl;
        this.createdAt = builder.createdAt;
        this.securityGroup = builder.securityGroup;
        this.subnetIds = builder.subnetIds;
        this.stateDetails = builder.stateDetails;
        this.failureReason = builder.failureReason;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ID of the endpoint.
     * </p>
     * 
     * @return The ID of the endpoint.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The name of the endpoint.
     * </p>
     * 
     * @return The name of the endpoint.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ARN of the endpoint.
     * </p>
     * 
     * @return The ARN of the endpoint.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The ID of the endpoint's virtual cluster.
     * </p>
     * 
     * @return The ID of the endpoint's virtual cluster.
     */
    public final String virtualClusterId() {
        return virtualClusterId;
    }

    /**
     * <p>
     * The type of the endpoint.
     * </p>
     * 
     * @return The type of the endpoint.
     */
    public final String type() {
        return type;
    }

    /**
     * <p>
     * The state of the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link EndpointState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the endpoint.
     * @see EndpointState
     */
    public final EndpointState state() {
        return EndpointState.fromValue(state);
    }

    /**
     * <p>
     * The state of the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link EndpointState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the endpoint.
     * @see EndpointState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The EMR release version to be used for the endpoint.
     * </p>
     * 
     * @return The EMR release version to be used for the endpoint.
     */
    public final String releaseLabel() {
        return releaseLabel;
    }

    /**
     * <p>
     * The execution role ARN of the endpoint.
     * </p>
     * 
     * @return The execution role ARN of the endpoint.
     */
    public final String executionRoleArn() {
        return executionRoleArn;
    }

    /**
     * <p>
     * The certificate ARN of the endpoint. This field is under deprecation and will be removed in future.
     * </p>
     * 
     * @return The certificate ARN of the endpoint. This field is under deprecation and will be removed in future.
     * @deprecated Customer provided certificate-arn is deprecated and would be removed in future.
     */
    @Deprecated
    public final String certificateArn() {
        return certificateArn;
    }

    /**
     * <p>
     * The certificate generated by emr control plane on customer behalf to secure the managed endpoint.
     * </p>
     * 
     * @return The certificate generated by emr control plane on customer behalf to secure the managed endpoint.
     */
    public final Certificate certificateAuthority() {
        return certificateAuthority;
    }

    /**
     * <p>
     * The configuration settings that are used to override existing configurations for endpoints.
     * </p>
     * 
     * @return The configuration settings that are used to override existing configurations for endpoints.
     */
    public final ConfigurationOverrides configurationOverrides() {
        return configurationOverrides;
    }

    /**
     * <p>
     * The server URL of the endpoint.
     * </p>
     * 
     * @return The server URL of the endpoint.
     */
    public final String serverUrl() {
        return serverUrl;
    }

    /**
     * <p>
     * The date and time when the endpoint was created.
     * </p>
     * 
     * @return The date and time when the endpoint was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The security group configuration of the endpoint.
     * </p>
     * 
     * @return The security group configuration of the endpoint.
     */
    public final String securityGroup() {
        return securityGroup;
    }

    /**
     * For responses, this returns true if the service returned a value for the SubnetIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The subnet IDs of the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnetIds} method.
     * </p>
     * 
     * @return The subnet IDs of the endpoint.
     */
    public final List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * <p>
     * Additional details of the endpoint state.
     * </p>
     * 
     * @return Additional details of the endpoint state.
     */
    public final String stateDetails() {
        return stateDetails;
    }

    /**
     * <p>
     * The reasons why the endpoint has failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #failureReason}
     * will return {@link FailureReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #failureReasonAsString}.
     * </p>
     * 
     * @return The reasons why the endpoint has failed.
     * @see FailureReason
     */
    public final FailureReason failureReason() {
        return FailureReason.fromValue(failureReason);
    }

    /**
     * <p>
     * The reasons why the endpoint has failed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #failureReason}
     * will return {@link FailureReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #failureReasonAsString}.
     * </p>
     * 
     * @return The reasons why the endpoint has failed.
     * @see FailureReason
     */
    public final String failureReasonAsString() {
        return failureReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags of the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags of the endpoint.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(virtualClusterId());
        hashCode = 31 * hashCode + Objects.hashCode(type());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(releaseLabel());
        hashCode = 31 * hashCode + Objects.hashCode(executionRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(certificateArn());
        hashCode = 31 * hashCode + Objects.hashCode(certificateAuthority());
        hashCode = 31 * hashCode + Objects.hashCode(configurationOverrides());
        hashCode = 31 * hashCode + Objects.hashCode(serverUrl());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(securityGroup());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetIds() ? subnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stateDetails());
        hashCode = 31 * hashCode + Objects.hashCode(failureReasonAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Endpoint)) {
            return false;
        }
        Endpoint other = (Endpoint) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(virtualClusterId(), other.virtualClusterId()) && Objects.equals(type(), other.type())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(releaseLabel(), other.releaseLabel())
                && Objects.equals(executionRoleArn(), other.executionRoleArn())
                && Objects.equals(certificateArn(), other.certificateArn())
                && Objects.equals(certificateAuthority(), other.certificateAuthority())
                && Objects.equals(configurationOverrides(), other.configurationOverrides())
                && Objects.equals(serverUrl(), other.serverUrl()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(securityGroup(), other.securityGroup()) && hasSubnetIds() == other.hasSubnetIds()
                && Objects.equals(subnetIds(), other.subnetIds()) && Objects.equals(stateDetails(), other.stateDetails())
                && Objects.equals(failureReasonAsString(), other.failureReasonAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Endpoint").add("Id", id()).add("Name", name()).add("Arn", arn())
                .add("VirtualClusterId", virtualClusterId()).add("Type", type()).add("State", stateAsString())
                .add("ReleaseLabel", releaseLabel()).add("ExecutionRoleArn", executionRoleArn())
                .add("CertificateArn", certificateArn()).add("CertificateAuthority", certificateAuthority())
                .add("ConfigurationOverrides", configurationOverrides()).add("ServerUrl", serverUrl())
                .add("CreatedAt", createdAt()).add("SecurityGroup", securityGroup())
                .add("SubnetIds", hasSubnetIds() ? subnetIds() : null).add("StateDetails", stateDetails())
                .add("FailureReason", failureReasonAsString()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "virtualClusterId":
            return Optional.ofNullable(clazz.cast(virtualClusterId()));
        case "type":
            return Optional.ofNullable(clazz.cast(type()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "releaseLabel":
            return Optional.ofNullable(clazz.cast(releaseLabel()));
        case "executionRoleArn":
            return Optional.ofNullable(clazz.cast(executionRoleArn()));
        case "certificateArn":
            return Optional.ofNullable(clazz.cast(certificateArn()));
        case "certificateAuthority":
            return Optional.ofNullable(clazz.cast(certificateAuthority()));
        case "configurationOverrides":
            return Optional.ofNullable(clazz.cast(configurationOverrides()));
        case "serverUrl":
            return Optional.ofNullable(clazz.cast(serverUrl()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "securityGroup":
            return Optional.ofNullable(clazz.cast(securityGroup()));
        case "subnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "stateDetails":
            return Optional.ofNullable(clazz.cast(stateDetails()));
        case "failureReason":
            return Optional.ofNullable(clazz.cast(failureReasonAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Endpoint, T> g) {
        return obj -> g.apply((Endpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Endpoint> {
        /**
         * <p>
         * The ID of the endpoint.
         * </p>
         * 
         * @param id
         *        The ID of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the endpoint.
         * </p>
         * 
         * @param name
         *        The name of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ARN of the endpoint.
         * </p>
         * 
         * @param arn
         *        The ARN of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The ID of the endpoint's virtual cluster.
         * </p>
         * 
         * @param virtualClusterId
         *        The ID of the endpoint's virtual cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder virtualClusterId(String virtualClusterId);

        /**
         * <p>
         * The type of the endpoint.
         * </p>
         * 
         * @param type
         *        The type of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder type(String type);

        /**
         * <p>
         * The state of the endpoint.
         * </p>
         * 
         * @param state
         *        The state of the endpoint.
         * @see EndpointState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the endpoint.
         * </p>
         * 
         * @param state
         *        The state of the endpoint.
         * @see EndpointState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointState
         */
        Builder state(EndpointState state);

        /**
         * <p>
         * The EMR release version to be used for the endpoint.
         * </p>
         * 
         * @param releaseLabel
         *        The EMR release version to be used for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder releaseLabel(String releaseLabel);

        /**
         * <p>
         * The execution role ARN of the endpoint.
         * </p>
         * 
         * @param executionRoleArn
         *        The execution role ARN of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRoleArn(String executionRoleArn);

        /**
         * <p>
         * The certificate ARN of the endpoint. This field is under deprecation and will be removed in future.
         * </p>
         * 
         * @param certificateArn
         *        The certificate ARN of the endpoint. This field is under deprecation and will be removed in future.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated Customer provided certificate-arn is deprecated and would be removed in future.
         */
        @Deprecated
        Builder certificateArn(String certificateArn);

        /**
         * <p>
         * The certificate generated by emr control plane on customer behalf to secure the managed endpoint.
         * </p>
         * 
         * @param certificateAuthority
         *        The certificate generated by emr control plane on customer behalf to secure the managed endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateAuthority(Certificate certificateAuthority);

        /**
         * <p>
         * The certificate generated by emr control plane on customer behalf to secure the managed endpoint.
         * </p>
         * This is a convenience method that creates an instance of the {@link Certificate.Builder} avoiding the need to
         * create one manually via {@link Certificate#builder()}.
         *
         * When the {@link Consumer} completes, {@link Certificate.Builder#build()} is called immediately and its result
         * is passed to {@link #certificateAuthority(Certificate)}.
         * 
         * @param certificateAuthority
         *        a consumer that will call methods on {@link Certificate.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #certificateAuthority(Certificate)
         */
        default Builder certificateAuthority(Consumer<Certificate.Builder> certificateAuthority) {
            return certificateAuthority(Certificate.builder().applyMutation(certificateAuthority).build());
        }

        /**
         * <p>
         * The configuration settings that are used to override existing configurations for endpoints.
         * </p>
         * 
         * @param configurationOverrides
         *        The configuration settings that are used to override existing configurations for endpoints.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationOverrides(ConfigurationOverrides configurationOverrides);

        /**
         * <p>
         * The configuration settings that are used to override existing configurations for endpoints.
         * </p>
         * This is a convenience method that creates an instance of the {@link ConfigurationOverrides.Builder} avoiding
         * the need to create one manually via {@link ConfigurationOverrides#builder()}.
         *
         * When the {@link Consumer} completes, {@link ConfigurationOverrides.Builder#build()} is called immediately and
         * its result is passed to {@link #configurationOverrides(ConfigurationOverrides)}.
         * 
         * @param configurationOverrides
         *        a consumer that will call methods on {@link ConfigurationOverrides.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configurationOverrides(ConfigurationOverrides)
         */
        default Builder configurationOverrides(Consumer<ConfigurationOverrides.Builder> configurationOverrides) {
            return configurationOverrides(ConfigurationOverrides.builder().applyMutation(configurationOverrides).build());
        }

        /**
         * <p>
         * The server URL of the endpoint.
         * </p>
         * 
         * @param serverUrl
         *        The server URL of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverUrl(String serverUrl);

        /**
         * <p>
         * The date and time when the endpoint was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time when the endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The security group configuration of the endpoint.
         * </p>
         * 
         * @param securityGroup
         *        The security group configuration of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroup(String securityGroup);

        /**
         * <p>
         * The subnet IDs of the endpoint.
         * </p>
         * 
         * @param subnetIds
         *        The subnet IDs of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * <p>
         * The subnet IDs of the endpoint.
         * </p>
         * 
         * @param subnetIds
         *        The subnet IDs of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * <p>
         * Additional details of the endpoint state.
         * </p>
         * 
         * @param stateDetails
         *        Additional details of the endpoint state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateDetails(String stateDetails);

        /**
         * <p>
         * The reasons why the endpoint has failed.
         * </p>
         * 
         * @param failureReason
         *        The reasons why the endpoint has failed.
         * @see FailureReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FailureReason
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * The reasons why the endpoint has failed.
         * </p>
         * 
         * @param failureReason
         *        The reasons why the endpoint has failed.
         * @see FailureReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FailureReason
         */
        Builder failureReason(FailureReason failureReason);

        /**
         * <p>
         * The tags of the endpoint.
         * </p>
         * 
         * @param tags
         *        The tags of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String name;

        private String arn;

        private String virtualClusterId;

        private String type;

        private String state;

        private String releaseLabel;

        private String executionRoleArn;

        private String certificateArn;

        private Certificate certificateAuthority;

        private ConfigurationOverrides configurationOverrides;

        private String serverUrl;

        private Instant createdAt;

        private String securityGroup;

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private String stateDetails;

        private String failureReason;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Endpoint model) {
            id(model.id);
            name(model.name);
            arn(model.arn);
            virtualClusterId(model.virtualClusterId);
            type(model.type);
            state(model.state);
            releaseLabel(model.releaseLabel);
            executionRoleArn(model.executionRoleArn);
            certificateArn(model.certificateArn);
            certificateAuthority(model.certificateAuthority);
            configurationOverrides(model.configurationOverrides);
            serverUrl(model.serverUrl);
            createdAt(model.createdAt);
            securityGroup(model.securityGroup);
            subnetIds(model.subnetIds);
            stateDetails(model.stateDetails);
            failureReason(model.failureReason);
            tags(model.tags);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getVirtualClusterId() {
            return virtualClusterId;
        }

        public final void setVirtualClusterId(String virtualClusterId) {
            this.virtualClusterId = virtualClusterId;
        }

        @Override
        public final Builder virtualClusterId(String virtualClusterId) {
            this.virtualClusterId = virtualClusterId;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(EndpointState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getReleaseLabel() {
            return releaseLabel;
        }

        public final void setReleaseLabel(String releaseLabel) {
            this.releaseLabel = releaseLabel;
        }

        @Override
        public final Builder releaseLabel(String releaseLabel) {
            this.releaseLabel = releaseLabel;
            return this;
        }

        public final String getExecutionRoleArn() {
            return executionRoleArn;
        }

        public final void setExecutionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
        }

        @Override
        public final Builder executionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
            return this;
        }

        @Deprecated
        public final String getCertificateArn() {
            return certificateArn;
        }

        @Deprecated
        public final void setCertificateArn(String certificateArn) {
            this.certificateArn = certificateArn;
        }

        @Override
        @Deprecated
        public final Builder certificateArn(String certificateArn) {
            this.certificateArn = certificateArn;
            return this;
        }

        public final Certificate.Builder getCertificateAuthority() {
            return certificateAuthority != null ? certificateAuthority.toBuilder() : null;
        }

        public final void setCertificateAuthority(Certificate.BuilderImpl certificateAuthority) {
            this.certificateAuthority = certificateAuthority != null ? certificateAuthority.build() : null;
        }

        @Override
        public final Builder certificateAuthority(Certificate certificateAuthority) {
            this.certificateAuthority = certificateAuthority;
            return this;
        }

        public final ConfigurationOverrides.Builder getConfigurationOverrides() {
            return configurationOverrides != null ? configurationOverrides.toBuilder() : null;
        }

        public final void setConfigurationOverrides(ConfigurationOverrides.BuilderImpl configurationOverrides) {
            this.configurationOverrides = configurationOverrides != null ? configurationOverrides.build() : null;
        }

        @Override
        public final Builder configurationOverrides(ConfigurationOverrides configurationOverrides) {
            this.configurationOverrides = configurationOverrides;
            return this;
        }

        public final String getServerUrl() {
            return serverUrl;
        }

        public final void setServerUrl(String serverUrl) {
            this.serverUrl = serverUrl;
        }

        @Override
        public final Builder serverUrl(String serverUrl) {
            this.serverUrl = serverUrl;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getSecurityGroup() {
            return securityGroup;
        }

        public final void setSecurityGroup(String securityGroup) {
            this.securityGroup = securityGroup;
        }

        @Override
        public final Builder securityGroup(String securityGroup) {
            this.securityGroup = securityGroup;
            return this;
        }

        public final Collection<String> getSubnetIds() {
            if (subnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetIds;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final String getStateDetails() {
            return stateDetails;
        }

        public final void setStateDetails(String stateDetails) {
            this.stateDetails = stateDetails;
        }

        @Override
        public final Builder stateDetails(String stateDetails) {
            this.stateDetails = stateDetails;
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        @Override
        public final Builder failureReason(FailureReason failureReason) {
            this.failureReason(failureReason == null ? null : failureReason.toString());
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        @Override
        public Endpoint build() {
            return new Endpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
