/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emrcontainers;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.emrcontainers.internal.EmrContainersServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.emrcontainers.model.CancelJobRunRequest;
import software.amazon.awssdk.services.emrcontainers.model.CancelJobRunResponse;
import software.amazon.awssdk.services.emrcontainers.model.CreateJobTemplateRequest;
import software.amazon.awssdk.services.emrcontainers.model.CreateJobTemplateResponse;
import software.amazon.awssdk.services.emrcontainers.model.CreateManagedEndpointRequest;
import software.amazon.awssdk.services.emrcontainers.model.CreateManagedEndpointResponse;
import software.amazon.awssdk.services.emrcontainers.model.CreateSecurityConfigurationRequest;
import software.amazon.awssdk.services.emrcontainers.model.CreateSecurityConfigurationResponse;
import software.amazon.awssdk.services.emrcontainers.model.CreateVirtualClusterRequest;
import software.amazon.awssdk.services.emrcontainers.model.CreateVirtualClusterResponse;
import software.amazon.awssdk.services.emrcontainers.model.DeleteJobTemplateRequest;
import software.amazon.awssdk.services.emrcontainers.model.DeleteJobTemplateResponse;
import software.amazon.awssdk.services.emrcontainers.model.DeleteManagedEndpointRequest;
import software.amazon.awssdk.services.emrcontainers.model.DeleteManagedEndpointResponse;
import software.amazon.awssdk.services.emrcontainers.model.DeleteVirtualClusterRequest;
import software.amazon.awssdk.services.emrcontainers.model.DeleteVirtualClusterResponse;
import software.amazon.awssdk.services.emrcontainers.model.DescribeJobRunRequest;
import software.amazon.awssdk.services.emrcontainers.model.DescribeJobRunResponse;
import software.amazon.awssdk.services.emrcontainers.model.DescribeJobTemplateRequest;
import software.amazon.awssdk.services.emrcontainers.model.DescribeJobTemplateResponse;
import software.amazon.awssdk.services.emrcontainers.model.DescribeManagedEndpointRequest;
import software.amazon.awssdk.services.emrcontainers.model.DescribeManagedEndpointResponse;
import software.amazon.awssdk.services.emrcontainers.model.DescribeSecurityConfigurationRequest;
import software.amazon.awssdk.services.emrcontainers.model.DescribeSecurityConfigurationResponse;
import software.amazon.awssdk.services.emrcontainers.model.DescribeVirtualClusterRequest;
import software.amazon.awssdk.services.emrcontainers.model.DescribeVirtualClusterResponse;
import software.amazon.awssdk.services.emrcontainers.model.EksRequestThrottledException;
import software.amazon.awssdk.services.emrcontainers.model.EmrContainersException;
import software.amazon.awssdk.services.emrcontainers.model.GetManagedEndpointSessionCredentialsRequest;
import software.amazon.awssdk.services.emrcontainers.model.GetManagedEndpointSessionCredentialsResponse;
import software.amazon.awssdk.services.emrcontainers.model.InternalServerException;
import software.amazon.awssdk.services.emrcontainers.model.ListJobRunsRequest;
import software.amazon.awssdk.services.emrcontainers.model.ListJobRunsResponse;
import software.amazon.awssdk.services.emrcontainers.model.ListJobTemplatesRequest;
import software.amazon.awssdk.services.emrcontainers.model.ListJobTemplatesResponse;
import software.amazon.awssdk.services.emrcontainers.model.ListManagedEndpointsRequest;
import software.amazon.awssdk.services.emrcontainers.model.ListManagedEndpointsResponse;
import software.amazon.awssdk.services.emrcontainers.model.ListSecurityConfigurationsRequest;
import software.amazon.awssdk.services.emrcontainers.model.ListSecurityConfigurationsResponse;
import software.amazon.awssdk.services.emrcontainers.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.emrcontainers.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.emrcontainers.model.ListVirtualClustersRequest;
import software.amazon.awssdk.services.emrcontainers.model.ListVirtualClustersResponse;
import software.amazon.awssdk.services.emrcontainers.model.RequestThrottledException;
import software.amazon.awssdk.services.emrcontainers.model.ResourceNotFoundException;
import software.amazon.awssdk.services.emrcontainers.model.StartJobRunRequest;
import software.amazon.awssdk.services.emrcontainers.model.StartJobRunResponse;
import software.amazon.awssdk.services.emrcontainers.model.TagResourceRequest;
import software.amazon.awssdk.services.emrcontainers.model.TagResourceResponse;
import software.amazon.awssdk.services.emrcontainers.model.UntagResourceRequest;
import software.amazon.awssdk.services.emrcontainers.model.UntagResourceResponse;
import software.amazon.awssdk.services.emrcontainers.model.ValidationException;
import software.amazon.awssdk.services.emrcontainers.transform.CancelJobRunRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.CreateJobTemplateRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.CreateManagedEndpointRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.CreateSecurityConfigurationRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.CreateVirtualClusterRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.DeleteJobTemplateRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.DeleteManagedEndpointRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.DeleteVirtualClusterRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.DescribeJobRunRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.DescribeJobTemplateRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.DescribeManagedEndpointRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.DescribeSecurityConfigurationRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.DescribeVirtualClusterRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.GetManagedEndpointSessionCredentialsRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.ListJobRunsRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.ListJobTemplatesRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.ListManagedEndpointsRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.ListSecurityConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.ListVirtualClustersRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.StartJobRunRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.emrcontainers.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link EmrContainersClient}.
 *
 * @see EmrContainersClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEmrContainersClient implements EmrContainersClient {
    private static final Logger log = Logger.loggerFor(DefaultEmrContainersClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEmrContainersClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels a job run. A job run is a unit of work, such as a Spark jar, PySpark script, or SparkSQL query, that you
     * submit to Amazon EMR on EKS.
     * </p>
     *
     * @param cancelJobRunRequest
     * @return Result of the CancelJobRun operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.CancelJobRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/CancelJobRun" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CancelJobRunResponse cancelJobRun(CancelJobRunRequest cancelJobRunRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelJobRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelJobRunResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelJobRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJobRun");

            return clientHandler.execute(new ClientExecutionParams<CancelJobRunRequest, CancelJobRunResponse>()
                    .withOperationName("CancelJobRun").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(cancelJobRunRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelJobRunRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a job template. Job template stores values of StartJobRun API request in a template and can be used to
     * start a job run. Job template allows two use cases: avoid repeating recurring StartJobRun API request values,
     * enforcing certain values in StartJobRun API request.
     * </p>
     *
     * @param createJobTemplateRequest
     * @return Result of the CreateJobTemplate operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.CreateJobTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/CreateJobTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateJobTemplateResponse createJobTemplate(CreateJobTemplateRequest createJobTemplateRequest)
            throws ValidationException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateJobTemplateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateJobTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createJobTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJobTemplate");

            return clientHandler.execute(new ClientExecutionParams<CreateJobTemplateRequest, CreateJobTemplateResponse>()
                    .withOperationName("CreateJobTemplate").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createJobTemplateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateJobTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a managed endpoint. A managed endpoint is a gateway that connects Amazon EMR Studio to Amazon EMR on EKS
     * so that Amazon EMR Studio can communicate with your virtual cluster.
     * </p>
     *
     * @param createManagedEndpointRequest
     * @return Result of the CreateManagedEndpoint operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.CreateManagedEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/CreateManagedEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateManagedEndpointResponse createManagedEndpoint(CreateManagedEndpointRequest createManagedEndpointRequest)
            throws ValidationException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateManagedEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateManagedEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createManagedEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createManagedEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateManagedEndpoint");

            return clientHandler.execute(new ClientExecutionParams<CreateManagedEndpointRequest, CreateManagedEndpointResponse>()
                    .withOperationName("CreateManagedEndpoint").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createManagedEndpointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateManagedEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a security configuration. Security configurations in Amazon EMR on EKS are templates for different
     * security setups. You can use security configurations to configure the Lake Formation integration setup. You can
     * also create a security configuration to re-use a security setup each time you create a virtual cluster.
     * </p>
     *
     * @param createSecurityConfigurationRequest
     * @return Result of the CreateSecurityConfiguration operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.CreateSecurityConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/CreateSecurityConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateSecurityConfigurationResponse createSecurityConfiguration(
            CreateSecurityConfigurationRequest createSecurityConfigurationRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSecurityConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSecurityConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSecurityConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSecurityConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSecurityConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateSecurityConfigurationRequest, CreateSecurityConfigurationResponse>()
                            .withOperationName("CreateSecurityConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createSecurityConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateSecurityConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a virtual cluster. Virtual cluster is a managed entity on Amazon EMR on EKS. You can create, describe,
     * list and delete virtual clusters. They do not consume any additional resource in your system. A single virtual
     * cluster maps to a single Kubernetes namespace. Given this relationship, you can model virtual clusters the same
     * way you model Kubernetes namespaces to meet your requirements.
     * </p>
     *
     * @param createVirtualClusterRequest
     * @return Result of the CreateVirtualCluster operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws EksRequestThrottledException
     *         The request exceeded the Amazon EKS API operation limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.CreateVirtualCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/CreateVirtualCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateVirtualClusterResponse createVirtualCluster(CreateVirtualClusterRequest createVirtualClusterRequest)
            throws ValidationException, ResourceNotFoundException, InternalServerException, EksRequestThrottledException,
            AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVirtualClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateVirtualClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVirtualClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVirtualClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVirtualCluster");

            return clientHandler.execute(new ClientExecutionParams<CreateVirtualClusterRequest, CreateVirtualClusterResponse>()
                    .withOperationName("CreateVirtualCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createVirtualClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateVirtualClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a job template. Job template stores values of StartJobRun API request in a template and can be used to
     * start a job run. Job template allows two use cases: avoid repeating recurring StartJobRun API request values,
     * enforcing certain values in StartJobRun API request.
     * </p>
     *
     * @param deleteJobTemplateRequest
     * @return Result of the DeleteJobTemplate operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.DeleteJobTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/DeleteJobTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteJobTemplateResponse deleteJobTemplate(DeleteJobTemplateRequest deleteJobTemplateRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteJobTemplateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteJobTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteJobTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteJobTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteJobTemplate");

            return clientHandler.execute(new ClientExecutionParams<DeleteJobTemplateRequest, DeleteJobTemplateResponse>()
                    .withOperationName("DeleteJobTemplate").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteJobTemplateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteJobTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a managed endpoint. A managed endpoint is a gateway that connects Amazon EMR Studio to Amazon EMR on EKS
     * so that Amazon EMR Studio can communicate with your virtual cluster.
     * </p>
     *
     * @param deleteManagedEndpointRequest
     * @return Result of the DeleteManagedEndpoint operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.DeleteManagedEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/DeleteManagedEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteManagedEndpointResponse deleteManagedEndpoint(DeleteManagedEndpointRequest deleteManagedEndpointRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteManagedEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteManagedEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteManagedEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteManagedEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteManagedEndpoint");

            return clientHandler.execute(new ClientExecutionParams<DeleteManagedEndpointRequest, DeleteManagedEndpointResponse>()
                    .withOperationName("DeleteManagedEndpoint").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteManagedEndpointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteManagedEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a virtual cluster. Virtual cluster is a managed entity on Amazon EMR on EKS. You can create, describe,
     * list and delete virtual clusters. They do not consume any additional resource in your system. A single virtual
     * cluster maps to a single Kubernetes namespace. Given this relationship, you can model virtual clusters the same
     * way you model Kubernetes namespaces to meet your requirements.
     * </p>
     *
     * @param deleteVirtualClusterRequest
     * @return Result of the DeleteVirtualCluster operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.DeleteVirtualCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/DeleteVirtualCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteVirtualClusterResponse deleteVirtualCluster(DeleteVirtualClusterRequest deleteVirtualClusterRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVirtualClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteVirtualClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVirtualClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVirtualClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVirtualCluster");

            return clientHandler.execute(new ClientExecutionParams<DeleteVirtualClusterRequest, DeleteVirtualClusterResponse>()
                    .withOperationName("DeleteVirtualCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteVirtualClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVirtualClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays detailed information about a job run. A job run is a unit of work, such as a Spark jar, PySpark script,
     * or SparkSQL query, that you submit to Amazon EMR on EKS.
     * </p>
     *
     * @param describeJobRunRequest
     * @return Result of the DescribeJobRun operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.DescribeJobRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/DescribeJobRun" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeJobRunResponse describeJobRun(DescribeJobRunRequest describeJobRunRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeJobRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeJobRunResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeJobRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobRun");

            return clientHandler.execute(new ClientExecutionParams<DescribeJobRunRequest, DescribeJobRunResponse>()
                    .withOperationName("DescribeJobRun").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeJobRunRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeJobRunRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays detailed information about a specified job template. Job template stores values of StartJobRun API
     * request in a template and can be used to start a job run. Job template allows two use cases: avoid repeating
     * recurring StartJobRun API request values, enforcing certain values in StartJobRun API request.
     * </p>
     *
     * @param describeJobTemplateRequest
     * @return Result of the DescribeJobTemplate operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.DescribeJobTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/DescribeJobTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeJobTemplateResponse describeJobTemplate(DescribeJobTemplateRequest describeJobTemplateRequest)
            throws ValidationException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeJobTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeJobTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeJobTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobTemplate");

            return clientHandler.execute(new ClientExecutionParams<DescribeJobTemplateRequest, DescribeJobTemplateResponse>()
                    .withOperationName("DescribeJobTemplate").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeJobTemplateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeJobTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays detailed information about a managed endpoint. A managed endpoint is a gateway that connects Amazon EMR
     * Studio to Amazon EMR on EKS so that Amazon EMR Studio can communicate with your virtual cluster.
     * </p>
     *
     * @param describeManagedEndpointRequest
     * @return Result of the DescribeManagedEndpoint operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.DescribeManagedEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/DescribeManagedEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeManagedEndpointResponse describeManagedEndpoint(DescribeManagedEndpointRequest describeManagedEndpointRequest)
            throws ValidationException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeManagedEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeManagedEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeManagedEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeManagedEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeManagedEndpoint");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeManagedEndpointRequest, DescribeManagedEndpointResponse>()
                            .withOperationName("DescribeManagedEndpoint").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeManagedEndpointRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeManagedEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays detailed information about a specified security configuration. Security configurations in Amazon EMR on
     * EKS are templates for different security setups. You can use security configurations to configure the Lake
     * Formation integration setup. You can also create a security configuration to re-use a security setup each time
     * you create a virtual cluster.
     * </p>
     *
     * @param describeSecurityConfigurationRequest
     * @return Result of the DescribeSecurityConfiguration operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.DescribeSecurityConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/DescribeSecurityConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeSecurityConfigurationResponse describeSecurityConfiguration(
            DescribeSecurityConfigurationRequest describeSecurityConfigurationRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSecurityConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSecurityConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSecurityConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeSecurityConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSecurityConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeSecurityConfigurationRequest, DescribeSecurityConfigurationResponse>()
                            .withOperationName("DescribeSecurityConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeSecurityConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeSecurityConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays detailed information about a specified virtual cluster. Virtual cluster is a managed entity on Amazon
     * EMR on EKS. You can create, describe, list and delete virtual clusters. They do not consume any additional
     * resource in your system. A single virtual cluster maps to a single Kubernetes namespace. Given this relationship,
     * you can model virtual clusters the same way you model Kubernetes namespaces to meet your requirements.
     * </p>
     *
     * @param describeVirtualClusterRequest
     * @return Result of the DescribeVirtualCluster operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.DescribeVirtualCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/DescribeVirtualCluster"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeVirtualClusterResponse describeVirtualCluster(DescribeVirtualClusterRequest describeVirtualClusterRequest)
            throws ValidationException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeVirtualClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeVirtualClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeVirtualClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVirtualClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVirtualCluster");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeVirtualClusterRequest, DescribeVirtualClusterResponse>()
                            .withOperationName("DescribeVirtualCluster").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeVirtualClusterRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeVirtualClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Generate a session token to connect to a managed endpoint.
     * </p>
     *
     * @param getManagedEndpointSessionCredentialsRequest
     * @return Result of the GetManagedEndpointSessionCredentials operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws RequestThrottledException
     *         The request throttled.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.GetManagedEndpointSessionCredentials
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/GetManagedEndpointSessionCredentials"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetManagedEndpointSessionCredentialsResponse getManagedEndpointSessionCredentials(
            GetManagedEndpointSessionCredentialsRequest getManagedEndpointSessionCredentialsRequest) throws ValidationException,
            RequestThrottledException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetManagedEndpointSessionCredentialsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetManagedEndpointSessionCredentialsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getManagedEndpointSessionCredentialsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getManagedEndpointSessionCredentialsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetManagedEndpointSessionCredentials");

            return clientHandler
                    .execute(new ClientExecutionParams<GetManagedEndpointSessionCredentialsRequest, GetManagedEndpointSessionCredentialsResponse>()
                            .withOperationName("GetManagedEndpointSessionCredentials").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getManagedEndpointSessionCredentialsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetManagedEndpointSessionCredentialsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists job runs based on a set of parameters. A job run is a unit of work, such as a Spark jar, PySpark script, or
     * SparkSQL query, that you submit to Amazon EMR on EKS.
     * </p>
     *
     * @param listJobRunsRequest
     * @return Result of the ListJobRuns operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.ListJobRuns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/ListJobRuns" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListJobRunsResponse listJobRuns(ListJobRunsRequest listJobRunsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJobRunsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListJobRunsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listJobRunsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobRunsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobRuns");

            return clientHandler.execute(new ClientExecutionParams<ListJobRunsRequest, ListJobRunsResponse>()
                    .withOperationName("ListJobRuns").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listJobRunsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListJobRunsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists job templates based on a set of parameters. Job template stores values of StartJobRun API request in a
     * template and can be used to start a job run. Job template allows two use cases: avoid repeating recurring
     * StartJobRun API request values, enforcing certain values in StartJobRun API request.
     * </p>
     *
     * @param listJobTemplatesRequest
     * @return Result of the ListJobTemplates operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.ListJobTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/ListJobTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListJobTemplatesResponse listJobTemplates(ListJobTemplatesRequest listJobTemplatesRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJobTemplatesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListJobTemplatesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listJobTemplatesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobTemplatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobTemplates");

            return clientHandler.execute(new ClientExecutionParams<ListJobTemplatesRequest, ListJobTemplatesResponse>()
                    .withOperationName("ListJobTemplates").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listJobTemplatesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListJobTemplatesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists managed endpoints based on a set of parameters. A managed endpoint is a gateway that connects Amazon EMR
     * Studio to Amazon EMR on EKS so that Amazon EMR Studio can communicate with your virtual cluster.
     * </p>
     *
     * @param listManagedEndpointsRequest
     * @return Result of the ListManagedEndpoints operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.ListManagedEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/ListManagedEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListManagedEndpointsResponse listManagedEndpoints(ListManagedEndpointsRequest listManagedEndpointsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListManagedEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListManagedEndpointsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listManagedEndpointsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listManagedEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListManagedEndpoints");

            return clientHandler.execute(new ClientExecutionParams<ListManagedEndpointsRequest, ListManagedEndpointsResponse>()
                    .withOperationName("ListManagedEndpoints").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listManagedEndpointsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListManagedEndpointsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists security configurations based on a set of parameters. Security configurations in Amazon EMR on EKS are
     * templates for different security setups. You can use security configurations to configure the Lake Formation
     * integration setup. You can also create a security configuration to re-use a security setup each time you create a
     * virtual cluster.
     * </p>
     *
     * @param listSecurityConfigurationsRequest
     * @return Result of the ListSecurityConfigurations operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.ListSecurityConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/ListSecurityConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSecurityConfigurationsResponse listSecurityConfigurations(
            ListSecurityConfigurationsRequest listSecurityConfigurationsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSecurityConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSecurityConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSecurityConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSecurityConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSecurityConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListSecurityConfigurationsRequest, ListSecurityConfigurationsResponse>()
                            .withOperationName("ListSecurityConfigurations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listSecurityConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListSecurityConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags assigned to the resources.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists information about the specified virtual cluster. Virtual cluster is a managed entity on Amazon EMR on EKS.
     * You can create, describe, list and delete virtual clusters. They do not consume any additional resource in your
     * system. A single virtual cluster maps to a single Kubernetes namespace. Given this relationship, you can model
     * virtual clusters the same way you model Kubernetes namespaces to meet your requirements.
     * </p>
     *
     * @param listVirtualClustersRequest
     * @return Result of the ListVirtualClusters operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.ListVirtualClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/ListVirtualClusters"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListVirtualClustersResponse listVirtualClusters(ListVirtualClustersRequest listVirtualClustersRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVirtualClustersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListVirtualClustersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVirtualClustersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVirtualClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVirtualClusters");

            return clientHandler.execute(new ClientExecutionParams<ListVirtualClustersRequest, ListVirtualClustersResponse>()
                    .withOperationName("ListVirtualClusters").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listVirtualClustersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVirtualClustersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a job run. A job run is a unit of work, such as a Spark jar, PySpark script, or SparkSQL query, that you
     * submit to Amazon EMR on EKS.
     * </p>
     *
     * @param startJobRunRequest
     * @return Result of the StartJobRun operation returned by the service.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.StartJobRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/StartJobRun" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartJobRunResponse startJobRun(StartJobRunRequest startJobRunRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartJobRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartJobRunResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startJobRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startJobRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartJobRun");

            return clientHandler.execute(new ClientExecutionParams<StartJobRunRequest, StartJobRunResponse>()
                    .withOperationName("StartJobRun").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(startJobRunRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartJobRunRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns tags to resources. A tag is a label that you assign to an Amazon Web Services resource. Each tag consists
     * of a key and an optional value, both of which you define. Tags enable you to categorize your Amazon Web Services
     * resources by attributes such as purpose, owner, or environment. When you have many resources of the same type,
     * you can quickly identify a specific resource based on the tags you've assigned to it. For example, you can define
     * a set of tags for your Amazon EMR on EKS clusters to help you track each cluster's owner and stack level. We
     * recommend that you devise a consistent set of tag keys for each resource type. You can then search and filter the
     * resources based on the tags that you add.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from resources.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     *         This is an internal server exception.
     * @throws ValidationException
     *         There are invalid parameters in the client request.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EmrContainersException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EmrContainersClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-containers-2020-10-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EmrContainersException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR containers");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        EmrContainersServiceClientConfigurationBuilder serviceConfigBuilder = new EmrContainersServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EmrContainersException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestThrottledException")
                                .exceptionBuilderSupplier(RequestThrottledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EKSRequestThrottledException")
                                .exceptionBuilderSupplier(EksRequestThrottledException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final EmrContainersServiceClientConfiguration serviceClientConfiguration() {
        return new EmrContainersServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
