/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emrcontainers.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This entity describes a virtual cluster. A virtual cluster is a Kubernetes namespace that Amazon EMR is registered
 * with. Amazon EMR uses virtual clusters to run jobs and host endpoints. Multiple virtual clusters can be backed by the
 * same physical cluster. However, each virtual cluster maps to one namespace on an Amazon EKS cluster. Virtual clusters
 * do not create any active resources that contribute to your bill or that require lifecycle management outside the
 * service.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VirtualCluster implements SdkPojo, Serializable, ToCopyableBuilder<VirtualCluster.Builder, VirtualCluster> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(VirtualCluster::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(VirtualCluster::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(VirtualCluster::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(VirtualCluster::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<ContainerProvider> CONTAINER_PROVIDER_FIELD = SdkField
            .<ContainerProvider> builder(MarshallingType.SDK_POJO).memberName("containerProvider")
            .getter(getter(VirtualCluster::containerProvider)).setter(setter(Builder::containerProvider))
            .constructor(ContainerProvider::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerProvider").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(VirtualCluster::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(VirtualCluster::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> SECURITY_CONFIGURATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("securityConfigurationId").getter(getter(VirtualCluster::securityConfigurationId))
            .setter(setter(Builder::securityConfigurationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("securityConfigurationId").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, NAME_FIELD,
            ARN_FIELD, STATE_FIELD, CONTAINER_PROVIDER_FIELD, CREATED_AT_FIELD, TAGS_FIELD, SECURITY_CONFIGURATION_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String name;

    private final String arn;

    private final String state;

    private final ContainerProvider containerProvider;

    private final Instant createdAt;

    private final Map<String, String> tags;

    private final String securityConfigurationId;

    private VirtualCluster(BuilderImpl builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.arn = builder.arn;
        this.state = builder.state;
        this.containerProvider = builder.containerProvider;
        this.createdAt = builder.createdAt;
        this.tags = builder.tags;
        this.securityConfigurationId = builder.securityConfigurationId;
    }

    /**
     * <p>
     * The ID of the virtual cluster.
     * </p>
     * 
     * @return The ID of the virtual cluster.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The name of the virtual cluster.
     * </p>
     * 
     * @return The name of the virtual cluster.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ARN of the virtual cluster.
     * </p>
     * 
     * @return The ARN of the virtual cluster.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The state of the virtual cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link VirtualClusterState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the virtual cluster.
     * @see VirtualClusterState
     */
    public final VirtualClusterState state() {
        return VirtualClusterState.fromValue(state);
    }

    /**
     * <p>
     * The state of the virtual cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link VirtualClusterState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the virtual cluster.
     * @see VirtualClusterState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The container provider of the virtual cluster.
     * </p>
     * 
     * @return The container provider of the virtual cluster.
     */
    public final ContainerProvider containerProvider() {
        return containerProvider;
    }

    /**
     * <p>
     * The date and time when the virtual cluster is created.
     * </p>
     * 
     * @return The date and time when the virtual cluster is created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The assigned tags of the virtual cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The assigned tags of the virtual cluster.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The ID of the security configuration.
     * </p>
     * 
     * @return The ID of the security configuration.
     */
    public final String securityConfigurationId() {
        return securityConfigurationId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(containerProvider());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(securityConfigurationId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VirtualCluster)) {
            return false;
        }
        VirtualCluster other = (VirtualCluster) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(containerProvider(), other.containerProvider())
                && Objects.equals(createdAt(), other.createdAt()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(securityConfigurationId(), other.securityConfigurationId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("VirtualCluster").add("Id", id()).add("Name", name()).add("Arn", arn())
                .add("State", stateAsString()).add("ContainerProvider", containerProvider()).add("CreatedAt", createdAt())
                .add("Tags", hasTags() ? tags() : null).add("SecurityConfigurationId", securityConfigurationId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "containerProvider":
            return Optional.ofNullable(clazz.cast(containerProvider()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "securityConfigurationId":
            return Optional.ofNullable(clazz.cast(securityConfigurationId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<VirtualCluster, T> g) {
        return obj -> g.apply((VirtualCluster) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VirtualCluster> {
        /**
         * <p>
         * The ID of the virtual cluster.
         * </p>
         * 
         * @param id
         *        The ID of the virtual cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the virtual cluster.
         * </p>
         * 
         * @param name
         *        The name of the virtual cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ARN of the virtual cluster.
         * </p>
         * 
         * @param arn
         *        The ARN of the virtual cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The state of the virtual cluster.
         * </p>
         * 
         * @param state
         *        The state of the virtual cluster.
         * @see VirtualClusterState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VirtualClusterState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the virtual cluster.
         * </p>
         * 
         * @param state
         *        The state of the virtual cluster.
         * @see VirtualClusterState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VirtualClusterState
         */
        Builder state(VirtualClusterState state);

        /**
         * <p>
         * The container provider of the virtual cluster.
         * </p>
         * 
         * @param containerProvider
         *        The container provider of the virtual cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerProvider(ContainerProvider containerProvider);

        /**
         * <p>
         * The container provider of the virtual cluster.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerProvider.Builder} avoiding the
         * need to create one manually via {@link ContainerProvider#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerProvider.Builder#build()} is called immediately and its
         * result is passed to {@link #containerProvider(ContainerProvider)}.
         * 
         * @param containerProvider
         *        a consumer that will call methods on {@link ContainerProvider.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #containerProvider(ContainerProvider)
         */
        default Builder containerProvider(Consumer<ContainerProvider.Builder> containerProvider) {
            return containerProvider(ContainerProvider.builder().applyMutation(containerProvider).build());
        }

        /**
         * <p>
         * The date and time when the virtual cluster is created.
         * </p>
         * 
         * @param createdAt
         *        The date and time when the virtual cluster is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The assigned tags of the virtual cluster.
         * </p>
         * 
         * @param tags
         *        The assigned tags of the virtual cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The ID of the security configuration.
         * </p>
         * 
         * @param securityConfigurationId
         *        The ID of the security configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityConfigurationId(String securityConfigurationId);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String name;

        private String arn;

        private String state;

        private ContainerProvider containerProvider;

        private Instant createdAt;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String securityConfigurationId;

        private BuilderImpl() {
        }

        private BuilderImpl(VirtualCluster model) {
            id(model.id);
            name(model.name);
            arn(model.arn);
            state(model.state);
            containerProvider(model.containerProvider);
            createdAt(model.createdAt);
            tags(model.tags);
            securityConfigurationId(model.securityConfigurationId);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(VirtualClusterState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final ContainerProvider.Builder getContainerProvider() {
            return containerProvider != null ? containerProvider.toBuilder() : null;
        }

        public final void setContainerProvider(ContainerProvider.BuilderImpl containerProvider) {
            this.containerProvider = containerProvider != null ? containerProvider.build() : null;
        }

        @Override
        public final Builder containerProvider(ContainerProvider containerProvider) {
            this.containerProvider = containerProvider;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final String getSecurityConfigurationId() {
            return securityConfigurationId;
        }

        public final void setSecurityConfigurationId(String securityConfigurationId) {
            this.securityConfigurationId = securityConfigurationId;
        }

        @Override
        public final Builder securityConfigurationId(String securityConfigurationId) {
            this.securityConfigurationId = securityConfigurationId;
            return this;
        }

        @Override
        public VirtualCluster build() {
            return new VirtualCluster(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
