/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emrserverless;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.emrserverless.internal.EmrServerlessServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.emrserverless.model.CancelJobRunRequest;
import software.amazon.awssdk.services.emrserverless.model.CancelJobRunResponse;
import software.amazon.awssdk.services.emrserverless.model.ConflictException;
import software.amazon.awssdk.services.emrserverless.model.CreateApplicationRequest;
import software.amazon.awssdk.services.emrserverless.model.CreateApplicationResponse;
import software.amazon.awssdk.services.emrserverless.model.DeleteApplicationRequest;
import software.amazon.awssdk.services.emrserverless.model.DeleteApplicationResponse;
import software.amazon.awssdk.services.emrserverless.model.EmrServerlessException;
import software.amazon.awssdk.services.emrserverless.model.GetApplicationRequest;
import software.amazon.awssdk.services.emrserverless.model.GetApplicationResponse;
import software.amazon.awssdk.services.emrserverless.model.GetDashboardForJobRunRequest;
import software.amazon.awssdk.services.emrserverless.model.GetDashboardForJobRunResponse;
import software.amazon.awssdk.services.emrserverless.model.GetJobRunRequest;
import software.amazon.awssdk.services.emrserverless.model.GetJobRunResponse;
import software.amazon.awssdk.services.emrserverless.model.InternalServerException;
import software.amazon.awssdk.services.emrserverless.model.ListApplicationsRequest;
import software.amazon.awssdk.services.emrserverless.model.ListApplicationsResponse;
import software.amazon.awssdk.services.emrserverless.model.ListJobRunAttemptsRequest;
import software.amazon.awssdk.services.emrserverless.model.ListJobRunAttemptsResponse;
import software.amazon.awssdk.services.emrserverless.model.ListJobRunsRequest;
import software.amazon.awssdk.services.emrserverless.model.ListJobRunsResponse;
import software.amazon.awssdk.services.emrserverless.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.emrserverless.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.emrserverless.model.ResourceNotFoundException;
import software.amazon.awssdk.services.emrserverless.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.emrserverless.model.StartApplicationRequest;
import software.amazon.awssdk.services.emrserverless.model.StartApplicationResponse;
import software.amazon.awssdk.services.emrserverless.model.StartJobRunRequest;
import software.amazon.awssdk.services.emrserverless.model.StartJobRunResponse;
import software.amazon.awssdk.services.emrserverless.model.StopApplicationRequest;
import software.amazon.awssdk.services.emrserverless.model.StopApplicationResponse;
import software.amazon.awssdk.services.emrserverless.model.TagResourceRequest;
import software.amazon.awssdk.services.emrserverless.model.TagResourceResponse;
import software.amazon.awssdk.services.emrserverless.model.UntagResourceRequest;
import software.amazon.awssdk.services.emrserverless.model.UntagResourceResponse;
import software.amazon.awssdk.services.emrserverless.model.UpdateApplicationRequest;
import software.amazon.awssdk.services.emrserverless.model.UpdateApplicationResponse;
import software.amazon.awssdk.services.emrserverless.model.ValidationException;
import software.amazon.awssdk.services.emrserverless.transform.CancelJobRunRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.DeleteApplicationRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.GetApplicationRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.GetDashboardForJobRunRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.GetJobRunRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.ListJobRunAttemptsRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.ListJobRunsRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.StartApplicationRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.StartJobRunRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.StopApplicationRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.emrserverless.transform.UpdateApplicationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link EmrServerlessAsyncClient}.
 *
 * @see EmrServerlessAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEmrServerlessAsyncClient implements EmrServerlessAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultEmrServerlessAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEmrServerlessAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels a job run.
     * </p>
     *
     * @param cancelJobRunRequest
     * @return A Java Future containing the result of the CancelJobRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.CancelJobRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/CancelJobRun" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelJobRunResponse> cancelJobRun(CancelJobRunRequest cancelJobRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelJobRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJobRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelJobRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelJobRunResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CancelJobRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelJobRunRequest, CancelJobRunResponse>()
                            .withOperationName("CancelJobRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelJobRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelJobRunRequest));
            CompletableFuture<CancelJobRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an application.
     * </p>
     *
     * @param createApplicationRequest
     * @return A Java Future containing the result of the CreateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.CreateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/CreateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApplicationResponse> createApplication(CreateApplicationRequest createApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateApplicationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                            .withOperationName("CreateApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createApplicationRequest));
            CompletableFuture<CreateApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an application. An application has to be in a stopped or created state in order to be deleted.
     * </p>
     *
     * @param deleteApplicationRequest
     * @return A Java Future containing the result of the DeleteApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.DeleteApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/DeleteApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApplicationResponse> deleteApplication(DeleteApplicationRequest deleteApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteApplicationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApplicationRequest, DeleteApplicationResponse>()
                            .withOperationName("DeleteApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteApplicationRequest));
            CompletableFuture<DeleteApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays detailed information about a specified application.
     * </p>
     *
     * @param getApplicationRequest
     * @return A Java Future containing the result of the GetApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.GetApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/GetApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetApplicationResponse> getApplication(GetApplicationRequest getApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getApplicationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetApplicationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetApplicationRequest, GetApplicationResponse>()
                            .withOperationName("GetApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getApplicationRequest));
            CompletableFuture<GetApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates and returns a URL that you can use to access the application UIs for a job run.
     * </p>
     * <p>
     * For jobs in a running state, the application UI is a live user interface such as the Spark or Tez web UI. For
     * completed jobs, the application UI is a persistent application user interface such as the Spark History Server or
     * persistent Tez UI.
     * </p>
     * <note>
     * <p>
     * The URL is valid for one hour after you generate it. To access the application UI after that hour elapses, you
     * must invoke the API again to generate a new URL.
     * </p>
     * </note>
     *
     * @param getDashboardForJobRunRequest
     * @return A Java Future containing the result of the GetDashboardForJobRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.GetDashboardForJobRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/GetDashboardForJobRun"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDashboardForJobRunResponse> getDashboardForJobRun(
            GetDashboardForJobRunRequest getDashboardForJobRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDashboardForJobRunRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDashboardForJobRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDashboardForJobRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDashboardForJobRunResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDashboardForJobRunResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetDashboardForJobRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDashboardForJobRunRequest, GetDashboardForJobRunResponse>()
                            .withOperationName("GetDashboardForJobRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDashboardForJobRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDashboardForJobRunRequest));
            CompletableFuture<GetDashboardForJobRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays detailed information about a job run.
     * </p>
     *
     * @param getJobRunRequest
     * @return A Java Future containing the result of the GetJobRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.GetJobRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/GetJobRun" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobRunResponse> getJobRun(GetJobRunRequest getJobRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getJobRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJobRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetJobRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetJobRunResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetJobRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetJobRunRequest, GetJobRunResponse>().withOperationName("GetJobRun")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetJobRunRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getJobRunRequest));
            CompletableFuture<GetJobRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists applications based on a set of parameters.
     * </p>
     *
     * @param listApplicationsRequest
     * @return A Java Future containing the result of the ListApplications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/ListApplications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationsResponse> listApplications(ListApplicationsRequest listApplicationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listApplicationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplications");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApplicationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListApplicationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                            .withOperationName("ListApplications").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listApplicationsRequest));
            CompletableFuture<ListApplicationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all attempt of a job run.
     * </p>
     *
     * @param listJobRunAttemptsRequest
     * @return A Java Future containing the result of the ListJobRunAttempts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.ListJobRunAttempts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/ListJobRunAttempts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobRunAttemptsResponse> listJobRunAttempts(ListJobRunAttemptsRequest listJobRunAttemptsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listJobRunAttemptsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobRunAttemptsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobRunAttempts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobRunAttemptsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListJobRunAttemptsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListJobRunAttemptsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobRunAttemptsRequest, ListJobRunAttemptsResponse>()
                            .withOperationName("ListJobRunAttempts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListJobRunAttemptsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listJobRunAttemptsRequest));
            CompletableFuture<ListJobRunAttemptsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists job runs based on a set of parameters.
     * </p>
     *
     * @param listJobRunsRequest
     * @return A Java Future containing the result of the ListJobRuns operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.ListJobRuns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/ListJobRuns" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobRunsResponse> listJobRuns(ListJobRunsRequest listJobRunsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listJobRunsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobRunsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobRuns");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobRunsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListJobRunsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListJobRunsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobRunsRequest, ListJobRunsResponse>()
                            .withOperationName("ListJobRuns").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListJobRunsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listJobRunsRequest));
            CompletableFuture<ListJobRunsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags assigned to the resources.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a specified application and initializes initial capacity if configured.
     * </p>
     *
     * @param startApplicationRequest
     * @return A Java Future containing the result of the StartApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>ServiceQuotaExceededException The maximum number of resources per account has been reached.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.StartApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/StartApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartApplicationResponse> startApplication(StartApplicationRequest startApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartApplicationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartApplicationRequest, StartApplicationResponse>()
                            .withOperationName("StartApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startApplicationRequest));
            CompletableFuture<StartApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a job run.
     * </p>
     *
     * @param startJobRunRequest
     * @return A Java Future containing the result of the StartJobRun operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.StartJobRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/StartJobRun" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartJobRunResponse> startJobRun(StartJobRunRequest startJobRunRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startJobRunRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startJobRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartJobRun");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartJobRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartJobRunResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartJobRunResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartJobRunRequest, StartJobRunResponse>()
                            .withOperationName("StartJobRun").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartJobRunRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startJobRunRequest));
            CompletableFuture<StartJobRunResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a specified application and releases initial capacity if configured. All scheduled and running jobs must be
     * completed or cancelled before stopping an application.
     * </p>
     *
     * @param stopApplicationRequest
     * @return A Java Future containing the result of the StopApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.StopApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/StopApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopApplicationResponse> stopApplication(StopApplicationRequest stopApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopApplicationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StopApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopApplicationRequest, StopApplicationResponse>()
                            .withOperationName("StopApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopApplicationRequest));
            CompletableFuture<StopApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns tags to resources. A tag is a label that you assign to an Amazon Web Services resource. Each tag consists
     * of a key and an optional value, both of which you define. Tags enable you to categorize your Amazon Web Services
     * resources by attributes such as purpose, owner, or environment. When you have many resources of the same type,
     * you can quickly identify a specific resource based on the tags you've assigned to it.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from resources.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a specified application. An application has to be in a stopped or created state in order to be updated.
     * </p>
     *
     * @param updateApplicationRequest
     * @return A Java Future containing the result of the UpdateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServerException Request processing failed because of an error or failure with the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EmrServerlessException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EmrServerlessAsyncClient.UpdateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/emr-serverless-2021-07-13/UpdateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApplicationResponse> updateApplication(UpdateApplicationRequest updateApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EMR Serverless");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateApplicationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApplicationRequest, UpdateApplicationResponse>()
                            .withOperationName("UpdateApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateApplicationRequest));
            CompletableFuture<UpdateApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final EmrServerlessServiceClientConfiguration serviceClientConfiguration() {
        return new EmrServerlessServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(EmrServerlessException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        EmrServerlessServiceClientConfigurationBuilder serviceConfigBuilder = new EmrServerlessServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
