/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.entityresolution;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.entityresolution.model.AccessDeniedException;
import software.amazon.awssdk.services.entityresolution.model.ConflictException;
import software.amazon.awssdk.services.entityresolution.model.CreateMatchingWorkflowRequest;
import software.amazon.awssdk.services.entityresolution.model.CreateMatchingWorkflowResponse;
import software.amazon.awssdk.services.entityresolution.model.CreateSchemaMappingRequest;
import software.amazon.awssdk.services.entityresolution.model.CreateSchemaMappingResponse;
import software.amazon.awssdk.services.entityresolution.model.DeleteMatchingWorkflowRequest;
import software.amazon.awssdk.services.entityresolution.model.DeleteMatchingWorkflowResponse;
import software.amazon.awssdk.services.entityresolution.model.DeleteSchemaMappingRequest;
import software.amazon.awssdk.services.entityresolution.model.DeleteSchemaMappingResponse;
import software.amazon.awssdk.services.entityresolution.model.EntityResolutionException;
import software.amazon.awssdk.services.entityresolution.model.ExceedsLimitException;
import software.amazon.awssdk.services.entityresolution.model.GetMatchIdRequest;
import software.amazon.awssdk.services.entityresolution.model.GetMatchIdResponse;
import software.amazon.awssdk.services.entityresolution.model.GetMatchingJobRequest;
import software.amazon.awssdk.services.entityresolution.model.GetMatchingJobResponse;
import software.amazon.awssdk.services.entityresolution.model.GetMatchingWorkflowRequest;
import software.amazon.awssdk.services.entityresolution.model.GetMatchingWorkflowResponse;
import software.amazon.awssdk.services.entityresolution.model.GetSchemaMappingRequest;
import software.amazon.awssdk.services.entityresolution.model.GetSchemaMappingResponse;
import software.amazon.awssdk.services.entityresolution.model.InternalServerException;
import software.amazon.awssdk.services.entityresolution.model.ListMatchingJobsRequest;
import software.amazon.awssdk.services.entityresolution.model.ListMatchingJobsResponse;
import software.amazon.awssdk.services.entityresolution.model.ListMatchingWorkflowsRequest;
import software.amazon.awssdk.services.entityresolution.model.ListMatchingWorkflowsResponse;
import software.amazon.awssdk.services.entityresolution.model.ListSchemaMappingsRequest;
import software.amazon.awssdk.services.entityresolution.model.ListSchemaMappingsResponse;
import software.amazon.awssdk.services.entityresolution.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.entityresolution.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.entityresolution.model.ResourceNotFoundException;
import software.amazon.awssdk.services.entityresolution.model.StartMatchingJobRequest;
import software.amazon.awssdk.services.entityresolution.model.StartMatchingJobResponse;
import software.amazon.awssdk.services.entityresolution.model.TagResourceRequest;
import software.amazon.awssdk.services.entityresolution.model.TagResourceResponse;
import software.amazon.awssdk.services.entityresolution.model.ThrottlingException;
import software.amazon.awssdk.services.entityresolution.model.UntagResourceRequest;
import software.amazon.awssdk.services.entityresolution.model.UntagResourceResponse;
import software.amazon.awssdk.services.entityresolution.model.UpdateMatchingWorkflowRequest;
import software.amazon.awssdk.services.entityresolution.model.UpdateMatchingWorkflowResponse;
import software.amazon.awssdk.services.entityresolution.model.ValidationException;
import software.amazon.awssdk.services.entityresolution.transform.CreateMatchingWorkflowRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.CreateSchemaMappingRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.DeleteMatchingWorkflowRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.DeleteSchemaMappingRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.GetMatchIdRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.GetMatchingJobRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.GetMatchingWorkflowRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.GetSchemaMappingRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.ListMatchingJobsRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.ListMatchingWorkflowsRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.ListSchemaMappingsRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.StartMatchingJobRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.UpdateMatchingWorkflowRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link EntityResolutionClient}.
 *
 * @see EntityResolutionClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEntityResolutionClient implements EntityResolutionClient {
    private static final Logger log = Logger.loggerFor(DefaultEntityResolutionClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final EntityResolutionServiceClientConfiguration serviceClientConfiguration;

    protected DefaultEntityResolutionClient(EntityResolutionServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a <code>MatchingWorkflow</code> object which stores the configuration of the data processing job to be
     * run. It is important to note that there should not be a pre-existing <code>MatchingWorkflow</code> with the same
     * name. To modify an existing workflow, utilize the <code>UpdateMatchingWorkflow</code> API.
     * </p>
     *
     * @param createMatchingWorkflowRequest
     * @return Result of the CreateMatchingWorkflow operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. <code>HTTP Status Code: 429</code>
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. <code>HTTP Status Code: 403</code>
     * @throws ExceedsLimitException
     *         The request was rejected because it attempted to create resources beyond the current Entity Resolution
     *         account limits. The error message describes the limit exceeded. <code>HTTP Status Code: 402</code>
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource. Example:
     *         Workflow already exists, Schema already exists, Workflow is currently running, etc.
     *         <code>HTTP Status Code: 400</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.CreateMatchingWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/CreateMatchingWorkflow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateMatchingWorkflowResponse createMatchingWorkflow(CreateMatchingWorkflowRequest createMatchingWorkflowRequest)
            throws ThrottlingException, InternalServerException, AccessDeniedException, ExceedsLimitException, ConflictException,
            ValidationException, AwsServiceException, SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMatchingWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateMatchingWorkflowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMatchingWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMatchingWorkflow");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateMatchingWorkflowRequest, CreateMatchingWorkflowResponse>()
                            .withOperationName("CreateMatchingWorkflow").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createMatchingWorkflowRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateMatchingWorkflowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a schema mapping, which defines the schema of the input customer records table. The
     * <code>SchemaMapping</code> also provides Entity Resolution with some metadata about the table, such as the
     * attribute types of the columns and which columns to match on.
     * </p>
     *
     * @param createSchemaMappingRequest
     * @return Result of the CreateSchemaMapping operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. <code>HTTP Status Code: 429</code>
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. <code>HTTP Status Code: 403</code>
     * @throws ExceedsLimitException
     *         The request was rejected because it attempted to create resources beyond the current Entity Resolution
     *         account limits. The error message describes the limit exceeded. <code>HTTP Status Code: 402</code>
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource. Example:
     *         Workflow already exists, Schema already exists, Workflow is currently running, etc.
     *         <code>HTTP Status Code: 400</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.CreateSchemaMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/CreateSchemaMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateSchemaMappingResponse createSchemaMapping(CreateSchemaMappingRequest createSchemaMappingRequest)
            throws ThrottlingException, InternalServerException, AccessDeniedException, ExceedsLimitException, ConflictException,
            ValidationException, AwsServiceException, SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSchemaMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSchemaMappingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSchemaMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSchemaMapping");

            return clientHandler.execute(new ClientExecutionParams<CreateSchemaMappingRequest, CreateSchemaMappingResponse>()
                    .withOperationName("CreateSchemaMapping").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createSchemaMappingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSchemaMappingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the <code>MatchingWorkflow</code> with a given name. This operation will succeed even if a workflow with
     * the given name does not exist.
     * </p>
     *
     * @param deleteMatchingWorkflowRequest
     * @return Result of the DeleteMatchingWorkflow operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. <code>HTTP Status Code: 429</code>
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. <code>HTTP Status Code: 403</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.DeleteMatchingWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/DeleteMatchingWorkflow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteMatchingWorkflowResponse deleteMatchingWorkflow(DeleteMatchingWorkflowRequest deleteMatchingWorkflowRequest)
            throws ThrottlingException, InternalServerException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMatchingWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteMatchingWorkflowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMatchingWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMatchingWorkflow");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteMatchingWorkflowRequest, DeleteMatchingWorkflowResponse>()
                            .withOperationName("DeleteMatchingWorkflow").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteMatchingWorkflowRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteMatchingWorkflowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the <code>SchemaMapping</code> with a given name. This operation will succeed even if a schema with the
     * given name does not exist. This operation will fail if there is a <code>DataIntegrationWorkflow</code> object
     * that references the <code>SchemaMapping</code> in the workflow's <code>InputSourceConfig</code>.
     * </p>
     *
     * @param deleteSchemaMappingRequest
     * @return Result of the DeleteSchemaMapping operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. <code>HTTP Status Code: 429</code>
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. <code>HTTP Status Code: 403</code>
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource. Example:
     *         Workflow already exists, Schema already exists, Workflow is currently running, etc.
     *         <code>HTTP Status Code: 400</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.DeleteSchemaMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/DeleteSchemaMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteSchemaMappingResponse deleteSchemaMapping(DeleteSchemaMappingRequest deleteSchemaMappingRequest)
            throws ThrottlingException, InternalServerException, AccessDeniedException, ConflictException, ValidationException,
            AwsServiceException, SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSchemaMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSchemaMappingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSchemaMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSchemaMapping");

            return clientHandler.execute(new ClientExecutionParams<DeleteSchemaMappingRequest, DeleteSchemaMappingResponse>()
                    .withOperationName("DeleteSchemaMapping").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteSchemaMappingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSchemaMappingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the corresponding Match ID of a customer record if the record has been processed.
     * </p>
     *
     * @param getMatchIdRequest
     * @return Result of the GetMatchId operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. <code>HTTP Status Code: 429</code>
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws ResourceNotFoundException
     *         The resource could not be found. <code>HTTP Status Code: 404</code>
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. <code>HTTP Status Code: 403</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.GetMatchId
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/GetMatchId" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetMatchIdResponse getMatchId(GetMatchIdRequest getMatchIdRequest) throws ThrottlingException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMatchIdResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMatchIdResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMatchIdRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMatchId");

            return clientHandler
                    .execute(new ClientExecutionParams<GetMatchIdRequest, GetMatchIdResponse>().withOperationName("GetMatchId")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getMatchIdRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetMatchIdRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the status, metrics, and errors (if there are any) that are associated with a job.
     * </p>
     *
     * @param getMatchingJobRequest
     * @return Result of the GetMatchingJob operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. <code>HTTP Status Code: 429</code>
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws ResourceNotFoundException
     *         The resource could not be found. <code>HTTP Status Code: 404</code>
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. <code>HTTP Status Code: 403</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.GetMatchingJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/GetMatchingJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMatchingJobResponse getMatchingJob(GetMatchingJobRequest getMatchingJobRequest) throws ThrottlingException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMatchingJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMatchingJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMatchingJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMatchingJob");

            return clientHandler.execute(new ClientExecutionParams<GetMatchingJobRequest, GetMatchingJobResponse>()
                    .withOperationName("GetMatchingJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getMatchingJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMatchingJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the <code>MatchingWorkflow</code> with a given name, if it exists.
     * </p>
     *
     * @param getMatchingWorkflowRequest
     * @return Result of the GetMatchingWorkflow operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. <code>HTTP Status Code: 429</code>
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws ResourceNotFoundException
     *         The resource could not be found. <code>HTTP Status Code: 404</code>
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. <code>HTTP Status Code: 403</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.GetMatchingWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/GetMatchingWorkflow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMatchingWorkflowResponse getMatchingWorkflow(GetMatchingWorkflowRequest getMatchingWorkflowRequest)
            throws ThrottlingException, InternalServerException, ResourceNotFoundException, AccessDeniedException,
            ValidationException, AwsServiceException, SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMatchingWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetMatchingWorkflowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMatchingWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMatchingWorkflow");

            return clientHandler.execute(new ClientExecutionParams<GetMatchingWorkflowRequest, GetMatchingWorkflowResponse>()
                    .withOperationName("GetMatchingWorkflow").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getMatchingWorkflowRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMatchingWorkflowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the SchemaMapping of a given name.
     * </p>
     *
     * @param getSchemaMappingRequest
     * @return Result of the GetSchemaMapping operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. <code>HTTP Status Code: 429</code>
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws ResourceNotFoundException
     *         The resource could not be found. <code>HTTP Status Code: 404</code>
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. <code>HTTP Status Code: 403</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.GetSchemaMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/GetSchemaMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSchemaMappingResponse getSchemaMapping(GetSchemaMappingRequest getSchemaMappingRequest) throws ThrottlingException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSchemaMappingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSchemaMappingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSchemaMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSchemaMapping");

            return clientHandler.execute(new ClientExecutionParams<GetSchemaMappingRequest, GetSchemaMappingResponse>()
                    .withOperationName("GetSchemaMapping").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSchemaMappingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSchemaMappingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all jobs for a given workflow.
     * </p>
     *
     * @param listMatchingJobsRequest
     * @return Result of the ListMatchingJobs operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. <code>HTTP Status Code: 429</code>
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws ResourceNotFoundException
     *         The resource could not be found. <code>HTTP Status Code: 404</code>
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. <code>HTTP Status Code: 403</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.ListMatchingJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/ListMatchingJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMatchingJobsResponse listMatchingJobs(ListMatchingJobsRequest listMatchingJobsRequest) throws ThrottlingException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMatchingJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListMatchingJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMatchingJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMatchingJobs");

            return clientHandler.execute(new ClientExecutionParams<ListMatchingJobsRequest, ListMatchingJobsResponse>()
                    .withOperationName("ListMatchingJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listMatchingJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMatchingJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all the <code>MatchingWorkflows</code> that have been created for an Amazon Web Services
     * account.
     * </p>
     *
     * @param listMatchingWorkflowsRequest
     * @return Result of the ListMatchingWorkflows operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. <code>HTTP Status Code: 429</code>
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. <code>HTTP Status Code: 403</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.ListMatchingWorkflows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/ListMatchingWorkflows"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMatchingWorkflowsResponse listMatchingWorkflows(ListMatchingWorkflowsRequest listMatchingWorkflowsRequest)
            throws ThrottlingException, InternalServerException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMatchingWorkflowsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListMatchingWorkflowsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMatchingWorkflowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMatchingWorkflows");

            return clientHandler.execute(new ClientExecutionParams<ListMatchingWorkflowsRequest, ListMatchingWorkflowsResponse>()
                    .withOperationName("ListMatchingWorkflows").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listMatchingWorkflowsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMatchingWorkflowsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all the <code>SchemaMappings</code> that have been created for an Amazon Web Services account.
     * </p>
     *
     * @param listSchemaMappingsRequest
     * @return Result of the ListSchemaMappings operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. <code>HTTP Status Code: 429</code>
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. <code>HTTP Status Code: 403</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.ListSchemaMappings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/ListSchemaMappings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSchemaMappingsResponse listSchemaMappings(ListSchemaMappingsRequest listSchemaMappingsRequest)
            throws ThrottlingException, InternalServerException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSchemaMappingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSchemaMappingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSchemaMappingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSchemaMappings");

            return clientHandler.execute(new ClientExecutionParams<ListSchemaMappingsRequest, ListSchemaMappingsResponse>()
                    .withOperationName("ListSchemaMappings").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSchemaMappingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSchemaMappingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays the tags associated with an Entity Resolution resource. In Entity Resolution, <code>SchemaMapping</code>
     * , and <code>MatchingWorkflow</code> can be tagged.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws ResourceNotFoundException
     *         The resource could not be found. <code>HTTP Status Code: 404</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the <code>MatchingJob</code> of a workflow. The workflow must have previously been created using the
     * <code>CreateMatchingWorkflow</code> endpoint.
     * </p>
     *
     * @param startMatchingJobRequest
     * @return Result of the StartMatchingJob operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. <code>HTTP Status Code: 429</code>
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws ResourceNotFoundException
     *         The resource could not be found. <code>HTTP Status Code: 404</code>
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. <code>HTTP Status Code: 403</code>
     * @throws ExceedsLimitException
     *         The request was rejected because it attempted to create resources beyond the current Entity Resolution
     *         account limits. The error message describes the limit exceeded. <code>HTTP Status Code: 402</code>
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource. Example:
     *         Workflow already exists, Schema already exists, Workflow is currently running, etc.
     *         <code>HTTP Status Code: 400</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.StartMatchingJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/StartMatchingJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartMatchingJobResponse startMatchingJob(StartMatchingJobRequest startMatchingJobRequest) throws ThrottlingException,
            InternalServerException, ResourceNotFoundException, AccessDeniedException, ExceedsLimitException, ConflictException,
            ValidationException, AwsServiceException, SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartMatchingJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartMatchingJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMatchingJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMatchingJob");

            return clientHandler.execute(new ClientExecutionParams<StartMatchingJobRequest, StartMatchingJobResponse>()
                    .withOperationName("StartMatchingJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startMatchingJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartMatchingJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified Entity Resolution resource. Tags can help you
     * organize and categorize your resources. You can also use them to scope user permissions by granting a user
     * permission to access or change only resources with certain tag values. In Entity Resolution,
     * <code>SchemaMapping</code> and <code>MatchingWorkflow</code> can be tagged. Tags don't have any semantic meaning
     * to Amazon Web Services and are interpreted strictly as strings of characters. You can use the
     * <code>TagResource</code> action with a resource that already has tags. If you specify a new tag key, this tag is
     * appended to the list of tags associated with the resource. If you specify a tag key that is already associated
     * with the resource, the new tag value that you specify replaces the previous value for that tag.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws ResourceNotFoundException
     *         The resource could not be found. <code>HTTP Status Code: 404</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified Entity Resolution resource. In Entity Resolution,
     * <code>SchemaMapping</code>, and <code>MatchingWorkflow</code> can be tagged.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws ResourceNotFoundException
     *         The resource could not be found. <code>HTTP Status Code: 404</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing <code>MatchingWorkflow</code>. This method is identical to
     * <code>CreateMatchingWorkflow</code>, except it uses an HTTP <code>PUT</code> request instead of a
     * <code>POST</code> request, and the <code>MatchingWorkflow</code> must already exist for the method to succeed.
     * </p>
     *
     * @param updateMatchingWorkflowRequest
     * @return Result of the UpdateMatchingWorkflow operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. <code>HTTP Status Code: 429</code>
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Entity Resolution service.
     *         <code>HTTP Status Code: 500</code>
     * @throws ResourceNotFoundException
     *         The resource could not be found. <code>HTTP Status Code: 404</code>
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action. <code>HTTP Status Code: 403</code>
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EntityResolutionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EntityResolutionClient.UpdateMatchingWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/UpdateMatchingWorkflow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateMatchingWorkflowResponse updateMatchingWorkflow(UpdateMatchingWorkflowRequest updateMatchingWorkflowRequest)
            throws ThrottlingException, InternalServerException, ResourceNotFoundException, AccessDeniedException,
            ValidationException, AwsServiceException, SdkClientException, EntityResolutionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateMatchingWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateMatchingWorkflowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMatchingWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMatchingWorkflow");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateMatchingWorkflowRequest, UpdateMatchingWorkflowResponse>()
                            .withOperationName("UpdateMatchingWorkflow").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateMatchingWorkflowRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateMatchingWorkflowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EntityResolutionException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExceedsLimitException")
                                .exceptionBuilderSupplier(ExceedsLimitException::builder).httpStatusCode(402).build());
    }

    @Override
    public final EntityResolutionServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
