/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.entityresolution;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.entityresolution.internal.EntityResolutionServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.entityresolution.model.AccessDeniedException;
import software.amazon.awssdk.services.entityresolution.model.ConflictException;
import software.amazon.awssdk.services.entityresolution.model.CreateMatchingWorkflowRequest;
import software.amazon.awssdk.services.entityresolution.model.CreateMatchingWorkflowResponse;
import software.amazon.awssdk.services.entityresolution.model.CreateSchemaMappingRequest;
import software.amazon.awssdk.services.entityresolution.model.CreateSchemaMappingResponse;
import software.amazon.awssdk.services.entityresolution.model.DeleteMatchingWorkflowRequest;
import software.amazon.awssdk.services.entityresolution.model.DeleteMatchingWorkflowResponse;
import software.amazon.awssdk.services.entityresolution.model.DeleteSchemaMappingRequest;
import software.amazon.awssdk.services.entityresolution.model.DeleteSchemaMappingResponse;
import software.amazon.awssdk.services.entityresolution.model.EntityResolutionException;
import software.amazon.awssdk.services.entityresolution.model.ExceedsLimitException;
import software.amazon.awssdk.services.entityresolution.model.GetMatchIdRequest;
import software.amazon.awssdk.services.entityresolution.model.GetMatchIdResponse;
import software.amazon.awssdk.services.entityresolution.model.GetMatchingJobRequest;
import software.amazon.awssdk.services.entityresolution.model.GetMatchingJobResponse;
import software.amazon.awssdk.services.entityresolution.model.GetMatchingWorkflowRequest;
import software.amazon.awssdk.services.entityresolution.model.GetMatchingWorkflowResponse;
import software.amazon.awssdk.services.entityresolution.model.GetSchemaMappingRequest;
import software.amazon.awssdk.services.entityresolution.model.GetSchemaMappingResponse;
import software.amazon.awssdk.services.entityresolution.model.InternalServerException;
import software.amazon.awssdk.services.entityresolution.model.ListMatchingJobsRequest;
import software.amazon.awssdk.services.entityresolution.model.ListMatchingJobsResponse;
import software.amazon.awssdk.services.entityresolution.model.ListMatchingWorkflowsRequest;
import software.amazon.awssdk.services.entityresolution.model.ListMatchingWorkflowsResponse;
import software.amazon.awssdk.services.entityresolution.model.ListSchemaMappingsRequest;
import software.amazon.awssdk.services.entityresolution.model.ListSchemaMappingsResponse;
import software.amazon.awssdk.services.entityresolution.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.entityresolution.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.entityresolution.model.ResourceNotFoundException;
import software.amazon.awssdk.services.entityresolution.model.StartMatchingJobRequest;
import software.amazon.awssdk.services.entityresolution.model.StartMatchingJobResponse;
import software.amazon.awssdk.services.entityresolution.model.TagResourceRequest;
import software.amazon.awssdk.services.entityresolution.model.TagResourceResponse;
import software.amazon.awssdk.services.entityresolution.model.ThrottlingException;
import software.amazon.awssdk.services.entityresolution.model.UntagResourceRequest;
import software.amazon.awssdk.services.entityresolution.model.UntagResourceResponse;
import software.amazon.awssdk.services.entityresolution.model.UpdateMatchingWorkflowRequest;
import software.amazon.awssdk.services.entityresolution.model.UpdateMatchingWorkflowResponse;
import software.amazon.awssdk.services.entityresolution.model.ValidationException;
import software.amazon.awssdk.services.entityresolution.transform.CreateMatchingWorkflowRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.CreateSchemaMappingRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.DeleteMatchingWorkflowRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.DeleteSchemaMappingRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.GetMatchIdRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.GetMatchingJobRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.GetMatchingWorkflowRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.GetSchemaMappingRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.ListMatchingJobsRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.ListMatchingWorkflowsRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.ListSchemaMappingsRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.StartMatchingJobRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.entityresolution.transform.UpdateMatchingWorkflowRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link EntityResolutionAsyncClient}.
 *
 * @see EntityResolutionAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEntityResolutionAsyncClient implements EntityResolutionAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultEntityResolutionAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final EntityResolutionServiceClientConfiguration serviceClientConfiguration;

    protected DefaultEntityResolutionAsyncClient(EntityResolutionServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a <code>MatchingWorkflow</code> object which stores the configuration of the data processing job to be
     * run. It is important to note that there should not be a pre-existing <code>MatchingWorkflow</code> with the same
     * name. To modify an existing workflow, utilize the <code>UpdateMatchingWorkflow</code> API.
     * </p>
     *
     * @param createMatchingWorkflowRequest
     * @return A Java Future containing the result of the CreateMatchingWorkflow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.
     *         <code>HTTP Status Code: 429</code></li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.
     *         <code>HTTP Status Code: 403</code></li>
     *         <li>ExceedsLimitException The request was rejected because it attempted to create resources beyond the
     *         current Entity Resolution account limits. The error message describes the limit exceeded.
     *         <code>HTTP Status Code: 402</code></li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource. Example: Workflow already exists, Schema already exists, Workflow is currently running, etc.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.CreateMatchingWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/CreateMatchingWorkflow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMatchingWorkflowResponse> createMatchingWorkflow(
            CreateMatchingWorkflowRequest createMatchingWorkflowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMatchingWorkflowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMatchingWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMatchingWorkflow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMatchingWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMatchingWorkflowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMatchingWorkflowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMatchingWorkflowRequest, CreateMatchingWorkflowResponse>()
                            .withOperationName("CreateMatchingWorkflow")
                            .withMarshaller(new CreateMatchingWorkflowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createMatchingWorkflowRequest));
            CompletableFuture<CreateMatchingWorkflowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a schema mapping, which defines the schema of the input customer records table. The
     * <code>SchemaMapping</code> also provides Entity Resolution with some metadata about the table, such as the
     * attribute types of the columns and which columns to match on.
     * </p>
     *
     * @param createSchemaMappingRequest
     * @return A Java Future containing the result of the CreateSchemaMapping operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.
     *         <code>HTTP Status Code: 429</code></li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.
     *         <code>HTTP Status Code: 403</code></li>
     *         <li>ExceedsLimitException The request was rejected because it attempted to create resources beyond the
     *         current Entity Resolution account limits. The error message describes the limit exceeded.
     *         <code>HTTP Status Code: 402</code></li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource. Example: Workflow already exists, Schema already exists, Workflow is currently running, etc.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.CreateSchemaMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/CreateSchemaMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSchemaMappingResponse> createSchemaMapping(
            CreateSchemaMappingRequest createSchemaMappingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSchemaMappingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSchemaMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSchemaMapping");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSchemaMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSchemaMappingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSchemaMappingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSchemaMappingRequest, CreateSchemaMappingResponse>()
                            .withOperationName("CreateSchemaMapping")
                            .withMarshaller(new CreateSchemaMappingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSchemaMappingRequest));
            CompletableFuture<CreateSchemaMappingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the <code>MatchingWorkflow</code> with a given name. This operation will succeed even if a workflow with
     * the given name does not exist.
     * </p>
     *
     * @param deleteMatchingWorkflowRequest
     * @return A Java Future containing the result of the DeleteMatchingWorkflow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.
     *         <code>HTTP Status Code: 429</code></li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.
     *         <code>HTTP Status Code: 403</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.DeleteMatchingWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/DeleteMatchingWorkflow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMatchingWorkflowResponse> deleteMatchingWorkflow(
            DeleteMatchingWorkflowRequest deleteMatchingWorkflowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMatchingWorkflowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMatchingWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMatchingWorkflow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMatchingWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMatchingWorkflowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMatchingWorkflowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMatchingWorkflowRequest, DeleteMatchingWorkflowResponse>()
                            .withOperationName("DeleteMatchingWorkflow")
                            .withMarshaller(new DeleteMatchingWorkflowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMatchingWorkflowRequest));
            CompletableFuture<DeleteMatchingWorkflowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the <code>SchemaMapping</code> with a given name. This operation will succeed even if a schema with the
     * given name does not exist. This operation will fail if there is a <code>DataIntegrationWorkflow</code> object
     * that references the <code>SchemaMapping</code> in the workflow's <code>InputSourceConfig</code>.
     * </p>
     *
     * @param deleteSchemaMappingRequest
     * @return A Java Future containing the result of the DeleteSchemaMapping operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.
     *         <code>HTTP Status Code: 429</code></li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.
     *         <code>HTTP Status Code: 403</code></li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource. Example: Workflow already exists, Schema already exists, Workflow is currently running, etc.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.DeleteSchemaMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/DeleteSchemaMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSchemaMappingResponse> deleteSchemaMapping(
            DeleteSchemaMappingRequest deleteSchemaMappingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSchemaMappingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSchemaMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSchemaMapping");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSchemaMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSchemaMappingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSchemaMappingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSchemaMappingRequest, DeleteSchemaMappingResponse>()
                            .withOperationName("DeleteSchemaMapping")
                            .withMarshaller(new DeleteSchemaMappingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSchemaMappingRequest));
            CompletableFuture<DeleteSchemaMappingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the corresponding Match ID of a customer record if the record has been processed.
     * </p>
     *
     * @param getMatchIdRequest
     * @return A Java Future containing the result of the GetMatchId operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.
     *         <code>HTTP Status Code: 429</code></li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>ResourceNotFoundException The resource could not be found. <code>HTTP Status Code: 404</code></li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.
     *         <code>HTTP Status Code: 403</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.GetMatchId
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/GetMatchId" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMatchIdResponse> getMatchId(GetMatchIdRequest getMatchIdRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMatchIdRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMatchIdRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMatchId");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMatchIdResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMatchIdResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMatchIdResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMatchIdRequest, GetMatchIdResponse>().withOperationName("GetMatchId")
                            .withMarshaller(new GetMatchIdRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMatchIdRequest));
            CompletableFuture<GetMatchIdResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the status, metrics, and errors (if there are any) that are associated with a job.
     * </p>
     *
     * @param getMatchingJobRequest
     * @return A Java Future containing the result of the GetMatchingJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.
     *         <code>HTTP Status Code: 429</code></li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>ResourceNotFoundException The resource could not be found. <code>HTTP Status Code: 404</code></li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.
     *         <code>HTTP Status Code: 403</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.GetMatchingJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/GetMatchingJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMatchingJobResponse> getMatchingJob(GetMatchingJobRequest getMatchingJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMatchingJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMatchingJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMatchingJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMatchingJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMatchingJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMatchingJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMatchingJobRequest, GetMatchingJobResponse>()
                            .withOperationName("GetMatchingJob")
                            .withMarshaller(new GetMatchingJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMatchingJobRequest));
            CompletableFuture<GetMatchingJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the <code>MatchingWorkflow</code> with a given name, if it exists.
     * </p>
     *
     * @param getMatchingWorkflowRequest
     * @return A Java Future containing the result of the GetMatchingWorkflow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.
     *         <code>HTTP Status Code: 429</code></li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>ResourceNotFoundException The resource could not be found. <code>HTTP Status Code: 404</code></li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.
     *         <code>HTTP Status Code: 403</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.GetMatchingWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/GetMatchingWorkflow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMatchingWorkflowResponse> getMatchingWorkflow(
            GetMatchingWorkflowRequest getMatchingWorkflowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMatchingWorkflowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMatchingWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMatchingWorkflow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMatchingWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMatchingWorkflowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMatchingWorkflowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMatchingWorkflowRequest, GetMatchingWorkflowResponse>()
                            .withOperationName("GetMatchingWorkflow")
                            .withMarshaller(new GetMatchingWorkflowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMatchingWorkflowRequest));
            CompletableFuture<GetMatchingWorkflowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the SchemaMapping of a given name.
     * </p>
     *
     * @param getSchemaMappingRequest
     * @return A Java Future containing the result of the GetSchemaMapping operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.
     *         <code>HTTP Status Code: 429</code></li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>ResourceNotFoundException The resource could not be found. <code>HTTP Status Code: 404</code></li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.
     *         <code>HTTP Status Code: 403</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.GetSchemaMapping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/GetSchemaMapping"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSchemaMappingResponse> getSchemaMapping(GetSchemaMappingRequest getSchemaMappingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSchemaMappingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSchemaMappingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSchemaMapping");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSchemaMappingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSchemaMappingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSchemaMappingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSchemaMappingRequest, GetSchemaMappingResponse>()
                            .withOperationName("GetSchemaMapping")
                            .withMarshaller(new GetSchemaMappingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSchemaMappingRequest));
            CompletableFuture<GetSchemaMappingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all jobs for a given workflow.
     * </p>
     *
     * @param listMatchingJobsRequest
     * @return A Java Future containing the result of the ListMatchingJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.
     *         <code>HTTP Status Code: 429</code></li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>ResourceNotFoundException The resource could not be found. <code>HTTP Status Code: 404</code></li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.
     *         <code>HTTP Status Code: 403</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.ListMatchingJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/ListMatchingJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMatchingJobsResponse> listMatchingJobs(ListMatchingJobsRequest listMatchingJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMatchingJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMatchingJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMatchingJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMatchingJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMatchingJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMatchingJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMatchingJobsRequest, ListMatchingJobsResponse>()
                            .withOperationName("ListMatchingJobs")
                            .withMarshaller(new ListMatchingJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMatchingJobsRequest));
            CompletableFuture<ListMatchingJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all the <code>MatchingWorkflows</code> that have been created for an Amazon Web Services
     * account.
     * </p>
     *
     * @param listMatchingWorkflowsRequest
     * @return A Java Future containing the result of the ListMatchingWorkflows operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.
     *         <code>HTTP Status Code: 429</code></li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.
     *         <code>HTTP Status Code: 403</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.ListMatchingWorkflows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/ListMatchingWorkflows"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMatchingWorkflowsResponse> listMatchingWorkflows(
            ListMatchingWorkflowsRequest listMatchingWorkflowsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMatchingWorkflowsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMatchingWorkflowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMatchingWorkflows");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMatchingWorkflowsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMatchingWorkflowsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMatchingWorkflowsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMatchingWorkflowsRequest, ListMatchingWorkflowsResponse>()
                            .withOperationName("ListMatchingWorkflows")
                            .withMarshaller(new ListMatchingWorkflowsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMatchingWorkflowsRequest));
            CompletableFuture<ListMatchingWorkflowsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all the <code>SchemaMappings</code> that have been created for an Amazon Web Services account.
     * </p>
     *
     * @param listSchemaMappingsRequest
     * @return A Java Future containing the result of the ListSchemaMappings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.
     *         <code>HTTP Status Code: 429</code></li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.
     *         <code>HTTP Status Code: 403</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.ListSchemaMappings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/ListSchemaMappings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSchemaMappingsResponse> listSchemaMappings(ListSchemaMappingsRequest listSchemaMappingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSchemaMappingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSchemaMappingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSchemaMappings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSchemaMappingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSchemaMappingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSchemaMappingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSchemaMappingsRequest, ListSchemaMappingsResponse>()
                            .withOperationName("ListSchemaMappings")
                            .withMarshaller(new ListSchemaMappingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSchemaMappingsRequest));
            CompletableFuture<ListSchemaMappingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays the tags associated with an Entity Resolution resource. In Entity Resolution, <code>SchemaMapping</code>
     * , and <code>MatchingWorkflow</code> can be tagged.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>ResourceNotFoundException The resource could not be found. <code>HTTP Status Code: 404</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the <code>MatchingJob</code> of a workflow. The workflow must have previously been created using the
     * <code>CreateMatchingWorkflow</code> endpoint.
     * </p>
     *
     * @param startMatchingJobRequest
     * @return A Java Future containing the result of the StartMatchingJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.
     *         <code>HTTP Status Code: 429</code></li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>ResourceNotFoundException The resource could not be found. <code>HTTP Status Code: 404</code></li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.
     *         <code>HTTP Status Code: 403</code></li>
     *         <li>ExceedsLimitException The request was rejected because it attempted to create resources beyond the
     *         current Entity Resolution account limits. The error message describes the limit exceeded.
     *         <code>HTTP Status Code: 402</code></li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource. Example: Workflow already exists, Schema already exists, Workflow is currently running, etc.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.StartMatchingJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/StartMatchingJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartMatchingJobResponse> startMatchingJob(StartMatchingJobRequest startMatchingJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMatchingJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMatchingJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMatchingJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMatchingJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartMatchingJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartMatchingJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartMatchingJobRequest, StartMatchingJobResponse>()
                            .withOperationName("StartMatchingJob")
                            .withMarshaller(new StartMatchingJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startMatchingJobRequest));
            CompletableFuture<StartMatchingJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified Entity Resolution resource. Tags can help you
     * organize and categorize your resources. You can also use them to scope user permissions by granting a user
     * permission to access or change only resources with certain tag values. In Entity Resolution,
     * <code>SchemaMapping</code> and <code>MatchingWorkflow</code> can be tagged. Tags don't have any semantic meaning
     * to Amazon Web Services and are interpreted strictly as strings of characters. You can use the
     * <code>TagResource</code> action with a resource that already has tags. If you specify a new tag key, this tag is
     * appended to the list of tags associated with the resource. If you specify a tag key that is already associated
     * with the resource, the new tag value that you specify replaces the previous value for that tag.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>ResourceNotFoundException The resource could not be found. <code>HTTP Status Code: 404</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified Entity Resolution resource. In Entity Resolution,
     * <code>SchemaMapping</code>, and <code>MatchingWorkflow</code> can be tagged.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>ResourceNotFoundException The resource could not be found. <code>HTTP Status Code: 404</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing <code>MatchingWorkflow</code>. This method is identical to
     * <code>CreateMatchingWorkflow</code>, except it uses an HTTP <code>PUT</code> request instead of a
     * <code>POST</code> request, and the <code>MatchingWorkflow</code> must already exist for the method to succeed.
     * </p>
     *
     * @param updateMatchingWorkflowRequest
     * @return A Java Future containing the result of the UpdateMatchingWorkflow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.
     *         <code>HTTP Status Code: 429</code></li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Entity
     *         Resolution service. <code>HTTP Status Code: 500</code></li>
     *         <li>ResourceNotFoundException The resource could not be found. <code>HTTP Status Code: 404</code></li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.
     *         <code>HTTP Status Code: 403</code></li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Entity Resolution.
     *         <code>HTTP Status Code: 400</code></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EntityResolutionException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample EntityResolutionAsyncClient.UpdateMatchingWorkflow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/entityresolution-2018-05-10/UpdateMatchingWorkflow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMatchingWorkflowResponse> updateMatchingWorkflow(
            UpdateMatchingWorkflowRequest updateMatchingWorkflowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateMatchingWorkflowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMatchingWorkflowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EntityResolution");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMatchingWorkflow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMatchingWorkflowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateMatchingWorkflowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMatchingWorkflowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMatchingWorkflowRequest, UpdateMatchingWorkflowResponse>()
                            .withOperationName("UpdateMatchingWorkflow")
                            .withMarshaller(new UpdateMatchingWorkflowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateMatchingWorkflowRequest));
            CompletableFuture<UpdateMatchingWorkflowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final EntityResolutionServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EntityResolutionException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExceedsLimitException")
                                .exceptionBuilderSupplier(ExceedsLimitException::builder).httpStatusCode(402).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        EntityResolutionServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = EntityResolutionServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
