/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eventbridge.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The custom parameters to be used when the target is an Amazon ECS task.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EcsParameters implements SdkPojo, Serializable, ToCopyableBuilder<EcsParameters.Builder, EcsParameters> {
    private static final SdkField<String> TASK_DEFINITION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EcsParameters::taskDefinitionArn)).setter(setter(Builder::taskDefinitionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskDefinitionArn").build()).build();

    private static final SdkField<Integer> TASK_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(EcsParameters::taskCount)).setter(setter(Builder::taskCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskCount").build()).build();

    private static final SdkField<String> LAUNCH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EcsParameters::launchTypeAsString)).setter(setter(Builder::launchType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchType").build()).build();

    private static final SdkField<NetworkConfiguration> NETWORK_CONFIGURATION_FIELD = SdkField
            .<NetworkConfiguration> builder(MarshallingType.SDK_POJO).getter(getter(EcsParameters::networkConfiguration))
            .setter(setter(Builder::networkConfiguration)).constructor(NetworkConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkConfiguration").build())
            .build();

    private static final SdkField<String> PLATFORM_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EcsParameters::platformVersion)).setter(setter(Builder::platformVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlatformVersion").build()).build();

    private static final SdkField<String> GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EcsParameters::group)).setter(setter(Builder::group))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Group").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TASK_DEFINITION_ARN_FIELD,
            TASK_COUNT_FIELD, LAUNCH_TYPE_FIELD, NETWORK_CONFIGURATION_FIELD, PLATFORM_VERSION_FIELD, GROUP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String taskDefinitionArn;

    private final Integer taskCount;

    private final String launchType;

    private final NetworkConfiguration networkConfiguration;

    private final String platformVersion;

    private final String group;

    private EcsParameters(BuilderImpl builder) {
        this.taskDefinitionArn = builder.taskDefinitionArn;
        this.taskCount = builder.taskCount;
        this.launchType = builder.launchType;
        this.networkConfiguration = builder.networkConfiguration;
        this.platformVersion = builder.platformVersion;
        this.group = builder.group;
    }

    /**
     * <p>
     * The ARN of the task definition to use if the event target is an Amazon ECS task.
     * </p>
     * 
     * @return The ARN of the task definition to use if the event target is an Amazon ECS task.
     */
    public String taskDefinitionArn() {
        return taskDefinitionArn;
    }

    /**
     * <p>
     * The number of tasks to create based on <code>TaskDefinition</code>. The default is 1.
     * </p>
     * 
     * @return The number of tasks to create based on <code>TaskDefinition</code>. The default is 1.
     */
    public Integer taskCount() {
        return taskCount;
    }

    /**
     * <p>
     * Specifies the launch type on which your task is running. The launch type that you specify here must match one of
     * the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is supported only in the
     * Regions where AWS Fargate with Amazon ECS is supported. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS-Fargate.html">AWS Fargate on Amazon ECS</a>
     * in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #launchType} will
     * return {@link LaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #launchTypeAsString}.
     * </p>
     * 
     * @return Specifies the launch type on which your task is running. The launch type that you specify here must match
     *         one of the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is supported
     *         only in the Regions where AWS Fargate with Amazon ECS is supported. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS-Fargate.html">AWS Fargate on Amazon
     *         ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * @see LaunchType
     */
    public LaunchType launchType() {
        return LaunchType.fromValue(launchType);
    }

    /**
     * <p>
     * Specifies the launch type on which your task is running. The launch type that you specify here must match one of
     * the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is supported only in the
     * Regions where AWS Fargate with Amazon ECS is supported. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS-Fargate.html">AWS Fargate on Amazon ECS</a>
     * in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #launchType} will
     * return {@link LaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #launchTypeAsString}.
     * </p>
     * 
     * @return Specifies the launch type on which your task is running. The launch type that you specify here must match
     *         one of the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is supported
     *         only in the Regions where AWS Fargate with Amazon ECS is supported. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS-Fargate.html">AWS Fargate on Amazon
     *         ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * @see LaunchType
     */
    public String launchTypeAsString() {
        return launchType;
    }

    /**
     * <p>
     * Use this structure if the ECS task uses the <code>awsvpc</code> network mode. This structure specifies the VPC
     * subnets and security groups associated with the task, and whether a public IP address is to be used. This
     * structure is required if <code>LaunchType</code> is <code>FARGATE</code> because the <code>awsvpc</code> mode is
     * required for Fargate tasks.
     * </p>
     * <p>
     * If you specify <code>NetworkConfiguration</code> when the target ECS task does not use the <code>awsvpc</code>
     * network mode, the task fails.
     * </p>
     * 
     * @return Use this structure if the ECS task uses the <code>awsvpc</code> network mode. This structure specifies
     *         the VPC subnets and security groups associated with the task, and whether a public IP address is to be
     *         used. This structure is required if <code>LaunchType</code> is <code>FARGATE</code> because the
     *         <code>awsvpc</code> mode is required for Fargate tasks.</p>
     *         <p>
     *         If you specify <code>NetworkConfiguration</code> when the target ECS task does not use the
     *         <code>awsvpc</code> network mode, the task fails.
     */
    public NetworkConfiguration networkConfiguration() {
        return networkConfiguration;
    }

    /**
     * <p>
     * Specifies the platform version for the task. Specify only the numeric portion of the platform version, such as
     * <code>1.1.0</code>.
     * </p>
     * <p>
     * This structure is used only if <code>LaunchType</code> is <code>FARGATE</code>. For more information about valid
     * platform versions, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html">AWS Fargate Platform
     * Versions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return Specifies the platform version for the task. Specify only the numeric portion of the platform version,
     *         such as <code>1.1.0</code>.</p>
     *         <p>
     *         This structure is used only if <code>LaunchType</code> is <code>FARGATE</code>. For more information
     *         about valid platform versions, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html">AWS Fargate
     *         Platform Versions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public String platformVersion() {
        return platformVersion;
    }

    /**
     * <p>
     * Specifies an ECS task group for the task. The maximum length is 255 characters.
     * </p>
     * 
     * @return Specifies an ECS task group for the task. The maximum length is 255 characters.
     */
    public String group() {
        return group;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(taskDefinitionArn());
        hashCode = 31 * hashCode + Objects.hashCode(taskCount());
        hashCode = 31 * hashCode + Objects.hashCode(launchTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(networkConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(platformVersion());
        hashCode = 31 * hashCode + Objects.hashCode(group());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EcsParameters)) {
            return false;
        }
        EcsParameters other = (EcsParameters) obj;
        return Objects.equals(taskDefinitionArn(), other.taskDefinitionArn()) && Objects.equals(taskCount(), other.taskCount())
                && Objects.equals(launchTypeAsString(), other.launchTypeAsString())
                && Objects.equals(networkConfiguration(), other.networkConfiguration())
                && Objects.equals(platformVersion(), other.platformVersion()) && Objects.equals(group(), other.group());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("EcsParameters").add("TaskDefinitionArn", taskDefinitionArn()).add("TaskCount", taskCount())
                .add("LaunchType", launchTypeAsString()).add("NetworkConfiguration", networkConfiguration())
                .add("PlatformVersion", platformVersion()).add("Group", group()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TaskDefinitionArn":
            return Optional.ofNullable(clazz.cast(taskDefinitionArn()));
        case "TaskCount":
            return Optional.ofNullable(clazz.cast(taskCount()));
        case "LaunchType":
            return Optional.ofNullable(clazz.cast(launchTypeAsString()));
        case "NetworkConfiguration":
            return Optional.ofNullable(clazz.cast(networkConfiguration()));
        case "PlatformVersion":
            return Optional.ofNullable(clazz.cast(platformVersion()));
        case "Group":
            return Optional.ofNullable(clazz.cast(group()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EcsParameters, T> g) {
        return obj -> g.apply((EcsParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EcsParameters> {
        /**
         * <p>
         * The ARN of the task definition to use if the event target is an Amazon ECS task.
         * </p>
         * 
         * @param taskDefinitionArn
         *        The ARN of the task definition to use if the event target is an Amazon ECS task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskDefinitionArn(String taskDefinitionArn);

        /**
         * <p>
         * The number of tasks to create based on <code>TaskDefinition</code>. The default is 1.
         * </p>
         * 
         * @param taskCount
         *        The number of tasks to create based on <code>TaskDefinition</code>. The default is 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskCount(Integer taskCount);

        /**
         * <p>
         * Specifies the launch type on which your task is running. The launch type that you specify here must match one
         * of the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is supported only in
         * the Regions where AWS Fargate with Amazon ECS is supported. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS-Fargate.html">AWS Fargate on Amazon
         * ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param launchType
         *        Specifies the launch type on which your task is running. The launch type that you specify here must
         *        match one of the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is
         *        supported only in the Regions where AWS Fargate with Amazon ECS is supported. For more information,
         *        see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS-Fargate.html">AWS Fargate
         *        on Amazon ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @see LaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchType
         */
        Builder launchType(String launchType);

        /**
         * <p>
         * Specifies the launch type on which your task is running. The launch type that you specify here must match one
         * of the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is supported only in
         * the Regions where AWS Fargate with Amazon ECS is supported. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS-Fargate.html">AWS Fargate on Amazon
         * ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param launchType
         *        Specifies the launch type on which your task is running. The launch type that you specify here must
         *        match one of the launch type (compatibilities) of the target task. The <code>FARGATE</code> value is
         *        supported only in the Regions where AWS Fargate with Amazon ECS is supported. For more information,
         *        see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS-Fargate.html">AWS Fargate
         *        on Amazon ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @see LaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchType
         */
        Builder launchType(LaunchType launchType);

        /**
         * <p>
         * Use this structure if the ECS task uses the <code>awsvpc</code> network mode. This structure specifies the
         * VPC subnets and security groups associated with the task, and whether a public IP address is to be used. This
         * structure is required if <code>LaunchType</code> is <code>FARGATE</code> because the <code>awsvpc</code> mode
         * is required for Fargate tasks.
         * </p>
         * <p>
         * If you specify <code>NetworkConfiguration</code> when the target ECS task does not use the
         * <code>awsvpc</code> network mode, the task fails.
         * </p>
         * 
         * @param networkConfiguration
         *        Use this structure if the ECS task uses the <code>awsvpc</code> network mode. This structure specifies
         *        the VPC subnets and security groups associated with the task, and whether a public IP address is to be
         *        used. This structure is required if <code>LaunchType</code> is <code>FARGATE</code> because the
         *        <code>awsvpc</code> mode is required for Fargate tasks.</p>
         *        <p>
         *        If you specify <code>NetworkConfiguration</code> when the target ECS task does not use the
         *        <code>awsvpc</code> network mode, the task fails.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkConfiguration(NetworkConfiguration networkConfiguration);

        /**
         * <p>
         * Use this structure if the ECS task uses the <code>awsvpc</code> network mode. This structure specifies the
         * VPC subnets and security groups associated with the task, and whether a public IP address is to be used. This
         * structure is required if <code>LaunchType</code> is <code>FARGATE</code> because the <code>awsvpc</code> mode
         * is required for Fargate tasks.
         * </p>
         * <p>
         * If you specify <code>NetworkConfiguration</code> when the target ECS task does not use the
         * <code>awsvpc</code> network mode, the task fails.
         * </p>
         * This is a convenience that creates an instance of the {@link NetworkConfiguration.Builder} avoiding the need
         * to create one manually via {@link NetworkConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link NetworkConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #networkConfiguration(NetworkConfiguration)}.
         * 
         * @param networkConfiguration
         *        a consumer that will call methods on {@link NetworkConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkConfiguration(NetworkConfiguration)
         */
        default Builder networkConfiguration(Consumer<NetworkConfiguration.Builder> networkConfiguration) {
            return networkConfiguration(NetworkConfiguration.builder().applyMutation(networkConfiguration).build());
        }

        /**
         * <p>
         * Specifies the platform version for the task. Specify only the numeric portion of the platform version, such
         * as <code>1.1.0</code>.
         * </p>
         * <p>
         * This structure is used only if <code>LaunchType</code> is <code>FARGATE</code>. For more information about
         * valid platform versions, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html">AWS Fargate
         * Platform Versions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param platformVersion
         *        Specifies the platform version for the task. Specify only the numeric portion of the platform version,
         *        such as <code>1.1.0</code>.</p>
         *        <p>
         *        This structure is used only if <code>LaunchType</code> is <code>FARGATE</code>. For more information
         *        about valid platform versions, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html">AWS Fargate
         *        Platform Versions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformVersion(String platformVersion);

        /**
         * <p>
         * Specifies an ECS task group for the task. The maximum length is 255 characters.
         * </p>
         * 
         * @param group
         *        Specifies an ECS task group for the task. The maximum length is 255 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder group(String group);
    }

    static final class BuilderImpl implements Builder {
        private String taskDefinitionArn;

        private Integer taskCount;

        private String launchType;

        private NetworkConfiguration networkConfiguration;

        private String platformVersion;

        private String group;

        private BuilderImpl() {
        }

        private BuilderImpl(EcsParameters model) {
            taskDefinitionArn(model.taskDefinitionArn);
            taskCount(model.taskCount);
            launchType(model.launchType);
            networkConfiguration(model.networkConfiguration);
            platformVersion(model.platformVersion);
            group(model.group);
        }

        public final String getTaskDefinitionArn() {
            return taskDefinitionArn;
        }

        @Override
        public final Builder taskDefinitionArn(String taskDefinitionArn) {
            this.taskDefinitionArn = taskDefinitionArn;
            return this;
        }

        public final void setTaskDefinitionArn(String taskDefinitionArn) {
            this.taskDefinitionArn = taskDefinitionArn;
        }

        public final Integer getTaskCount() {
            return taskCount;
        }

        @Override
        public final Builder taskCount(Integer taskCount) {
            this.taskCount = taskCount;
            return this;
        }

        public final void setTaskCount(Integer taskCount) {
            this.taskCount = taskCount;
        }

        public final String getLaunchTypeAsString() {
            return launchType;
        }

        @Override
        public final Builder launchType(String launchType) {
            this.launchType = launchType;
            return this;
        }

        @Override
        public final Builder launchType(LaunchType launchType) {
            this.launchType(launchType == null ? null : launchType.toString());
            return this;
        }

        public final void setLaunchType(String launchType) {
            this.launchType = launchType;
        }

        public final NetworkConfiguration.Builder getNetworkConfiguration() {
            return networkConfiguration != null ? networkConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder networkConfiguration(NetworkConfiguration networkConfiguration) {
            this.networkConfiguration = networkConfiguration;
            return this;
        }

        public final void setNetworkConfiguration(NetworkConfiguration.BuilderImpl networkConfiguration) {
            this.networkConfiguration = networkConfiguration != null ? networkConfiguration.build() : null;
        }

        public final String getPlatformVersion() {
            return platformVersion;
        }

        @Override
        public final Builder platformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
            return this;
        }

        public final void setPlatformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
        }

        public final String getGroup() {
            return group;
        }

        @Override
        public final Builder group(String group) {
            this.group = group;
            return this;
        }

        public final void setGroup(String group) {
            this.group = group;
        }

        @Override
        public EcsParameters build() {
            return new EcsParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
