/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eventbridge.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an event to be submitted.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutEventsRequestEntry implements SdkPojo, Serializable,
        ToCopyableBuilder<PutEventsRequestEntry.Builder, PutEventsRequestEntry> {
    private static final SdkField<Instant> TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT).memberName("Time")
            .getter(getter(PutEventsRequestEntry::time)).setter(setter(Builder::time))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Time").build()).build();

    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Source")
            .getter(getter(PutEventsRequestEntry::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final SdkField<List<String>> RESOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Resources")
            .getter(getter(PutEventsRequestEntry::resources))
            .setter(setter(Builder::resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DETAIL_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DetailType").getter(getter(PutEventsRequestEntry::detailType)).setter(setter(Builder::detailType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DetailType").build()).build();

    private static final SdkField<String> DETAIL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Detail")
            .getter(getter(PutEventsRequestEntry::detail)).setter(setter(Builder::detail))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Detail").build()).build();

    private static final SdkField<String> EVENT_BUS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventBusName").getter(getter(PutEventsRequestEntry::eventBusName)).setter(setter(Builder::eventBusName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventBusName").build()).build();

    private static final SdkField<String> TRACE_HEADER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TraceHeader").getter(getter(PutEventsRequestEntry::traceHeader)).setter(setter(Builder::traceHeader))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TraceHeader").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TIME_FIELD, SOURCE_FIELD,
            RESOURCES_FIELD, DETAIL_TYPE_FIELD, DETAIL_FIELD, EVENT_BUS_NAME_FIELD, TRACE_HEADER_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant time;

    private final String source;

    private final List<String> resources;

    private final String detailType;

    private final String detail;

    private final String eventBusName;

    private final String traceHeader;

    private PutEventsRequestEntry(BuilderImpl builder) {
        this.time = builder.time;
        this.source = builder.source;
        this.resources = builder.resources;
        this.detailType = builder.detailType;
        this.detail = builder.detail;
        this.eventBusName = builder.eventBusName;
        this.traceHeader = builder.traceHeader;
    }

    /**
     * <p>
     * The time stamp of the event, per <a href="https://www.rfc-editor.org/rfc/rfc3339.txt">RFC3339</a>. If no time
     * stamp is provided, the time stamp of the <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEvents.html">PutEvents</a> call is used.
     * </p>
     * 
     * @return The time stamp of the event, per <a href="https://www.rfc-editor.org/rfc/rfc3339.txt">RFC3339</a>. If no
     *         time stamp is provided, the time stamp of the <a
     *         href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEvents.html">PutEvents</a> call
     *         is used.
     */
    public final Instant time() {
        return time;
    }

    /**
     * <p>
     * The source of the event.
     * </p>
     * 
     * @return The source of the event.
     */
    public final String source() {
        return source;
    }

    /**
     * Returns true if the Resources property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasResources() {
        return resources != null && !(resources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Amazon Web Services resources, identified by Amazon Resource Name (ARN), which the event primarily concerns. Any
     * number, including zero, may be present.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResources()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Amazon Web Services resources, identified by Amazon Resource Name (ARN), which the event primarily
     *         concerns. Any number, including zero, may be present.
     */
    public final List<String> resources() {
        return resources;
    }

    /**
     * <p>
     * Free-form string used to decide what fields to expect in the event detail.
     * </p>
     * 
     * @return Free-form string used to decide what fields to expect in the event detail.
     */
    public final String detailType() {
        return detailType;
    }

    /**
     * <p>
     * A valid JSON string. There is no other schema imposed. The JSON string may contain fields and nested subobjects.
     * </p>
     * 
     * @return A valid JSON string. There is no other schema imposed. The JSON string may contain fields and nested
     *         subobjects.
     */
    public final String detail() {
        return detail;
    }

    /**
     * <p>
     * The name or ARN of the event bus to receive the event. Only the rules that are associated with this event bus are
     * used to match the event. If you omit this, the default event bus is used.
     * </p>
     * 
     * @return The name or ARN of the event bus to receive the event. Only the rules that are associated with this event
     *         bus are used to match the event. If you omit this, the default event bus is used.
     */
    public final String eventBusName() {
        return eventBusName;
    }

    /**
     * <p>
     * An X-Ray trade header, which is an http header (X-Amzn-Trace-Id) that contains the trace-id associated with the
     * event.
     * </p>
     * <p>
     * To learn more about X-Ray trace headers, see <a
     * href="https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-tracingheader">Tracing
     * header</a> in the X-Ray Developer Guide.
     * </p>
     * 
     * @return An X-Ray trade header, which is an http header (X-Amzn-Trace-Id) that contains the trace-id associated
     *         with the event.</p>
     *         <p>
     *         To learn more about X-Ray trace headers, see <a
     *         href="https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-tracingheader"
     *         >Tracing header</a> in the X-Ray Developer Guide.
     */
    public final String traceHeader() {
        return traceHeader;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(time());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(hasResources() ? resources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(detailType());
        hashCode = 31 * hashCode + Objects.hashCode(detail());
        hashCode = 31 * hashCode + Objects.hashCode(eventBusName());
        hashCode = 31 * hashCode + Objects.hashCode(traceHeader());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutEventsRequestEntry)) {
            return false;
        }
        PutEventsRequestEntry other = (PutEventsRequestEntry) obj;
        return Objects.equals(time(), other.time()) && Objects.equals(source(), other.source())
                && hasResources() == other.hasResources() && Objects.equals(resources(), other.resources())
                && Objects.equals(detailType(), other.detailType()) && Objects.equals(detail(), other.detail())
                && Objects.equals(eventBusName(), other.eventBusName()) && Objects.equals(traceHeader(), other.traceHeader());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PutEventsRequestEntry").add("Time", time()).add("Source", source())
                .add("Resources", hasResources() ? resources() : null).add("DetailType", detailType()).add("Detail", detail())
                .add("EventBusName", eventBusName()).add("TraceHeader", traceHeader()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Time":
            return Optional.ofNullable(clazz.cast(time()));
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "DetailType":
            return Optional.ofNullable(clazz.cast(detailType()));
        case "Detail":
            return Optional.ofNullable(clazz.cast(detail()));
        case "EventBusName":
            return Optional.ofNullable(clazz.cast(eventBusName()));
        case "TraceHeader":
            return Optional.ofNullable(clazz.cast(traceHeader()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutEventsRequestEntry, T> g) {
        return obj -> g.apply((PutEventsRequestEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PutEventsRequestEntry> {
        /**
         * <p>
         * The time stamp of the event, per <a href="https://www.rfc-editor.org/rfc/rfc3339.txt">RFC3339</a>. If no time
         * stamp is provided, the time stamp of the <a
         * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEvents.html">PutEvents</a> call is
         * used.
         * </p>
         * 
         * @param time
         *        The time stamp of the event, per <a href="https://www.rfc-editor.org/rfc/rfc3339.txt">RFC3339</a>. If
         *        no time stamp is provided, the time stamp of the <a
         *        href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEvents.html">PutEvents</a>
         *        call is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder time(Instant time);

        /**
         * <p>
         * The source of the event.
         * </p>
         * 
         * @param source
         *        The source of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * Amazon Web Services resources, identified by Amazon Resource Name (ARN), which the event primarily concerns.
         * Any number, including zero, may be present.
         * </p>
         * 
         * @param resources
         *        Amazon Web Services resources, identified by Amazon Resource Name (ARN), which the event primarily
         *        concerns. Any number, including zero, may be present.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<String> resources);

        /**
         * <p>
         * Amazon Web Services resources, identified by Amazon Resource Name (ARN), which the event primarily concerns.
         * Any number, including zero, may be present.
         * </p>
         * 
         * @param resources
         *        Amazon Web Services resources, identified by Amazon Resource Name (ARN), which the event primarily
         *        concerns. Any number, including zero, may be present.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(String... resources);

        /**
         * <p>
         * Free-form string used to decide what fields to expect in the event detail.
         * </p>
         * 
         * @param detailType
         *        Free-form string used to decide what fields to expect in the event detail.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detailType(String detailType);

        /**
         * <p>
         * A valid JSON string. There is no other schema imposed. The JSON string may contain fields and nested
         * subobjects.
         * </p>
         * 
         * @param detail
         *        A valid JSON string. There is no other schema imposed. The JSON string may contain fields and nested
         *        subobjects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detail(String detail);

        /**
         * <p>
         * The name or ARN of the event bus to receive the event. Only the rules that are associated with this event bus
         * are used to match the event. If you omit this, the default event bus is used.
         * </p>
         * 
         * @param eventBusName
         *        The name or ARN of the event bus to receive the event. Only the rules that are associated with this
         *        event bus are used to match the event. If you omit this, the default event bus is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventBusName(String eventBusName);

        /**
         * <p>
         * An X-Ray trade header, which is an http header (X-Amzn-Trace-Id) that contains the trace-id associated with
         * the event.
         * </p>
         * <p>
         * To learn more about X-Ray trace headers, see <a
         * href="https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-tracingheader"
         * >Tracing header</a> in the X-Ray Developer Guide.
         * </p>
         * 
         * @param traceHeader
         *        An X-Ray trade header, which is an http header (X-Amzn-Trace-Id) that contains the trace-id associated
         *        with the event.</p>
         *        <p>
         *        To learn more about X-Ray trace headers, see <a href=
         *        "https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-tracingheader"
         *        >Tracing header</a> in the X-Ray Developer Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder traceHeader(String traceHeader);
    }

    static final class BuilderImpl implements Builder {
        private Instant time;

        private String source;

        private List<String> resources = DefaultSdkAutoConstructList.getInstance();

        private String detailType;

        private String detail;

        private String eventBusName;

        private String traceHeader;

        private BuilderImpl() {
        }

        private BuilderImpl(PutEventsRequestEntry model) {
            time(model.time);
            source(model.source);
            resources(model.resources);
            detailType(model.detailType);
            detail(model.detail);
            eventBusName(model.eventBusName);
            traceHeader(model.traceHeader);
        }

        public final Instant getTime() {
            return time;
        }

        public final void setTime(Instant time) {
            this.time = time;
        }

        @Override
        @Transient
        public final Builder time(Instant time) {
            this.time = time;
            return this;
        }

        public final String getSource() {
            return source;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        @Override
        @Transient
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final Collection<String> getResources() {
            if (resources instanceof SdkAutoConstructList) {
                return null;
            }
            return resources;
        }

        public final void setResources(Collection<String> resources) {
            this.resources = EventResourceListCopier.copy(resources);
        }

        @Override
        @Transient
        public final Builder resources(Collection<String> resources) {
            this.resources = EventResourceListCopier.copy(resources);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder resources(String... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        public final String getDetailType() {
            return detailType;
        }

        public final void setDetailType(String detailType) {
            this.detailType = detailType;
        }

        @Override
        @Transient
        public final Builder detailType(String detailType) {
            this.detailType = detailType;
            return this;
        }

        public final String getDetail() {
            return detail;
        }

        public final void setDetail(String detail) {
            this.detail = detail;
        }

        @Override
        @Transient
        public final Builder detail(String detail) {
            this.detail = detail;
            return this;
        }

        public final String getEventBusName() {
            return eventBusName;
        }

        public final void setEventBusName(String eventBusName) {
            this.eventBusName = eventBusName;
        }

        @Override
        @Transient
        public final Builder eventBusName(String eventBusName) {
            this.eventBusName = eventBusName;
            return this;
        }

        public final String getTraceHeader() {
            return traceHeader;
        }

        public final void setTraceHeader(String traceHeader) {
            this.traceHeader = traceHeader;
        }

        @Override
        @Transient
        public final Builder traceHeader(String traceHeader) {
            this.traceHeader = traceHeader;
            return this;
        }

        @Override
        public PutEventsRequestEntry build() {
            return new PutEventsRequestEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
