/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchevents;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.protocol.json.AwsJsonProtocol;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolMetadata;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.internal.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cloudwatchevents.model.CloudWatchEventsException;
import software.amazon.awssdk.services.cloudwatchevents.model.ConcurrentModificationException;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeEventBusRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeEventBusResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DisableRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DisableRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.EnableRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.EnableRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.InternalException;
import software.amazon.awssdk.services.cloudwatchevents.model.InvalidEventPatternException;
import software.amazon.awssdk.services.cloudwatchevents.model.LimitExceededException;
import software.amazon.awssdk.services.cloudwatchevents.model.ListRuleNamesByTargetRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListRuleNamesByTargetResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListRulesRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListRulesResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListTargetsByRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListTargetsByRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.PolicyLengthExceededException;
import software.amazon.awssdk.services.cloudwatchevents.model.PutEventsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutEventsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.PutPermissionRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutPermissionResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.PutRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.PutTargetsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutTargetsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.RemovePermissionRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.RemovePermissionResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.RemoveTargetsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.RemoveTargetsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cloudwatchevents.model.TestEventPatternRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.TestEventPatternResponse;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeleteRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeleteRuleResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeEventBusRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeEventBusResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeRuleResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DisableRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DisableRuleResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.EnableRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.EnableRuleResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListRuleNamesByTargetRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListRuleNamesByTargetResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListRulesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListRulesResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListTargetsByRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListTargetsByRuleResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutEventsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutEventsResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutPermissionRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutPermissionResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutRuleResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutTargetsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutTargetsResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.RemovePermissionRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.RemovePermissionResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.RemoveTargetsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.RemoveTargetsResponseUnmarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.TestEventPatternRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.TestEventPatternResponseUnmarshaller;

/**
 * Internal implementation of {@link CloudWatchEventsClient}.
 *
 * @see CloudWatchEventsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudWatchEventsClient implements CloudWatchEventsClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudWatchEventsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.protocolFactory = init();
        this.clientConfiguration = clientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes the specified rule.
     * </p>
     * <p>
     * You must remove all targets from a rule using <a>RemoveTargets</a> before you can delete the rule.
     * </p>
     * <p>
     * When you delete a rule, incoming events might continue to match to the deleted rule. Please allow a short period
     * of time for changes to take effect.
     * </p>
     *
     * @param deleteRuleRequest
     * @return Result of the DeleteRule operation returned by the service.
     * @throws ConcurrentModificationException
     *         There is concurrent modification on a rule or target.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.DeleteRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DeleteRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteRuleResponse deleteRule(DeleteRuleRequest deleteRuleRequest) throws ConcurrentModificationException,
            InternalException, AwsServiceException, SdkClientException, CloudWatchEventsException {

        HttpResponseHandler<DeleteRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteRuleResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DeleteRuleRequest, DeleteRuleResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(deleteRuleRequest)
                .withMarshaller(new DeleteRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Displays the external AWS accounts that are permitted to write events to your account using your account's event
     * bus, and the associated policy. To enable your account to receive events from other accounts, use
     * <a>PutPermission</a>.
     * </p>
     *
     * @param describeEventBusRequest
     * @return Result of the DescribeEventBus operation returned by the service.
     * @throws ResourceNotFoundException
     *         An entity that you specified does not exist.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.DescribeEventBus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribeEventBus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeEventBusResponse describeEventBus(DescribeEventBusRequest describeEventBusRequest)
            throws ResourceNotFoundException, InternalException, AwsServiceException, SdkClientException,
            CloudWatchEventsException {

        HttpResponseHandler<DescribeEventBusResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeEventBusResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeEventBusRequest, DescribeEventBusResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeEventBusRequest).withMarshaller(new DescribeEventBusRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes the specified rule.
     * </p>
     *
     * @param describeRuleRequest
     * @return Result of the DescribeRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         An entity that you specified does not exist.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.DescribeRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribeRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeRuleResponse describeRule(DescribeRuleRequest describeRuleRequest) throws ResourceNotFoundException,
            InternalException, AwsServiceException, SdkClientException, CloudWatchEventsException {

        HttpResponseHandler<DescribeRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeRuleResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeRuleRequest, DescribeRuleResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeRuleRequest).withMarshaller(new DescribeRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Disables the specified rule. A disabled rule won't match any events, and won't self-trigger if it has a schedule
     * expression.
     * </p>
     * <p>
     * When you disable a rule, incoming events might continue to match to the disabled rule. Please allow a short
     * period of time for changes to take effect.
     * </p>
     *
     * @param disableRuleRequest
     * @return Result of the DisableRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         An entity that you specified does not exist.
     * @throws ConcurrentModificationException
     *         There is concurrent modification on a rule or target.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.DisableRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DisableRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DisableRuleResponse disableRule(DisableRuleRequest disableRuleRequest) throws ResourceNotFoundException,
            ConcurrentModificationException, InternalException, AwsServiceException, SdkClientException,
            CloudWatchEventsException {

        HttpResponseHandler<DisableRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DisableRuleResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DisableRuleRequest, DisableRuleResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(disableRuleRequest).withMarshaller(new DisableRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Enables the specified rule. If the rule does not exist, the operation fails.
     * </p>
     * <p>
     * When you enable a rule, incoming events might not immediately start matching to a newly enabled rule. Please
     * allow a short period of time for changes to take effect.
     * </p>
     *
     * @param enableRuleRequest
     * @return Result of the EnableRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         An entity that you specified does not exist.
     * @throws ConcurrentModificationException
     *         There is concurrent modification on a rule or target.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.EnableRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/EnableRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public EnableRuleResponse enableRule(EnableRuleRequest enableRuleRequest) throws ResourceNotFoundException,
            ConcurrentModificationException, InternalException, AwsServiceException, SdkClientException,
            CloudWatchEventsException {

        HttpResponseHandler<EnableRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new EnableRuleResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<EnableRuleRequest, EnableRuleResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(enableRuleRequest)
                .withMarshaller(new EnableRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the rules for the specified target. You can see which of the rules in Amazon CloudWatch Events can invoke a
     * specific target in your account.
     * </p>
     *
     * @param listRuleNamesByTargetRequest
     * @return Result of the ListRuleNamesByTarget operation returned by the service.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.ListRuleNamesByTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListRuleNamesByTarget" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListRuleNamesByTargetResponse listRuleNamesByTarget(ListRuleNamesByTargetRequest listRuleNamesByTargetRequest)
            throws InternalException, AwsServiceException, SdkClientException, CloudWatchEventsException {

        HttpResponseHandler<ListRuleNamesByTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListRuleNamesByTargetResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListRuleNamesByTargetRequest, ListRuleNamesByTargetResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listRuleNamesByTargetRequest)
                .withMarshaller(new ListRuleNamesByTargetRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists your Amazon CloudWatch Events rules. You can either list all the rules or you can provide a prefix to match
     * to the rule names.
     * </p>
     *
     * @param listRulesRequest
     * @return Result of the ListRules operation returned by the service.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.ListRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListRules" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListRulesResponse listRules(ListRulesRequest listRulesRequest) throws InternalException, AwsServiceException,
            SdkClientException, CloudWatchEventsException {

        HttpResponseHandler<ListRulesResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListRulesResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListRulesRequest, ListRulesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(listRulesRequest)
                .withMarshaller(new ListRulesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the targets assigned to the specified rule.
     * </p>
     *
     * @param listTargetsByRuleRequest
     * @return Result of the ListTargetsByRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         An entity that you specified does not exist.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.ListTargetsByRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListTargetsByRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTargetsByRuleResponse listTargetsByRule(ListTargetsByRuleRequest listTargetsByRuleRequest)
            throws ResourceNotFoundException, InternalException, AwsServiceException, SdkClientException,
            CloudWatchEventsException {

        HttpResponseHandler<ListTargetsByRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListTargetsByRuleResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListTargetsByRuleRequest, ListTargetsByRuleResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listTargetsByRuleRequest).withMarshaller(new ListTargetsByRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Sends custom events to Amazon CloudWatch Events so that they can be matched to rules.
     * </p>
     *
     * @param putEventsRequest
     * @return Result of the PutEvents operation returned by the service.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.PutEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutEventsResponse putEvents(PutEventsRequest putEventsRequest) throws InternalException, AwsServiceException,
            SdkClientException, CloudWatchEventsException {

        HttpResponseHandler<PutEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new PutEventsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<PutEventsRequest, PutEventsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(putEventsRequest)
                .withMarshaller(new PutEventsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Running <code>PutPermission</code> permits the specified AWS account to put events to your account's default
     * <i>event bus</i>. CloudWatch Events rules in your account are triggered by these events arriving to your default
     * event bus.
     * </p>
     * <p>
     * For another account to send events to your account, that external account must have a CloudWatch Events rule with
     * your account's default event bus as a target.
     * </p>
     * <p>
     * To enable multiple AWS accounts to put events to your default event bus, run <code>PutPermission</code> once for
     * each of these accounts.
     * </p>
     * <p>
     * The permission policy on the default event bus cannot exceed 10KB in size.
     * </p>
     *
     * @param putPermissionRequest
     * @return Result of the PutPermission operation returned by the service.
     * @throws ResourceNotFoundException
     *         An entity that you specified does not exist.
     * @throws PolicyLengthExceededException
     *         The event bus policy is too long. For more information, see the limits.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws ConcurrentModificationException
     *         There is concurrent modification on a rule or target.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.PutPermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutPermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutPermissionResponse putPermission(PutPermissionRequest putPermissionRequest) throws ResourceNotFoundException,
            PolicyLengthExceededException, InternalException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CloudWatchEventsException {

        HttpResponseHandler<PutPermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new PutPermissionResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<PutPermissionRequest, PutPermissionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(putPermissionRequest).withMarshaller(new PutPermissionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates or updates the specified rule. Rules are enabled by default, or based on value of the state. You can
     * disable a rule using <a>DisableRule</a>.
     * </p>
     * <p>
     * When you create or update a rule, incoming events might not immediately start matching to new or updated rules.
     * Please allow a short period of time for changes to take effect.
     * </p>
     * <p>
     * A rule must contain at least an EventPattern or ScheduleExpression. Rules with EventPatterns are triggered when a
     * matching event is observed. Rules with ScheduleExpressions self-trigger based on the given schedule. A rule can
     * have both an EventPattern and a ScheduleExpression, in which case the rule triggers on matching events as well as
     * on a schedule.
     * </p>
     * <p>
     * Most services in AWS treat : or / as the same character in Amazon Resource Names (ARNs). However, CloudWatch
     * Events uses an exact match in event patterns and rules. Be sure to use the correct ARN characters when creating
     * event patterns so that they match the ARN syntax in the event you want to match.
     * </p>
     *
     * @param putRuleRequest
     * @return Result of the PutRule operation returned by the service.
     * @throws InvalidEventPatternException
     *         The event pattern is not valid.
     * @throws LimitExceededException
     *         You tried to create more rules or add more targets to a rule than is allowed.
     * @throws ConcurrentModificationException
     *         There is concurrent modification on a rule or target.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.PutRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutRuleResponse putRule(PutRuleRequest putRuleRequest) throws InvalidEventPatternException, LimitExceededException,
            ConcurrentModificationException, InternalException, AwsServiceException, SdkClientException,
            CloudWatchEventsException {

        HttpResponseHandler<PutRuleResponse> responseHandler = protocolFactory.createResponseHandler(new JsonOperationMetadata()
                .withPayloadJson(true).withHasStreamingSuccessResponse(false), new PutRuleResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<PutRuleRequest, PutRuleResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(putRuleRequest)
                .withMarshaller(new PutRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds the specified targets to the specified rule, or updates the targets if they are already associated with the
     * rule.
     * </p>
     * <p>
     * Targets are the resources that are invoked when a rule is triggered.
     * </p>
     * <p>
     * You can configure the following as targets for CloudWatch Events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * EC2 instances
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS Lambda functions
     * </p>
     * </li>
     * <li>
     * <p>
     * Streams in Amazon Kinesis Streams
     * </p>
     * </li>
     * <li>
     * <p>
     * Delivery streams in Amazon Kinesis Firehose
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon ECS tasks
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS Step Functions state machines
     * </p>
     * </li>
     * <li>
     * <p>
     * Pipelines in Amazon Code Pipeline
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Inspector assessment templates
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon SNS topics
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon SQS queues
     * </p>
     * </li>
     * <li>
     * <p>
     * The default event bus of another AWS account
     * </p>
     * </li>
     * </ul>
     * <p>
     * Note that creating rules with built-in targets is supported only in the AWS Management Console.
     * </p>
     * <p>
     * For some target types, <code>PutTargets</code> provides target-specific parameters. If the target is an Amazon
     * Kinesis stream, you can optionally specify which shard the event goes to by using the
     * <code>KinesisParameters</code> argument. To invoke a command on multiple EC2 instances with one rule, you can use
     * the <code>RunCommandParameters</code> field.
     * </p>
     * <p>
     * To be able to make API calls against the resources that you own, Amazon CloudWatch Events needs the appropriate
     * permissions. For AWS Lambda and Amazon SNS resources, CloudWatch Events relies on resource-based policies. For
     * EC2 instances, Amazon Kinesis streams, and AWS Step Functions state machines, CloudWatch Events relies on IAM
     * roles that you specify in the <code>RoleARN</code> argument in <code>PutTargets</code>. For more information, see
     * <a href="http://docs.aws.amazon.com/AmazonCloudWatch/latest/events/auth-and-access-control-cwe.html">
     * Authentication and Access Control</a> in the <i>Amazon CloudWatch Events User Guide</i>.
     * </p>
     * <p>
     * If another AWS account is in the same region and has granted you permission (using <code>PutPermission</code>),
     * you can send events to that account by setting that account's event bus as a target of the rules in your account.
     * To send the matched events to the other account, specify that account's event bus as the <code>Arn</code> when
     * you run <code>PutTargets</code>. If your account sends events to another account, your account is charged for
     * each sent event. Each event sent to antoher account is charged as a custom event. The account receiving the event
     * is not charged. For more information on pricing, see <a href="https://aws.amazon.com/cloudwatch/pricing/">Amazon
     * CloudWatch Pricing</a>.
     * </p>
     * <p>
     * For more information about enabling cross-account events, see <a>PutPermission</a>.
     * </p>
     * <p>
     * <b>Input</b>, <b>InputPath</b> and <b>InputTransformer</b> are mutually exclusive and optional parameters of a
     * target. When a rule is triggered due to a matched event:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If none of the following arguments are specified for a target, then the entire event is passed to the target in
     * JSON form (unless the target is Amazon EC2 Run Command or Amazon ECS task, in which case nothing from the event
     * is passed to the target).
     * </p>
     * </li>
     * <li>
     * <p>
     * If <b>Input</b> is specified in the form of valid JSON, then the matched event is overridden with this constant.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <b>InputPath</b> is specified in the form of JSONPath (for example, <code>&#36.detail</code>), then only the
     * part of the event specified in the path is passed to the target (for example, only the detail part of the event
     * is passed).
     * </p>
     * </li>
     * <li>
     * <p>
     * If <b>InputTransformer</b> is specified, then one or more specified JSONPaths are extracted from the event and
     * used as values in a template that you specify as the input to the target.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you specify <code>Input</code>, <code>InputPath</code>, or <code>InputTransformer</code>, you must use JSON
     * dot notation, not bracket notation.
     * </p>
     * <p>
     * When you add targets to a rule and the associated rule triggers soon after, new or updated targets might not be
     * immediately invoked. Please allow a short period of time for changes to take effect.
     * </p>
     * <p>
     * This action can partially fail if too many requests are made at the same time. If that happens,
     * <code>FailedEntryCount</code> is non-zero in the response and each entry in <code>FailedEntries</code> provides
     * the ID of the failed target and the error code.
     * </p>
     *
     * @param putTargetsRequest
     * @return Result of the PutTargets operation returned by the service.
     * @throws ResourceNotFoundException
     *         An entity that you specified does not exist.
     * @throws ConcurrentModificationException
     *         There is concurrent modification on a rule or target.
     * @throws LimitExceededException
     *         You tried to create more rules or add more targets to a rule than is allowed.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.PutTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutTargets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutTargetsResponse putTargets(PutTargetsRequest putTargetsRequest) throws ResourceNotFoundException,
            ConcurrentModificationException, LimitExceededException, InternalException, AwsServiceException, SdkClientException,
            CloudWatchEventsException {

        HttpResponseHandler<PutTargetsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new PutTargetsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<PutTargetsRequest, PutTargetsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(putTargetsRequest)
                .withMarshaller(new PutTargetsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Revokes the permission of another AWS account to be able to put events to your default event bus. Specify the
     * account to revoke by the <code>StatementId</code> value that you associated with the account when you granted it
     * permission with <code>PutPermission</code>. You can find the <code>StatementId</code> by using
     * <a>DescribeEventBus</a>.
     * </p>
     *
     * @param removePermissionRequest
     * @return Result of the RemovePermission operation returned by the service.
     * @throws ResourceNotFoundException
     *         An entity that you specified does not exist.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws ConcurrentModificationException
     *         There is concurrent modification on a rule or target.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.RemovePermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/RemovePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RemovePermissionResponse removePermission(RemovePermissionRequest removePermissionRequest)
            throws ResourceNotFoundException, InternalException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CloudWatchEventsException {

        HttpResponseHandler<RemovePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new RemovePermissionResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<RemovePermissionRequest, RemovePermissionResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(removePermissionRequest).withMarshaller(new RemovePermissionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes the specified targets from the specified rule. When the rule is triggered, those targets are no longer be
     * invoked.
     * </p>
     * <p>
     * When you remove a target, when the associated rule triggers, removed targets might continue to be invoked. Please
     * allow a short period of time for changes to take effect.
     * </p>
     * <p>
     * This action can partially fail if too many requests are made at the same time. If that happens,
     * <code>FailedEntryCount</code> is non-zero in the response and each entry in <code>FailedEntries</code> provides
     * the ID of the failed target and the error code.
     * </p>
     *
     * @param removeTargetsRequest
     * @return Result of the RemoveTargets operation returned by the service.
     * @throws ResourceNotFoundException
     *         An entity that you specified does not exist.
     * @throws ConcurrentModificationException
     *         There is concurrent modification on a rule or target.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.RemoveTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/RemoveTargets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RemoveTargetsResponse removeTargets(RemoveTargetsRequest removeTargetsRequest) throws ResourceNotFoundException,
            ConcurrentModificationException, InternalException, AwsServiceException, SdkClientException,
            CloudWatchEventsException {

        HttpResponseHandler<RemoveTargetsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new RemoveTargetsResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<RemoveTargetsRequest, RemoveTargetsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(removeTargetsRequest).withMarshaller(new RemoveTargetsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Tests whether the specified event pattern matches the provided event.
     * </p>
     * <p>
     * Most services in AWS treat : or / as the same character in Amazon Resource Names (ARNs). However, CloudWatch
     * Events uses an exact match in event patterns and rules. Be sure to use the correct ARN characters when creating
     * event patterns so that they match the ARN syntax in the event you want to match.
     * </p>
     *
     * @param testEventPatternRequest
     * @return Result of the TestEventPattern operation returned by the service.
     * @throws InvalidEventPatternException
     *         The event pattern is not valid.
     * @throws InternalException
     *         This exception occurs due to unexpected causes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchEventsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchEventsClient.TestEventPattern
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/TestEventPattern" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TestEventPatternResponse testEventPattern(TestEventPatternRequest testEventPatternRequest)
            throws InvalidEventPatternException, InternalException, AwsServiceException, SdkClientException,
            CloudWatchEventsException {

        HttpResponseHandler<TestEventPatternResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new TestEventPatternResponseUnmarshaller());

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<TestEventPatternRequest, TestEventPatternResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(testEventPatternRequest).withMarshaller(new TestEventPatternRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler() {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }

    private software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory init() {
        return new AwsJsonProtocolFactory(
                new JsonClientMetadata()
                        .withSupportsCbor(false)
                        .withSupportsIon(false)
                        .withBaseServiceExceptionClass(
                                software.amazon.awssdk.services.cloudwatchevents.model.CloudWatchEventsException.class)
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("ConcurrentModificationException").withModeledClass(
                                        ConcurrentModificationException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("ResourceNotFoundException").withModeledClass(
                                        ResourceNotFoundException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("InvalidEventPatternException").withModeledClass(
                                        InvalidEventPatternException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("InternalException").withModeledClass(
                                        InternalException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("PolicyLengthExceededException").withModeledClass(
                                        PolicyLengthExceededException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("LimitExceededException").withModeledClass(
                                        LimitExceededException.class)), AwsJsonProtocolMetadata.builder().protocolVersion("1.1")
                        .protocol(AwsJsonProtocol.AWS_JSON).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
