/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.evidently.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.JsonValueTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that contains the information about one evaluation event or custom event sent to Evidently. This is a
 * JSON payload. If this event specifies a pre-defined event type, the payload must follow the defined event schema.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Event implements SdkPojo, Serializable, ToCopyableBuilder<Event.Builder, Event> {
    private static final SdkField<String> DATA_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("data")
            .getter(getter(Event::data))
            .setter(setter(Builder::data))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("data").build(),
                    JsonValueTrait.create()).build();

    private static final SdkField<Instant> TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("timestamp").getter(getter(Event::timestamp)).setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timestamp").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(Event::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DATA_FIELD, TIMESTAMP_FIELD,
            TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String data;

    private final Instant timestamp;

    private final String type;

    private Event(BuilderImpl builder) {
        this.data = builder.data;
        this.timestamp = builder.timestamp;
        this.type = builder.type;
    }

    /**
     * <p>
     * The event data.
     * </p>
     * 
     * @return The event data.
     */
    public final String data() {
        return data;
    }

    /**
     * <p>
     * The timestamp of the event.
     * </p>
     * 
     * @return The timestamp of the event.
     */
    public final Instant timestamp() {
        return timestamp;
    }

    /**
     * <p>
     * <code>aws.evidently.evaluation</code> specifies an evaluation event, which determines which feature variation
     * that a user sees. <code>aws.evidently.custom</code> specifies a custom event, which generates metrics from user
     * actions such as clicks and checkouts.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link EventType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return <code>aws.evidently.evaluation</code> specifies an evaluation event, which determines which feature
     *         variation that a user sees. <code>aws.evidently.custom</code> specifies a custom event, which generates
     *         metrics from user actions such as clicks and checkouts.
     * @see EventType
     */
    public final EventType type() {
        return EventType.fromValue(type);
    }

    /**
     * <p>
     * <code>aws.evidently.evaluation</code> specifies an evaluation event, which determines which feature variation
     * that a user sees. <code>aws.evidently.custom</code> specifies a custom event, which generates metrics from user
     * actions such as clicks and checkouts.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link EventType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return <code>aws.evidently.evaluation</code> specifies an evaluation event, which determines which feature
     *         variation that a user sees. <code>aws.evidently.custom</code> specifies a custom event, which generates
     *         metrics from user actions such as clicks and checkouts.
     * @see EventType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(data());
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Event)) {
            return false;
        }
        Event other = (Event) obj;
        return Objects.equals(data(), other.data()) && Objects.equals(timestamp(), other.timestamp())
                && Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Event").add("Data", data()).add("Timestamp", timestamp()).add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "data":
            return Optional.ofNullable(clazz.cast(data()));
        case "timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Event, T> g) {
        return obj -> g.apply((Event) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Event> {
        /**
         * <p>
         * The event data.
         * </p>
         * 
         * @param data
         *        The event data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(String data);

        /**
         * <p>
         * The timestamp of the event.
         * </p>
         * 
         * @param timestamp
         *        The timestamp of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);

        /**
         * <p>
         * <code>aws.evidently.evaluation</code> specifies an evaluation event, which determines which feature variation
         * that a user sees. <code>aws.evidently.custom</code> specifies a custom event, which generates metrics from
         * user actions such as clicks and checkouts.
         * </p>
         * 
         * @param type
         *        <code>aws.evidently.evaluation</code> specifies an evaluation event, which determines which feature
         *        variation that a user sees. <code>aws.evidently.custom</code> specifies a custom event, which
         *        generates metrics from user actions such as clicks and checkouts.
         * @see EventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventType
         */
        Builder type(String type);

        /**
         * <p>
         * <code>aws.evidently.evaluation</code> specifies an evaluation event, which determines which feature variation
         * that a user sees. <code>aws.evidently.custom</code> specifies a custom event, which generates metrics from
         * user actions such as clicks and checkouts.
         * </p>
         * 
         * @param type
         *        <code>aws.evidently.evaluation</code> specifies an evaluation event, which determines which feature
         *        variation that a user sees. <code>aws.evidently.custom</code> specifies a custom event, which
         *        generates metrics from user actions such as clicks and checkouts.
         * @see EventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventType
         */
        Builder type(EventType type);
    }

    static final class BuilderImpl implements Builder {
        private String data;

        private Instant timestamp;

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(Event model) {
            data(model.data);
            timestamp(model.timestamp);
            type(model.type);
        }

        public final String getData() {
            return data;
        }

        public final void setData(String data) {
            this.data = data;
        }

        @Override
        @Transient
        public final Builder data(String data) {
            this.data = data;
            return this;
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        @Transient
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        @Transient
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        @Transient
        public final Builder type(EventType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public Event build() {
            return new Event(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
