/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.evidently.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that defines one treatment in an experiment. A treatment is a variation of the feature that you are
 * including in the experiment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Treatment implements SdkPojo, Serializable, ToCopyableBuilder<Treatment.Builder, Treatment> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(Treatment::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<Map<String, String>> FEATURE_VARIATIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("featureVariations")
            .getter(getter(Treatment::featureVariations))
            .setter(setter(Builder::featureVariations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("featureVariations").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Treatment::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            FEATURE_VARIATIONS_FIELD, NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String description;

    private final Map<String, String> featureVariations;

    private final String name;

    private Treatment(BuilderImpl builder) {
        this.description = builder.description;
        this.featureVariations = builder.featureVariations;
        this.name = builder.name;
    }

    /**
     * <p>
     * The description of the treatment.
     * </p>
     * 
     * @return The description of the treatment.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the FeatureVariations property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFeatureVariations() {
        return featureVariations != null && !(featureVariations instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The feature variation used for this treatment. This is a key-value pair. The key is the feature name, and the
     * value is the variation name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeatureVariations} method.
     * </p>
     * 
     * @return The feature variation used for this treatment. This is a key-value pair. The key is the feature name, and
     *         the value is the variation name.
     */
    public final Map<String, String> featureVariations() {
        return featureVariations;
    }

    /**
     * <p>
     * The name of this treatment.
     * </p>
     * 
     * @return The name of this treatment.
     */
    public final String name() {
        return name;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasFeatureVariations() ? featureVariations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Treatment)) {
            return false;
        }
        Treatment other = (Treatment) obj;
        return Objects.equals(description(), other.description()) && hasFeatureVariations() == other.hasFeatureVariations()
                && Objects.equals(featureVariations(), other.featureVariations()) && Objects.equals(name(), other.name());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Treatment").add("Description", description())
                .add("FeatureVariations", hasFeatureVariations() ? featureVariations() : null).add("Name", name()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "featureVariations":
            return Optional.ofNullable(clazz.cast(featureVariations()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Treatment, T> g) {
        return obj -> g.apply((Treatment) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Treatment> {
        /**
         * <p>
         * The description of the treatment.
         * </p>
         * 
         * @param description
         *        The description of the treatment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The feature variation used for this treatment. This is a key-value pair. The key is the feature name, and the
         * value is the variation name.
         * </p>
         * 
         * @param featureVariations
         *        The feature variation used for this treatment. This is a key-value pair. The key is the feature name,
         *        and the value is the variation name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featureVariations(Map<String, String> featureVariations);

        /**
         * <p>
         * The name of this treatment.
         * </p>
         * 
         * @param name
         *        The name of this treatment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);
    }

    static final class BuilderImpl implements Builder {
        private String description;

        private Map<String, String> featureVariations = DefaultSdkAutoConstructMap.getInstance();

        private String name;

        private BuilderImpl() {
        }

        private BuilderImpl(Treatment model) {
            description(model.description);
            featureVariations(model.featureVariations);
            name(model.name);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        @Transient
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Map<String, String> getFeatureVariations() {
            if (featureVariations instanceof SdkAutoConstructMap) {
                return null;
            }
            return featureVariations;
        }

        public final void setFeatureVariations(Map<String, String> featureVariations) {
            this.featureVariations = FeatureToVariationMapCopier.copy(featureVariations);
        }

        @Override
        @Transient
        public final Builder featureVariations(Map<String, String> featureVariations) {
            this.featureVariations = FeatureToVariationMapCopier.copy(featureVariations);
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public Treatment build() {
            return new Treatment(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
