/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.evidently.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure contains information about one Evidently feature in your account.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Feature implements SdkPojo, Serializable, ToCopyableBuilder<Feature.Builder, Feature> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Feature::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdTime").getter(getter(Feature::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdTime").build()).build();

    private static final SdkField<String> DEFAULT_VARIATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("defaultVariation").getter(getter(Feature::defaultVariation)).setter(setter(Builder::defaultVariation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("defaultVariation").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(Feature::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<Map<String, String>> ENTITY_OVERRIDES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("entityOverrides")
            .getter(getter(Feature::entityOverrides))
            .setter(setter(Builder::entityOverrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("entityOverrides").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<EvaluationRule>> EVALUATION_RULES_FIELD = SdkField
            .<List<EvaluationRule>> builder(MarshallingType.LIST)
            .memberName("evaluationRules")
            .getter(getter(Feature::evaluationRules))
            .setter(setter(Builder::evaluationRules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("evaluationRules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EvaluationRule> builder(MarshallingType.SDK_POJO)
                                            .constructor(EvaluationRule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> EVALUATION_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("evaluationStrategy").getter(getter(Feature::evaluationStrategyAsString))
            .setter(setter(Builder::evaluationStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("evaluationStrategy").build())
            .build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedTime").getter(getter(Feature::lastUpdatedTime)).setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedTime").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Feature::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> PROJECT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("project")
            .getter(getter(Feature::project)).setter(setter(Builder::project))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("project").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(Feature::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Feature::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> VALUE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("valueType").getter(getter(Feature::valueTypeAsString)).setter(setter(Builder::valueType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("valueType").build()).build();

    private static final SdkField<List<Variation>> VARIATIONS_FIELD = SdkField
            .<List<Variation>> builder(MarshallingType.LIST)
            .memberName("variations")
            .getter(getter(Feature::variations))
            .setter(setter(Builder::variations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("variations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Variation> builder(MarshallingType.SDK_POJO)
                                            .constructor(Variation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, CREATED_TIME_FIELD,
            DEFAULT_VARIATION_FIELD, DESCRIPTION_FIELD, ENTITY_OVERRIDES_FIELD, EVALUATION_RULES_FIELD,
            EVALUATION_STRATEGY_FIELD, LAST_UPDATED_TIME_FIELD, NAME_FIELD, PROJECT_FIELD, STATUS_FIELD, TAGS_FIELD,
            VALUE_TYPE_FIELD, VARIATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final Instant createdTime;

    private final String defaultVariation;

    private final String description;

    private final Map<String, String> entityOverrides;

    private final List<EvaluationRule> evaluationRules;

    private final String evaluationStrategy;

    private final Instant lastUpdatedTime;

    private final String name;

    private final String project;

    private final String status;

    private final Map<String, String> tags;

    private final String valueType;

    private final List<Variation> variations;

    private Feature(BuilderImpl builder) {
        this.arn = builder.arn;
        this.createdTime = builder.createdTime;
        this.defaultVariation = builder.defaultVariation;
        this.description = builder.description;
        this.entityOverrides = builder.entityOverrides;
        this.evaluationRules = builder.evaluationRules;
        this.evaluationStrategy = builder.evaluationStrategy;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.name = builder.name;
        this.project = builder.project;
        this.status = builder.status;
        this.tags = builder.tags;
        this.valueType = builder.valueType;
        this.variations = builder.variations;
    }

    /**
     * <p>
     * The ARN of the feature.
     * </p>
     * 
     * @return The ARN of the feature.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The date and time that the feature is created.
     * </p>
     * 
     * @return The date and time that the feature is created.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The name of the variation that is used as the default variation. The default variation is served to users who are
     * not allocated to any ongoing launches or experiments of this feature.
     * </p>
     * <p>
     * This variation must also be listed in the <code>variations</code> structure.
     * </p>
     * <p>
     * If you omit <code>defaultVariation</code>, the first variation listed in the <code>variations</code> structure is
     * used as the default variation.
     * </p>
     * 
     * @return The name of the variation that is used as the default variation. The default variation is served to users
     *         who are not allocated to any ongoing launches or experiments of this feature.</p>
     *         <p>
     *         This variation must also be listed in the <code>variations</code> structure.
     *         </p>
     *         <p>
     *         If you omit <code>defaultVariation</code>, the first variation listed in the <code>variations</code>
     *         structure is used as the default variation.
     */
    public final String defaultVariation() {
        return defaultVariation;
    }

    /**
     * <p>
     * The description of the feature.
     * </p>
     * 
     * @return The description of the feature.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the EntityOverrides property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEntityOverrides() {
        return entityOverrides != null && !(entityOverrides instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A set of key-value pairs that specify users who should always be served a specific variation of a feature. Each
     * key specifies a user using their user ID, account ID, or some other identifier. The value specifies the name of
     * the variation that the user is to be served.
     * </p>
     * <p>
     * For the override to be successful, the value of the key must match the <code>entityId</code> used in the <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_EvaluateFeature.html"
     * >EvaluateFeature</a> operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntityOverrides} method.
     * </p>
     * 
     * @return A set of key-value pairs that specify users who should always be served a specific variation of a
     *         feature. Each key specifies a user using their user ID, account ID, or some other identifier. The value
     *         specifies the name of the variation that the user is to be served.</p>
     *         <p>
     *         For the override to be successful, the value of the key must match the <code>entityId</code> used in the
     *         <a href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_EvaluateFeature.html">
     *         EvaluateFeature</a> operation.
     */
    public final Map<String, String> entityOverrides() {
        return entityOverrides;
    }

    /**
     * For responses, this returns true if the service returned a value for the EvaluationRules property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEvaluationRules() {
        return evaluationRules != null && !(evaluationRules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of structures that define the evaluation rules for the feature.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEvaluationRules} method.
     * </p>
     * 
     * @return An array of structures that define the evaluation rules for the feature.
     */
    public final List<EvaluationRule> evaluationRules() {
        return evaluationRules;
    }

    /**
     * <p>
     * If this value is <code>ALL_RULES</code>, the traffic allocation specified by any ongoing launches or experiments
     * is being used. If this is <code>DEFAULT_VARIATION</code>, the default variation is being served to all users.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #evaluationStrategy} will return {@link FeatureEvaluationStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #evaluationStrategyAsString}.
     * </p>
     * 
     * @return If this value is <code>ALL_RULES</code>, the traffic allocation specified by any ongoing launches or
     *         experiments is being used. If this is <code>DEFAULT_VARIATION</code>, the default variation is being
     *         served to all users.
     * @see FeatureEvaluationStrategy
     */
    public final FeatureEvaluationStrategy evaluationStrategy() {
        return FeatureEvaluationStrategy.fromValue(evaluationStrategy);
    }

    /**
     * <p>
     * If this value is <code>ALL_RULES</code>, the traffic allocation specified by any ongoing launches or experiments
     * is being used. If this is <code>DEFAULT_VARIATION</code>, the default variation is being served to all users.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #evaluationStrategy} will return {@link FeatureEvaluationStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #evaluationStrategyAsString}.
     * </p>
     * 
     * @return If this value is <code>ALL_RULES</code>, the traffic allocation specified by any ongoing launches or
     *         experiments is being used. If this is <code>DEFAULT_VARIATION</code>, the default variation is being
     *         served to all users.
     * @see FeatureEvaluationStrategy
     */
    public final String evaluationStrategyAsString() {
        return evaluationStrategy;
    }

    /**
     * <p>
     * The date and time that the feature was most recently updated.
     * </p>
     * 
     * @return The date and time that the feature was most recently updated.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * The name of the feature.
     * </p>
     * 
     * @return The name of the feature.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The name or ARN of the project that contains the feature.
     * </p>
     * 
     * @return The name or ARN of the project that contains the feature.
     */
    public final String project() {
        return project;
    }

    /**
     * <p>
     * The current state of the feature.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link FeatureStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current state of the feature.
     * @see FeatureStatus
     */
    public final FeatureStatus status() {
        return FeatureStatus.fromValue(status);
    }

    /**
     * <p>
     * The current state of the feature.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link FeatureStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current state of the feature.
     * @see FeatureStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The list of tag keys and values associated with this feature.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The list of tag keys and values associated with this feature.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * Defines the type of value used to define the different feature variations. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Evidently-variationtypes.html"
     * >Variation types</a>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #valueType} will
     * return {@link VariationValueType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #valueTypeAsString}.
     * </p>
     * 
     * @return Defines the type of value used to define the different feature variations. For more information, see <a
     *         href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Evidently-variationtypes.html"
     *         >Variation types</a>
     * @see VariationValueType
     */
    public final VariationValueType valueType() {
        return VariationValueType.fromValue(valueType);
    }

    /**
     * <p>
     * Defines the type of value used to define the different feature variations. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Evidently-variationtypes.html"
     * >Variation types</a>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #valueType} will
     * return {@link VariationValueType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #valueTypeAsString}.
     * </p>
     * 
     * @return Defines the type of value used to define the different feature variations. For more information, see <a
     *         href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Evidently-variationtypes.html"
     *         >Variation types</a>
     * @see VariationValueType
     */
    public final String valueTypeAsString() {
        return valueType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Variations property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasVariations() {
        return variations != null && !(variations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of structures that contain the configuration of the feature's different variations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVariations} method.
     * </p>
     * 
     * @return An array of structures that contain the configuration of the feature's different variations.
     */
    public final List<Variation> variations() {
        return variations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(defaultVariation());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasEntityOverrides() ? entityOverrides() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEvaluationRules() ? evaluationRules() : null);
        hashCode = 31 * hashCode + Objects.hashCode(evaluationStrategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(project());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(valueTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasVariations() ? variations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Feature)) {
            return false;
        }
        Feature other = (Feature) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(defaultVariation(), other.defaultVariation())
                && Objects.equals(description(), other.description()) && hasEntityOverrides() == other.hasEntityOverrides()
                && Objects.equals(entityOverrides(), other.entityOverrides())
                && hasEvaluationRules() == other.hasEvaluationRules()
                && Objects.equals(evaluationRules(), other.evaluationRules())
                && Objects.equals(evaluationStrategyAsString(), other.evaluationStrategyAsString())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime()) && Objects.equals(name(), other.name())
                && Objects.equals(project(), other.project()) && Objects.equals(statusAsString(), other.statusAsString())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(valueTypeAsString(), other.valueTypeAsString()) && hasVariations() == other.hasVariations()
                && Objects.equals(variations(), other.variations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Feature").add("Arn", arn()).add("CreatedTime", createdTime())
                .add("DefaultVariation", defaultVariation()).add("Description", description())
                .add("EntityOverrides", hasEntityOverrides() ? entityOverrides() : null)
                .add("EvaluationRules", hasEvaluationRules() ? evaluationRules() : null)
                .add("EvaluationStrategy", evaluationStrategyAsString()).add("LastUpdatedTime", lastUpdatedTime())
                .add("Name", name()).add("Project", project()).add("Status", statusAsString())
                .add("Tags", hasTags() ? tags() : null).add("ValueType", valueTypeAsString())
                .add("Variations", hasVariations() ? variations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "createdTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "defaultVariation":
            return Optional.ofNullable(clazz.cast(defaultVariation()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "entityOverrides":
            return Optional.ofNullable(clazz.cast(entityOverrides()));
        case "evaluationRules":
            return Optional.ofNullable(clazz.cast(evaluationRules()));
        case "evaluationStrategy":
            return Optional.ofNullable(clazz.cast(evaluationStrategyAsString()));
        case "lastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "project":
            return Optional.ofNullable(clazz.cast(project()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "valueType":
            return Optional.ofNullable(clazz.cast(valueTypeAsString()));
        case "variations":
            return Optional.ofNullable(clazz.cast(variations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Feature, T> g) {
        return obj -> g.apply((Feature) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Feature> {
        /**
         * <p>
         * The ARN of the feature.
         * </p>
         * 
         * @param arn
         *        The ARN of the feature.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The date and time that the feature is created.
         * </p>
         * 
         * @param createdTime
         *        The date and time that the feature is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The name of the variation that is used as the default variation. The default variation is served to users who
         * are not allocated to any ongoing launches or experiments of this feature.
         * </p>
         * <p>
         * This variation must also be listed in the <code>variations</code> structure.
         * </p>
         * <p>
         * If you omit <code>defaultVariation</code>, the first variation listed in the <code>variations</code>
         * structure is used as the default variation.
         * </p>
         * 
         * @param defaultVariation
         *        The name of the variation that is used as the default variation. The default variation is served to
         *        users who are not allocated to any ongoing launches or experiments of this feature.</p>
         *        <p>
         *        This variation must also be listed in the <code>variations</code> structure.
         *        </p>
         *        <p>
         *        If you omit <code>defaultVariation</code>, the first variation listed in the <code>variations</code>
         *        structure is used as the default variation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultVariation(String defaultVariation);

        /**
         * <p>
         * The description of the feature.
         * </p>
         * 
         * @param description
         *        The description of the feature.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A set of key-value pairs that specify users who should always be served a specific variation of a feature.
         * Each key specifies a user using their user ID, account ID, or some other identifier. The value specifies the
         * name of the variation that the user is to be served.
         * </p>
         * <p>
         * For the override to be successful, the value of the key must match the <code>entityId</code> used in the <a
         * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_EvaluateFeature.html">
         * EvaluateFeature</a> operation.
         * </p>
         * 
         * @param entityOverrides
         *        A set of key-value pairs that specify users who should always be served a specific variation of a
         *        feature. Each key specifies a user using their user ID, account ID, or some other identifier. The
         *        value specifies the name of the variation that the user is to be served.</p>
         *        <p>
         *        For the override to be successful, the value of the key must match the <code>entityId</code> used in
         *        the <a
         *        href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_EvaluateFeature.html"
         *        >EvaluateFeature</a> operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityOverrides(Map<String, String> entityOverrides);

        /**
         * <p>
         * An array of structures that define the evaluation rules for the feature.
         * </p>
         * 
         * @param evaluationRules
         *        An array of structures that define the evaluation rules for the feature.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationRules(Collection<EvaluationRule> evaluationRules);

        /**
         * <p>
         * An array of structures that define the evaluation rules for the feature.
         * </p>
         * 
         * @param evaluationRules
         *        An array of structures that define the evaluation rules for the feature.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationRules(EvaluationRule... evaluationRules);

        /**
         * <p>
         * An array of structures that define the evaluation rules for the feature.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<EvaluationRule>.Builder} avoiding
         * the need to create one manually via {@link List<EvaluationRule>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<EvaluationRule>.Builder#build()} is called immediately and
         * its result is passed to {@link #evaluationRules(List<EvaluationRule>)}.
         * 
         * @param evaluationRules
         *        a consumer that will call methods on {@link List<EvaluationRule>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #evaluationRules(List<EvaluationRule>)
         */
        Builder evaluationRules(Consumer<EvaluationRule.Builder>... evaluationRules);

        /**
         * <p>
         * If this value is <code>ALL_RULES</code>, the traffic allocation specified by any ongoing launches or
         * experiments is being used. If this is <code>DEFAULT_VARIATION</code>, the default variation is being served
         * to all users.
         * </p>
         * 
         * @param evaluationStrategy
         *        If this value is <code>ALL_RULES</code>, the traffic allocation specified by any ongoing launches or
         *        experiments is being used. If this is <code>DEFAULT_VARIATION</code>, the default variation is being
         *        served to all users.
         * @see FeatureEvaluationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FeatureEvaluationStrategy
         */
        Builder evaluationStrategy(String evaluationStrategy);

        /**
         * <p>
         * If this value is <code>ALL_RULES</code>, the traffic allocation specified by any ongoing launches or
         * experiments is being used. If this is <code>DEFAULT_VARIATION</code>, the default variation is being served
         * to all users.
         * </p>
         * 
         * @param evaluationStrategy
         *        If this value is <code>ALL_RULES</code>, the traffic allocation specified by any ongoing launches or
         *        experiments is being used. If this is <code>DEFAULT_VARIATION</code>, the default variation is being
         *        served to all users.
         * @see FeatureEvaluationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FeatureEvaluationStrategy
         */
        Builder evaluationStrategy(FeatureEvaluationStrategy evaluationStrategy);

        /**
         * <p>
         * The date and time that the feature was most recently updated.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The date and time that the feature was most recently updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * The name of the feature.
         * </p>
         * 
         * @param name
         *        The name of the feature.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The name or ARN of the project that contains the feature.
         * </p>
         * 
         * @param project
         *        The name or ARN of the project that contains the feature.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder project(String project);

        /**
         * <p>
         * The current state of the feature.
         * </p>
         * 
         * @param status
         *        The current state of the feature.
         * @see FeatureStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FeatureStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current state of the feature.
         * </p>
         * 
         * @param status
         *        The current state of the feature.
         * @see FeatureStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FeatureStatus
         */
        Builder status(FeatureStatus status);

        /**
         * <p>
         * The list of tag keys and values associated with this feature.
         * </p>
         * 
         * @param tags
         *        The list of tag keys and values associated with this feature.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * Defines the type of value used to define the different feature variations. For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Evidently-variationtypes.html"
         * >Variation types</a>
         * </p>
         * 
         * @param valueType
         *        Defines the type of value used to define the different feature variations. For more information, see
         *        <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Evidently-variationtypes.html"
         *        >Variation types</a>
         * @see VariationValueType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VariationValueType
         */
        Builder valueType(String valueType);

        /**
         * <p>
         * Defines the type of value used to define the different feature variations. For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Evidently-variationtypes.html"
         * >Variation types</a>
         * </p>
         * 
         * @param valueType
         *        Defines the type of value used to define the different feature variations. For more information, see
         *        <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Evidently-variationtypes.html"
         *        >Variation types</a>
         * @see VariationValueType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VariationValueType
         */
        Builder valueType(VariationValueType valueType);

        /**
         * <p>
         * An array of structures that contain the configuration of the feature's different variations.
         * </p>
         * 
         * @param variations
         *        An array of structures that contain the configuration of the feature's different variations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variations(Collection<Variation> variations);

        /**
         * <p>
         * An array of structures that contain the configuration of the feature's different variations.
         * </p>
         * 
         * @param variations
         *        An array of structures that contain the configuration of the feature's different variations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variations(Variation... variations);

        /**
         * <p>
         * An array of structures that contain the configuration of the feature's different variations.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Variation>.Builder} avoiding the
         * need to create one manually via {@link List<Variation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Variation>.Builder#build()} is called immediately and its
         * result is passed to {@link #variations(List<Variation>)}.
         * 
         * @param variations
         *        a consumer that will call methods on {@link List<Variation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #variations(List<Variation>)
         */
        Builder variations(Consumer<Variation.Builder>... variations);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private Instant createdTime;

        private String defaultVariation;

        private String description;

        private Map<String, String> entityOverrides = DefaultSdkAutoConstructMap.getInstance();

        private List<EvaluationRule> evaluationRules = DefaultSdkAutoConstructList.getInstance();

        private String evaluationStrategy;

        private Instant lastUpdatedTime;

        private String name;

        private String project;

        private String status;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String valueType;

        private List<Variation> variations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Feature model) {
            arn(model.arn);
            createdTime(model.createdTime);
            defaultVariation(model.defaultVariation);
            description(model.description);
            entityOverrides(model.entityOverrides);
            evaluationRules(model.evaluationRules);
            evaluationStrategy(model.evaluationStrategy);
            lastUpdatedTime(model.lastUpdatedTime);
            name(model.name);
            project(model.project);
            status(model.status);
            tags(model.tags);
            valueType(model.valueType);
            variations(model.variations);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final String getDefaultVariation() {
            return defaultVariation;
        }

        public final void setDefaultVariation(String defaultVariation) {
            this.defaultVariation = defaultVariation;
        }

        @Override
        public final Builder defaultVariation(String defaultVariation) {
            this.defaultVariation = defaultVariation;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Map<String, String> getEntityOverrides() {
            if (entityOverrides instanceof SdkAutoConstructMap) {
                return null;
            }
            return entityOverrides;
        }

        public final void setEntityOverrides(Map<String, String> entityOverrides) {
            this.entityOverrides = EntityOverrideMapCopier.copy(entityOverrides);
        }

        @Override
        public final Builder entityOverrides(Map<String, String> entityOverrides) {
            this.entityOverrides = EntityOverrideMapCopier.copy(entityOverrides);
            return this;
        }

        public final List<EvaluationRule.Builder> getEvaluationRules() {
            List<EvaluationRule.Builder> result = EvaluationRulesListCopier.copyToBuilder(this.evaluationRules);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEvaluationRules(Collection<EvaluationRule.BuilderImpl> evaluationRules) {
            this.evaluationRules = EvaluationRulesListCopier.copyFromBuilder(evaluationRules);
        }

        @Override
        public final Builder evaluationRules(Collection<EvaluationRule> evaluationRules) {
            this.evaluationRules = EvaluationRulesListCopier.copy(evaluationRules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder evaluationRules(EvaluationRule... evaluationRules) {
            evaluationRules(Arrays.asList(evaluationRules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder evaluationRules(Consumer<EvaluationRule.Builder>... evaluationRules) {
            evaluationRules(Stream.of(evaluationRules).map(c -> EvaluationRule.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getEvaluationStrategy() {
            return evaluationStrategy;
        }

        public final void setEvaluationStrategy(String evaluationStrategy) {
            this.evaluationStrategy = evaluationStrategy;
        }

        @Override
        public final Builder evaluationStrategy(String evaluationStrategy) {
            this.evaluationStrategy = evaluationStrategy;
            return this;
        }

        @Override
        public final Builder evaluationStrategy(FeatureEvaluationStrategy evaluationStrategy) {
            this.evaluationStrategy(evaluationStrategy == null ? null : evaluationStrategy.toString());
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getProject() {
            return project;
        }

        public final void setProject(String project) {
            this.project = project;
        }

        @Override
        public final Builder project(String project) {
            this.project = project;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(FeatureStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final String getValueType() {
            return valueType;
        }

        public final void setValueType(String valueType) {
            this.valueType = valueType;
        }

        @Override
        public final Builder valueType(String valueType) {
            this.valueType = valueType;
            return this;
        }

        @Override
        public final Builder valueType(VariationValueType valueType) {
            this.valueType(valueType == null ? null : valueType.toString());
            return this;
        }

        public final List<Variation.Builder> getVariations() {
            List<Variation.Builder> result = VariationsListCopier.copyToBuilder(this.variations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVariations(Collection<Variation.BuilderImpl> variations) {
            this.variations = VariationsListCopier.copyFromBuilder(variations);
        }

        @Override
        public final Builder variations(Collection<Variation> variations) {
            this.variations = VariationsListCopier.copy(variations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variations(Variation... variations) {
            variations(Arrays.asList(variations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variations(Consumer<Variation.Builder>... variations) {
            variations(Stream.of(variations).map(c -> Variation.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Feature build() {
            return new Feature(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
