/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.evidently.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetExperimentResultsRequest extends EvidentlyRequest implements
        ToCopyableBuilder<GetExperimentResultsRequest.Builder, GetExperimentResultsRequest> {
    private static final SdkField<String> BASE_STAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("baseStat").getter(getter(GetExperimentResultsRequest::baseStatAsString))
            .setter(setter(Builder::baseStat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("baseStat").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime").getter(getter(GetExperimentResultsRequest::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<String> EXPERIMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("experiment").getter(getter(GetExperimentResultsRequest::experiment)).setter(setter(Builder::experiment))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("experiment").build()).build();

    private static final SdkField<List<String>> METRIC_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("metricNames")
            .getter(getter(GetExperimentResultsRequest::metricNames))
            .setter(setter(Builder::metricNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metricNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> PERIOD_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("period")
            .getter(getter(GetExperimentResultsRequest::period)).setter(setter(Builder::period))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("period").build()).build();

    private static final SdkField<String> PROJECT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("project")
            .getter(getter(GetExperimentResultsRequest::project)).setter(setter(Builder::project))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("project").build()).build();

    private static final SdkField<List<String>> REPORT_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("reportNames")
            .getter(getter(GetExperimentResultsRequest::reportNamesAsStrings))
            .setter(setter(Builder::reportNamesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("reportNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> RESULT_STATS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("resultStats")
            .getter(getter(GetExperimentResultsRequest::resultStatsAsStrings))
            .setter(setter(Builder::resultStatsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resultStats").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(GetExperimentResultsRequest::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<List<String>> TREATMENT_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("treatmentNames")
            .getter(getter(GetExperimentResultsRequest::treatmentNames))
            .setter(setter(Builder::treatmentNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("treatmentNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BASE_STAT_FIELD,
            END_TIME_FIELD, EXPERIMENT_FIELD, METRIC_NAMES_FIELD, PERIOD_FIELD, PROJECT_FIELD, REPORT_NAMES_FIELD,
            RESULT_STATS_FIELD, START_TIME_FIELD, TREATMENT_NAMES_FIELD));

    private final String baseStat;

    private final Instant endTime;

    private final String experiment;

    private final List<String> metricNames;

    private final Long period;

    private final String project;

    private final List<String> reportNames;

    private final List<String> resultStats;

    private final Instant startTime;

    private final List<String> treatmentNames;

    private GetExperimentResultsRequest(BuilderImpl builder) {
        super(builder);
        this.baseStat = builder.baseStat;
        this.endTime = builder.endTime;
        this.experiment = builder.experiment;
        this.metricNames = builder.metricNames;
        this.period = builder.period;
        this.project = builder.project;
        this.reportNames = builder.reportNames;
        this.resultStats = builder.resultStats;
        this.startTime = builder.startTime;
        this.treatmentNames = builder.treatmentNames;
    }

    /**
     * <p>
     * The statistic used to calculate experiment results. Currently the only valid value is <code>mean</code>, which
     * uses the mean of the collected values as the statistic.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #baseStat} will
     * return {@link ExperimentBaseStat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #baseStatAsString}.
     * </p>
     * 
     * @return The statistic used to calculate experiment results. Currently the only valid value is <code>mean</code>,
     *         which uses the mean of the collected values as the statistic.
     * @see ExperimentBaseStat
     */
    public final ExperimentBaseStat baseStat() {
        return ExperimentBaseStat.fromValue(baseStat);
    }

    /**
     * <p>
     * The statistic used to calculate experiment results. Currently the only valid value is <code>mean</code>, which
     * uses the mean of the collected values as the statistic.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #baseStat} will
     * return {@link ExperimentBaseStat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #baseStatAsString}.
     * </p>
     * 
     * @return The statistic used to calculate experiment results. Currently the only valid value is <code>mean</code>,
     *         which uses the mean of the collected values as the statistic.
     * @see ExperimentBaseStat
     */
    public final String baseStatAsString() {
        return baseStat;
    }

    /**
     * <p>
     * The date and time that the experiment ended, if it is completed.
     * </p>
     * 
     * @return The date and time that the experiment ended, if it is completed.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The name of the experiment to retrieve the results of.
     * </p>
     * 
     * @return The name of the experiment to retrieve the results of.
     */
    public final String experiment() {
        return experiment;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMetricNames() {
        return metricNames != null && !(metricNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The names of the experiment metrics that you want to see the results of.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricNames} method.
     * </p>
     * 
     * @return The names of the experiment metrics that you want to see the results of.
     */
    public final List<String> metricNames() {
        return metricNames;
    }

    /**
     * <p>
     * In seconds, the amount of time to aggregate results together.
     * </p>
     * 
     * @return In seconds, the amount of time to aggregate results together.
     */
    public final Long period() {
        return period;
    }

    /**
     * <p>
     * The name or ARN of the project that contains the experiment that you want to see the results of.
     * </p>
     * 
     * @return The name or ARN of the project that contains the experiment that you want to see the results of.
     */
    public final String project() {
        return project;
    }

    /**
     * <p>
     * The names of the report types that you want to see. Currently, <code>BayesianInference</code> is the only valid
     * value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReportNames} method.
     * </p>
     * 
     * @return The names of the report types that you want to see. Currently, <code>BayesianInference</code> is the only
     *         valid value.
     */
    public final List<ExperimentReportName> reportNames() {
        return ExperimentReportNameListCopier.copyStringToEnum(reportNames);
    }

    /**
     * For responses, this returns true if the service returned a value for the ReportNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasReportNames() {
        return reportNames != null && !(reportNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The names of the report types that you want to see. Currently, <code>BayesianInference</code> is the only valid
     * value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReportNames} method.
     * </p>
     * 
     * @return The names of the report types that you want to see. Currently, <code>BayesianInference</code> is the only
     *         valid value.
     */
    public final List<String> reportNamesAsStrings() {
        return reportNames;
    }

    /**
     * <p>
     * The statistics that you want to see in the returned results.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PValue</code> specifies to use p-values for the results. A p-value is used in hypothesis testing to measure
     * how often you are willing to make a mistake in rejecting the null hypothesis. A general practice is to reject the
     * null hypothesis and declare that the results are statistically significant when the p-value is less than 0.05.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConfidenceInterval</code> specifies a confidence interval for the results. The confidence interval
     * represents the range of values for the chosen metric that is likely to contain the true difference between the
     * <code>baseStat</code> of a variation and the baseline. Evidently returns the 95% confidence interval.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TreatmentEffect</code> is the difference in the statistic specified by the <code>baseStat</code> parameter
     * between each variation and the default variation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BaseStat</code> returns the statistical values collected for the metric for each variation. The statistic
     * uses the same statistic specified in the <code>baseStat</code> parameter. Therefore, if <code>baseStat</code> is
     * <code>mean</code>, this returns the mean of the values collected for each variation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResultStats} method.
     * </p>
     * 
     * @return The statistics that you want to see in the returned results.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PValue</code> specifies to use p-values for the results. A p-value is used in hypothesis testing to
     *         measure how often you are willing to make a mistake in rejecting the null hypothesis. A general practice
     *         is to reject the null hypothesis and declare that the results are statistically significant when the
     *         p-value is less than 0.05.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ConfidenceInterval</code> specifies a confidence interval for the results. The confidence interval
     *         represents the range of values for the chosen metric that is likely to contain the true difference
     *         between the <code>baseStat</code> of a variation and the baseline. Evidently returns the 95% confidence
     *         interval.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TreatmentEffect</code> is the difference in the statistic specified by the <code>baseStat</code>
     *         parameter between each variation and the default variation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BaseStat</code> returns the statistical values collected for the metric for each variation. The
     *         statistic uses the same statistic specified in the <code>baseStat</code> parameter. Therefore, if
     *         <code>baseStat</code> is <code>mean</code>, this returns the mean of the values collected for each
     *         variation.
     *         </p>
     *         </li>
     */
    public final List<ExperimentResultRequestType> resultStats() {
        return ExperimentResultRequestTypeListCopier.copyStringToEnum(resultStats);
    }

    /**
     * For responses, this returns true if the service returned a value for the ResultStats property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResultStats() {
        return resultStats != null && !(resultStats instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The statistics that you want to see in the returned results.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PValue</code> specifies to use p-values for the results. A p-value is used in hypothesis testing to measure
     * how often you are willing to make a mistake in rejecting the null hypothesis. A general practice is to reject the
     * null hypothesis and declare that the results are statistically significant when the p-value is less than 0.05.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConfidenceInterval</code> specifies a confidence interval for the results. The confidence interval
     * represents the range of values for the chosen metric that is likely to contain the true difference between the
     * <code>baseStat</code> of a variation and the baseline. Evidently returns the 95% confidence interval.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TreatmentEffect</code> is the difference in the statistic specified by the <code>baseStat</code> parameter
     * between each variation and the default variation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BaseStat</code> returns the statistical values collected for the metric for each variation. The statistic
     * uses the same statistic specified in the <code>baseStat</code> parameter. Therefore, if <code>baseStat</code> is
     * <code>mean</code>, this returns the mean of the values collected for each variation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResultStats} method.
     * </p>
     * 
     * @return The statistics that you want to see in the returned results.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PValue</code> specifies to use p-values for the results. A p-value is used in hypothesis testing to
     *         measure how often you are willing to make a mistake in rejecting the null hypothesis. A general practice
     *         is to reject the null hypothesis and declare that the results are statistically significant when the
     *         p-value is less than 0.05.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ConfidenceInterval</code> specifies a confidence interval for the results. The confidence interval
     *         represents the range of values for the chosen metric that is likely to contain the true difference
     *         between the <code>baseStat</code> of a variation and the baseline. Evidently returns the 95% confidence
     *         interval.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TreatmentEffect</code> is the difference in the statistic specified by the <code>baseStat</code>
     *         parameter between each variation and the default variation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BaseStat</code> returns the statistical values collected for the metric for each variation. The
     *         statistic uses the same statistic specified in the <code>baseStat</code> parameter. Therefore, if
     *         <code>baseStat</code> is <code>mean</code>, this returns the mean of the values collected for each
     *         variation.
     *         </p>
     *         </li>
     */
    public final List<String> resultStatsAsStrings() {
        return resultStats;
    }

    /**
     * <p>
     * The date and time that the experiment started.
     * </p>
     * 
     * @return The date and time that the experiment started.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the TreatmentNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTreatmentNames() {
        return treatmentNames != null && !(treatmentNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The names of the experiment treatments that you want to see the results for.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTreatmentNames} method.
     * </p>
     * 
     * @return The names of the experiment treatments that you want to see the results for.
     */
    public final List<String> treatmentNames() {
        return treatmentNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(baseStatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(experiment());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricNames() ? metricNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(period());
        hashCode = 31 * hashCode + Objects.hashCode(project());
        hashCode = 31 * hashCode + Objects.hashCode(hasReportNames() ? reportNamesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResultStats() ? resultStatsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasTreatmentNames() ? treatmentNames() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetExperimentResultsRequest)) {
            return false;
        }
        GetExperimentResultsRequest other = (GetExperimentResultsRequest) obj;
        return Objects.equals(baseStatAsString(), other.baseStatAsString()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(experiment(), other.experiment()) && hasMetricNames() == other.hasMetricNames()
                && Objects.equals(metricNames(), other.metricNames()) && Objects.equals(period(), other.period())
                && Objects.equals(project(), other.project()) && hasReportNames() == other.hasReportNames()
                && Objects.equals(reportNamesAsStrings(), other.reportNamesAsStrings())
                && hasResultStats() == other.hasResultStats()
                && Objects.equals(resultStatsAsStrings(), other.resultStatsAsStrings())
                && Objects.equals(startTime(), other.startTime()) && hasTreatmentNames() == other.hasTreatmentNames()
                && Objects.equals(treatmentNames(), other.treatmentNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetExperimentResultsRequest").add("BaseStat", baseStatAsString()).add("EndTime", endTime())
                .add("Experiment", experiment()).add("MetricNames", hasMetricNames() ? metricNames() : null)
                .add("Period", period()).add("Project", project())
                .add("ReportNames", hasReportNames() ? reportNamesAsStrings() : null)
                .add("ResultStats", hasResultStats() ? resultStatsAsStrings() : null).add("StartTime", startTime())
                .add("TreatmentNames", hasTreatmentNames() ? treatmentNames() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "baseStat":
            return Optional.ofNullable(clazz.cast(baseStatAsString()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "experiment":
            return Optional.ofNullable(clazz.cast(experiment()));
        case "metricNames":
            return Optional.ofNullable(clazz.cast(metricNames()));
        case "period":
            return Optional.ofNullable(clazz.cast(period()));
        case "project":
            return Optional.ofNullable(clazz.cast(project()));
        case "reportNames":
            return Optional.ofNullable(clazz.cast(reportNamesAsStrings()));
        case "resultStats":
            return Optional.ofNullable(clazz.cast(resultStatsAsStrings()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "treatmentNames":
            return Optional.ofNullable(clazz.cast(treatmentNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetExperimentResultsRequest, T> g) {
        return obj -> g.apply((GetExperimentResultsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends EvidentlyRequest.Builder, SdkPojo, CopyableBuilder<Builder, GetExperimentResultsRequest> {
        /**
         * <p>
         * The statistic used to calculate experiment results. Currently the only valid value is <code>mean</code>,
         * which uses the mean of the collected values as the statistic.
         * </p>
         * 
         * @param baseStat
         *        The statistic used to calculate experiment results. Currently the only valid value is
         *        <code>mean</code>, which uses the mean of the collected values as the statistic.
         * @see ExperimentBaseStat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExperimentBaseStat
         */
        Builder baseStat(String baseStat);

        /**
         * <p>
         * The statistic used to calculate experiment results. Currently the only valid value is <code>mean</code>,
         * which uses the mean of the collected values as the statistic.
         * </p>
         * 
         * @param baseStat
         *        The statistic used to calculate experiment results. Currently the only valid value is
         *        <code>mean</code>, which uses the mean of the collected values as the statistic.
         * @see ExperimentBaseStat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExperimentBaseStat
         */
        Builder baseStat(ExperimentBaseStat baseStat);

        /**
         * <p>
         * The date and time that the experiment ended, if it is completed.
         * </p>
         * 
         * @param endTime
         *        The date and time that the experiment ended, if it is completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The name of the experiment to retrieve the results of.
         * </p>
         * 
         * @param experiment
         *        The name of the experiment to retrieve the results of.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder experiment(String experiment);

        /**
         * <p>
         * The names of the experiment metrics that you want to see the results of.
         * </p>
         * 
         * @param metricNames
         *        The names of the experiment metrics that you want to see the results of.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricNames(Collection<String> metricNames);

        /**
         * <p>
         * The names of the experiment metrics that you want to see the results of.
         * </p>
         * 
         * @param metricNames
         *        The names of the experiment metrics that you want to see the results of.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricNames(String... metricNames);

        /**
         * <p>
         * In seconds, the amount of time to aggregate results together.
         * </p>
         * 
         * @param period
         *        In seconds, the amount of time to aggregate results together.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder period(Long period);

        /**
         * <p>
         * The name or ARN of the project that contains the experiment that you want to see the results of.
         * </p>
         * 
         * @param project
         *        The name or ARN of the project that contains the experiment that you want to see the results of.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder project(String project);

        /**
         * <p>
         * The names of the report types that you want to see. Currently, <code>BayesianInference</code> is the only
         * valid value.
         * </p>
         * 
         * @param reportNames
         *        The names of the report types that you want to see. Currently, <code>BayesianInference</code> is the
         *        only valid value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reportNamesWithStrings(Collection<String> reportNames);

        /**
         * <p>
         * The names of the report types that you want to see. Currently, <code>BayesianInference</code> is the only
         * valid value.
         * </p>
         * 
         * @param reportNames
         *        The names of the report types that you want to see. Currently, <code>BayesianInference</code> is the
         *        only valid value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reportNamesWithStrings(String... reportNames);

        /**
         * <p>
         * The names of the report types that you want to see. Currently, <code>BayesianInference</code> is the only
         * valid value.
         * </p>
         * 
         * @param reportNames
         *        The names of the report types that you want to see. Currently, <code>BayesianInference</code> is the
         *        only valid value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reportNames(Collection<ExperimentReportName> reportNames);

        /**
         * <p>
         * The names of the report types that you want to see. Currently, <code>BayesianInference</code> is the only
         * valid value.
         * </p>
         * 
         * @param reportNames
         *        The names of the report types that you want to see. Currently, <code>BayesianInference</code> is the
         *        only valid value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reportNames(ExperimentReportName... reportNames);

        /**
         * <p>
         * The statistics that you want to see in the returned results.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PValue</code> specifies to use p-values for the results. A p-value is used in hypothesis testing to
         * measure how often you are willing to make a mistake in rejecting the null hypothesis. A general practice is
         * to reject the null hypothesis and declare that the results are statistically significant when the p-value is
         * less than 0.05.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ConfidenceInterval</code> specifies a confidence interval for the results. The confidence interval
         * represents the range of values for the chosen metric that is likely to contain the true difference between
         * the <code>baseStat</code> of a variation and the baseline. Evidently returns the 95% confidence interval.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TreatmentEffect</code> is the difference in the statistic specified by the <code>baseStat</code>
         * parameter between each variation and the default variation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BaseStat</code> returns the statistical values collected for the metric for each variation. The
         * statistic uses the same statistic specified in the <code>baseStat</code> parameter. Therefore, if
         * <code>baseStat</code> is <code>mean</code>, this returns the mean of the values collected for each variation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resultStats
         *        The statistics that you want to see in the returned results.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PValue</code> specifies to use p-values for the results. A p-value is used in hypothesis testing
         *        to measure how often you are willing to make a mistake in rejecting the null hypothesis. A general
         *        practice is to reject the null hypothesis and declare that the results are statistically significant
         *        when the p-value is less than 0.05.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ConfidenceInterval</code> specifies a confidence interval for the results. The confidence
         *        interval represents the range of values for the chosen metric that is likely to contain the true
         *        difference between the <code>baseStat</code> of a variation and the baseline. Evidently returns the
         *        95% confidence interval.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TreatmentEffect</code> is the difference in the statistic specified by the <code>baseStat</code>
         *        parameter between each variation and the default variation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BaseStat</code> returns the statistical values collected for the metric for each variation. The
         *        statistic uses the same statistic specified in the <code>baseStat</code> parameter. Therefore, if
         *        <code>baseStat</code> is <code>mean</code>, this returns the mean of the values collected for each
         *        variation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resultStatsWithStrings(Collection<String> resultStats);

        /**
         * <p>
         * The statistics that you want to see in the returned results.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PValue</code> specifies to use p-values for the results. A p-value is used in hypothesis testing to
         * measure how often you are willing to make a mistake in rejecting the null hypothesis. A general practice is
         * to reject the null hypothesis and declare that the results are statistically significant when the p-value is
         * less than 0.05.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ConfidenceInterval</code> specifies a confidence interval for the results. The confidence interval
         * represents the range of values for the chosen metric that is likely to contain the true difference between
         * the <code>baseStat</code> of a variation and the baseline. Evidently returns the 95% confidence interval.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TreatmentEffect</code> is the difference in the statistic specified by the <code>baseStat</code>
         * parameter between each variation and the default variation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BaseStat</code> returns the statistical values collected for the metric for each variation. The
         * statistic uses the same statistic specified in the <code>baseStat</code> parameter. Therefore, if
         * <code>baseStat</code> is <code>mean</code>, this returns the mean of the values collected for each variation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resultStats
         *        The statistics that you want to see in the returned results.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PValue</code> specifies to use p-values for the results. A p-value is used in hypothesis testing
         *        to measure how often you are willing to make a mistake in rejecting the null hypothesis. A general
         *        practice is to reject the null hypothesis and declare that the results are statistically significant
         *        when the p-value is less than 0.05.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ConfidenceInterval</code> specifies a confidence interval for the results. The confidence
         *        interval represents the range of values for the chosen metric that is likely to contain the true
         *        difference between the <code>baseStat</code> of a variation and the baseline. Evidently returns the
         *        95% confidence interval.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TreatmentEffect</code> is the difference in the statistic specified by the <code>baseStat</code>
         *        parameter between each variation and the default variation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BaseStat</code> returns the statistical values collected for the metric for each variation. The
         *        statistic uses the same statistic specified in the <code>baseStat</code> parameter. Therefore, if
         *        <code>baseStat</code> is <code>mean</code>, this returns the mean of the values collected for each
         *        variation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resultStatsWithStrings(String... resultStats);

        /**
         * <p>
         * The statistics that you want to see in the returned results.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PValue</code> specifies to use p-values for the results. A p-value is used in hypothesis testing to
         * measure how often you are willing to make a mistake in rejecting the null hypothesis. A general practice is
         * to reject the null hypothesis and declare that the results are statistically significant when the p-value is
         * less than 0.05.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ConfidenceInterval</code> specifies a confidence interval for the results. The confidence interval
         * represents the range of values for the chosen metric that is likely to contain the true difference between
         * the <code>baseStat</code> of a variation and the baseline. Evidently returns the 95% confidence interval.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TreatmentEffect</code> is the difference in the statistic specified by the <code>baseStat</code>
         * parameter between each variation and the default variation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BaseStat</code> returns the statistical values collected for the metric for each variation. The
         * statistic uses the same statistic specified in the <code>baseStat</code> parameter. Therefore, if
         * <code>baseStat</code> is <code>mean</code>, this returns the mean of the values collected for each variation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resultStats
         *        The statistics that you want to see in the returned results.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PValue</code> specifies to use p-values for the results. A p-value is used in hypothesis testing
         *        to measure how often you are willing to make a mistake in rejecting the null hypothesis. A general
         *        practice is to reject the null hypothesis and declare that the results are statistically significant
         *        when the p-value is less than 0.05.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ConfidenceInterval</code> specifies a confidence interval for the results. The confidence
         *        interval represents the range of values for the chosen metric that is likely to contain the true
         *        difference between the <code>baseStat</code> of a variation and the baseline. Evidently returns the
         *        95% confidence interval.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TreatmentEffect</code> is the difference in the statistic specified by the <code>baseStat</code>
         *        parameter between each variation and the default variation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BaseStat</code> returns the statistical values collected for the metric for each variation. The
         *        statistic uses the same statistic specified in the <code>baseStat</code> parameter. Therefore, if
         *        <code>baseStat</code> is <code>mean</code>, this returns the mean of the values collected for each
         *        variation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resultStats(Collection<ExperimentResultRequestType> resultStats);

        /**
         * <p>
         * The statistics that you want to see in the returned results.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PValue</code> specifies to use p-values for the results. A p-value is used in hypothesis testing to
         * measure how often you are willing to make a mistake in rejecting the null hypothesis. A general practice is
         * to reject the null hypothesis and declare that the results are statistically significant when the p-value is
         * less than 0.05.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ConfidenceInterval</code> specifies a confidence interval for the results. The confidence interval
         * represents the range of values for the chosen metric that is likely to contain the true difference between
         * the <code>baseStat</code> of a variation and the baseline. Evidently returns the 95% confidence interval.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TreatmentEffect</code> is the difference in the statistic specified by the <code>baseStat</code>
         * parameter between each variation and the default variation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BaseStat</code> returns the statistical values collected for the metric for each variation. The
         * statistic uses the same statistic specified in the <code>baseStat</code> parameter. Therefore, if
         * <code>baseStat</code> is <code>mean</code>, this returns the mean of the values collected for each variation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resultStats
         *        The statistics that you want to see in the returned results.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PValue</code> specifies to use p-values for the results. A p-value is used in hypothesis testing
         *        to measure how often you are willing to make a mistake in rejecting the null hypothesis. A general
         *        practice is to reject the null hypothesis and declare that the results are statistically significant
         *        when the p-value is less than 0.05.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ConfidenceInterval</code> specifies a confidence interval for the results. The confidence
         *        interval represents the range of values for the chosen metric that is likely to contain the true
         *        difference between the <code>baseStat</code> of a variation and the baseline. Evidently returns the
         *        95% confidence interval.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TreatmentEffect</code> is the difference in the statistic specified by the <code>baseStat</code>
         *        parameter between each variation and the default variation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BaseStat</code> returns the statistical values collected for the metric for each variation. The
         *        statistic uses the same statistic specified in the <code>baseStat</code> parameter. Therefore, if
         *        <code>baseStat</code> is <code>mean</code>, this returns the mean of the values collected for each
         *        variation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resultStats(ExperimentResultRequestType... resultStats);

        /**
         * <p>
         * The date and time that the experiment started.
         * </p>
         * 
         * @param startTime
         *        The date and time that the experiment started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The names of the experiment treatments that you want to see the results for.
         * </p>
         * 
         * @param treatmentNames
         *        The names of the experiment treatments that you want to see the results for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder treatmentNames(Collection<String> treatmentNames);

        /**
         * <p>
         * The names of the experiment treatments that you want to see the results for.
         * </p>
         * 
         * @param treatmentNames
         *        The names of the experiment treatments that you want to see the results for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder treatmentNames(String... treatmentNames);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends EvidentlyRequest.BuilderImpl implements Builder {
        private String baseStat;

        private Instant endTime;

        private String experiment;

        private List<String> metricNames = DefaultSdkAutoConstructList.getInstance();

        private Long period;

        private String project;

        private List<String> reportNames = DefaultSdkAutoConstructList.getInstance();

        private List<String> resultStats = DefaultSdkAutoConstructList.getInstance();

        private Instant startTime;

        private List<String> treatmentNames = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GetExperimentResultsRequest model) {
            super(model);
            baseStat(model.baseStat);
            endTime(model.endTime);
            experiment(model.experiment);
            metricNames(model.metricNames);
            period(model.period);
            project(model.project);
            reportNamesWithStrings(model.reportNames);
            resultStatsWithStrings(model.resultStats);
            startTime(model.startTime);
            treatmentNames(model.treatmentNames);
        }

        public final String getBaseStat() {
            return baseStat;
        }

        public final void setBaseStat(String baseStat) {
            this.baseStat = baseStat;
        }

        @Override
        public final Builder baseStat(String baseStat) {
            this.baseStat = baseStat;
            return this;
        }

        @Override
        public final Builder baseStat(ExperimentBaseStat baseStat) {
            this.baseStat(baseStat == null ? null : baseStat.toString());
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getExperiment() {
            return experiment;
        }

        public final void setExperiment(String experiment) {
            this.experiment = experiment;
        }

        @Override
        public final Builder experiment(String experiment) {
            this.experiment = experiment;
            return this;
        }

        public final Collection<String> getMetricNames() {
            if (metricNames instanceof SdkAutoConstructList) {
                return null;
            }
            return metricNames;
        }

        public final void setMetricNames(Collection<String> metricNames) {
            this.metricNames = MetricNameListCopier.copy(metricNames);
        }

        @Override
        public final Builder metricNames(Collection<String> metricNames) {
            this.metricNames = MetricNameListCopier.copy(metricNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricNames(String... metricNames) {
            metricNames(Arrays.asList(metricNames));
            return this;
        }

        public final Long getPeriod() {
            return period;
        }

        public final void setPeriod(Long period) {
            this.period = period;
        }

        @Override
        public final Builder period(Long period) {
            this.period = period;
            return this;
        }

        public final String getProject() {
            return project;
        }

        public final void setProject(String project) {
            this.project = project;
        }

        @Override
        public final Builder project(String project) {
            this.project = project;
            return this;
        }

        public final Collection<String> getReportNames() {
            if (reportNames instanceof SdkAutoConstructList) {
                return null;
            }
            return reportNames;
        }

        public final void setReportNames(Collection<String> reportNames) {
            this.reportNames = ExperimentReportNameListCopier.copy(reportNames);
        }

        @Override
        public final Builder reportNamesWithStrings(Collection<String> reportNames) {
            this.reportNames = ExperimentReportNameListCopier.copy(reportNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder reportNamesWithStrings(String... reportNames) {
            reportNamesWithStrings(Arrays.asList(reportNames));
            return this;
        }

        @Override
        public final Builder reportNames(Collection<ExperimentReportName> reportNames) {
            this.reportNames = ExperimentReportNameListCopier.copyEnumToString(reportNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder reportNames(ExperimentReportName... reportNames) {
            reportNames(Arrays.asList(reportNames));
            return this;
        }

        public final Collection<String> getResultStats() {
            if (resultStats instanceof SdkAutoConstructList) {
                return null;
            }
            return resultStats;
        }

        public final void setResultStats(Collection<String> resultStats) {
            this.resultStats = ExperimentResultRequestTypeListCopier.copy(resultStats);
        }

        @Override
        public final Builder resultStatsWithStrings(Collection<String> resultStats) {
            this.resultStats = ExperimentResultRequestTypeListCopier.copy(resultStats);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resultStatsWithStrings(String... resultStats) {
            resultStatsWithStrings(Arrays.asList(resultStats));
            return this;
        }

        @Override
        public final Builder resultStats(Collection<ExperimentResultRequestType> resultStats) {
            this.resultStats = ExperimentResultRequestTypeListCopier.copyEnumToString(resultStats);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resultStats(ExperimentResultRequestType... resultStats) {
            resultStats(Arrays.asList(resultStats));
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Collection<String> getTreatmentNames() {
            if (treatmentNames instanceof SdkAutoConstructList) {
                return null;
            }
            return treatmentNames;
        }

        public final void setTreatmentNames(Collection<String> treatmentNames) {
            this.treatmentNames = TreatmentNameListCopier.copy(treatmentNames);
        }

        @Override
        public final Builder treatmentNames(Collection<String> treatmentNames) {
            this.treatmentNames = TreatmentNameListCopier.copy(treatmentNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder treatmentNames(String... treatmentNames) {
            treatmentNames(Arrays.asList(treatmentNames));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetExperimentResultsRequest build() {
            return new GetExperimentResultsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
