/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.evidently;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.evidently.model.AccessDeniedException;
import software.amazon.awssdk.services.evidently.model.BatchEvaluateFeatureRequest;
import software.amazon.awssdk.services.evidently.model.BatchEvaluateFeatureResponse;
import software.amazon.awssdk.services.evidently.model.ConflictException;
import software.amazon.awssdk.services.evidently.model.CreateExperimentRequest;
import software.amazon.awssdk.services.evidently.model.CreateExperimentResponse;
import software.amazon.awssdk.services.evidently.model.CreateFeatureRequest;
import software.amazon.awssdk.services.evidently.model.CreateFeatureResponse;
import software.amazon.awssdk.services.evidently.model.CreateLaunchRequest;
import software.amazon.awssdk.services.evidently.model.CreateLaunchResponse;
import software.amazon.awssdk.services.evidently.model.CreateProjectRequest;
import software.amazon.awssdk.services.evidently.model.CreateProjectResponse;
import software.amazon.awssdk.services.evidently.model.DeleteExperimentRequest;
import software.amazon.awssdk.services.evidently.model.DeleteExperimentResponse;
import software.amazon.awssdk.services.evidently.model.DeleteFeatureRequest;
import software.amazon.awssdk.services.evidently.model.DeleteFeatureResponse;
import software.amazon.awssdk.services.evidently.model.DeleteLaunchRequest;
import software.amazon.awssdk.services.evidently.model.DeleteLaunchResponse;
import software.amazon.awssdk.services.evidently.model.DeleteProjectRequest;
import software.amazon.awssdk.services.evidently.model.DeleteProjectResponse;
import software.amazon.awssdk.services.evidently.model.EvaluateFeatureRequest;
import software.amazon.awssdk.services.evidently.model.EvaluateFeatureResponse;
import software.amazon.awssdk.services.evidently.model.EvidentlyException;
import software.amazon.awssdk.services.evidently.model.EvidentlyRequest;
import software.amazon.awssdk.services.evidently.model.GetExperimentRequest;
import software.amazon.awssdk.services.evidently.model.GetExperimentResponse;
import software.amazon.awssdk.services.evidently.model.GetExperimentResultsRequest;
import software.amazon.awssdk.services.evidently.model.GetExperimentResultsResponse;
import software.amazon.awssdk.services.evidently.model.GetFeatureRequest;
import software.amazon.awssdk.services.evidently.model.GetFeatureResponse;
import software.amazon.awssdk.services.evidently.model.GetLaunchRequest;
import software.amazon.awssdk.services.evidently.model.GetLaunchResponse;
import software.amazon.awssdk.services.evidently.model.GetProjectRequest;
import software.amazon.awssdk.services.evidently.model.GetProjectResponse;
import software.amazon.awssdk.services.evidently.model.InternalServerException;
import software.amazon.awssdk.services.evidently.model.ListExperimentsRequest;
import software.amazon.awssdk.services.evidently.model.ListExperimentsResponse;
import software.amazon.awssdk.services.evidently.model.ListFeaturesRequest;
import software.amazon.awssdk.services.evidently.model.ListFeaturesResponse;
import software.amazon.awssdk.services.evidently.model.ListLaunchesRequest;
import software.amazon.awssdk.services.evidently.model.ListLaunchesResponse;
import software.amazon.awssdk.services.evidently.model.ListProjectsRequest;
import software.amazon.awssdk.services.evidently.model.ListProjectsResponse;
import software.amazon.awssdk.services.evidently.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.evidently.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.evidently.model.PutProjectEventsRequest;
import software.amazon.awssdk.services.evidently.model.PutProjectEventsResponse;
import software.amazon.awssdk.services.evidently.model.ResourceNotFoundException;
import software.amazon.awssdk.services.evidently.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.evidently.model.ServiceUnavailableException;
import software.amazon.awssdk.services.evidently.model.StartExperimentRequest;
import software.amazon.awssdk.services.evidently.model.StartExperimentResponse;
import software.amazon.awssdk.services.evidently.model.StartLaunchRequest;
import software.amazon.awssdk.services.evidently.model.StartLaunchResponse;
import software.amazon.awssdk.services.evidently.model.StopExperimentRequest;
import software.amazon.awssdk.services.evidently.model.StopExperimentResponse;
import software.amazon.awssdk.services.evidently.model.StopLaunchRequest;
import software.amazon.awssdk.services.evidently.model.StopLaunchResponse;
import software.amazon.awssdk.services.evidently.model.TagResourceRequest;
import software.amazon.awssdk.services.evidently.model.TagResourceResponse;
import software.amazon.awssdk.services.evidently.model.ThrottlingException;
import software.amazon.awssdk.services.evidently.model.UntagResourceRequest;
import software.amazon.awssdk.services.evidently.model.UntagResourceResponse;
import software.amazon.awssdk.services.evidently.model.UpdateExperimentRequest;
import software.amazon.awssdk.services.evidently.model.UpdateExperimentResponse;
import software.amazon.awssdk.services.evidently.model.UpdateFeatureRequest;
import software.amazon.awssdk.services.evidently.model.UpdateFeatureResponse;
import software.amazon.awssdk.services.evidently.model.UpdateLaunchRequest;
import software.amazon.awssdk.services.evidently.model.UpdateLaunchResponse;
import software.amazon.awssdk.services.evidently.model.UpdateProjectDataDeliveryRequest;
import software.amazon.awssdk.services.evidently.model.UpdateProjectDataDeliveryResponse;
import software.amazon.awssdk.services.evidently.model.UpdateProjectRequest;
import software.amazon.awssdk.services.evidently.model.UpdateProjectResponse;
import software.amazon.awssdk.services.evidently.model.ValidationException;
import software.amazon.awssdk.services.evidently.paginators.ListExperimentsPublisher;
import software.amazon.awssdk.services.evidently.paginators.ListFeaturesPublisher;
import software.amazon.awssdk.services.evidently.paginators.ListLaunchesPublisher;
import software.amazon.awssdk.services.evidently.paginators.ListProjectsPublisher;
import software.amazon.awssdk.services.evidently.transform.BatchEvaluateFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateProjectRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteProjectRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.EvaluateFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetExperimentResultsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetProjectRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListExperimentsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListFeaturesRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListLaunchesRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListProjectsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.PutProjectEventsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.StartExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.StartLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.StopExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.StopLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateProjectDataDeliveryRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateProjectRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link EvidentlyAsyncClient}.
 *
 * @see EvidentlyAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEvidentlyAsyncClient implements EvidentlyAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultEvidentlyAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEvidentlyAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * This operation assigns feature variation to user sessions. For each user session, you pass in an
     * <code>entityID</code> that represents the user. Evidently then checks the evaluation rules and assigns the
     * variation.
     * </p>
     * <p>
     * The first rules that are evaluated are the override rules. If the user's <code>entityID</code> matches an
     * override rule, the user is served the variation specified by that rule.
     * </p>
     * <p>
     * Next, if there is a launch of the feature, the user might be assigned to a variation in the launch. The chance of
     * this depends on the percentage of users that are allocated to that launch. If the user is enrolled in the launch,
     * the variation they are served depends on the allocation of the various feature variations used for the launch.
     * </p>
     * <p>
     * If the user is not assigned to a launch, and there is an ongoing experiment for this feature, the user might be
     * assigned to a variation in the experiment. The chance of this depends on the percentage of users that are
     * allocated to that experiment. If the user is enrolled in the experiment, the variation they are served depends on
     * the allocation of the various feature variations used for the experiment.
     * </p>
     * <p>
     * If the user is not assigned to a launch or experiment, they are served the default variation.
     * </p>
     *
     * @param batchEvaluateFeatureRequest
     * @return A Java Future containing the result of the BatchEvaluateFeature operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.BatchEvaluateFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/BatchEvaluateFeature"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchEvaluateFeatureResponse> batchEvaluateFeature(
            BatchEvaluateFeatureRequest batchEvaluateFeatureRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchEvaluateFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchEvaluateFeature");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchEvaluateFeatureResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchEvaluateFeatureResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "dataplane.";
            String resolvedHostExpression = "dataplane.";

            CompletableFuture<BatchEvaluateFeatureResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchEvaluateFeatureRequest, BatchEvaluateFeatureResponse>()
                            .withOperationName("BatchEvaluateFeature")
                            .withMarshaller(new BatchEvaluateFeatureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchEvaluateFeatureRequest));
            CompletableFuture<BatchEvaluateFeatureResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Evidently <i>experiment</i>. Before you create an experiment, you must create the feature to use for
     * the experiment.
     * </p>
     * <p>
     * An experiment helps you make feature design decisions based on evidence and data. An experiment can test as many
     * as five variations at once. Evidently collects experiment data and analyzes it by statistical methods, and
     * provides clear recommendations about which variations perform better.
     * </p>
     * <p>
     * Don't use this operation to update an existing experiment. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateExperiment.html"
     * >UpdateExperiment</a>.
     * </p>
     *
     * @param createExperimentRequest
     * @return A Java Future containing the result of the CreateExperiment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.CreateExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateExperiment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateExperimentResponse> createExperiment(CreateExperimentRequest createExperimentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateExperiment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateExperimentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateExperimentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateExperimentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateExperimentRequest, CreateExperimentResponse>()
                            .withOperationName("CreateExperiment")
                            .withMarshaller(new CreateExperimentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createExperimentRequest));
            CompletableFuture<CreateExperimentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Evidently <i>feature</i> that you want to launch or test. You can define up to five variations of a
     * feature, and use these variations in your launches and experiments. A feature must be created in a project. For
     * information about creating a project, see <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateProject.html"
     * >CreateProject</a>.
     * </p>
     * <p>
     * Don't use this operation to update an existing feature. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateFeature.html"
     * >UpdateFeature</a>.
     * </p>
     *
     * @param createFeatureRequest
     * @return A Java Future containing the result of the CreateFeature operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.CreateFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFeatureResponse> createFeature(CreateFeatureRequest createFeatureRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFeature");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFeatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateFeatureResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFeatureResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFeatureRequest, CreateFeatureResponse>()
                            .withOperationName("CreateFeature")
                            .withMarshaller(new CreateFeatureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createFeatureRequest));
            CompletableFuture<CreateFeatureResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <i>launch</i> of a given feature. Before you create a launch, you must create the feature to use for
     * the launch.
     * </p>
     * <p>
     * You can use a launch to safely validate new features by serving them to a specified percentage of your users
     * while you roll out the feature. You can monitor the performance of the new feature to help you decide when to
     * ramp up traffic to more users. This helps you reduce risk and identify unintended consequences before you fully
     * launch the feature.
     * </p>
     * <p>
     * Don't use this operation to update an existing launch. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateLaunch.html"
     * >UpdateLaunch</a>.
     * </p>
     *
     * @param createLaunchRequest
     * @return A Java Future containing the result of the CreateLaunch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.CreateLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLaunchResponse> createLaunch(CreateLaunchRequest createLaunchRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLaunch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateLaunchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLaunchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLaunchRequest, CreateLaunchResponse>()
                            .withOperationName("CreateLaunch").withMarshaller(new CreateLaunchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLaunchRequest));
            CompletableFuture<CreateLaunchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a project, which is the logical object in Evidently that can contain features, launches, and experiments.
     * Use projects to group similar features together.
     * </p>
     * <p>
     * To update an existing project, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateProject.html"
     * >UpdateProject</a>.
     * </p>
     *
     * @param createProjectRequest
     * @return A Java Future containing the result of the CreateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.CreateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProjectResponse> createProject(CreateProjectRequest createProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProjectRequest, CreateProjectResponse>()
                            .withOperationName("CreateProject")
                            .withMarshaller(new CreateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createProjectRequest));
            CompletableFuture<CreateProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Evidently experiment. The feature used for the experiment is not deleted.
     * </p>
     * <p>
     * To stop an experiment without deleting it, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_StopExperiment.html"
     * >StopExperiment</a>.
     * </p>
     *
     * @param deleteExperimentRequest
     * @return A Java Future containing the result of the DeleteExperiment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceUnavailableException The service was unavailable. Retry the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.DeleteExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteExperiment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteExperimentResponse> deleteExperiment(DeleteExperimentRequest deleteExperimentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteExperiment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteExperimentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteExperimentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteExperimentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteExperimentRequest, DeleteExperimentResponse>()
                            .withOperationName("DeleteExperiment")
                            .withMarshaller(new DeleteExperimentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteExperimentRequest));
            CompletableFuture<DeleteExperimentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Evidently feature.
     * </p>
     *
     * @param deleteFeatureRequest
     * @return A Java Future containing the result of the DeleteFeature operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.DeleteFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFeatureResponse> deleteFeature(DeleteFeatureRequest deleteFeatureRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFeature");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFeatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteFeatureResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFeatureResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFeatureRequest, DeleteFeatureResponse>()
                            .withOperationName("DeleteFeature")
                            .withMarshaller(new DeleteFeatureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteFeatureRequest));
            CompletableFuture<DeleteFeatureResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Evidently launch. The feature used for the launch is not deleted.
     * </p>
     * <p>
     * To stop a launch without deleting it, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_StopLaunch.html">StopLaunch</a>.
     * </p>
     *
     * @param deleteLaunchRequest
     * @return A Java Future containing the result of the DeleteLaunch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.DeleteLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLaunchResponse> deleteLaunch(DeleteLaunchRequest deleteLaunchRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLaunch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteLaunchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLaunchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLaunchRequest, DeleteLaunchResponse>()
                            .withOperationName("DeleteLaunch").withMarshaller(new DeleteLaunchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLaunchRequest));
            CompletableFuture<DeleteLaunchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Evidently project. Before you can delete a project, you must delete all the features that the project
     * contains. To delete a feature, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_DeleteFeature.html"
     * >DeleteFeature</a>.
     * </p>
     *
     * @param deleteProjectRequest
     * @return A Java Future containing the result of the DeleteProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.DeleteProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProjectResponse> deleteProject(DeleteProjectRequest deleteProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProjectRequest, DeleteProjectResponse>()
                            .withOperationName("DeleteProject")
                            .withMarshaller(new DeleteProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteProjectRequest));
            CompletableFuture<DeleteProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation assigns a feature variation to one given user session. You pass in an <code>entityID</code> that
     * represents the user. Evidently then checks the evaluation rules and assigns the variation.
     * </p>
     * <p>
     * The first rules that are evaluated are the override rules. If the user's <code>entityID</code> matches an
     * override rule, the user is served the variation specified by that rule.
     * </p>
     * <p>
     * Next, if there is a launch of the feature, the user might be assigned to a variation in the launch. The chance of
     * this depends on the percentage of users that are allocated to that launch. If the user is enrolled in the launch,
     * the variation they are served depends on the allocation of the various feature variations used for the launch.
     * </p>
     * <p>
     * If the user is not assigned to a launch, and there is an ongoing experiment for this feature, the user might be
     * assigned to a variation in the experiment. The chance of this depends on the percentage of users that are
     * allocated to that experiment. If the user is enrolled in the experiment, the variation they are served depends on
     * the allocation of the various feature variations used for the experiment.
     * </p>
     * <p>
     * If the user is not assigned to a launch or experiment, they are served the default variation.
     * </p>
     *
     * @param evaluateFeatureRequest
     * @return A Java Future containing the result of the EvaluateFeature operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.EvaluateFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/EvaluateFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<EvaluateFeatureResponse> evaluateFeature(EvaluateFeatureRequest evaluateFeatureRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, evaluateFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EvaluateFeature");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EvaluateFeatureResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EvaluateFeatureResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "dataplane.";
            String resolvedHostExpression = "dataplane.";

            CompletableFuture<EvaluateFeatureResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EvaluateFeatureRequest, EvaluateFeatureResponse>()
                            .withOperationName("EvaluateFeature")
                            .withMarshaller(new EvaluateFeatureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(evaluateFeatureRequest));
            CompletableFuture<EvaluateFeatureResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details about one experiment. You must already know the experiment name. To retrieve a list of
     * experiments in your account, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_ListExperiments.html"
     * >ListExperiments</a>.
     * </p>
     *
     * @param getExperimentRequest
     * @return A Java Future containing the result of the GetExperiment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.GetExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetExperimentResponse> getExperiment(GetExperimentRequest getExperimentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExperiment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetExperimentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetExperimentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetExperimentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetExperimentRequest, GetExperimentResponse>()
                            .withOperationName("GetExperiment")
                            .withMarshaller(new GetExperimentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getExperimentRequest));
            CompletableFuture<GetExperimentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the results of a running or completed experiment. No results are available until there have been 100
     * events for each variation and at least 10 minutes have passed since the start of the experiment.
     * </p>
     * <p>
     * Experiment results are available up to 63 days after the start of the experiment. They are not available after
     * that because of CloudWatch data retention policies.
     * </p>
     *
     * @param getExperimentResultsRequest
     * @return A Java Future containing the result of the GetExperimentResults operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.GetExperimentResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetExperimentResults"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetExperimentResultsResponse> getExperimentResults(
            GetExperimentResultsRequest getExperimentResultsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExperimentResultsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExperimentResults");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetExperimentResultsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetExperimentResultsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetExperimentResultsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetExperimentResultsRequest, GetExperimentResultsResponse>()
                            .withOperationName("GetExperimentResults")
                            .withMarshaller(new GetExperimentResultsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getExperimentResultsRequest));
            CompletableFuture<GetExperimentResultsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details about one feature. You must already know the feature name. To retrieve a list of features in
     * your account, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_ListFeatures.html"
     * >ListFeatures</a>.
     * </p>
     *
     * @param getFeatureRequest
     * @return A Java Future containing the result of the GetFeature operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.GetFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetFeatureResponse> getFeature(GetFeatureRequest getFeatureRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFeature");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFeatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFeatureResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFeatureResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFeatureRequest, GetFeatureResponse>().withOperationName("GetFeature")
                            .withMarshaller(new GetFeatureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getFeatureRequest));
            CompletableFuture<GetFeatureResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details about one launch. You must already know the launch name. To retrieve a list of launches in
     * your account, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_ListLaunches.html"
     * >ListLaunches</a>.
     * </p>
     *
     * @param getLaunchRequest
     * @return A Java Future containing the result of the GetLaunch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.GetLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLaunchResponse> getLaunch(GetLaunchRequest getLaunchRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLaunch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLaunchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLaunchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLaunchRequest, GetLaunchResponse>().withOperationName("GetLaunch")
                            .withMarshaller(new GetLaunchRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLaunchRequest));
            CompletableFuture<GetLaunchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details about one launch. You must already know the project name. To retrieve a list of projects in
     * your account, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_ListProjects.html"
     * >ListProjects</a>.
     * </p>
     *
     * @param getProjectRequest
     * @return A Java Future containing the result of the GetProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.GetProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetProjectResponse> getProject(GetProjectRequest getProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProjectRequest, GetProjectResponse>().withOperationName("GetProject")
                            .withMarshaller(new GetProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getProjectRequest));
            CompletableFuture<GetProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration details about all the experiments in the specified project.
     * </p>
     *
     * @param listExperimentsRequest
     * @return A Java Future containing the result of the ListExperiments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListExperiments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListExperiments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListExperimentsResponse> listExperiments(ListExperimentsRequest listExperimentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExperimentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExperiments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListExperimentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListExperimentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListExperimentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListExperimentsRequest, ListExperimentsResponse>()
                            .withOperationName("ListExperiments")
                            .withMarshaller(new ListExperimentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listExperimentsRequest));
            CompletableFuture<ListExperimentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration details about all the experiments in the specified project.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listExperiments(software.amazon.awssdk.services.evidently.model.ListExperimentsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evidently.paginators.ListExperimentsPublisher publisher = client.listExperimentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evidently.paginators.ListExperimentsPublisher publisher = client.listExperimentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.evidently.model.ListExperimentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.evidently.model.ListExperimentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listExperiments(software.amazon.awssdk.services.evidently.model.ListExperimentsRequest)} operation.</b>
     * </p>
     *
     * @param listExperimentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListExperiments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListExperiments" target="_top">AWS API
     *      Documentation</a>
     */
    public ListExperimentsPublisher listExperimentsPaginator(ListExperimentsRequest listExperimentsRequest) {
        return new ListExperimentsPublisher(this, applyPaginatorUserAgent(listExperimentsRequest));
    }

    /**
     * <p>
     * Returns configuration details about all the features in the specified project.
     * </p>
     *
     * @param listFeaturesRequest
     * @return A Java Future containing the result of the ListFeatures operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListFeatures
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListFeatures" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFeaturesResponse> listFeatures(ListFeaturesRequest listFeaturesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFeaturesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFeatures");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFeaturesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFeaturesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFeaturesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFeaturesRequest, ListFeaturesResponse>()
                            .withOperationName("ListFeatures").withMarshaller(new ListFeaturesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFeaturesRequest));
            CompletableFuture<ListFeaturesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration details about all the features in the specified project.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listFeatures(software.amazon.awssdk.services.evidently.model.ListFeaturesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evidently.paginators.ListFeaturesPublisher publisher = client.listFeaturesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evidently.paginators.ListFeaturesPublisher publisher = client.listFeaturesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.evidently.model.ListFeaturesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.evidently.model.ListFeaturesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFeatures(software.amazon.awssdk.services.evidently.model.ListFeaturesRequest)} operation.</b>
     * </p>
     *
     * @param listFeaturesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListFeatures
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListFeatures" target="_top">AWS API
     *      Documentation</a>
     */
    public ListFeaturesPublisher listFeaturesPaginator(ListFeaturesRequest listFeaturesRequest) {
        return new ListFeaturesPublisher(this, applyPaginatorUserAgent(listFeaturesRequest));
    }

    /**
     * <p>
     * Returns configuration details about all the launches in the specified project.
     * </p>
     *
     * @param listLaunchesRequest
     * @return A Java Future containing the result of the ListLaunches operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListLaunches
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListLaunches" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLaunchesResponse> listLaunches(ListLaunchesRequest listLaunchesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLaunchesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLaunches");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLaunchesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLaunchesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLaunchesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLaunchesRequest, ListLaunchesResponse>()
                            .withOperationName("ListLaunches").withMarshaller(new ListLaunchesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLaunchesRequest));
            CompletableFuture<ListLaunchesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration details about all the launches in the specified project.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listLaunches(software.amazon.awssdk.services.evidently.model.ListLaunchesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evidently.paginators.ListLaunchesPublisher publisher = client.listLaunchesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evidently.paginators.ListLaunchesPublisher publisher = client.listLaunchesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.evidently.model.ListLaunchesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.evidently.model.ListLaunchesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLaunches(software.amazon.awssdk.services.evidently.model.ListLaunchesRequest)} operation.</b>
     * </p>
     *
     * @param listLaunchesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListLaunches
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListLaunches" target="_top">AWS API
     *      Documentation</a>
     */
    public ListLaunchesPublisher listLaunchesPaginator(ListLaunchesRequest listLaunchesRequest) {
        return new ListLaunchesPublisher(this, applyPaginatorUserAgent(listLaunchesRequest));
    }

    /**
     * <p>
     * Returns configuration details about all the projects in the current Region in your account.
     * </p>
     *
     * @param listProjectsRequest
     * @return A Java Future containing the result of the ListProjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListProjectsResponse> listProjects(ListProjectsRequest listProjectsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListProjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProjectsRequest, ListProjectsResponse>()
                            .withOperationName("ListProjects").withMarshaller(new ListProjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProjectsRequest));
            CompletableFuture<ListProjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration details about all the projects in the current Region in your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listProjects(software.amazon.awssdk.services.evidently.model.ListProjectsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evidently.paginators.ListProjectsPublisher publisher = client.listProjectsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.evidently.paginators.ListProjectsPublisher publisher = client.listProjectsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.evidently.model.ListProjectsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.evidently.model.ListProjectsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProjects(software.amazon.awssdk.services.evidently.model.ListProjectsRequest)} operation.</b>
     * </p>
     *
     * @param listProjectsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    public ListProjectsPublisher listProjectsPaginator(ListProjectsRequest listProjectsRequest) {
        return new ListProjectsPublisher(this, applyPaginatorUserAgent(listProjectsRequest));
    }

    /**
     * <p>
     * Displays the tags associated with an Evidently resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends performance events to Evidently. These events can be used to evaluate a launch or an experiment.
     * </p>
     *
     * @param putProjectEventsRequest
     * @return A Java Future containing the result of the PutProjectEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.PutProjectEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/PutProjectEvents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutProjectEventsResponse> putProjectEvents(PutProjectEventsRequest putProjectEventsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putProjectEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutProjectEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutProjectEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutProjectEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "dataplane.";
            String resolvedHostExpression = "dataplane.";

            CompletableFuture<PutProjectEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutProjectEventsRequest, PutProjectEventsResponse>()
                            .withOperationName("PutProjectEvents")
                            .withMarshaller(new PutProjectEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(putProjectEventsRequest));
            CompletableFuture<PutProjectEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an existing experiment. To create an experiment, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateExperiment.html"
     * >CreateExperiment</a>.
     * </p>
     *
     * @param startExperimentRequest
     * @return A Java Future containing the result of the StartExperiment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.StartExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/StartExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartExperimentResponse> startExperiment(StartExperimentRequest startExperimentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartExperiment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartExperimentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartExperimentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartExperimentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartExperimentRequest, StartExperimentResponse>()
                            .withOperationName("StartExperiment")
                            .withMarshaller(new StartExperimentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startExperimentRequest));
            CompletableFuture<StartExperimentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an existing launch. To create a launch, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateLaunch.html"
     * >CreateLaunch</a>.
     * </p>
     *
     * @param startLaunchRequest
     * @return A Java Future containing the result of the StartLaunch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.StartLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/StartLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartLaunchResponse> startLaunch(StartLaunchRequest startLaunchRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartLaunch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartLaunchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartLaunchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartLaunchRequest, StartLaunchResponse>()
                            .withOperationName("StartLaunch").withMarshaller(new StartLaunchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startLaunchRequest));
            CompletableFuture<StartLaunchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops an experiment that is currently running. If you stop an experiment, you can't resume it or restart it.
     * </p>
     *
     * @param stopExperimentRequest
     * @return A Java Future containing the result of the StopExperiment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.StopExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/StopExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopExperimentResponse> stopExperiment(StopExperimentRequest stopExperimentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopExperiment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopExperimentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopExperimentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopExperimentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopExperimentRequest, StopExperimentResponse>()
                            .withOperationName("StopExperiment")
                            .withMarshaller(new StopExperimentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopExperimentRequest));
            CompletableFuture<StopExperimentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a launch that is currently running. After you stop a launch, you will not be able to resume it or restart
     * it. Also, it will not be evaluated as a rule for traffic allocation, and the traffic that was allocated to the
     * launch will instead be available to the feature's experiment, if there is one. Otherwise, all traffic will be
     * served the default variation after the launch is stopped.
     * </p>
     *
     * @param stopLaunchRequest
     * @return A Java Future containing the result of the StopLaunch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.StopLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/StopLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopLaunchResponse> stopLaunch(StopLaunchRequest stopLaunchRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopLaunch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopLaunchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopLaunchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopLaunchRequest, StopLaunchResponse>().withOperationName("StopLaunch")
                            .withMarshaller(new StopLaunchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopLaunchRequest));
            CompletableFuture<StopLaunchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch Evidently resource. Projects, features,
     * launches, and experiments can be tagged.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key for the resource, this tag is appended to the list of tags associated with the alarm. If you specify a tag
     * key that is already associated with the resource, the new tag value that you specify replaces the previous value
     * for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon
     * Web Services resources</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an Evidently experiment.
     * </p>
     * <p>
     * Don't use this operation to update an experiment's tag. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateExperimentRequest
     * @return A Java Future containing the result of the UpdateExperiment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.UpdateExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateExperiment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateExperimentResponse> updateExperiment(UpdateExperimentRequest updateExperimentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateExperiment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateExperimentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateExperimentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateExperimentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateExperimentRequest, UpdateExperimentResponse>()
                            .withOperationName("UpdateExperiment")
                            .withMarshaller(new UpdateExperimentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateExperimentRequest));
            CompletableFuture<UpdateExperimentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing feature.
     * </p>
     * <p>
     * You can't use this operation to update the tags of an existing feature. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateFeatureRequest
     * @return A Java Future containing the result of the UpdateFeature operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.UpdateFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFeatureResponse> updateFeature(UpdateFeatureRequest updateFeatureRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFeature");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFeatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateFeatureResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFeatureResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFeatureRequest, UpdateFeatureResponse>()
                            .withOperationName("UpdateFeature")
                            .withMarshaller(new UpdateFeatureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFeatureRequest));
            CompletableFuture<UpdateFeatureResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a launch of a given feature.
     * </p>
     * <p>
     * Don't use this operation to update the tags of an existing launch. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateLaunchRequest
     * @return A Java Future containing the result of the UpdateLaunch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.UpdateLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLaunchResponse> updateLaunch(UpdateLaunchRequest updateLaunchRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLaunch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateLaunchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLaunchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLaunchRequest, UpdateLaunchResponse>()
                            .withOperationName("UpdateLaunch").withMarshaller(new UpdateLaunchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLaunchRequest));
            CompletableFuture<UpdateLaunchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the description of an existing project.
     * </p>
     * <p>
     * To create a new project, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateProject.html"
     * >CreateProject</a>.
     * </p>
     * <p>
     * Don't use this operation to update the data storage options of a project. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateProjectDataDelivery.html"
     * >UpdateProjectDataDelivery</a>.
     * </p>
     * <p>
     * Don't use this operation to update the tags of a project. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateProjectRequest
     * @return A Java Future containing the result of the UpdateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.UpdateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProjectResponse> updateProject(UpdateProjectRequest updateProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProjectRequest, UpdateProjectResponse>()
                            .withOperationName("UpdateProject")
                            .withMarshaller(new UpdateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateProjectRequest));
            CompletableFuture<UpdateProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the data storage options for this project. If you store evaluation events, you an keep them and analyze
     * them on your own. If you choose not to store evaluation events, Evidently deletes them after using them to
     * produce metrics and other experiment results that you can view.
     * </p>
     * <p>
     * You can't specify both <code>cloudWatchLogs</code> and <code>s3Destination</code> in the same operation.
     * </p>
     *
     * @param updateProjectDataDeliveryRequest
     * @return A Java Future containing the result of the UpdateProjectDataDelivery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.UpdateProjectDataDelivery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateProjectDataDelivery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProjectDataDeliveryResponse> updateProjectDataDelivery(
            UpdateProjectDataDeliveryRequest updateProjectDataDeliveryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProjectDataDeliveryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProjectDataDelivery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProjectDataDeliveryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateProjectDataDeliveryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProjectDataDeliveryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProjectDataDeliveryRequest, UpdateProjectDataDeliveryResponse>()
                            .withOperationName("UpdateProjectDataDelivery")
                            .withMarshaller(new UpdateProjectDataDeliveryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateProjectDataDeliveryRequest));
            CompletableFuture<UpdateProjectDataDeliveryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EvidentlyException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends EvidentlyRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
