/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.evidently;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.evidently.internal.EvidentlyServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.evidently.model.AccessDeniedException;
import software.amazon.awssdk.services.evidently.model.BatchEvaluateFeatureRequest;
import software.amazon.awssdk.services.evidently.model.BatchEvaluateFeatureResponse;
import software.amazon.awssdk.services.evidently.model.ConflictException;
import software.amazon.awssdk.services.evidently.model.CreateExperimentRequest;
import software.amazon.awssdk.services.evidently.model.CreateExperimentResponse;
import software.amazon.awssdk.services.evidently.model.CreateFeatureRequest;
import software.amazon.awssdk.services.evidently.model.CreateFeatureResponse;
import software.amazon.awssdk.services.evidently.model.CreateLaunchRequest;
import software.amazon.awssdk.services.evidently.model.CreateLaunchResponse;
import software.amazon.awssdk.services.evidently.model.CreateProjectRequest;
import software.amazon.awssdk.services.evidently.model.CreateProjectResponse;
import software.amazon.awssdk.services.evidently.model.CreateSegmentRequest;
import software.amazon.awssdk.services.evidently.model.CreateSegmentResponse;
import software.amazon.awssdk.services.evidently.model.DeleteExperimentRequest;
import software.amazon.awssdk.services.evidently.model.DeleteExperimentResponse;
import software.amazon.awssdk.services.evidently.model.DeleteFeatureRequest;
import software.amazon.awssdk.services.evidently.model.DeleteFeatureResponse;
import software.amazon.awssdk.services.evidently.model.DeleteLaunchRequest;
import software.amazon.awssdk.services.evidently.model.DeleteLaunchResponse;
import software.amazon.awssdk.services.evidently.model.DeleteProjectRequest;
import software.amazon.awssdk.services.evidently.model.DeleteProjectResponse;
import software.amazon.awssdk.services.evidently.model.DeleteSegmentRequest;
import software.amazon.awssdk.services.evidently.model.DeleteSegmentResponse;
import software.amazon.awssdk.services.evidently.model.EvaluateFeatureRequest;
import software.amazon.awssdk.services.evidently.model.EvaluateFeatureResponse;
import software.amazon.awssdk.services.evidently.model.EvidentlyException;
import software.amazon.awssdk.services.evidently.model.GetExperimentRequest;
import software.amazon.awssdk.services.evidently.model.GetExperimentResponse;
import software.amazon.awssdk.services.evidently.model.GetExperimentResultsRequest;
import software.amazon.awssdk.services.evidently.model.GetExperimentResultsResponse;
import software.amazon.awssdk.services.evidently.model.GetFeatureRequest;
import software.amazon.awssdk.services.evidently.model.GetFeatureResponse;
import software.amazon.awssdk.services.evidently.model.GetLaunchRequest;
import software.amazon.awssdk.services.evidently.model.GetLaunchResponse;
import software.amazon.awssdk.services.evidently.model.GetProjectRequest;
import software.amazon.awssdk.services.evidently.model.GetProjectResponse;
import software.amazon.awssdk.services.evidently.model.GetSegmentRequest;
import software.amazon.awssdk.services.evidently.model.GetSegmentResponse;
import software.amazon.awssdk.services.evidently.model.InternalServerException;
import software.amazon.awssdk.services.evidently.model.ListExperimentsRequest;
import software.amazon.awssdk.services.evidently.model.ListExperimentsResponse;
import software.amazon.awssdk.services.evidently.model.ListFeaturesRequest;
import software.amazon.awssdk.services.evidently.model.ListFeaturesResponse;
import software.amazon.awssdk.services.evidently.model.ListLaunchesRequest;
import software.amazon.awssdk.services.evidently.model.ListLaunchesResponse;
import software.amazon.awssdk.services.evidently.model.ListProjectsRequest;
import software.amazon.awssdk.services.evidently.model.ListProjectsResponse;
import software.amazon.awssdk.services.evidently.model.ListSegmentReferencesRequest;
import software.amazon.awssdk.services.evidently.model.ListSegmentReferencesResponse;
import software.amazon.awssdk.services.evidently.model.ListSegmentsRequest;
import software.amazon.awssdk.services.evidently.model.ListSegmentsResponse;
import software.amazon.awssdk.services.evidently.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.evidently.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.evidently.model.PutProjectEventsRequest;
import software.amazon.awssdk.services.evidently.model.PutProjectEventsResponse;
import software.amazon.awssdk.services.evidently.model.ResourceNotFoundException;
import software.amazon.awssdk.services.evidently.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.evidently.model.ServiceUnavailableException;
import software.amazon.awssdk.services.evidently.model.StartExperimentRequest;
import software.amazon.awssdk.services.evidently.model.StartExperimentResponse;
import software.amazon.awssdk.services.evidently.model.StartLaunchRequest;
import software.amazon.awssdk.services.evidently.model.StartLaunchResponse;
import software.amazon.awssdk.services.evidently.model.StopExperimentRequest;
import software.amazon.awssdk.services.evidently.model.StopExperimentResponse;
import software.amazon.awssdk.services.evidently.model.StopLaunchRequest;
import software.amazon.awssdk.services.evidently.model.StopLaunchResponse;
import software.amazon.awssdk.services.evidently.model.TagResourceRequest;
import software.amazon.awssdk.services.evidently.model.TagResourceResponse;
import software.amazon.awssdk.services.evidently.model.TestSegmentPatternRequest;
import software.amazon.awssdk.services.evidently.model.TestSegmentPatternResponse;
import software.amazon.awssdk.services.evidently.model.ThrottlingException;
import software.amazon.awssdk.services.evidently.model.UntagResourceRequest;
import software.amazon.awssdk.services.evidently.model.UntagResourceResponse;
import software.amazon.awssdk.services.evidently.model.UpdateExperimentRequest;
import software.amazon.awssdk.services.evidently.model.UpdateExperimentResponse;
import software.amazon.awssdk.services.evidently.model.UpdateFeatureRequest;
import software.amazon.awssdk.services.evidently.model.UpdateFeatureResponse;
import software.amazon.awssdk.services.evidently.model.UpdateLaunchRequest;
import software.amazon.awssdk.services.evidently.model.UpdateLaunchResponse;
import software.amazon.awssdk.services.evidently.model.UpdateProjectDataDeliveryRequest;
import software.amazon.awssdk.services.evidently.model.UpdateProjectDataDeliveryResponse;
import software.amazon.awssdk.services.evidently.model.UpdateProjectRequest;
import software.amazon.awssdk.services.evidently.model.UpdateProjectResponse;
import software.amazon.awssdk.services.evidently.model.ValidationException;
import software.amazon.awssdk.services.evidently.transform.BatchEvaluateFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateProjectRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.CreateSegmentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteProjectRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.DeleteSegmentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.EvaluateFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetExperimentResultsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetProjectRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.GetSegmentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListExperimentsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListFeaturesRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListLaunchesRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListProjectsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListSegmentReferencesRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListSegmentsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.PutProjectEventsRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.StartExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.StartLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.StopExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.StopLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.TestSegmentPatternRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateExperimentRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateFeatureRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateLaunchRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateProjectDataDeliveryRequestMarshaller;
import software.amazon.awssdk.services.evidently.transform.UpdateProjectRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link EvidentlyAsyncClient}.
 *
 * @see EvidentlyAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEvidentlyAsyncClient implements EvidentlyAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultEvidentlyAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEvidentlyAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * This operation assigns feature variation to user sessions. For each user session, you pass in an
     * <code>entityID</code> that represents the user. Evidently then checks the evaluation rules and assigns the
     * variation.
     * </p>
     * <p>
     * The first rules that are evaluated are the override rules. If the user's <code>entityID</code> matches an
     * override rule, the user is served the variation specified by that rule.
     * </p>
     * <p>
     * Next, if there is a launch of the feature, the user might be assigned to a variation in the launch. The chance of
     * this depends on the percentage of users that are allocated to that launch. If the user is enrolled in the launch,
     * the variation they are served depends on the allocation of the various feature variations used for the launch.
     * </p>
     * <p>
     * If the user is not assigned to a launch, and there is an ongoing experiment for this feature, the user might be
     * assigned to a variation in the experiment. The chance of this depends on the percentage of users that are
     * allocated to that experiment. If the user is enrolled in the experiment, the variation they are served depends on
     * the allocation of the various feature variations used for the experiment.
     * </p>
     * <p>
     * If the user is not assigned to a launch or experiment, they are served the default variation.
     * </p>
     *
     * @param batchEvaluateFeatureRequest
     * @return A Java Future containing the result of the BatchEvaluateFeature operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.BatchEvaluateFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/BatchEvaluateFeature"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchEvaluateFeatureResponse> batchEvaluateFeature(
            BatchEvaluateFeatureRequest batchEvaluateFeatureRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchEvaluateFeatureRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchEvaluateFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchEvaluateFeature");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchEvaluateFeatureResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchEvaluateFeatureResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "dataplane.";
            String resolvedHostExpression = "dataplane.";

            CompletableFuture<BatchEvaluateFeatureResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchEvaluateFeatureRequest, BatchEvaluateFeatureResponse>()
                            .withOperationName("BatchEvaluateFeature").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchEvaluateFeatureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(batchEvaluateFeatureRequest));
            CompletableFuture<BatchEvaluateFeatureResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Evidently <i>experiment</i>. Before you create an experiment, you must create the feature to use for
     * the experiment.
     * </p>
     * <p>
     * An experiment helps you make feature design decisions based on evidence and data. An experiment can test as many
     * as five variations at once. Evidently collects experiment data and analyzes it by statistical methods, and
     * provides clear recommendations about which variations perform better.
     * </p>
     * <p>
     * You can optionally specify a <code>segment</code> to have the experiment consider only certain audience types in
     * the experiment, such as using only user sessions from a certain location or who use a certain internet browser.
     * </p>
     * <p>
     * Don't use this operation to update an existing experiment. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateExperiment.html"
     * >UpdateExperiment</a>.
     * </p>
     *
     * @param createExperimentRequest
     * @return A Java Future containing the result of the CreateExperiment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.CreateExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateExperiment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateExperimentResponse> createExperiment(CreateExperimentRequest createExperimentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createExperimentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateExperiment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateExperimentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateExperimentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateExperimentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateExperimentRequest, CreateExperimentResponse>()
                            .withOperationName("CreateExperiment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateExperimentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createExperimentRequest));
            CompletableFuture<CreateExperimentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Evidently <i>feature</i> that you want to launch or test. You can define up to five variations of a
     * feature, and use these variations in your launches and experiments. A feature must be created in a project. For
     * information about creating a project, see <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateProject.html"
     * >CreateProject</a>.
     * </p>
     * <p>
     * Don't use this operation to update an existing feature. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateFeature.html"
     * >UpdateFeature</a>.
     * </p>
     *
     * @param createFeatureRequest
     * @return A Java Future containing the result of the CreateFeature operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.CreateFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFeatureResponse> createFeature(CreateFeatureRequest createFeatureRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createFeatureRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFeature");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFeatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateFeatureResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFeatureResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFeatureRequest, CreateFeatureResponse>()
                            .withOperationName("CreateFeature").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateFeatureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createFeatureRequest));
            CompletableFuture<CreateFeatureResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <i>launch</i> of a given feature. Before you create a launch, you must create the feature to use for
     * the launch.
     * </p>
     * <p>
     * You can use a launch to safely validate new features by serving them to a specified percentage of your users
     * while you roll out the feature. You can monitor the performance of the new feature to help you decide when to
     * ramp up traffic to more users. This helps you reduce risk and identify unintended consequences before you fully
     * launch the feature.
     * </p>
     * <p>
     * Don't use this operation to update an existing launch. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateLaunch.html"
     * >UpdateLaunch</a>.
     * </p>
     *
     * @param createLaunchRequest
     * @return A Java Future containing the result of the CreateLaunch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.CreateLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLaunchResponse> createLaunch(CreateLaunchRequest createLaunchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLaunchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLaunch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateLaunchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLaunchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLaunchRequest, CreateLaunchResponse>()
                            .withOperationName("CreateLaunch").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateLaunchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createLaunchRequest));
            CompletableFuture<CreateLaunchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a project, which is the logical object in Evidently that can contain features, launches, and experiments.
     * Use projects to group similar features together.
     * </p>
     * <p>
     * To update an existing project, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateProject.html"
     * >UpdateProject</a>.
     * </p>
     *
     * @param createProjectRequest
     * @return A Java Future containing the result of the CreateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.CreateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProjectResponse> createProject(CreateProjectRequest createProjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProjectRequest, CreateProjectResponse>()
                            .withOperationName("CreateProject").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createProjectRequest));
            CompletableFuture<CreateProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to define a <i>segment</i> of your audience. A segment is a portion of your audience that
     * share one or more characteristics. Examples could be Chrome browser users, users in Europe, or Firefox browser
     * users in Europe who also fit other criteria that your application collects, such as age.
     * </p>
     * <p>
     * Using a segment in an experiment limits that experiment to evaluate only the users who match the segment
     * criteria. Using one or more segments in a launch allows you to define different traffic splits for the different
     * audience segments.
     * </p>
     * <p>
     * For more information about segment pattern syntax, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Evidently-segments.html#CloudWatch-Evidently-segments-syntax.html"
     * > Segment rule pattern syntax</a>.
     * </p>
     * <p>
     * The pattern that you define for a segment is matched against the value of <code>evaluationContext</code>, which
     * is passed into Evidently in the <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_EvaluateFeature.html"
     * >EvaluateFeature</a> operation, when Evidently assigns a feature variation to a user.
     * </p>
     *
     * @param createSegmentRequest
     * @return A Java Future containing the result of the CreateSegment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.CreateSegment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/CreateSegment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSegmentResponse> createSegment(CreateSegmentRequest createSegmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSegmentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSegmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSegment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSegmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateSegmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSegmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSegmentRequest, CreateSegmentResponse>()
                            .withOperationName("CreateSegment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSegmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSegmentRequest));
            CompletableFuture<CreateSegmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Evidently experiment. The feature used for the experiment is not deleted.
     * </p>
     * <p>
     * To stop an experiment without deleting it, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_StopExperiment.html"
     * >StopExperiment</a>.
     * </p>
     *
     * @param deleteExperimentRequest
     * @return A Java Future containing the result of the DeleteExperiment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceUnavailableException The service was unavailable. Retry the request.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.DeleteExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteExperiment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteExperimentResponse> deleteExperiment(DeleteExperimentRequest deleteExperimentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteExperimentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteExperiment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteExperimentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteExperimentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteExperimentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteExperimentRequest, DeleteExperimentResponse>()
                            .withOperationName("DeleteExperiment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteExperimentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteExperimentRequest));
            CompletableFuture<DeleteExperimentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Evidently feature.
     * </p>
     *
     * @param deleteFeatureRequest
     * @return A Java Future containing the result of the DeleteFeature operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.DeleteFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFeatureResponse> deleteFeature(DeleteFeatureRequest deleteFeatureRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteFeatureRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFeature");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFeatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteFeatureResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFeatureResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFeatureRequest, DeleteFeatureResponse>()
                            .withOperationName("DeleteFeature").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteFeatureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteFeatureRequest));
            CompletableFuture<DeleteFeatureResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Evidently launch. The feature used for the launch is not deleted.
     * </p>
     * <p>
     * To stop a launch without deleting it, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_StopLaunch.html">StopLaunch</a>.
     * </p>
     *
     * @param deleteLaunchRequest
     * @return A Java Future containing the result of the DeleteLaunch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.DeleteLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLaunchResponse> deleteLaunch(DeleteLaunchRequest deleteLaunchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLaunchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLaunch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteLaunchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLaunchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLaunchRequest, DeleteLaunchResponse>()
                            .withOperationName("DeleteLaunch").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteLaunchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteLaunchRequest));
            CompletableFuture<DeleteLaunchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Evidently project. Before you can delete a project, you must delete all the features that the project
     * contains. To delete a feature, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_DeleteFeature.html"
     * >DeleteFeature</a>.
     * </p>
     *
     * @param deleteProjectRequest
     * @return A Java Future containing the result of the DeleteProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.DeleteProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProjectResponse> deleteProject(DeleteProjectRequest deleteProjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProjectRequest, DeleteProjectResponse>()
                            .withOperationName("DeleteProject").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteProjectRequest));
            CompletableFuture<DeleteProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a segment. You can't delete a segment that is being used in a launch or experiment, even if that launch
     * or experiment is not currently running.
     * </p>
     *
     * @param deleteSegmentRequest
     * @return A Java Future containing the result of the DeleteSegment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.DeleteSegment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/DeleteSegment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSegmentResponse> deleteSegment(DeleteSegmentRequest deleteSegmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSegmentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSegmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSegment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSegmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteSegmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSegmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSegmentRequest, DeleteSegmentResponse>()
                            .withOperationName("DeleteSegment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSegmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSegmentRequest));
            CompletableFuture<DeleteSegmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation assigns a feature variation to one given user session. You pass in an <code>entityID</code> that
     * represents the user. Evidently then checks the evaluation rules and assigns the variation.
     * </p>
     * <p>
     * The first rules that are evaluated are the override rules. If the user's <code>entityID</code> matches an
     * override rule, the user is served the variation specified by that rule.
     * </p>
     * <p>
     * If there is a current launch with this feature that uses segment overrides, and if the user session's
     * <code>evaluationContext</code> matches a segment rule defined in a segment override, the configuration in the
     * segment overrides is used. For more information about segments, see <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateSegment.html"
     * >CreateSegment</a> and <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Evidently-segments.html">Use
     * segments to focus your audience</a>.
     * </p>
     * <p>
     * If there is a launch with no segment overrides, the user might be assigned to a variation in the launch. The
     * chance of this depends on the percentage of users that are allocated to that launch. If the user is enrolled in
     * the launch, the variation they are served depends on the allocation of the various feature variations used for
     * the launch.
     * </p>
     * <p>
     * If the user is not assigned to a launch, and there is an ongoing experiment for this feature, the user might be
     * assigned to a variation in the experiment. The chance of this depends on the percentage of users that are
     * allocated to that experiment.
     * </p>
     * <p>
     * If the experiment uses a segment, then only user sessions with <code>evaluationContext</code> values that match
     * the segment rule are used in the experiment.
     * </p>
     * <p>
     * If the user is enrolled in the experiment, the variation they are served depends on the allocation of the various
     * feature variations used for the experiment.
     * </p>
     * <p>
     * If the user is not assigned to a launch or experiment, they are served the default variation.
     * </p>
     *
     * @param evaluateFeatureRequest
     * @return A Java Future containing the result of the EvaluateFeature operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.EvaluateFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/EvaluateFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<EvaluateFeatureResponse> evaluateFeature(EvaluateFeatureRequest evaluateFeatureRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(evaluateFeatureRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, evaluateFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EvaluateFeature");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EvaluateFeatureResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EvaluateFeatureResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "dataplane.";
            String resolvedHostExpression = "dataplane.";

            CompletableFuture<EvaluateFeatureResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EvaluateFeatureRequest, EvaluateFeatureResponse>()
                            .withOperationName("EvaluateFeature").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new EvaluateFeatureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(evaluateFeatureRequest));
            CompletableFuture<EvaluateFeatureResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details about one experiment. You must already know the experiment name. To retrieve a list of
     * experiments in your account, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_ListExperiments.html"
     * >ListExperiments</a>.
     * </p>
     *
     * @param getExperimentRequest
     * @return A Java Future containing the result of the GetExperiment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.GetExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetExperimentResponse> getExperiment(GetExperimentRequest getExperimentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getExperimentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExperiment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetExperimentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetExperimentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetExperimentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetExperimentRequest, GetExperimentResponse>()
                            .withOperationName("GetExperiment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetExperimentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getExperimentRequest));
            CompletableFuture<GetExperimentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the results of a running or completed experiment. No results are available until there have been 100
     * events for each variation and at least 10 minutes have passed since the start of the experiment. To increase the
     * statistical power, Evidently performs an additional offline p-value analysis at the end of the experiment.
     * Offline p-value analysis can detect statistical significance in some cases where the anytime p-values used during
     * the experiment do not find statistical significance.
     * </p>
     * <p>
     * Experiment results are available up to 63 days after the start of the experiment. They are not available after
     * that because of CloudWatch data retention policies.
     * </p>
     *
     * @param getExperimentResultsRequest
     * @return A Java Future containing the result of the GetExperimentResults operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.GetExperimentResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetExperimentResults"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetExperimentResultsResponse> getExperimentResults(
            GetExperimentResultsRequest getExperimentResultsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getExperimentResultsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExperimentResultsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExperimentResults");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetExperimentResultsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetExperimentResultsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetExperimentResultsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetExperimentResultsRequest, GetExperimentResultsResponse>()
                            .withOperationName("GetExperimentResults").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetExperimentResultsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getExperimentResultsRequest));
            CompletableFuture<GetExperimentResultsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details about one feature. You must already know the feature name. To retrieve a list of features in
     * your account, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_ListFeatures.html"
     * >ListFeatures</a>.
     * </p>
     *
     * @param getFeatureRequest
     * @return A Java Future containing the result of the GetFeature operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.GetFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetFeatureResponse> getFeature(GetFeatureRequest getFeatureRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getFeatureRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFeature");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFeatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFeatureResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFeatureResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFeatureRequest, GetFeatureResponse>().withOperationName("GetFeature")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetFeatureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getFeatureRequest));
            CompletableFuture<GetFeatureResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details about one launch. You must already know the launch name. To retrieve a list of launches in
     * your account, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_ListLaunches.html"
     * >ListLaunches</a>.
     * </p>
     *
     * @param getLaunchRequest
     * @return A Java Future containing the result of the GetLaunch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.GetLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLaunchResponse> getLaunch(GetLaunchRequest getLaunchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLaunchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLaunch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLaunchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLaunchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLaunchRequest, GetLaunchResponse>().withOperationName("GetLaunch")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetLaunchRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getLaunchRequest));
            CompletableFuture<GetLaunchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details about one launch. You must already know the project name. To retrieve a list of projects in
     * your account, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_ListProjects.html"
     * >ListProjects</a>.
     * </p>
     *
     * @param getProjectRequest
     * @return A Java Future containing the result of the GetProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.GetProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetProjectResponse> getProject(GetProjectRequest getProjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProjectRequest, GetProjectResponse>().withOperationName("GetProject")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getProjectRequest));
            CompletableFuture<GetProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified segment. Specify the segment you want to view by specifying its ARN.
     * </p>
     *
     * @param getSegmentRequest
     * @return A Java Future containing the result of the GetSegment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.GetSegment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/GetSegment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSegmentResponse> getSegment(GetSegmentRequest getSegmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSegmentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSegmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSegment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSegmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSegmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSegmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSegmentRequest, GetSegmentResponse>().withOperationName("GetSegment")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSegmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSegmentRequest));
            CompletableFuture<GetSegmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration details about all the experiments in the specified project.
     * </p>
     *
     * @param listExperimentsRequest
     * @return A Java Future containing the result of the ListExperiments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListExperiments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListExperiments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListExperimentsResponse> listExperiments(ListExperimentsRequest listExperimentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listExperimentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExperimentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExperiments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListExperimentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListExperimentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListExperimentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListExperimentsRequest, ListExperimentsResponse>()
                            .withOperationName("ListExperiments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListExperimentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listExperimentsRequest));
            CompletableFuture<ListExperimentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration details about all the features in the specified project.
     * </p>
     *
     * @param listFeaturesRequest
     * @return A Java Future containing the result of the ListFeatures operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListFeatures
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListFeatures" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFeaturesResponse> listFeatures(ListFeaturesRequest listFeaturesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFeaturesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFeaturesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFeatures");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFeaturesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFeaturesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFeaturesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFeaturesRequest, ListFeaturesResponse>()
                            .withOperationName("ListFeatures").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListFeaturesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listFeaturesRequest));
            CompletableFuture<ListFeaturesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration details about all the launches in the specified project.
     * </p>
     *
     * @param listLaunchesRequest
     * @return A Java Future containing the result of the ListLaunches operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListLaunches
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListLaunches" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLaunchesResponse> listLaunches(ListLaunchesRequest listLaunchesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLaunchesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLaunchesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLaunches");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLaunchesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLaunchesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLaunchesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLaunchesRequest, ListLaunchesResponse>()
                            .withOperationName("ListLaunches").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListLaunchesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLaunchesRequest));
            CompletableFuture<ListLaunchesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration details about all the projects in the current Region in your account.
     * </p>
     *
     * @param listProjectsRequest
     * @return A Java Future containing the result of the ListProjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListProjectsResponse> listProjects(ListProjectsRequest listProjectsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProjectsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListProjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProjectsRequest, ListProjectsResponse>()
                            .withOperationName("ListProjects").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListProjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listProjectsRequest));
            CompletableFuture<ListProjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to find which experiments or launches are using a specified segment.
     * </p>
     *
     * @param listSegmentReferencesRequest
     * @return A Java Future containing the result of the ListSegmentReferences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListSegmentReferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListSegmentReferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSegmentReferencesResponse> listSegmentReferences(
            ListSegmentReferencesRequest listSegmentReferencesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSegmentReferencesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSegmentReferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSegmentReferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSegmentReferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSegmentReferencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSegmentReferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSegmentReferencesRequest, ListSegmentReferencesResponse>()
                            .withOperationName("ListSegmentReferences").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSegmentReferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSegmentReferencesRequest));
            CompletableFuture<ListSegmentReferencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of audience segments that you have created in your account in this Region.
     * </p>
     *
     * @param listSegmentsRequest
     * @return A Java Future containing the result of the ListSegments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListSegments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListSegments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSegmentsResponse> listSegments(ListSegmentsRequest listSegmentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSegmentsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSegmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSegments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSegmentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSegmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSegmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSegmentsRequest, ListSegmentsResponse>()
                            .withOperationName("ListSegments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSegmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSegmentsRequest));
            CompletableFuture<ListSegmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays the tags associated with an Evidently resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends performance events to Evidently. These events can be used to evaluate a launch or an experiment.
     * </p>
     *
     * @param putProjectEventsRequest
     * @return A Java Future containing the result of the PutProjectEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.PutProjectEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/PutProjectEvents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutProjectEventsResponse> putProjectEvents(PutProjectEventsRequest putProjectEventsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putProjectEventsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putProjectEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutProjectEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutProjectEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutProjectEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "dataplane.";
            String resolvedHostExpression = "dataplane.";

            CompletableFuture<PutProjectEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutProjectEventsRequest, PutProjectEventsResponse>()
                            .withOperationName("PutProjectEvents").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutProjectEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(putProjectEventsRequest));
            CompletableFuture<PutProjectEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an existing experiment. To create an experiment, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateExperiment.html"
     * >CreateExperiment</a>.
     * </p>
     *
     * @param startExperimentRequest
     * @return A Java Future containing the result of the StartExperiment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.StartExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/StartExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartExperimentResponse> startExperiment(StartExperimentRequest startExperimentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startExperimentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartExperiment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartExperimentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartExperimentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartExperimentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartExperimentRequest, StartExperimentResponse>()
                            .withOperationName("StartExperiment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartExperimentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startExperimentRequest));
            CompletableFuture<StartExperimentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an existing launch. To create a launch, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateLaunch.html"
     * >CreateLaunch</a>.
     * </p>
     *
     * @param startLaunchRequest
     * @return A Java Future containing the result of the StartLaunch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.StartLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/StartLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartLaunchResponse> startLaunch(StartLaunchRequest startLaunchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startLaunchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartLaunch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartLaunchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartLaunchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartLaunchRequest, StartLaunchResponse>()
                            .withOperationName("StartLaunch").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartLaunchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startLaunchRequest));
            CompletableFuture<StartLaunchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops an experiment that is currently running. If you stop an experiment, you can't resume it or restart it.
     * </p>
     *
     * @param stopExperimentRequest
     * @return A Java Future containing the result of the StopExperiment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.StopExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/StopExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopExperimentResponse> stopExperiment(StopExperimentRequest stopExperimentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopExperimentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopExperiment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopExperimentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopExperimentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopExperimentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopExperimentRequest, StopExperimentResponse>()
                            .withOperationName("StopExperiment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopExperimentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopExperimentRequest));
            CompletableFuture<StopExperimentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a launch that is currently running. After you stop a launch, you will not be able to resume it or restart
     * it. Also, it will not be evaluated as a rule for traffic allocation, and the traffic that was allocated to the
     * launch will instead be available to the feature's experiment, if there is one. Otherwise, all traffic will be
     * served the default variation after the launch is stopped.
     * </p>
     *
     * @param stopLaunchRequest
     * @return A Java Future containing the result of the StopLaunch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.StopLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/StopLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopLaunchResponse> stopLaunch(StopLaunchRequest stopLaunchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopLaunchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopLaunch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopLaunchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopLaunchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopLaunchRequest, StopLaunchResponse>().withOperationName("StopLaunch")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopLaunchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopLaunchRequest));
            CompletableFuture<StopLaunchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch Evidently resource. Projects, features,
     * launches, and experiments can be tagged.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key for the resource, this tag is appended to the list of tags associated with the alarm. If you specify a tag
     * key that is already associated with the resource, the new tag value that you specify replaces the previous value
     * for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon
     * Web Services resources</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to test a rules pattern that you plan to use to create an audience segment. For more
     * information about segments, see <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateSegment.html"
     * >CreateSegment</a>.
     * </p>
     *
     * @param testSegmentPatternRequest
     * @return A Java Future containing the result of the TestSegmentPattern operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.TestSegmentPattern
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/TestSegmentPattern" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TestSegmentPatternResponse> testSegmentPattern(TestSegmentPatternRequest testSegmentPatternRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(testSegmentPatternRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, testSegmentPatternRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TestSegmentPattern");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TestSegmentPatternResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TestSegmentPatternResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TestSegmentPatternResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TestSegmentPatternRequest, TestSegmentPatternResponse>()
                            .withOperationName("TestSegmentPattern").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TestSegmentPatternRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(testSegmentPatternRequest));
            CompletableFuture<TestSegmentPatternResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an Evidently experiment.
     * </p>
     * <p>
     * Don't use this operation to update an experiment's tag. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateExperimentRequest
     * @return A Java Future containing the result of the UpdateExperiment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.UpdateExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateExperiment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateExperimentResponse> updateExperiment(UpdateExperimentRequest updateExperimentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateExperimentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateExperiment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateExperimentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateExperimentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateExperimentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateExperimentRequest, UpdateExperimentResponse>()
                            .withOperationName("UpdateExperiment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateExperimentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateExperimentRequest));
            CompletableFuture<UpdateExperimentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing feature.
     * </p>
     * <p>
     * You can't use this operation to update the tags of an existing feature. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateFeatureRequest
     * @return A Java Future containing the result of the UpdateFeature operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.UpdateFeature
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateFeature" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFeatureResponse> updateFeature(UpdateFeatureRequest updateFeatureRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateFeatureRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFeatureRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFeature");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFeatureResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateFeatureResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFeatureResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFeatureRequest, UpdateFeatureResponse>()
                            .withOperationName("UpdateFeature").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateFeatureRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateFeatureRequest));
            CompletableFuture<UpdateFeatureResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a launch of a given feature.
     * </p>
     * <p>
     * Don't use this operation to update the tags of an existing launch. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateLaunchRequest
     * @return A Java Future containing the result of the UpdateLaunch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.UpdateLaunch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateLaunch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLaunchResponse> updateLaunch(UpdateLaunchRequest updateLaunchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateLaunchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLaunchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLaunch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLaunchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateLaunchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLaunchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLaunchRequest, UpdateLaunchResponse>()
                            .withOperationName("UpdateLaunch").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateLaunchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateLaunchRequest));
            CompletableFuture<UpdateLaunchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the description of an existing project.
     * </p>
     * <p>
     * To create a new project, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_CreateProject.html"
     * >CreateProject</a>.
     * </p>
     * <p>
     * Don't use this operation to update the data storage options of a project. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_UpdateProjectDataDelivery.html"
     * >UpdateProjectDataDelivery</a>.
     * </p>
     * <p>
     * Don't use this operation to update the tags of a project. Instead, use <a
     * href="https://docs.aws.amazon.com/cloudwatchevidently/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     *
     * @param updateProjectRequest
     * @return A Java Future containing the result of the UpdateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.UpdateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProjectResponse> updateProject(UpdateProjectRequest updateProjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProjectRequest, UpdateProjectResponse>()
                            .withOperationName("UpdateProject").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateProjectRequest));
            CompletableFuture<UpdateProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the data storage options for this project. If you store evaluation events, you an keep them and analyze
     * them on your own. If you choose not to store evaluation events, Evidently deletes them after using them to
     * produce metrics and other experiment results that you can view.
     * </p>
     * <p>
     * You can't specify both <code>cloudWatchLogs</code> and <code>s3Destination</code> in the same operation.
     * </p>
     *
     * @param updateProjectDataDeliveryRequest
     * @return A Java Future containing the result of the UpdateProjectDataDelivery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EvidentlyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EvidentlyAsyncClient.UpdateProjectDataDelivery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/evidently-2021-02-01/UpdateProjectDataDelivery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProjectDataDeliveryResponse> updateProjectDataDelivery(
            UpdateProjectDataDeliveryRequest updateProjectDataDeliveryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateProjectDataDeliveryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProjectDataDeliveryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Evidently");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProjectDataDelivery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProjectDataDeliveryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateProjectDataDeliveryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProjectDataDeliveryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProjectDataDeliveryRequest, UpdateProjectDataDeliveryResponse>()
                            .withOperationName("UpdateProjectDataDelivery").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateProjectDataDeliveryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateProjectDataDeliveryRequest));
            CompletableFuture<UpdateProjectDataDeliveryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final EvidentlyServiceClientConfiguration serviceClientConfiguration() {
        return new EvidentlyServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EvidentlyException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        EvidentlyServiceClientConfigurationBuilder serviceConfigBuilder = new EvidentlyServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
